/*
 * $Log: histo.c,v $
 * Revision 1.9  1992/12/10  03:48:05  moore
 * nuke unused vars.
 *
 * Revision 1.8  1992/12/09  05:01:55  moore
 * add a cast in a call to XtAddEventHandler; add a missing arg to a call
 * to hist_SetHairLine.
 *
 * Revision 1.7  1992/09/18  01:44:55  moore
 * add support for hairline
 * fix bugs in elapsed_time()
 * change computation of minRightEdge and maxRightEdge
 * optimize draw_bar
 * clear old window when resizing.  eventually we should
 *     redraw only as much as necessary
 *
 * Revision 1.6  1992/08/24  22:07:09  moore
 * labels now scroll horizontally along with the bargraph data
 *
 * Revision 1.5  1992/08/05  22:35:54  moore
 * remove #ifdefs that are no longer needed
 *
 * Revision 1.4  1992/08/05  21:49:49  moore
 * - deal with time in units of msec rather than seconds (requires new
 *   tracefile code)
 * - change scales to go down to 100 msec/div
 * - fix monochrome pixmaps so that vertical tics through them look right.
 * - fix edges on "white" bar in monochrome mode so that they coincide with
 *   the edges of bars drawn with pixmaps.
 * - change interface to draw_bar.  add new routines before(), after(),
 *   elapsed_time() to do computations with new time struct.
 *
 * Revision 1.3  1992/07/23  20:52:16  moore
 * on moves in monochrome mode, make sure moves are in increments of
 * the least common multiple of the width or height of the stipple
 * patterns used to draw the bars.
 *
 * Revision 1.2  1992/07/22  18:48:48  moore
 * fix some histogram drawing bugs:
 * - draw a bar even when event time == 0 (from getState)
 * - don't draw bars after end of time
 *
 * Revision 1.1  1992/07/10  21:56:22  moore
 * Initial revision
 *
 */

#define SCROLLING_LABELS
#define BIGHAIR			/* do hairline by calling graph_redraw */

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Scrollbar.h>
#include "comn.h"
#include "histo.h"

#include <X11/bitmaps/gray>
#include <X11/bitmaps/light_gray>
#include <X11/bitmaps/hlines3>
#include "xbm/up.xbm"
#include "xbm/down.xbm"

#define max(a,b) ((a) > (b) ? (a) : (b))
#define min(a,b) ((a) < (b) ? (a) : (b))

static char *fallbacks[] = {
    "*font:fixed",
    "*reverseVideo:off",
    "*hostList.background: slategrey",
    "*hostList.foreground: cyan",
    "*hostGraph.background: white",
    "*buttonBox.background: red",
    0,
};

/************************************************************************/

static Dimension currentWidth = 600;	/* width of popup window */
static Dimension currentHeight = 400;	/* height of popup window */
static Dimension scrollBarSize = 20;	/* thickness of a scrollbar */
static Dimension listWidth = 100;	/* width of the host list widget */
static Dimension borderWidth = 1; /* width of interior lines in popup */
static int rowSpacing = 26;	/* distance between bars */
static int barThickness = 20;	/* thickness of bars */
static long pixelsPerTic = 25;
#ifdef SCROLLING_LABELS
static int ticsPerTicMark = 5;
#endif
static int horizMoveChunk = 1;
static int vertMoveChunk = 1;

static struct foo {
    int width;
    int height;
    int scrollBarSize;
    int listWidth;
    int borderWidth;
    int rowSpacing;
    int barThickness;
    int pixelsPerTic;
    Pixel errorColor;
    Pixel runningColor;
    Pixel idleColor;
    Pixel deadColor;
    Pixel ticColor;
#ifdef HAIRLINE
    Pixel hairLineColor;
    int hairLineWidth;
#endif
    Dimension ticWidth;
    Font listFont;
/*
 * XXX this needs to be specific to hostList.
 */
    Pixel foreground;		
} appDefaults;

#define offset(field) XtOffsetOf (struct foo, field)

static XtResource resourceSpecs [] = {
    { XtNwidth, XtCWidth, XtRInt, sizeof(int),
	  offset(width), XtRImmediate, (XtPointer) 600 },
    { XtNheight, XtCHeight, XtRInt, sizeof(int),
	  offset(height), XtRImmediate, (XtPointer) 400 },
    { "scrollBarSize", "ScrollBarSize", XtRInt, sizeof (int),
	  offset(scrollBarSize), XtRImmediate, (XtPointer) 20 },
    { "hostListWidth", "HostListWidth", XtRInt, sizeof (int),
	  offset(listWidth), XtRImmediate, (XtPointer) 125 },
    { "borderWidth", "BorderWidth", XtRInt, sizeof (int),
	  offset(borderWidth), XtRImmediate, (XtPointer) 1},
    { "rowSpacing", "RowSpacing", XtRInt, sizeof (int),
	  offset(rowSpacing), XtRImmediate, (XtPointer) 26 },
    { "barThickness", "BarThickness", XtRInt, sizeof (int),
	  offset(barThickness), XtRImmediate, (XtPointer) 20 },
    { "pixelsPerTic", "PixelsPerTic", XtRInt, sizeof (int),
	  offset(pixelsPerTic), XtRImmediate, (XtPointer) 25 },
    { "errorColor", "BarColor", XtRPixel, sizeof(Pixel),
	  offset(errorColor), XtRString, "red" },
    { "runningColor", "BarColor", XtRPixel, sizeof(Pixel),
	  offset(runningColor), XtRString, "green" },
    { "idleColor", "BarColor", XtRPixel, sizeof(Pixel),
	  offset(idleColor), XtRString, "yellow" },
    { "deadColor", "BarColor", XtRPixel, sizeof(Pixel),
	  offset(deadColor), XtRString, "gray" },
    { "ticColor", "TicColor", XtRPixel, sizeof(Pixel),
	  offset(ticColor), XtRString, "black" },
#ifdef HAIRLINE
    { "hairLineColor", "HairLineColor", XtRPixel, sizeof(Pixel),
	  offset(hairLineColor), XtRString, "red" },
    { "hairLineWidth", "HairLineWidth", XtRInt, sizeof (int),
	  offset(hairLineWidth), XtRImmediate, (XtPointer) 2 },
#endif
    { "ticWidth", "TicWidth", XtRDimension, sizeof(Dimension),
	  offset(ticWidth), XtRImmediate, (XtPointer) 1 },
    { XtNfont, XtCFont, XtRFont, sizeof(Font),
	  offset(listFont), XtRString, XtDefaultFont },
    { XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	  offset(foreground), XtRString, XtDefaultForeground },
};

#undef offset

static void
initialize_defaults (parent)
Widget parent;
{
    XtGetApplicationResources (parent,
			       &appDefaults,
			       resourceSpecs,
			       XtNumber (resourceSpecs),
			       NULL, 0);

    currentWidth = appDefaults.width;
    currentHeight = appDefaults.height;
    scrollBarSize = appDefaults.scrollBarSize;
    listWidth = appDefaults.listWidth;
    borderWidth = appDefaults.borderWidth;
    rowSpacing = appDefaults.rowSpacing;
    barThickness = appDefaults.barThickness;
    pixelsPerTic = appDefaults.pixelsPerTic;
}


static Widget outerForm = 0; 	/* container form widget */
static Widget leftBar;		/* "host" scrollbar on left  */
static Widget hostList;		/* window containing list of hosts */
static Widget hostGraph;	/* window containing bar graphs */
static Widget buttonBox;	/* container for scale buttons */
static Widget upButton;		/* up scale button */
static Widget downButton;	/* down scale button */
static Widget scaleLabel;	/* "scale" label button */
static Widget bottomBar;	/* horiz scrollbar on bottom */
#ifdef SCROLLING_LABELS
static Widget labelBox;		/* container for labels */
#else
static Widget leftLegend;	/* label with time at left side */
static Widget centerLegend;	/* label with #sec / tic */
static Widget rightLegend;	/* label with time at right side */
#endif
static Pixmap up_pixmap;	/* pixmap for up button */
static Pixmap down_pixmap;	/* pixmap for down button */
static GC errorGC, runningGC, idleGC, deadGC, ticGC;
#ifdef HAIRLINE
static GC hairLineGC, hairLineClearGC;
#endif
static Dimension graphWindowHeight = 0;
				/* height of graph subwindow */
static Dimension graphWindowWidth = 0;
				/* width of graph subwindow */
static int graphHeight = 0;	/* numHosts * rowSpacing */
static struct EventTime startTime = { 0, 0 };
				/* abs start time of trace (secs) */
static struct EventTime endTime = { 5000, 0 };
				/* time of last trace event (secs) */
#ifdef HAIRLINE
static struct EventTime hairLine = { 0, 0 };
#endif
static long rightEdge = 0;	/* time of right edge (in *PIXELS*) */
static int topEdge = 0;		/*
				 * x coordinate of top edge, relative
				 * to the top of the entire graph
				 */
static long msecPerTic = 1000;
static struct Event *savedState;
static char **hostNames;
static int numHosts = 0;

#if 0
static Pixmap hairLineSaveUnder = 0;

static int
window_depth (w)
Widget w;
{
    XWindowAttributes foo;
    XGetWindowAttributes (XtDisplay (w), XtWindow(w), &foo);
    return foo.depth;
}
#endif


#ifdef SCROLLING_LABELS
/*
 * draw a label for the indicated pixel location, centered on pixel.
 */

static void
draw_label (pixel)
long pixel;
{
    char buf[30];
    char *ptr;
    int msec = ((pixel * msecPerTic) / pixelsPerTic) % 1000;
    long sec = ((pixel * msecPerTic) / pixelsPerTic) / 1000;
    char *sign = "";
    int width;
    static GC textGC = 0;
    static int y = 0;
    static int initialized = 0;
    static XFontStruct *font_info;
    int direction, ascent, descent;
    XCharStruct stringSize;

    if (initialized == 0) {
	XGCValues values;
	int font_height;

	XtVaGetValues (labelBox,
		       XtNbackground, &(values.background),
		       NULL);
	/*
	 * XXX we really should have separate resources for the label
	 * font than for the list font.  likewise for the color of the
	 * label text.
	 */
	values.font = appDefaults.listFont;
	values.foreground = appDefaults.foreground;
	textGC = XtGetGC (labelBox,
			  (unsigned) GCForeground | GCBackground | GCFont,
			  &values);
	font_info = XQueryFont (XtDisplay (labelBox), values.font);
	font_height = font_info->ascent + font_info->descent;
	y = ((scrollBarSize - font_height) / 2) + font_info->ascent;
	initialized = 1;
    }

    /* don't draw a label if time < 0 */
    if (sec < 0L || (sec == 0L && msec < 0))		
	return;

    /*
     * generate the characters of the time in reverse order.
     * this is so we can express it in a reasonably short form --
     * hours and fractional parts of seconds are not displayed
     * unless they are nonzero.
     */

    ptr = buf + sizeof(buf);
    *--ptr = '\0';		/* trailing NUL */

    if (msec > 0) {		/* fractional part of seconds (up to 999) */
	while (msec % 10 == 0)
	    msec /= 10;
	while (msec > 0) {
	    *--ptr = msec % 10 + '0';
	    msec /= 10;
	}
	*--ptr = '.';		/* decimal point */
    }
    *--ptr = sec % 10 + '0';	/* seconds */
    sec /= 10;
    *--ptr = sec % 6 + '0';	/* tens of seconds */
    sec /= 6;
    *--ptr = ':';		/* colon */
    *--ptr = sec % 10 + '0';	/* minutes */
    sec /= 10;
    *--ptr = sec % 6 + '0';	/* tens of minutes */
    sec /= 6;
    if (sec > 0) {
	*--ptr = ':';		/* colon */
	while (sec > 0) {
	    *--ptr = sec % 10 + '0'; /* hours */
	    sec /= 10;
	}
    }
    XTextExtents (font_info, ptr, strlen(ptr), &direction, &ascent,
		  &descent, &stringSize);
    width = stringSize.rbearing - stringSize.lbearing;
    XDrawString (XtDisplay (labelBox), XtWindow (labelBox), textGC,
		 pixel - (width / 2) + graphWindowWidth - rightEdge, y,
		 ptr, strlen (ptr));
}

static void
update_labels ()
{
    long pixel;
    long pixelsPerTicMark = pixelsPerTic * ticsPerTicMark;

    /*
     * for now, always clear and redraw window.
     * XXX eventually do moves when we can
     */
    XClearArea (XtDisplay (labelBox), XtWindow (labelBox),
		0, 0, 0, 0, False);
    /*
     * start drawing labels one tic past right edge of window, because
     * the label for that tic might protrude into the window.  Finish
     * drawing labels one tic to the left of the window.  One tic should
     * be enough unless the labels get so wide that they overlap --
     * which we should probably prevent, anyway.
     */
    pixel = rightEdge - (rightEdge % pixelsPerTicMark) + pixelsPerTicMark;
    do {
	draw_label (pixel);
	pixel -= pixelsPerTicMark;
    } while ((pixel + pixelsPerTicMark + graphWindowWidth) > rightEdge);
}

static void
label_expose ()
{
    update_labels ();
}
#else
/*
 * set either the left or the right time legend
 */

static void
set_time_label (w, t)
Widget w;
long t;
{
    int msec = ((t * msecPerTic) / pixelsPerTic) % 1000;
    long sec = ((t * msecPerTic) / pixelsPerTic) / 1000;
    char foo[20];
    char *sign = "";

    if (t < 0L) {
	t = -t;
	sign = "-";
    }
    if (msecPerTic >= 1000) {
	sprintf (foo, "%s%02d:%02d:%02d", sign,
		 sec / (60 * 60),	/* hours */
		 (sec / 60) % 60,	/* mins */
		 sec % 60		/* secs */
		 );
    }
    else {
	sprintf (foo, "%s%02d:%02d:%02d.%03d", sign,
		 sec / (60 * 60),	/* hours */
		 (sec / 60) % 60,	/* mins */
		 sec % 60,		/* secs */
		 msec			/* msec */
		 );
    }
    XtVaSetValues (w, XtNlabel, foo, NULL);
}
#endif


static struct EventTime *
elapsed_time (t1, t2)
struct EventTime *t1, *t2;
{
    static struct EventTime result;

    result.secs = 0;
    result.millisecs = t2->millisecs - t1->millisecs;
    if (result.millisecs < 0) {
		result.millisecs += 1000;
		result.secs = -1;
    }
    result.secs = result.secs + t2->secs - t1->secs;
    return &result;
}

static int
time_to_pixels (t)
struct EventTime *t;
{
    return (((t->secs * 1000) + t->millisecs) * pixelsPerTic) /
	msecPerTic;
}

static void
time_from_x (t, x, msec)
struct EventTime *t;
int x;
int msec;
{
    msec += ((rightEdge - (int) graphWindowWidth + x) * msecPerTic) /
	pixelsPerTic;
    t->secs = (msec / 1000) + startTime.secs;
    t->millisecs = (msec % 1000) + startTime.millisecs;
    if (t->millisecs > 1000) {
	t->secs++;
	t->millisecs -= 1000;
    }
    if (t->millisecs < 0) {
	t->secs--;
	t->millisecs += 1000;
    }
}

static int
before (t1, t2)
struct EventTime *t1, *t2;
{
    if (t1->secs < t2->secs)
	return 1;
    else if (t1->secs > t2->secs)
	return 0;
    else
	return (t1->millisecs < t2->millisecs);
}

static int 
after (t1, t2)
struct EventTime *t1, *t2;
{
    if (t1->secs > t2->secs)
	return 1;
    else if (t1->secs < t2->secs)
	return 0;
    else
	return (t1->millisecs > t2->millisecs);
}

/*
 * make sure at least part of the graph is within the visible
 * portion of the window.
 * called whenever the displayed portion of the graph has changed.
 * (resize, change of scale, move scrollbar)
 */

static void
check_horiz_bounds (setThumb)
int setThumb;
{
    long minRightEdge, maxRightEdge;
    float topOfThumb, thumbWidth;
    long leftEdge;


#ifdef SCROLLING_LABELS
    minRightEdge = graphWindowWidth - pixelsPerTic;
    maxRightEdge = time_to_pixels (elapsed_time (&startTime, &endTime))
	+ (graphWindowWidth - pixelsPerTic);
    if (maxRightEdge < minRightEdge)
	maxRightEdge = minRightEdge;
#else
    minRightEdge = graphWindowWidth;
    maxRightEdge = time_to_pixels (elapsed_time (&startTime, &endTime))
	- pixelsPerTic + (int) graphWindowWidth;
#endif

    if (rightEdge > maxRightEdge)
	rightEdge = maxRightEdge;
    else if (rightEdge < minRightEdge)
	rightEdge = minRightEdge;

    if (setThumb == True) {
	thumbWidth = ((float) graphWindowWidth) /
	    ((float) time_to_pixels (elapsed_time (&startTime, &endTime)));
	if (thumbWidth > 1.0)
	    thumbWidth = 1.0;

	topOfThumb = ((float) (rightEdge - minRightEdge)) /
	    ((float) (maxRightEdge - minRightEdge));

	XawScrollbarSetThumb (bottomBar, topOfThumb, thumbWidth);
    }

    leftEdge = rightEdge - (int) graphWindowWidth;

#ifdef SCROLLING_LABELS
    update_labels ();
#else
    set_time_label (rightLegend, rightEdge);
    set_time_label (leftLegend, leftEdge);
#endif
}


static void
clear_area (w, x, y, wd, ht)
Widget w;
int x, y, wd, ht;
{
    if (wd > 0 && ht > 0)
	XClearArea (XtDisplay (w), XtWindow(w),
		    x, y, wd, ht, False);
}

static void
check_vert_bounds (setThumb)
int setThumb;
{
    int maxTopEdge;

    maxTopEdge = graphHeight - graphWindowHeight;
    if (maxTopEdge < 0)
	maxTopEdge = 0;

    if (topEdge < 0)
	topEdge = 0;
    else if (topEdge > maxTopEdge)
	topEdge = maxTopEdge;

    if (setThumb == True) {
	float shown;
	shown = (float) graphWindowHeight / (float) graphHeight;
	if (shown > 1.0)
	    shown = 1.0;
	XawScrollbarSetThumb (leftBar,
			      (float) topEdge / (float) maxTopEdge,
			      shown);
    }
}

static void graph_redraw ();

static void
move_graph_window (newTopEdge, newRightEdge)
int newTopEdge;
int newRightEdge;
{
    int oldTopEdge = topEdge;
    int oldRightEdge = rightEdge;
    int copyWidth;
    int copyHeight;
    int srcX, srcY, dstX, dstY;
    int clrX, clrY;
    int i;

    static GC copyGC = 0;

    /* if we don't have a GC yet, allocate one */

    if (copyGC == 0) {
	XGCValues values;
	values.graphics_exposures = True;
	copyGC = XtGetGC (hostGraph, GCGraphicsExposures, &values);
    }

    /*
     * Make sure new coordinates are in bounds before copying anything.
     * Also make sure src and destination are "chunk" pixels apart.
     * If not, align move on chunk boundaries to make sure stipple patterns
     * mesh.
     */

    if (oldRightEdge != newRightEdge) {
	rightEdge = newRightEdge;
	check_horiz_bounds (True);
	/*
	 * make sure src and destination are "chunk" pixels apart.
	 */
	for (i = 0; i < horizMoveChunk; ++i) {
	    if (abs((rightEdge + i) - oldRightEdge) % horizMoveChunk == 0){
		rightEdge += i;
		break;
	    }
	    if (abs((rightEdge - i) - oldRightEdge) % horizMoveChunk == 0){
		rightEdge -= i;
		break;
	    }
	}
	newRightEdge = rightEdge;
    }
    if (oldTopEdge != newTopEdge) {
	topEdge = newTopEdge;
	check_vert_bounds (True);
	/*
	 * make sure src and destination are "chunk" pixels apart.
	 */
	for (i = 0; i < vertMoveChunk; ++i) {
	    if (abs((topEdge + i) - oldTopEdge) % vertMoveChunk == 0) {
		topEdge += i;
		break;
	    }
	    if (abs((topEdge - i) - oldTopEdge) % vertMoveChunk == 0) {
		topEdge -= i;
		break;
	    }
	}
	newTopEdge = topEdge;
    }

    /* if we're not moving anywhere, don't bother redrawing */

    if (newTopEdge == oldTopEdge && newRightEdge == oldRightEdge)
	return;

    /*
     * if we're moving so far that there's no overlap, just expose the
     * entire area and be done with it.
     */

    if (abs (newRightEdge - oldRightEdge) > (int) graphWindowWidth ||
	abs (newTopEdge - oldTopEdge) > (int) graphWindowHeight) {
	XClearArea (XtDisplay (hostGraph), XtWindow (hostGraph),
		    0, 0, 0, 0, True);
	return;
    }

    copyWidth = (int) graphWindowWidth - abs (newRightEdge - oldRightEdge);
    copyHeight = (int) graphWindowHeight - abs (newTopEdge - oldTopEdge);

    /*
     * figure source and destination, and do the copy
     */

    if (newRightEdge > oldRightEdge) {
	/* right - copy to left */
	srcX = newRightEdge - oldRightEdge;
	dstX = 0;
	clrX = copyWidth;
    }
    else {
	/* left - copy to right */
	srcX = 0;
	dstX = oldRightEdge - newRightEdge;
	clrX = 0;
    }
    if (newTopEdge > oldTopEdge) {
	/* down - copy up */
	srcY = newTopEdge - oldTopEdge;
	dstY = 0;
	clrY = copyHeight;
    }
    else {
	/* up - copy down */
	srcY = 0;
	dstY = oldTopEdge - newTopEdge;
	clrY = 0;
    }
    XCopyArea (XtDisplay (hostGraph),
	       XtWindow (hostGraph), XtWindow (hostGraph), copyGC,
	       srcX, srcY, copyWidth, copyHeight, dstX, dstY);

    /*
     * clear the area either to the left or right of where the copied
     * region ended up.
     */

    clear_area (hostGraph, clrX, 0, graphWindowWidth - copyWidth,
		graphWindowHeight);
    graph_redraw (hostGraph, clrX, 0,
		  clrX + (int) graphWindowWidth - copyWidth,
		  0 + graphWindowHeight, NULL);
    /*
     * clear the area above or below the copied region.
     */

    clear_area (hostGraph, 0, clrY, copyWidth,
		graphWindowHeight - copyHeight);
    graph_redraw (hostGraph, 0, clrY, 0 + copyWidth,
		  clrY + graphWindowHeight - copyHeight, NULL);
}


/*
 * change the graph scale.
 * called when either the up or down scale button is pressed.
 */

static
change_scale (w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    static long scales[] = {
	100, 200, 500,
	1000, 2000, 5000,
	10000, 20000, 50000,
	100000, 200000, 500000,
	1000000, 2000000, 5000000,
	10000000, 20000000, 50000000,
	100000000
    };
    long oldScale = msecPerTic;
    int i;

    /*
     * find index of current scale
     */
    for (i = 0; i < sizeof (scales) / sizeof (*scales); ++i)
	if (msecPerTic == scales[i])
	    break;
    if (i == (sizeof (scales) / sizeof (*scales))) /* sanity */
	i = 0;

    /*
     * change scale according to button pushed
     */
    if (w == upButton) {
	if (++i >= (sizeof (scales) / sizeof (*scales)))
	    i = (sizeof (scales) / sizeof (*scales)) - 1;
    }
    else if (w == downButton) {
	if (--i < 0)
	    i = 0;
    }
    msecPerTic = scales[i];

    if (oldScale != msecPerTic) {
	char foo[40];
#ifdef SCROLLING_LABELS
	update_labels ();
#endif
	/*
	 * change time scale legend
	 */
	if (msecPerTic >= 1000)
	    sprintf (foo, "%d sec\n", msecPerTic / 1000);
	else
	    sprintf (foo, "%d msec\n", msecPerTic);
#ifdef SCROLLING_LABELS
	XtVaSetValues (scaleLabel,
		       XtNlabel, foo,
		       XtNjustify, XtJustifyCenter,
		       NULL);
#else
	XtVaSetValues (centerLegend,
		       XtNlabel, foo,
		       XtNjustify, XtJustifyCenter,
		       NULL);
#endif
	check_horiz_bounds (True);

	/*
	 * force redraw of hostGraph window
	 */
	XClearArea (XtDisplay (hostGraph), XtWindow (hostGraph),
		    0, 0, 0, 0, True);
    }
}

/*
 * callback for the "jump" proc of the vertical (left) scrollbar.
 * This is what happens when you click on the scrollbar with the middle
 * button.
 */

static void
vert_jump (w, client_data, perc)
Widget w;
XtPointer client_data;
XtPointer perc;			/* float */
{
    float percent = *((float *) perc);
    int maxTopEdge = graphHeight - graphWindowHeight;


    move_graph_window ((int) (percent * maxTopEdge), rightEdge);
    XClearArea (XtDisplay (hostList), XtWindow (hostList),
		0, 0, 0, 0, True);
}

/*
 * callback for the "scroll" proc of the vertical/left scrollbar.
 * this is what happens when you click with left or right button.
 */

static void
vert_scroll (w, client_data, position)
Widget w;
XtPointer client_data;
XtPointer position;		/* int */
{
    int pos = (int) position;
    int maxTopEdge = graphHeight - graphWindowHeight;

    move_graph_window (topEdge + pos, rightEdge);
    XClearArea (XtDisplay (hostList), XtWindow (hostList),
		0, 0, 0, 0, True);
}


/*
 * callback for the "jump" proc of the horizontal (right) scrollbar.
 * This is what happens when you click on the scrollbar with the middle
 * button.
 */

static
horiz_jump (w, client_data, perc)
Widget w;
XtPointer client_data;
XtPointer perc;			/* float */
{
    float percent = *((float *) perc);
    long minRightEdge;
    long maxRightEdge;

    /*
     * Grrr.  Figure out reasonable bounds on "percent".
     *
     * When percent == 0, set bars so that startTime is exactly
     * at the left of the display, like this:
     *
     * +------------------------------------------+
     * |                                          |
     * +------------------------------------------|
     * |                                          |
     * +------------------------------------------|
     * |                                          |
     * +------------------------------------------+
     * ^ startTime                         
     * +==========================================> graphWindowWidth
     *                                            ^ rightEdge
     *
     * This implies that rightEdge is the same as
     * the visible width of the graph.
     *
     * When percent == 1, set bars so that endTime is one tic
     * right of the left edge of the display, like this:
     *
     *                  +---------------------------+
     *                  |                           |
     * +----------------|-----+                     |
     * |                |     |                     |
     * +----------------|-----+                     |
     *                  |                           |
     *                  +---------------------------+
     * ^ startTime    endTime ^
     * +======================> (endTime - startTime) * pixels/sec
     *                  <====== pixelsPerTic
     *                  +===========================> graphWindowWidth
     *                                              ^ rightEdge
     */

#ifdef SCROLLING_LABELS
    minRightEdge = graphWindowWidth - pixelsPerTic;
    maxRightEdge = time_to_pixels (elapsed_time (&startTime, &endTime)) 
	+ (graphWindowWidth - pixelsPerTic);
    if (maxRightEdge < minRightEdge)
	maxRightEdge = minRightEdge;
#else
    minRightEdge = graphWindowWidth;
    maxRightEdge = time_to_pixels (elapsed_time (&startTime, &endTime)) -
	pixelsPerTic + graphWindowWidth;
#endif

    /*
     * sanity clause ("ain't no such thing as sanity clause" - Groucho)
     */

    if (minRightEdge > maxRightEdge)
	minRightEdge = maxRightEdge;

    move_graph_window (topEdge,
		       (int) (percent * (float)(maxRightEdge - minRightEdge))
		       + minRightEdge);
}


/*
 * callback for the "scroll" proc of the horizontal/right scrollbar.
 * this is what happens when you click with left or right button.
 */

static void
horiz_scroll (w, client_data, position)
Widget w;
XtPointer client_data;
XtPointer position;		/* int */
{
    int pos = (int) position;

    move_graph_window (topEdge, rightEdge + pos);
}


/*
 * translate an event time into an x-coordinate
 */

static int
xpos (t)
struct EventTime *t;
{
    int leftEdge = rightEdge - (int) graphWindowWidth;

    if (t == 0 || (t->secs == 0 && t->millisecs == 0))
	*t = startTime;
    return time_to_pixels (elapsed_time (&startTime, t)) - leftEdge;
}
/*
 * translate a host number into an y-coordinate for drawing bars
 * (drawing text requires a slightly different fudge factor)
 */

static int
ypos (h)
int h;
{
    return (h * rowSpacing) + (rowSpacing / 2) - topEdge;
}

static int
gcd (x, y)
{
    if (y == 0)
	return x;
    else
	return gcd (y, x % y);    
}


static int
lcm (x, y)
{
    return ( x * y ) / gcd (x, y);
}


static void
initialize_graph ()
{
    XSetWindowAttributes xswa;
    XGCValues values;

    savedState = (struct Event *)
	malloc (numHosts * sizeof (struct Event));

    /*
     * now that widget is realized, fill in resources
     */

    /*
     * set up GCs
     */

    if (inColor) {
	values.line_width = barThickness;
	values.foreground = appDefaults.errorColor;
	errorGC = XtGetGC (hostGraph,
			   GCForeground | GCLineWidth, &values);
	values.foreground = appDefaults.runningColor;
	runningGC = XtGetGC (hostGraph,
			     GCForeground | GCLineWidth, &values);
	values.foreground = appDefaults.idleColor;
	idleGC = XtGetGC (hostGraph,
			  GCForeground | GCLineWidth, &values);
	values.foreground = appDefaults.deadColor;
	deadGC = XtGetGC (hostGraph,
			  GCForeground | GCLineWidth, &values);
	values.line_width = 1;
	values.foreground = appDefaults.ticColor;
	ticGC = XtGetGC (hostGraph,
			 GCForeground | GCLineWidth, &values);
#ifdef HAIRLINE
	values.foreground = appDefaults.hairLineColor;
	values.line_width = appDefaults.hairLineWidth;
	hairLineGC = XtGetGC (hostGraph,
			      GCForeground | GCLineWidth, &values);
	XtVaGetValues (hostGraph,
		       XtNbackground, &(values.foreground),
		       NULL);
	hairLineClearGC = XtGetGC (hostGraph,
				   GCForeground | GCLineWidth,
				   &values);
#endif

	horizMoveChunk = vertMoveChunk = 1;
    }
    else {
	int whitePixel = WhitePixel (XtDisplay (hostGraph),
				     DefaultScreen (XtDisplay (hostGraph)));
	int blackPixel = BlackPixel (XtDisplay (hostGraph),
				     DefaultScreen (XtDisplay (hostGraph)));
	int depth = DefaultDepth (XtDisplay (hostGraph),
				  DefaultScreen (XtDisplay (hostGraph)));

	Pixmap grayPixmap = 
	    XCreatePixmapFromBitmapData (XtDisplay (hostGraph),
					 XtWindow (hostGraph),
					 gray_bits,
					 gray_width, gray_height,
					 blackPixel, whitePixel, depth);

	Pixmap lightGrayPixmap =
	    XCreatePixmapFromBitmapData (XtDisplay (hostGraph),
					 XtWindow(hostGraph),
					 light_gray_bits,
					 light_gray_width, light_gray_height,
					 blackPixel, whitePixel, depth);

	Pixmap hlinesPixmap =
	    XCreatePixmapFromBitmapData (XtDisplay (hostGraph),
					 XtWindow(hostGraph),
					 hlines3_bits,
					 hlines3_width, hlines3_height,
					 blackPixel, whitePixel, depth);

	horizMoveChunk = lcm (lcm (gray_width, light_gray_width),
			      hlines3_width);
	vertMoveChunk = lcm (lcm (gray_height, light_gray_height),
			     hlines3_height);

	values.line_width = barThickness;
	values.fill_style = FillTiled;
	values.function = GXcopy;

#define GCFLAGS (GCFunction|GCLineWidth|GCFillStyle)

	values.tile = hlinesPixmap;
	errorGC = XtGetGC (hostGraph, GCFLAGS | GCTile, &values);

	values.tile = grayPixmap;
	runningGC = XtGetGC (hostGraph, GCFLAGS | GCTile, &values);

	values.tile = lightGrayPixmap;
	idleGC = XtGetGC (hostGraph, GCFLAGS | GCTile, &values);


	values.line_width = 1;
	values.fill_style = FillSolid;
	values.function = GXcopy;
	ticGC = XtGetGC (hostGraph, GCFLAGS, &values);

	values.function = GXcopy;
	deadGC = XtGetGC (hostGraph, GCFLAGS, &values);

#ifdef HAIRLINE
	values.fill_style = FillTiled;
	values.line_width = appDefaults.hairLineWidth;
	hairLineGC = XtGetGC (hostGraph, GCFLAGS, &values);

	XtVaGetValues (hostGraph,
		       XtNbackground, &(values.foreground),
		       NULL);
	hairLineClearGC = XtGetGC (hostGraph,
				   GCFLAGS | GCForeground,
				   &values);
#endif
    }

    /*
     * Set window attributes so that future expose events
     * will copy the existing graph to the *right* side of
     * the window and issue an expose event on the left
     * side of the window.
     *
     * XXX use ForgetGravity now, NorthEastGravity eventually
     */
    xswa.bit_gravity = ForgetGravity;
    XChangeWindowAttributes (XtDisplay (hostGraph), XtWindow(hostGraph),
			     CWBitGravity, &xswa);
    check_horiz_bounds (True);
}


draw_bar (w, gc, from, to, host)
Widget w;
GC gc;
struct EventTime *from, *to;
int host;
{
    register int xfrom = xpos (from);
    register int xto = xpos (to);
    register int yhost = ypos (host);

#if 0
    /*
     * clear the area above and below the bar
     */
    int fudge = (rowSpacing - barThickness + 1) / 2;
    XClearArea (XtDisplay (w), XtWindow (w),
		xfrom, yhost - barThickness / 2 - fudge,
		xto - xfrom, fudge,
		0);
    XClearArea (XtDisplay (w), XtWindow (w),
		xfrom, yhost + (barThickness - 1) / 2 + 1,
		xto - xfrom, fudge,
		0);
#endif

    /*
     * On monochrome displays, we don't have enough different kinds of
     * stipple patterns for everything to look good.  So for "dead" hosts,
     * we draw a "white" bar with black lines for borders.
     *
     * "I'm ashamed of this."  -SMK
     */

    if (!inColor && gc == deadGC) {
	XDrawLine (XtDisplay (w), XtWindow (w), gc,
		   xfrom, yhost - barThickness / 2,
		   xto, yhost - barThickness / 2);
	XDrawLine (XtDisplay (w), XtWindow (w), gc,
		   xfrom, yhost + (barThickness - 1)  / 2,
		   xto, yhost + (barThickness - 1) / 2);
    }
    else {
	XDrawLine (XtDisplay (w), XtWindow (w), gc,
		   xfrom, yhost,
		   xto, yhost);
    }
}


#ifdef HAIRLINE
static void
draw_hairline (w, t, gc)
Widget w;
struct EventTime *t;
GC gc;
{
#ifdef BIGHAIR
    int x = xpos (t);
    XDrawLine (XtDisplay (w), XtWindow (w), gc,
	       x, 0, x, graphWindowHeight);
#else
    int y0;			/* bottom of previous host bar */
    int y1;			/* top of current host bar */
    int x;
    int hostNumber;
    int fudge = (rowSpacing - barThickness) / 2;
    int imin;
    int imax;

    imin = (topEdge - (rowSpacing + barThickness) / 2) /
	rowSpacing - 1;
    imax = (topEdge + graphWindowHeight - (rowSpacing + barThickness) / 2) /
	rowSpacing + 1;

    if (imin < 0)
	imin = 0;
    if (imin >= numHosts)
	imin = numHosts - 1;
    if (imax < 0)
	imax = 0;
    if (imax >= numHosts)
	imax = numHosts - 1;

    x = xpos (t);
    for (hostNumber = imin; hostNumber <= imax; ++hostNumber) {
	y0 = hostNumber * rowSpacing - fudge;
	y1 = hostNumber * rowSpacing + fudge;
	if ((0 < y0 && y0 < graphWindowHeight) ||
	    (0 < y1 && y1 < graphWindowHeight))
	    XDrawLine (XtDisplay (w), XtWindow (w), gc, x, y0, x, y1);
    }
#endif
}
#endif

static int
event_is_visible (t)
struct EventTime *t;
{
    if (t->secs == 0 && t->millisecs == 0)
	return 0;
    else
	return (0 <= xpos(t) && xpos(t) <= graphWindowWidth);
}

/*
 * come here when it is necessary to redraw a portion of the graph
 */

static void
graph_redraw (w, minX, minY, maxX, maxY, region)
Widget w;
int minX, minY, maxX, maxY;
Region region;
{
    int i;
    Dimension tic;
    int imin, imax;
    struct EventTime tmin, tmax;

    static GC gcs[4];
    static Region emptyRegion;
    static Region clipRegion;
    static int initialized = 0;

    if (initialized == 0) {
	initialize_graph ();
	gcs[HISTO_ERROR] = errorGC;
	gcs[HISTO_RUNNING] = runningGC;
	gcs[HISTO_IDLE] = idleGC;
	gcs[HISTO_OFF] = deadGC;
	emptyRegion = XCreateRegion ();
	clipRegion = XCreateRegion ();
	initialized = 1;
    }

    if (minX == maxX || minY == maxY)
	return;

    if (region == NULL) {
	XRectangle rect;

	rect.x = minX;
	rect.y = minY;
	rect.width = maxX - minX;
	rect.height = maxY - minY;
	XIntersectRegion (emptyRegion, clipRegion, clipRegion);
	XUnionRectWithRegion (&rect, emptyRegion, clipRegion);
	region = clipRegion;
    }

    for (i = 0; i < 4; ++i)
	XSetRegion (XtDisplay (w), gcs[i], region);
    XSetRegion (XtDisplay (w), ticGC, region);

    imin = (topEdge + minY - (rowSpacing + barThickness) / 2) /
	rowSpacing - 1;
    imax = (topEdge + maxY - (rowSpacing + barThickness) / 2) /
	rowSpacing + 1;

    if (imin < 0)
	imin = 0;
    if (imin >= numHosts)
	imin = numHosts - 1;
    if (imax < 0)
	imax = 0;
    if (imax >= numHosts)
	imax = numHosts - 1;

    /*
     * compute bounds on the earliest and latest events
     * to be displayed
     */
    time_from_x (&tmin, minX, -1);
    time_from_x (&tmax, maxX, 1);

    if (before (&tmin, &startTime)) {
	tmin = startTime;
    }
    if (after (&tmax, &endTime)) {
	tmax = endTime;
    }

    /*
     * don't draw bars at all if we're outside the time zone
     */

    if (!after (&tmin, &endTime) && !before (&tmax, &startTime)) {
	/*
	 * Get state of each host at some time to the left edge of the graph.
	 * Loop reading events until the event is past the right edge of the
	 * graph.  For each event that would appear on the screen, draw a
	 * bar showing how long that host was in its previous state.
	 * Keep track of the last known state of each host.
	 */
	getState (savedState, numHosts, &tmin);
	do {
	    int hostNumber;
	    struct Event oldEvent, newEvent;
	    
	    if (getNextEvent (&newEvent) < 0)
		break;
	    hostNumber = newEvent.hostno;
	    if (hostNumber < 0 && hostNumber >= numHosts)
		continue;
	    oldEvent = savedState[hostNumber];
	    
	    if (imin <= hostNumber && hostNumber <= imax)
		draw_bar (hostGraph, gcs[oldEvent.state],
			  &oldEvent.time, &newEvent.time, hostNumber);
	    savedState[hostNumber] = newEvent;
	    if (after (&newEvent.time, &tmax))
		break;
	} while (1);

	/*
	 * now extend the line for the last known state of each host
	 * to the right edge of the graph.
	 */
	for (i = imin; i <= imax; ++i) {
	    draw_bar (hostGraph, gcs[savedState[i].state],
		      &(savedState[i].time), &tmax, i);
	}
    }

    /*
     * draw tick marks, avoiding fenceposts
     */

    minY = min(ypos(imin) - barThickness / 2, minY);
    maxY = max(ypos(imax) + barThickness / 2, maxY);
    if (minY < 0)
	minY = 0;
    if (maxY > graphWindowHeight)
	maxY = graphWindowHeight;

    tic = (int) graphWindowWidth - (rightEdge % pixelsPerTic);
    if (tic == (int) graphWindowWidth) /* don't draw tic at right edge */
	tic -= pixelsPerTic;
    tic += pixelsPerTic;	/* compensate for pre-decrement */
    do {
#if 0
	struct EventTime foo;
#endif

	if (tic < pixelsPerTic)
	    break;
	tic -= pixelsPerTic;
	if (tic > maxX)
	    continue;
	if (tic < minX)
	    break;

#if 0
	/* don't draw tics left of zero time */
	time_from_x (&foo, tic, 0);
	if (before (&foo, &startTime))
	    break;
#endif
	XDrawLine (XtDisplay (hostGraph), XtWindow (hostGraph), ticGC,
		   tic, minY, tic, maxY);
    } while (1);

#ifdef HAIRLINE
    /*
     * draw hairline
     */
    if (event_is_visible (&hairLine))
	draw_hairline (hostGraph, &hairLine, hairLineGC);
#endif
}

static void
graph_expose (w, client_data, xevent, continue_to_dispatch)
Widget w;
XtPointer client_data;
XEvent *xevent;
Boolean *continue_to_dispatch;
{
    XConfigureEvent *cev =  &(xevent->xconfigure);
    XEvent xev;
    XExposeEvent xexp;
    XGraphicsExposeEvent xgrexp;
    int minX, maxX, minY, maxY;

    static Region emptyRegion, clipRegion;
    static int initialized = 0;

    if (initialized == 0) {
	emptyRegion = XCreateRegion ();
	clipRegion = XCreateRegion ();
	initialized = 1;
    }

    switch (xevent->type) {

    case Expose:
	xexp = xevent->xexpose;
	minX = xexp.x;
	maxX = xexp.x + xexp.width;
	minY = xexp.y;
	maxY = xexp.y + xexp.height;
	XIntersectRegion (emptyRegion, clipRegion, clipRegion);
	XtAddExposureToRegion(xevent, clipRegion);

	while (xexp.count > 0) {
	    XNextEvent (XtDisplay (w), &xev);
	    xexp = xev.xexpose;
	    minX = min (xexp.x, minX);
	    maxX = max (xexp.x + xexp.width, maxX);
	    minY = min (xexp.y, minY);
	    maxY = max (xexp.y + xexp.height, maxY);
	    XtAddExposureToRegion (&xev, clipRegion);
	}
	graph_redraw (w, minX, minY, maxX, maxY, clipRegion);
	return;

    case GraphicsExpose:
	xgrexp = xevent->xgraphicsexpose;
	minX = xgrexp.x;
	maxX = xgrexp.x + xgrexp.width;
	minY = xgrexp.y;
	maxY = xgrexp.y + xgrexp.height;
	XIntersectRegion (emptyRegion, clipRegion, clipRegion);
	XtAddExposureToRegion(xevent, clipRegion);

	while (xgrexp.count > 0) {
	    XNextEvent (XtDisplay (w), &xev);
	    xgrexp = xev.xgraphicsexpose;
	    minX = min (xgrexp.x, minX);
	    maxX = max (xgrexp.x + xgrexp.width, maxX);
	    minY = min (xgrexp.y, minY);
	    maxY = max (xgrexp.y + xgrexp.height, maxY);
	    XtAddExposureToRegion (&xev, clipRegion);
	}
	graph_redraw (w, minX, minY, maxX, maxY, clipRegion);
	return;

    case ConfigureNotify:
	graphWindowWidth = cev->width;
	graphWindowHeight = cev->height;

	check_horiz_bounds (True);
	check_vert_bounds (True);

	/*
	 * XXX eventually we should do something useful with the left-over
	 * trash in the resized window -- in which case, the XClearWindow()
	 * call should go away.
	 */
	XClearWindow (XtDisplay (w), XtWindow (w));
	graph_redraw (w, 0, 0, graphWindowWidth, graphWindowHeight,
		      NULL);
	return;

    case MapNotify:
    case NoExpose:
	return;

    default:
	fprintf (stderr, "graph_expose: event type %d\n", xevent->type);
	return;
    }
}


/*
 * come here to redraw a portion of the host list
 */

static void
list_redraw (w, minX, minY, maxX, maxY)
Widget w;
int minX, minY, maxX, maxY;
{
    int i, imin, imax;

    static int textBaseLine;	/* dist. from top of character to baseline */
    static GC textGC = 0;
    static int initialized = 0;

    if (initialized == 0) {
	/*
	 * If we haven't allocated a GC for drawing text yet, do so here.
	 * grab the background color from the hostList widget, and the
	 * foreground color and font from the resources database.
	 */

	XGCValues values;
	XFontStruct *fsptr;
	
	XtVaGetValues (hostList,
		       XtNbackground, &(values.background),
		       NULL);
	values.font = appDefaults.listFont;
	values.foreground = appDefaults.foreground;

	textGC = XtGetGC (hostList,
			  (unsigned) GCForeground | GCBackground | GCFont,
			  &values);
	fsptr = XQueryFont (XtDisplay (hostList), values.font);
	if (fsptr == NULL) {
	    textBaseLine = rowSpacing + 10;
	}
	else {
	    /*
	     * put the first line of text so that the portion of the text
	     * above the base line is even with the vertical center of
	     * the bar.
	     */
	    textBaseLine = (rowSpacing + fsptr->ascent) / 2;
	    XFreeFontInfo (NULL, fsptr, 1);
	}
	initialized = 1;
    }

    imin = (topEdge + minY - (rowSpacing + barThickness) / 2) /
	rowSpacing - 1;
    imax = (topEdge + maxY - (rowSpacing + barThickness) / 2) /
	rowSpacing + 1;

    if (imin < 0)
	imin = 0;
    if (imin >= numHosts)
	imin = 0;
    if (imax < 0)
	imax = 0;
    if (imax >= numHosts)
	imax = numHosts - 1;

    for (i = imin; i <= imax ; ++i) {
	XDrawString (XtDisplay (hostList), XtWindow(hostList), textGC, 10,
		     i * rowSpacing + textBaseLine - topEdge, hostNames[i],
		     strlen(hostNames[i]));
    }
}

static void
list_expose (w, client_data, xevent, continue_to_dispatch)
Widget w;
XtPointer client_data;
XEvent *xevent;
Boolean *continue_to_dispatch;
{
    XConfigureEvent *cev = (XConfigureEvent *) xevent;

    switch (xevent->type) {
    case Expose:
	if (xevent->xexpose.count != 0)
	    return;
	list_redraw (w, 0, 0, listWidth, graphWindowHeight);
	return;
    case GraphicsExpose:
	if (xevent->xgraphicsexpose.count != 0)
	    return;
	list_redraw (w, 0, 0, listWidth, graphWindowHeight);
	return;
    case ConfigureNotify:
	graphWindowHeight = cev->height;
	check_vert_bounds (True);
	clear_area (w, 0, 0, listWidth, graphWindowHeight);
	list_redraw (w, 0, 0, listWidth, graphWindowHeight);
	return;
    }
}

#define XtVaCMW XtVaCreateManagedWidget

/*
 * create a histogram popup
 */

void
hist_Create (parent, hosts, nhosts)
Widget parent;
char **hosts;
int nhosts;
{
    XtAppContext appContext;

    hostNames = hosts;
    numHosts = nhosts;

    initialize_defaults (parent);

    appContext = XtWidgetToApplicationContext (parent);

    up_pixmap =
	XCreateBitmapFromData (XtDisplay (parent),
			       DefaultRootWindow (XtDisplay (parent)),
			       up_bits, up_width, up_height);
    down_pixmap =
	XCreateBitmapFromData (XtDisplay (parent),
			       DefaultRootWindow (XtDisplay (parent)),
			       down_bits, down_width, down_height);

    graphHeight = numHosts * rowSpacing;

    /* make graph smaller if there aren't many hosts */

    currentHeight = min (graphHeight + (2 * scrollBarSize) + (2 * borderWidth),
			 currentHeight);

    graphWindowHeight =
	currentHeight - (2 * scrollBarSize) - (2 * borderWidth);
    graphWindowWidth =
	currentWidth - listWidth - scrollBarSize - (4 * borderWidth);

    /*
     * create top-level Form widget to hold everything together
     */

    outerForm = XtVaCMW ("form", formWidgetClass, parent,
			 XtNdefaultDistance, 0,
			 NULL);
						  
    /*
     * create a vertical scrollbar to the left of the host list.
     * set up callbacks
     */

    leftBar =
	XtVaCMW ("leftBar", scrollbarWidgetClass, outerForm,
		 XtNorientation, XtorientVertical,
		 XtNborderWidth, borderWidth,
		 XtNminimumThumb, scrollBarSize - 2 * borderWidth,
		 /* geometry */
		 XtNx, 0,
		 XtNy, 0,
		 XtNwidth, scrollBarSize,
		 XtNheight, graphWindowHeight,
		 /* form constraints */
		 XtNfromHoriz, NULL,
		 XtNfromVert, NULL,
		 XtNleft, XtChainLeft,
		 XtNright, XtChainLeft,
		 XtNtop, XtChainTop,
		 XtNbottom, XtChainBottom,
		 NULL);
    XtAddCallback (leftBar, XtNjumpProc, vert_jump, (XtPointer) NULL);
    XtAddCallback (leftBar, XtNscrollProc, vert_scroll, (XtPointer) NULL);

    hostList =
	XtVaCMW ("hostList", simpleWidgetClass, outerForm,
		 XtNborderWidth, borderWidth,
		 /* geometry */
		 XtNx, scrollBarSize + 2 * borderWidth,
		 XtNy, 0,
		 XtNwidth, listWidth,
		 XtNheight, graphWindowHeight,
		 /* form constraints */
		 XtNfromHoriz, leftBar,
		 XtNfromVert, NULL,
		 XtNleft, XtChainLeft,
		 XtNright, XtChainLeft,
		 XtNtop, XtChainTop,
		 XtNbottom, XtChainBottom,
		 NULL);
    XtAddEventHandler (hostList, StructureNotifyMask|ExposureMask,
		       True, list_expose, 0);

    hostGraph =
	XtVaCMW ("hostGraph", simpleWidgetClass, outerForm,
		 XtNborderWidth, borderWidth,
		 XtNx, scrollBarSize + listWidth,
		 XtNy, 0,
		 XtNwidth, graphWindowWidth,
		 XtNheight, graphWindowHeight,
		 /* form constraints */
		 XtNfromHoriz, hostList,
		 XtNfromVert, NULL,
		 XtNleft, XtChainLeft,
		 XtNright, XtChainRight,
		 XtNtop, XtChainTop,
		 XtNbottom, XtChainBottom,
		 NULL);
    XtAddEventHandler (hostGraph, StructureNotifyMask|ExposureMask,
		       True, graph_expose, 0);

    buttonBox =
	XtVaCMW ("buttonBox", formWidgetClass, outerForm,
		 XtNborderWidth, 1,
		 XtNdefaultDistance, 0,
		 XtNresizable, False,
		 XtNwidth, scrollBarSize + listWidth + 2 * borderWidth,
		 XtNheight, 2 * scrollBarSize + 2 * borderWidth,
		 /* form constraints */
		 XtNfromVert, hostList,
		 XtNfromHoriz, NULL,
		 XtNleft, XtChainLeft,
		 XtNright, XtChainLeft,
		 XtNtop, XtChainBottom,
		 XtNbottom, XtChainBottom,
		 NULL);

    downButton =
	XtVaCMW ("downButton", commandWidgetClass, buttonBox,
		 XtNwidth, 2 * down_width,
		 XtNheight, 2 * down_height,
		 XtNbitmap, down_pixmap,
		 XtNjustify, XtJustifyCenter,
		 XtNborderWidth, 0,
		 XtNhighlightThickness, 0,
		 /* form constraints */
		 XtNfromHoriz, NULL,
		 XtNfromVert, NULL,
		 XtNleft, XtChainLeft,
		 XtNright, XtChainLeft,
		 XtNtop, XtChainBottom,
		 XtNbottom, XtChainBottom,
		 NULL);
    XtAddCallback (downButton, XtNcallback, (XtCallbackProc) change_scale,
		   (XtPointer) 0);

    scaleLabel =
	XtVaCMW ("scaleLabel", labelWidgetClass, buttonBox,
#ifdef SCROLLING_LABELS
		 XtNlabel, "1 sec",
#else
		 XtNlabel, "scale",
#endif
		 XtNresize, False,
		 XtNwidth, ((listWidth + scrollBarSize + 2 * borderWidth) -
			    2 * (up_width + down_width)),
		 XtNheight, 2 * scrollBarSize,
		 XtNborderWidth, 0,
		 /* form constraints */
		 XtNfromHoriz, downButton,
		 XtNfromVert, NULL,
		 XtNleft, XtChainLeft,
		 XtNright, XtChainRight,
		 XtNtop, XtChainTop,
		 XtNbottom, XtChainBottom,
		 NULL);

    upButton =
	XtVaCMW ("upButton", commandWidgetClass, buttonBox,
		 XtNwidth, 2 * up_width,
		 XtNheight, 2 * up_height,
		 XtNbitmap, up_pixmap,
		 XtNjustify, XtJustifyCenter,
		 XtNborderWidth, 0,
		 XtNhighlightThickness, 0,
		 /* form constraints */
		 XtNfromHoriz, scaleLabel,
		 XtNfromVert, NULL,
		 XtNleft, XtChainRight,
		 XtNright, XtChainRight,
		 XtNtop, XtChainBottom,
		 XtNbottom, XtChainBottom,
		 NULL);
    XtAddCallback (upButton, XtNcallback, (XtCallbackProc) change_scale,
		   (XtPointer) 0);

    /* the bottom scroll bar is fixed height, and as wide as
       the bar graph (not including the host list) */

#ifdef SCROLLING_LABELS
    labelBox =
	XtVaCMW ("labelBox", simpleWidgetClass, outerForm,
		 XtNborderWidth, 0,
		 XtNx, scrollBarSize + listWidth + 4 * borderWidth,
		 XtNy, graphWindowHeight + 2 * borderWidth,
		 XtNheight, scrollBarSize,
		 XtNwidth, graphWindowWidth,
		 XtNfromVert, hostList,
		 XtNfromHoriz, buttonBox,
		 XtNleft, XtChainLeft,
		 XtNright, XtChainRight,
		 XtNtop, XtChainBottom,
		 XtNbottom, XtChainBottom,
		 NULL);
    XtAddEventHandler (labelBox, ExposureMask, True,
		       (XtEventHandler) label_expose,
		       (XtPointer) 0);
#endif

    bottomBar =
	XtVaCMW ("timeScrollBar", scrollbarWidgetClass, outerForm,
		 XtNorientation, XtorientHorizontal,
		 XtNborderWidth, borderWidth,
		 XtNminimumThumb, scrollBarSize - 2 * borderWidth,
		 /* geometry */
		 XtNx, scrollBarSize + listWidth + 4 * borderWidth,
#ifdef SCROLLING_LABELS
		 XtNy, graphWindowHeight + 4 * borderWidth + scrollBarSize,
#else
		 XtNy, graphWindowHeight + 2 * borderWidth,
#endif
		 XtNheight, scrollBarSize,
		 XtNwidth, graphWindowWidth,
		 /* form constraints */
#ifdef SCROLLING_LABELS
		 XtNfromVert, labelBox,
#else
		 XtNfromVert, hostList,
#endif
		 XtNfromHoriz, buttonBox,
		 XtNleft, XtChainLeft,
		 XtNright, XtChainRight,
		 XtNtop, XtChainBottom,
		 XtNbottom, XtChainBottom,
		 NULL);
    XtAddCallback (bottomBar, XtNjumpProc, (XtCallbackProc) horiz_jump,
		   (XtPointer) NULL);
    XtAddCallback (bottomBar, XtNscrollProc, (XtCallbackProc) horiz_scroll,
		   (XtPointer) NULL);

#ifndef SCROLLING_LABELS
    leftLegend =
	XtVaCMW ("leftLegend", labelWidgetClass, outerForm,
		 XtNborderWidth, 0,
		 XtNlabel, "00:00:00",
		 XtNjustify, XtJustifyLeft,
		 XtNresize, False,
		 /* geometry */
		 XtNx, scrollBarSize + listWidth + 4 * borderWidth,
		 XtNy, graphWindowHeight + scrollBarSize + 4 * borderWidth,
		 XtNheight, scrollBarSize,
		 XtNwidth, graphWindowWidth / 3,
		 /* form constraints */
		 XtNfromVert, bottomBar,
		 XtNfromHoriz, buttonBox,
		 XtNleft, XtChainLeft,
		 XtNtop, XtChainBottom,
		 XtNbottom, XtChainBottom,
		 NULL);
    centerLegend =
	XtVaCMW ("centerLegend", labelWidgetClass, outerForm,
		 XtNborderWidth, 0,
		 XtNlabel, "1 sec/div",
		 XtNjustify, XtJustifyCenter,
		 XtNresize, False,
		 /* geometry */
		 XtNx, scrollBarSize + listWidth + 4 * borderWidth,
		 XtNy, graphWindowHeight + scrollBarSize + 4 * borderWidth,
		 XtNheight, scrollBarSize,
		 XtNwidth, graphWindowWidth - 2 * (graphWindowWidth / 3),
		 /* form constraints */
		 XtNfromVert, bottomBar,
		 XtNfromHoriz, leftLegend,
		 XtNtop, XtChainBottom,
		 XtNbottom, XtChainBottom,
		 NULL);
    rightLegend =
	XtVaCMW ("rightLegend", labelWidgetClass, outerForm,
		 XtNborderWidth, 0,
		 XtNlabel, "99:99:99",
		 XtNjustify, XtJustifyRight,
		 XtNresize, False,
		 /* geometry */
		 XtNx, scrollBarSize + listWidth + 4 * borderWidth,
		 XtNy, graphWindowHeight + scrollBarSize + 4 * borderWidth,
		 XtNheight, scrollBarSize,
		 XtNwidth, graphWindowWidth / 3,
		 /* form constraints */
		 XtNfromVert, bottomBar,
		 XtNfromHoriz, centerLegend,
		 XtNright, XtChainRight,
		 XtNtop, XtChainBottom,
		 XtNbottom, XtChainBottom,
		 NULL);
#endif

}

void
hist_SetEpoch (t)
struct EventTime *t;
{
    startTime = *t;
}

void
hist_SetEndOfTime (t)
struct EventTime *t;
{
    struct EventTime oldEndTime;

    oldEndTime = endTime;

    endTime = *t;
    if (before (&endTime, &startTime))
	endTime = startTime;
    if (outerForm != NULL && XtIsRealized (outerForm)) {
	long x0, x1, leftEdge;

	/* if old end time was on the screen, redraw screen */
	x0 = xpos (&oldEndTime);
	leftEdge = rightEdge - graphWindowWidth;
	if (x0 >= leftEdge) {
	    x1 = xpos (&endTime);
	    if (x1 >= rightEdge) {
		/* if new event is off screen, scroll */
		move_graph_window (topEdge, rightEdge + graphWindowWidth / 2);
	    }
	    else
		XClearArea (XtDisplay (hostGraph), XtWindow (hostGraph),
			    0, 0, 0, 0, True);
	}
    }
}

#ifdef HAIRLINE
void
hist_SetHairLine (t, moveok)
struct EventTime *t;
int moveok;
{
    if (outerForm != NULL && XtIsRealized (outerForm)) {
	if (moveok && !event_is_visible (t))
	    move_graph_window (topEdge,
			       time_to_pixels (elapsed_time (&startTime, t))
			       + graphWindowWidth / 2);
#ifdef BIGHAIR

	/*
	 * "undraw" the old hairline by drawing a white line
	 * where the hairline used to be (this clears out the space
	 * *between* the horizontal bars), then calling graph_redraw()
	 * to fill in the space.  Note that we have to change the
	 * value of hairline before calling graph_redraw(), else it
	 * will put the hairline back where it was!
	 * graph_redraw() also takes care of drawing the new hairline
	 * if it is visible.
	 */
	if (event_is_visible (&hairLine)) {
	    int minx, maxx;

	    minx =  xpos (&hairLine) - appDefaults.hairLineWidth / 2;
	    maxx = minx + appDefaults.hairLineWidth;
	    draw_hairline (hostGraph, &hairLine, hairLineClearGC);
	    hairLine = *t;
	    graph_redraw (hostGraph, minx, 0, maxx, graphWindowHeight,
			  NULL);
	}
	else
	    hairLine = *t;
#else
	if (event_is_visible (&hairLine))
	    draw_hairline (hostGraph, &hairLine, hairLineClearGC);
	hairLine = *t;
	if (event_is_visible (&hairLine))
	    draw_hairline (hostGraph, &hairLine, hairLineGC);
#endif
    }
    else
	hairLine = *t;
}

void
hist_ClearHairLine ()
{
#ifdef BIGHAIR
    struct EventTime t;
    t.secs = 0;
    t.millisecs = 0;
    hist_SetHairLine (&t, 0);
#else
    if (outerForm != NULL && XtIsRealized (outerForm)) {
	if (event_is_visible (&hairLine))
	    draw_hairline (hostGraph, &hairLine, hairLineClearGC);
    }
    hairLine.secs = 0;
    hairLine.millisecs = 0;
#endif
}
#endif

int 
hist_MsecPerPixel (msec)
long *msec;
{
    if (outerForm != NULL && XtIsRealized (outerForm)) {
	*msec = (msecPerTic + pixelsPerTic / 2) / pixelsPerTic;
	return 0;
    }
    return -1;
}
