#ifndef FuncPr
#if defined(__STDC__)
#define FuncPr 1
#else
#define FuncPr 0
#endif
#endif

#include <stdio.h>
#include "rb.h"
#include "param.h"
#include "exp.h"
#include "graph.h"
#include "graphP.h"
#include "misc.h"

static int error_count = 0;		/* # of nodes in error */

static char *
nodeTypeName (
#if FuncPr
	int n)
#else
	n)
	int n;
#endif
{
	switch (n) {
	case NODE_NORMAL:
		return "NORMAL";
	case NODE_COND:
		return "BEGINSWITCH";
	case NODE_ENDCOND:
		return "ENDSWITCH";
	case NODE_LOOP:
		return "BEGINLOOP";
	case NODE_ENDLOOP:
		return "ENDLOOP";
	case NODE_FANOUT:
		return "FANOUT";
	case NODE_FANIN:
		return "FANIN";
	case NODE_PIPE:
		return "BEGINPIPE";
	case NODE_ENDPIPE:
		return "ENDPIPE";
	}
}

static int
isLeftSpecial (
#if FuncPr
	Node n)
#else
	n)
	Node n;
#endif
{
	if (n == (Node) NULL)
		return 0;
	switch (n->node_type) {
	case NODE_COND:
	case NODE_LOOP:
	case NODE_FANOUT:
	case NODE_PIPE:
		return 1;
	}
	return 0;
}

static int
isRightSpecial (
#if FuncPr
	Node n)
#else
	n)
	Node n;

#endif
{
	if (n == (Node) NULL)
		return 0;
	switch (n->node_type) {
	case NODE_ENDCOND:
	case NODE_ENDLOOP:
	case NODE_FANIN:
	case NODE_ENDPIPE:
		return 1;
	}
	return 0;
}

static int
matchingNodeType (
#if FuncPr
	Node n)
#else
	n)
	Node n;
#endif
{
	if (n == (Node) NULL)
		return -1;
	switch (n->node_type) {
	case NODE_NORMAL:
		return -1;
	case NODE_ENDCOND:
		return NODE_COND;
	case NODE_ENDLOOP:
		return NODE_LOOP;
	case NODE_FANIN:
		return NODE_FANOUT;
	case NODE_ENDPIPE:
		return NODE_PIPE;
	case NODE_COND:
		return NODE_ENDCOND;
	case NODE_LOOP:
		return NODE_ENDLOOP;
	case NODE_FANOUT:
		return NODE_FANIN;
	case NODE_PIPE:
		return NODE_ENDPIPE;
	}
}

/*
 * this routine marks a node so that it will show up on-screen
 * as an error, and bumps the error count.  If the node already
 * indicates an error, it doesn't bump the count, so that error_count
 * always indicates the number of nodes in error.
 *
 * returns 1 if the node didn't have the error flag set.
 */

static int
markError (
#if FuncPr
	Node n)
#else
	n)
	Node n;
#endif
{
	if (n->state == ST_ERROR)
		return 0;
	n->state = ST_ERROR;
	++error_count;
	return 1;
}

/*
 * find the set of terminal nodes of each of a node's children
 * if the cardinality of the set != 1, figure out which is
 * "most likely" to be the correct terminal node.  Normally this
 * will be the one that occurs most frequently, but if the start
 * node is a special node then the end node must be a matching pair.
 *
 * - if a node has no children, its terminal node is NULL.
 * - if a node has one child which is a right special, its terminal node
 *   is that right special node.
 * - if a node is a left special node, its terminal node is the terminal
 *   node of the matching right special node.
 * - otherwise, the terminal node of a node is the terminal node of each
 *   of its children, which should all be the same.
 */

#if FuncPr
static Node findTerminalNode (Node n, Node left);
#else
static Node findTerminalNode ();
#endif

static Node
reallyFindTerminalNode (
#if FuncPr
	Node n, Node left)
#else
	n, left)
	Node n;
	Node left;
#endif
{
	struct nf {
		Node node;
		int count;
	} *nf;
	TreeNode tn;
	int i;
	int result;
	int nfsize;
	Node result_node;
	int max;

	/*
	 * special case: 0 children
	 */

	if (n->nchildren == 0) {
		if (isLeftSpecial (n)) {
			if (markError (n))
				msg_Format ("node %d: special node has no children\n",
							n->nk.id);
		}
		return (Node) NULL;
	}

	nf = talloc (n->nchildren, struct nf);
	nfsize = 0;

	/*
	 * find each child's terminal node and keep track of how many times
	 * we've seen each one.
	 */
	for (tn = rb_First (n->children); tn != n->children; tn = rb_Next (tn)) {
		Node start = (Node) rb_Value (tn);
		Node end; 

		if (isRightSpecial (start)) {
			if (n->nchildren > 1) {
				/*
				 * if node has more than one child, none of them can be a
				 * right special
				 */
				if (markError (n))
					msg_Format ("node %d: only one child can be a %s\n",
								n->nk.id, nodeTypeName (start->node_type));
			}
			if (left == (Node) NULL) {
				if (markError (start))
					msg_Format ("node %d: no path to a matching %s node through node %d\n",
								start->nk.id,
								nodeTypeName (matchingNodeType(start)),
								n->nk.id);
			}
			else {
				free (nf);
				return start;
			}
		}
		if (isLeftSpecial (n))
			end = findTerminalNode (start, n);
		else
			end = findTerminalNode (start, left);

		for (i = 0 ; i < nfsize; ++i)
			if (end == nf[i].node)
				++nf[i].count;
		if (i == nfsize) {
			/* haven't seen this node yet - add it to the list */
			nf[nfsize].node = end;
			nf[nfsize].count = 1;
			++nfsize;
		}
	}

	/*
	 * now scan each node in the list and find the most frequently
	 * occuring matching node that qualifies.
	 */
	result = -1;
	max = 0;
	for (i = 0; i < nfsize; ++i) {
		if (isLeftSpecial (n)) {
			if (nf[i].node != (Node) NULL &&
				nf[i].node->node_type == matchingNodeType (n) &&
				nf[i].count > max)
				result = i;
		}
		else {
			if (nf[i].count > max)
				result = i;
		}
	}

	/*
	 * if there was no qualifying match, just use the first node
	 * in the list.
	 */
	if (result == -1)
		result = 0;
	result_node = nf[result].node;

	/*
	 * for all of the problem children, mark their terminal nodes
	 * as being in error
	 */

	for (tn = rb_First (n->children); tn != n->children; tn = rb_Next (tn)) {
		Node child = (Node) rb_Value (tn);
		Node end = child->scope ? child->scope : child;

		/*
		 * findTerminalNode(), already called above for each child,
		 * guarantees that the child's scope field is valid
		 */
		if (child->scope != result_node) {
			if (left) {
				if (markError (end)) {
					msg_Format ("node %d: no path to %s matching %s node %d\n",
								end->nk.id,
								nodeTypeName (matchingNodeType (left)),
								nodeTypeName (left->node_type),
								left->nk.id);
				}
			}
			else {
				if (markError (end))
					msg_Format ("node %d: branch out of subgraph\n",
								end->nk.id);
			}
		}
	}

	/*
	 * for special node pairs, set their pair fields.
	 */
	if (isLeftSpecial (n)) {
		if (isRightSpecial (result_node) &&
			n->node_type == matchingNodeType (result_node)) {
			n->pair = result_node;
			n->pair->pair = n;
			result_node = findTerminalNode (result_node, left);
		}
		else {
			if (markError (n))
				msg_Format ("node %d: no match for special node\n",
							n->nk.id);
		}
	}

	/*
	 * return the most likely terminal node
	 */
	free (nf);
	return result_node;
}

/*
 * this is a wrapper for the reallyFindTerminalNode function that
 * does loop detection, makes sure the scope field is set, and
 * takes advantage of previously computed terminal nodes.
 */

static Node
findTerminalNode (
#if FuncPr
	Node n, Node left)
#else
	n, left)
	Node n;
	Node left;
#endif
{
	Node result;

	if (n->flags & NF_BEEN_HERE) {
		if (markError (n)) {
			/* XXX need to mark all nodes in the loop to make
			   it obvious to the user. */
			msg_Format ("node %d: loop detected\n", n->nk.id);
		}
		result = n;
	}
	else {
		n->flags |= NF_BEEN_HERE;

		if (n->flags & NF_SCOPE_VALID)
			result = n->scope;
		else {
			result = reallyFindTerminalNode (n, left);
			n->scope = result;
			n->flags |= NF_SCOPE_VALID;
		}

		n->flags &= ~NF_BEEN_HERE;
	}
	return result;
}


static int
makeSureTerminalNodeIsNULL (
#if FuncPr
	TreeNode tn)
#else
	tn)
	TreeNode tn;
#endif
{
	Node n = (Node) rb_Value (tn);

	if (findTerminalNode (n, NULL) != (Node) NULL) {
		if (markError (n)) {
			if (isRightSpecial (n))
				msg_Format ("Node %d: %s has no ancestors.\n",
							n->nk.id, nodeTypeName (n->node_type));
			else {
				msg_Format ("Node %d: path from head ends in loop\n",
							n->nk.id);
			}
		}
	}
	return 0;
}

static int
clearFlags (
#if FuncPr
	TreeNode tn)
#else
	tn)
	TreeNode tn;
#endif
{
	Node n = (Node) rb_Value (tn);
	n->flags &= ~(NF_SCOPE_VALID|NF_BEEN_HERE);
	n->state = ST_NOT_BEGUN;
	n->pair = (Node) NULL;
	n->scope = (Node) NULL;
	return 0;
}

/*
 * scan a graph.
 * check for loops, unmatched special node pairs, branches out of
 * subgraphs, etc.
 *
 * return 0 if all is well.
 * return nonzero if there were any errors
 *
 * side effects:
 * - for every node n of the graph, sets n->scope to the terminal node
 *   of node n, and sets the NF_SCOPE_VALID flag.
 * - for special nodes, sets n->pair to the matching special node
 * - sets state of every node to either ST_NOT_BEGUN or ST_ERROR, so
 *   they will show up right on the screen.
 * - prints error messages
 */

int
gr_Critic (
#if FuncPr
	Graph g)
#else
	g)
	Graph g;
#endif
{
	if (g == (Graph) NULL ||
		g->nlist == (Tree) NULL || g->heads == (Tree) NULL) {
		msg_Format ("Graph is NULL.\n");
		return 1;
	}
	(void) rb_Traverse (g->nlist, clearFlags);
	error_count = 0;
	if (rb_Empty (g->nlist)) {
		msg_Format ("graph is empty.\n");
		return 0;
	}
	else if (rb_Empty (g->heads)) {
		/*
		 * this is lame.  there's no one node that is incorrect,
		 * but we have to mark something else the user won't
		 * understand why his program doesn't run.
		 */
		if (markError (rb_Value (rb_First (g->nlist))))
			msg_Format ("graph has no root.\n");
		return 1;
	}
	(void) rb_Traverse (g->heads, makeSureTerminalNodeIsNULL);
	return error_count;
}

static int
checkForCycles (
#if FuncPr
	TreeNode tn)
#else
	tn)
	TreeNode tn;
#endif
{
	Node n = (Node) rb_Value (tn);
	if (n->flags & NF_BEEN_HERE)
		return 1;
	if (n->nchildren == 0)
		return 0;
	n->flags |= NF_BEEN_HERE;
	if (rb_Traverse (n->children, checkForCycles))
		return 1;
	n->flags &= ~NF_BEEN_HERE;
	return 0;
}


static int
clearBeenHereFlag (
#if FuncPr
	TreeNode tn)
#else
	tn)
	TreeNode tn;
#endif
{
	Node n = (Node) rb_Value (tn);
	n->flags &= ~NF_BEEN_HERE;
	return 0;
}

int
gr_HasCycles (
#if FuncPr
	Graph g)
#else
	g)
	Graph g;
#endif
{
	int x;

	if (rb_Next (g->nlist) == g->nlist)
		return 0;
	else if (rb_Next (g->heads) == g->heads)
		return 1;
	(void) rb_Traverse (g->nlist, clearBeenHereFlag);
	x = rb_Traverse (g->heads, checkForCycles);
	(void) rb_Traverse (g->nlist, clearBeenHereFlag);
	return x;
}

#if 0
static int
printApair (
#if FuncPr
	TreeNode tn)
#else
	tn)
	TreeNode tn;
#endif
{
	Node n = (Node) rb_Value (tn);
	if (isLeftSpecial (n) && n->pair != (Node) NULL) {
		msg_Format ("pair: %d <-> %d\n", n->nk.id, n->pair->nk.id);
		if (n->pair->pair == (Node) NULL || n->pair->pair != n) {
			msg_Format ("non symmetrical\n");
		}
	}
	return 0;
}

static int
printMatchingPairs (
#if FuncPr
	Graph g)
#else
	g)
	Graph g;
#endif
{
	(void) rb_Traverse (g->nlist, printApair);
}
#endif /* 0 */

/*
 * Local variables:
 * tab-width:4
 * End:
 */
