/*
 *	HeNCE Tool
 *
 *	costmat.c - Main widget stuff.
 *		This is stuff for dealing with cost matrices.
 *
 *	Jun 1991  Robert Manchek  manchek@CS.UTK.EDU.
 *
 *	Revision Log
 *
$Log: costmat.c,v $
 * Revision 1.6  1993/05/16  23:03:49  moore
 * add hack to host file generation needed by pvm3
 * (even though pvm3 doesn't use it yet)
 * HeNCE v1.4
 *
 * Revision 1.5  1992/12/09  04:58:33  moore
 * don't write out an entry from cost matrix if cost == 0
 *
 * make host name compares case-insensitive.
 *
 * new implementation of cm_BuildPvmHostFile() looks up ip addresses
 * and uses those for comparison, rather than host names, when generating
 * host files to feed to pvm.
 *
 * Revision 1.4  1992/08/05  21:56:53  moore
 * cm_WriteHosts(): check SUBHOST_EN flag instead of SUBHOST_INPVM flag.
 * Also, check for flag != 0 rather than flag == 0.
 *
 * Revision 1.3  1992/05/19  05:31:57  moore
 * cm_WriteHosts(): new function to write out a list of hosts being
 * used in the current cost matrix.  Used by "make everywhere".
 *
 * Revision 1.2  1992/04/08  06:48:47  moore
 * incorporates changes up to 8 April 1992
 *
 *
 */

#include <stdio.h>
#include <ctype.h>
#include "xincl.h"
#include "xcomn.h"
#include "rb.h"
#include "param.h"
#include "exp.h"				/* XXX Yuk! */
#include "graph.h"
#include "graphP.h"				/* XXX Yuk! */
#include "comn.h"
#include "costmat.h"
#include <netdb.h>
#include <netinet/in.h>

static int
hostcmp (h1, h2)
char *h1, *h2;
{
	return strcasecmp (h1, h2);
}

/*	cm_New ()
 *
 *	Make a new cost matrix.  Add a single entry with null hostname
 *	and function name.
 */

struct costm*
cm_New ()
{
	struct costm *cm;

	if (cm = talloc(1, struct costm)) {
		cm->nhost = 0;
		cm->nsub = 0;
		cm->hosts = 0;
		cm->subs = 0;
		cm->mat = 0;
		cm_AddEntry (cm, (char*)0, (char*)0, 0);
		cm->isModified = 0;
	}
	return cm;
}

/*	cm_Free ()
 *
 *	Free any storage associated with a cost matrix.
 */

void
cm_Free (cm)
struct costm *cm;
{
	int i;

	if (cm) {
		if (cm->hosts) {
			for (i = cm->nhost; i-- > 0; )
				if (cm->hosts[i])
					free(cm->hosts[i]);
			free(cm->hosts);
		}
		if (cm->subs) {
			for (i = cm->nsub; i-- > 0; )
				if (cm->subs[i])
					free(cm->subs[i]);
			free(cm->subs);
		}
		if (cm->mat)
			free(cm->mat);
		free(cm);
	}
}

/*	cm_AddEntry ()
 *
 *	Add (or change) the cost of a host/subroutine pair in a cost matrix.
 *	Creates a new row or column if necessary.
 *
 *	Returns 0 if ok
 *		1 if can't get memory.
 */

int
cm_AddEntry (cm, host, sub, val)
struct costm *cm;
char *host;
char *sub;
int val;
{
	int h, s;
	char **pp;
	int *ip;
	int i, j, k;
	int n;

	cm->isModified = 1;
	if (host) {
		for (h = cm->nhost; h-- > 0; )
			if (cm->hosts[h] && !hostcmp(cm->hosts[h], host))
				break;
	}
	else {
		for (h = cm->nhost; h-- > 0; )
			if (!cm->hosts[h])
				break;
	}
	if (sub) {
		for (s = cm->nsub; s-- > 0; )
			if (cm->subs[s] && !strcmp(cm->subs[s], sub))
				break;
	}
	else {
		for (s = cm->nsub; s-- > 0; )
			if (!cm->subs[s])
				break;
	}

	if (h < 0 || s < 0) {
		n = (k = ((s < 0) ? cm->nsub + 1 : cm->nsub))
		* ((h < 0) ? cm->nhost + 1 : cm->nhost);
		if (!(ip = talloc(n, int)))
			return 1;
		bzero(ip, n * sizeof(int));
		for (i = cm->nsub; i-- > 0; )
			for (j = cm->nhost; j-- > 0; )
				ip[i + j * k] = cm->mat[i + j * cm->nsub];
		free(cm->mat);
		cm->mat = ip;
	}
	if (h < 0) {
		h = cm->nhost++;
		if (!(pp = talloc(h + 1, char*))
		|| !(ip = talloc(h + 1, int)))
			return 1;
		if (h) {
			bcopy(cm->hosts, pp, h * sizeof(char*));
			free(cm->hosts);
			bcopy(cm->hflag, ip, h * sizeof(int));
			free(cm->hflag);
		}
		cm->hosts = pp;
		cm->hflag = ip;
		cm->hosts[h] = host ? strsave(host) : 0;
		cm->hflag[h] = SUBHOST_EN;
	}
	if (s < 0) {
		s = cm->nsub++;
		if (!(pp = talloc(s + 1, char*))
		|| !(ip = talloc(s + 1, int)))
			return 1;
		if (s) {
			bcopy(cm->subs, pp, s * sizeof(char*));
			free(cm->subs);
			bcopy(cm->sflag, ip, s * sizeof(int));
			free(cm->sflag);
		}
		cm->subs = pp;
		cm->sflag = ip;
		cm->subs[s] = sub ? strsave(sub) : 0;
		cm->sflag[s] = 0;
	}
	cm->mat[s + h * cm->nsub] = val;
	return 0;
}

/*	cm_ReadFile ()
 *
 *	Read a file into a new cost matrix and return it.
 *
 *	Returns 0 if ok
 *		1 if can't read file
 *		2 if errors in file
 *		3 if can't get memory
 */

int
cm_ReadFile (fn, cmp)
char *fn;
struct costm **cmp;
{
	struct costm *cm;
	FILE *ff;
	char buf[1024];
	int ac;
	char *av[3];
	char *p;
	int err = 0;

	if (!(cm = cm_New ()))
		return 3;
	if (!(ff = fopen(fn, "r")))
		return 1;

	while (fgets(buf, sizeof(buf), ff)) {
		for (p = buf; *p && isspace(*p); p++);
		if (!*p || *p == '#')	/* skip comments */
			continue;
		ac = 3;
		if (acav(p, &ac, av) || ac != 3) {
			err++;
			continue;
		}
		cm_AddEntry (cm, av[0], av[1], atoi(av[2]));
	}
	fclose(ff);
	cm->isModified = 0;

	if (err) {
		cm_Free (cm);
		return 2;
	}

	if (cmp)
		*cmp = cm;
	else
		cm_Free (cm);
	return 0;
}

/*	cm_WriteFile ()
 *
 *	Write a cost matrix into a file.
 *
 *	Returns 0 if ok
 *		1 if can't write file.
 *		2 if null cost matrix.
 */

int
cm_WriteFile (fn, cmp)
char *fn;
struct costm *cmp;
{
	FILE *ff;
	int h, s;

	if (!cmp)
		return 2;
	if (!(ff = fopen(fn, "w")))
		return 1;

	for (h = 0; h < cmp->nhost; h++)
		if (cmp->hosts[h] && *cmp->hosts[h])
			for (s = 0; s < cmp->nsub; s++)
				/* only write out the entry if cost > 0 */
				if (cmp->subs[s] && *cmp->subs[s] &&
					cmp->mat[s + h * cmp->nsub] > 0)
					fprintf(ff, "%s %s %d\n", cmp->hosts[h], cmp->subs[s],
						cmp->mat[s + h * cmp->nsub]);

	(void)fclose(ff);
	cmp->isModified = 0;
	return 0;
}


/*	cm_CheckGraph ()
 *
 *	Check that every node in a graph has is mapped to at least one
 *	host (SUBHOST_EN is true and the cost is > 0).
 *	Mark all needed hosts with SUBHOST_HOST and needed subs with
 *	SUBHOST_SUB.
 *
 *	Returns 0 if ok, else 1;
 */

int
cm_CheckGraph (grf, cm)
Graph grf;
struct costm *cm;
{
	Node n;
	TreeNode tn;
	int err = 0;
	int i, j, k;

	if (grf == NULL) 
		return 0;

	if (grf->heads == NULL || gr_Empty (grf))
		return 0;

	/* clear all 'needed' flags */

	for (i = cm->nsub; i-- > 0; )
		cm->sflag[i] &= ~SUBHOST_SUB;
	for (i = cm->nhost; i-- > 0; )
		cm->hflag[i] &= ~SUBHOST_HOST;

	/*	for each node, mark function used and also
		all enabled hosts with positive cost */

	for (tn = rb_First(grf->nlist); tn != grf->nlist; tn = rb_Next (tn)) {
		n = (Node) rb_Value (tn);

		if (n == NULL || n->node_type != NODE_NORMAL)
			continue;

		for (i = cm->nsub; i-- > 0; )
			if (cm->subs[i] != NULL && n->sub_name != NULL &&
				strcmp(n->sub_name, cm->subs[i]) == 0) {
				cm->sflag[i] |= SUBHOST_SUB;
				k = 0;
				for (j = cm->nhost; j-- > 0; )
					if ((cm->hflag[j] & SUBHOST_EN)
					&& cm->hosts[j]
					&& cm->mat[i + j * cm->nsub] > 0) {
						cm->hflag[j] |= SUBHOST_HOST;
						k++;
					}
				if (!k)
					i = -1;
				break;
			}
		if (i < 0) {
			if (n->sub_name == NULL)
				msg_Format ("node %d: missing function name\n", n->nk.id);
			else
				msg_Format ("node %d: no mapping for function %s\n",
							n->nk.id, n->sub_name);
			err++;
		}
	}
	return err ? 1 : 0;
}

#if 0
/*	cm_MakePvmHostFile ()
 *
 *	Convert a cost matrix into a pvm host file.
 *	Makes sure the localhost has an entry and is first.
 *
 *	XXX domain names would be a problem here... what to do...
 *
 *	Write an entry for each needed host.
 *	Returns:
 *		0 for ok.
 *		1 if can't write host file.
 *
 *	XXX This function should go away.  The pvm hosts file should
 *  be supplied by the user, and the default cost matrix should
 *  be derived from the pvm hosts file instead of the other way
 *  'round.
 */

int
cm_MakePvmHostFile (cm, fn)
	struct costm *cm;	/* cost matrix */
	char *fn;			/* filename */
{
	FILE *ff;
	int h;
	char buf[256];

	gethostname(buf, sizeof(buf)-1);
	buf[sizeof(buf)-1] = 0;

	if (!(ff = fopen(fn, "w")))
		return 1;

	fputs("#PVM host file written by HeNCE tool\n", ff);
	fprintf(ff, "%s\n", buf);
	for (h = cm->nhost; h-- > 0; )
		if (cm->hflag[h] & SUBHOST_HOST
		&& hostcmp(cm->hosts[h], buf))
			fprintf(ff, "%s\n", cm->hosts[h]);

	fclose(ff);
	return 0;
}
#endif

/*
 * return 1 if hostname is mentioned in cost matrix, else 0
 * This is called by cm_BuildPvmHostFile().
 *
 * Also, set the INPVM bit as a side effect.  cm_BuildPvmHostFile()
 * will call cm_ClearPvmFlag() before starting, and will call
 * cm_CheckPvmFlag() after building the host file.  If there is
 * a host that is defined in the cost matrix, but not in the pvm
 * hosts file, cm_CheckPvmFlag() will complain about it.
 */

int
cm_HostIsPresent (cm, hostname)
struct costm *cm;
char *hostname;
{
	int h;
	for (h = cm->nhost; h-- > 0; ) {
		if ((cm->hflag[h] & SUBHOST_HOST) &&
			hostcmp (cm->hosts[h], hostname) == 0) {
			cm->hflag[h] |= SUBHOST_INPVM;
			return 1;
		}
	}
	return 0;
}

void
cm_ClearPvmFlag (cm)
struct costm *cm;
{
	int h;
	for (h = cm->nhost; h-- > 0; )
		cm->hflag[h] &= ~SUBHOST_INPVM;
}

int
cm_CheckPvmFlag (cm)
struct costm *cm;
{
	int h;
	int error = 0;

	for (h = cm->nhost; h-- > 0; ) {
		if (cm->hosts[h] && (cm->hflag[h] & SUBHOST_INPVM) == 0) {
			msg_Format ("\
%s appears in the cost matrix, but not in the pvm hosts file.\n",
						cm->hosts[h]);
			error = -1;
		}
	}
	return error;
}

int
cm_SubIsPresent (cm, sub)
struct costm *cm;
char *sub;
{
	int s;
	for (s = cm->nsub; s-- > 0; ) {
		if ((cm->sflag[s] & SUBHOST_SUB) &&
			strcmp (cm->subs[s], sub) == 0) {
			return 1;
		}
	}
	return 0;
}


#if 0
/*	defcostnames()
 *
 *	Make sure that each node function in a graph has a cost matrix column.
 */

defcostnames(grf, cm)
	Node grf;
	struct costm *cm;
{
	node_t *n;

	if (cm) {
		for (n = grf; n; n = n->next)
			if (n->type == NTYPE_NO && n->fun && *n->fun)
				cm_AddEntry (cm, (char*)0, n->fun, 0);
	}
}
#endif

/*
 * make sure there is a "sub" entry in the cost matrix for every
 * function called by the HeNCE graph.  Don't call cm_AddEntry()
 * for any functions that area already there, since this has the
 * side effect of setting the isModified bit.
 *
 * returns 1 if any functions were added, else 0.
 */

int
cm_AddNamesFromGraph (g, cm)
Graph g;
struct costm *cm;
{
	TreeNode tn;
	int modified = 0;

	if (g->nlist == NULL || rb_Empty (g->nlist))
		return 0;
	for (tn = rb_First (g->nlist); tn != g->nlist; tn = rb_Next (tn)) {
		Node n = (Node) rb_Value (tn);

		if (n->node_type == NODE_NORMAL && n->sub_name && *n->sub_name) {
			int s;

			for (s = cm->nsub; s-- > 0; )
				if (cm->subs[s] && strcmp (cm->subs[s], n->sub_name) == 0)
					break;
			if (s <= 0) {
				cm_AddEntry (cm, (char *) NULL, n->sub_name, 0);
				modified = 1;
			}
		}
	}
	return modified;
}

#if 0
/*
 * Given a "master" pvm hosts file, emit a replica of that host file
 * containing only those hosts we need for this HeNCE program, as
 * determined by the cost matrix.
 *
 * if "in" is NULL, it means the pvm hosts file did not exist, so
 * we simply write out a host file containing everything in the
 * cost matrix.
 */

int
cm_BuildPvmHostFile (in, out, cm)
FILE *in, *out;
struct costm *cm;
{
	char line[1024];
	char hostname[1024];
	char myhostname[256];
	int h;

	/*
	 * reset all "host is in pvm" flags in the cost matrix
	 */

	cm_ClearPvmFlag (cm);

	/*
	 * pass 1: read and parse the master pvm hosts file, setting
	 * the "host is in pvm" flag for every host that appears and
	 * is also in the cost matrix.
	 */

	if (in) {
		while (fgets (line, sizeof (line), in)) {
			if (*line == '\0' || *line == '#')
				continue;
			sscanf (line, "%[^# \t\n] %*[^\n]", hostname);

			(void) cm_HostIsPresent (cm, hostname);
		}

		if (ferror (in) || ferror (out))
			return -1;
	}


	/*
	 * Now, prepare to write the temp pvm hosts file.  First the
	 * local host name is written to the file, then the names of
	 * any hosts that were in the cost matrix but were not in the
	 * master pvm hosts file.
	 */

	for (h = cm->nhost; h-- > 0; ) {
		if (cm->hosts[h] && (cm->hflag[h] & SUBHOST_INPVM) == 0) {
#if 0
			msg_Format ("\
warning: %s appears in the cost matrix, but not in the pvm hosts file.\n",
						cm->hosts[h]);
#endif
			fprintf (out, "%s\n", cm->hosts[h]);
#if 0
			fprintf (stderr, "%s\n", cm->hosts[h]);
#endif
		}
	}

	/*
	 * pass 2: copy the line from the pvm hosts file for any host
	 * (other than the local host) that exists in the cost matrix.
	 * if the local host is not in the cost matrix, copy it out last.
	 */

	if (in) {
		int found_local_entry = 0;

		rewind (in);
		while (fgets (line, sizeof (line), in)) {
			int need_this_host;

			if (*line == '\0' || *line == '#')
				continue;
			sscanf (line, "%[^# \t\n] %*[^\n]", hostname);

			need_this_host = 0;
			if (strcmp (hostname, "*") == 0)
				need_this_host = 1;
			else {
				if (host_is_me (hostname)) {
					need_this_host = 1;
					found_local_entry = 1;
				}
				if (cm_HostIsPresent (cm, hostname))
					need_this_host = 1;
			}
			fputs (line, out);
#if 0
			fputs (line, stderr);
#endif

		}
		if (ferror (in) || ferror (out))
			return -1;
		if (found_local_entry == 0) {
			char myhostname[1024];
			gethostname (myhostname);
			fputs (myhostname, out);
		}
	}
	return 0;
}
#else
/*
 * struct used to keep track of network addresses for each host
 * these are stored in an rb-tree, indexed by ip address.
 *
 * XXX if a host has > 1 ip address, each gets a separate copy
 * of this structure...fix this.
 */

struct host_addr_entry {
	struct in_addr in_addr;
	char *cm_hostname;			/* hostname used in cost matrix */
	int flags;
	int cm_host_num;			/* index into cost matrix */
#define HOST_IS_ME	 01
#define HOST_WRITTEN 02
};

/*
 * utility function to compare two IP addresses
 */

static int
compare_ip_addresses (k1, k2)
Key *k1, *k2;
{
	struct host_addr_entry *h1, *h2;
	h1 = k1->pkey;
	h2 = k2->pkey;
	return memcmp (&(h1->in_addr), &(h2->in_addr), sizeof (struct in_addr));
}

/*
 * look up the ip addresses for a particular hostname and store them
 * in the tree.  if (islocal == 1), we are looking up the result of
 * gethostname() so we will know our own address.  hostnum is the
 * index of the host in the current cost matrix -- or -1 if the
 * hostname didn't come from the cost matrix.
 */

static void
add_ip_addresses (t, hostname, hostnum, islocal)
Tree t;
char *hostname;
int hostnum;
int islocal;
{
	struct hostent *hp;
	int i;
	
	if ((hp = gethostbyname (hostname)) == NULL) {
		msg_Format ("can't find address for host %s -- ignored\n", hostname);
		return;
	}
	for (i = 0; hp->h_addr_list[i]; ++i) {
		struct host_addr_entry *ptr;
		Key k;
		TreeNode tn;
		int fnd = 0;

		k.pkey = hp->h_addr_list[i];
		tn = rb_Find (t, &k, compare_ip_addresses, &fnd);
		if (fnd) {
			ptr = rb_Value (tn);
			if (ptr->flags & HOST_IS_ME) {
				if (ptr->cm_hostname)
					free (ptr->cm_hostname);
				ptr->cm_hostname = strsave (hostname);
				ptr->cm_host_num = hostnum;
				continue;
			}
			msg_Format ("warning: host %s (net address %s) appears %s\n",
						hostname, inet_ntoa (hp->h_addr_list[i]),
						"more than once in cost matrix");
			continue;
		}
#if 0
		msg_Format ("adding host %s (%s)\n", hostname,
					inet_ntoa (hp->h_addr_list[i]));
#endif
		ptr = talloc (1, struct host_addr_entry);
		memcpy (&(ptr->in_addr), hp->h_addr_list[i], sizeof (struct in_addr));
		ptr->flags = islocal ? HOST_IS_ME : 0;
		ptr->cm_hostname = strsave (hostname);
		ptr->cm_host_num = hostnum;
		k.pkey = &(ptr->in_addr);
		rb_InsertBefore (tn, &k, ptr);
	}
}

/*
 * given a line from the user's pvm hosts file, write it out
 * (using the name given in the cost matrix rather than the one
 * in the host file) and set the HOST_WRITTEN bit.  Also mark
 * the host as being in the current pvm machine definition;
 * so that later on we can add any hosts that were in the
 * cost matrix but weren't in the pvm hosts file.
 *
 * return 1 iff the host is the local host.
 */

static int
write_host_if_needed (out, t, line, cm)
FILE *out;
Tree t;
char *line;
struct costm *cm;
{
	char *p;
	struct hostent *hp;
	int i;
	char tmp;

	for (p = line; *p && *p != ' ' && *p != '\t' && *p != '\n'; ++p);
	tmp = *p;
	*p = '\0';
	if ((hp = gethostbyname (line)) == NULL) {
		msg_Format ("warning: can't find address for host %s\n",
					line);
		return 0;
	}
	*p = tmp;
	for (i = 0; hp->h_addr_list[i]; ++i) {
		TreeNode tn;
		int fnd;
		Key k;

		k.pkey = hp->h_addr_list[i];
		tn = rb_Find (t, &k, compare_ip_addresses, &fnd);
		if (fnd) {
			struct host_addr_entry *ptr;

			ptr = rb_Value (tn);
			if (ptr->flags & HOST_WRITTEN) {
				msg_Format ("warning: host %s appears twice in pvm host file\n",
							line);
				return 0;
			}
			fprintf (out, "%s%s", ptr->cm_hostname, p);
			if (ptr->cm_host_num >= 0)
				cm->hflag[ptr->cm_host_num] |= SUBHOST_INPVM;
			ptr->flags |= HOST_WRITTEN;
			return ptr->flags & HOST_IS_ME ? 1 : 0;
		}
	}
	return 0;
}

/*
 * return 1 iff the host named is the local host.
 */

static int
host_is_me (t, name)
Tree t;
char *name;
{
	struct hostent *hp;
	int i;

	if ((hp = gethostbyname (name)) == NULL)
		return 0;
	for (i = 0; hp->h_addr_list[i]; ++i) {
		TreeNode tn;
		int fnd;
		Key k;

		k.pkey = hp->h_addr_list[i];
		tn = rb_Find (t, &k, compare_ip_addresses, &fnd);
		if (fnd) {
			struct host_addr_entry *ptr;

			ptr = rb_Value (tn);
			if (ptr->flags & HOST_IS_ME)
				return 1;
		}
	}
	return 0;
}

/*
 * delete the tree and every object allocated by entries in the tree.
 */

static void
delete_tree (t)
Tree t;
{
	TreeNode tn, next_tn;
	struct host_addr_entry *ptr;

	for (tn = rb_First (t); tn != t; tn = next_tn) {
		ptr = (struct host_addr_entry *) rb_Value (tn);
		free (ptr->cm_hostname);
		free ((void *) ptr);
		next_tn = rb_Next (tn);
		rb_DeleteNode (tn);
	}
	rb_FreeTree (t);
}

/*
 * given the current cost matrix and optionally a "master" pvm hosts
 * file, emit a replica of that hosts file containing only those hosts
 * specified by the cost matrix (plus the local host).
 *
 * If "in" is NULL, it means the htool user did not specify a pvm
 * hosts file, so we just write out a host file containing everything
 * in the cost matrix.
 *
 * returns 0 on success, -1 on error.
 */

int
cm_BuildPvmHostFile (in, out, cm)
FILE *in, *out;
struct costm *cm;
{
	int h, s;
	int found_local;
	Tree hostAddrList;
	struct hostent *hp;
	int have_localhost;
	char myhostname[1024];
	char line[1024];
	/*
	 * This flag says whether we have seen an asterisk in the
	 * user-supplied hosts file.  If we haven't seen one, don't output
	 * one before supplying entries that are in the cost matrix but
	 * not in the hosts file.  Some versions of pvm choke on asterisks.
	 *
	 * This is named after Nathan Hale who said...
	 * "I regret that I have but one asterisk for my country".
	 */
	int nathan = 0;

	/*
	 * for each host in the cost matrix, reset the "host is needed"
	 * and the "host is in pvm" bits...then set the "host is needed"
	 * bit if there are any nonzero entries in the cost matrix.
	 *
	 * XXX maybe should check first to see if any of the named subs
	 * are actually used in the graph?
	 */
	for (h = 0; h < cm->nhost; ++h) {
		cm->hflag[h] &= ~(SUBHOST_HOST | SUBHOST_INPVM);
		if (cm->hosts[h] == NULL)
			continue;
		for (s = 0; s < cm->nsub; ++s) {
			if (cm->mat[h * cm->nsub + s] != 0) {
				cm->hflag[h] |= SUBHOST_HOST;
				break;
			}
		}
	}
	/*
	 * start building a tree to keep track of everybody's IP address
	 * initialize by putting the local host in the tree.
	 */
	hostAddrList = rb_MakeTree ();

	gethostname (myhostname, sizeof (myhostname));
	add_ip_addresses (hostAddrList, myhostname, -1, 1);

	/*
	 * now add the addresses for each host needed to run the program
	 */
	for (h = 0; h < cm->nhost; ++h) {
		if (cm->hflag[h] & SUBHOST_HOST)
			add_ip_addresses (hostAddrList, cm->hosts[h], h, 0);
	}

	/*
	 * scan each entry in the "master" pvm hosts file.  if we need
	 * that host to run the program, copy its entry to the hosts file
	 * we are going to feed to pvm.  Keep track of whether we've
	 * actually seen the local host -- pvm requires that it be there.
	 * If the local host is in the master hosts file we want to use
	 * it (if only so the ep= option will have an effect), but if
	 * the host doesn't appear there we have to copy out a dummy line
	 * later.
	 */
	have_localhost = 0;
	if (in) {
		while (fgets (line, sizeof (line), in) != NULL) {
			if (*line == '#')
				continue;
			else if (*line == '*') {
				fprintf (out, "%s", line);
				nathan = 1;
				continue;
			}
			if (write_host_if_needed (out, hostAddrList, line, cm))
				have_localhost = 1;
		}
	}
	/*
	 * for each host defined in the cost matrix,
	 * if that host is not already in the pvm, write out a line containing
	 * that host's name to the pvm host file.  Finally, if we haven't
	 * output the name of the local host, do that last.
	 */
	if (nathan)
		fprintf (out, "*\n");		/* disable any defaults */

	for (h = 0; h < cm->nhost; ++h) {
		if ((cm->hosts[h] == NULL) ||
			(cm->hflag[h] & SUBHOST_HOST) == 0 ||
			(cm->hflag[h] & SUBHOST_INPVM))
			continue;
		if (have_localhost == 0 && host_is_me (hostAddrList, cm->hosts[h]))
			have_localhost = 1;
		fprintf (out, "%s\n", cm->hosts[h]);
	}
	if (have_localhost == 0)
		fprintf (out, "%s\n", myhostname);

	/*
	 * clean up
	 */
	delete_tree (hostAddrList);
	return 0;
}
#endif

int
cm_WriteHosts (fp, cm)
FILE *fp;
struct costm *cm;
{
	int h;

	for (h = cm->nhost; h-- > 0; ) {
		if (cm->hosts[h] && ((cm->hflag[h] & SUBHOST_EN) != 0)) {
#if 0
			fprintf (stderr, "%s\n", cm->hosts[h]);
#endif
			fprintf (fp, "%s\n", cm->hosts[h]);
		}
	}
	return 0;
}


/*
 * Local variables:
 * tab-width:4
 * End:
 */
