/*
 * $Source: /a/thud/chalk/homes/moore/src/hence/master/RCS/anc.c,v $
 * $Revision: 1.1 $
 * $Date: 1992/04/08 05:51:36 $
 * $Author: moore $
 *
 * Anc.c defines routines to deal with getting information from ancestor 
 * nodes. This includes maintaining an ordered list of ancestors for the 
 * reading of parmaters, and maintaining an ordered list of 
 * ancestor/array pairs for the reading of array parameter values. 
 * 
 * Anc.c works with three basic structs: 
 * 
 * The "Anc" (defined in hence.h): Each node has a pointer to one of 
 * these, which gets initialized by their parent nodes when the parents 
 * complete execution. It contains pointers to two ordered lists. The 
 * first is "order", which is an ordered list of ancestor pointers. It is
 * ordered by the "distance" of the ancestor to the node, where distance 
 * is defined to be the longest path from the ancestor to the node. The 
 * second list is "elts", which is keyed on the ancestor's node pointer 
 * (used as an integer). Its value is a pointer to the rb_node in "order"
 * which contains the node. New_anc_list, free_anc_list, and 
 * update_anc_list are the routines dealing with Anc's. They are 
 * commented below. 
 * 
 * The Array_anc (defined in hence.h): Each node has a pointer to a list 
 * of Array_anc structs. This list contains the ancestor nodes which the 
 * current node must query to get its parameter values. This list is 
 * passed as initial information to the pvm process of the node, which 
 * traverses it (in slave.c), and queries the ancestors to initialize its
 * array elements. Each array_anc struct contains fields to identify the 
 * ancestor node (anc, sub_name, pvminum), and fields to identify the 
 * parameter (apnum and mypnum). Note that since parameters are 
 * referenced by their numbers, array_anc structs cannot be initialized 
 * until after the parameters are numbered (in execute.c). Make_new_aa, 
 * and put_into_aa_list are the specific routines dealing with 
 * array_anc's. 
 * 
 * The tmp_array_anc (defined below): The array_anc field is initialized 
 * using two passes. In the first pass, which is done in one of the first
 * passes of reduce_params (in execute.c), an initial list of condidate 
 * array/ancestor pairs is created. This is a list of tmp_array_anc 
 * structs. After the first pass, then a second pass is made where each 
 * candidate pair is considered and included if and only if the ancestor 
 * is holding a part of the array that the node desires. In that case, an
 * Array_anc struct is created for that pair, and put into the node's 
 * array_anc list. Add_array_anc() is used by execute.c to consider a new
 * candidate array/ancestor pair. Make_aa_list() is used to make the 
 * array_anc list from the tmp_array_anc list. 
 */

#include <stdio.h>
#include "std.h"
#include "rb.h"
#include "dlist.h"
#include "hence.h"
#include "rbhelp.h"

/* ---------------------------------------------------------------------- */
/* Dealing with Anc's */

/* ---------------------------------------------------------------------- */
/* New_anc_list allocates and returns a new Anc struct */

Anc new_anc_list()
{
  Anc a;
 
  a = talloc(struct anc_list, 1);
  a->elts = make_rb();
  a->order = make_rb();
  return a;
}

/* ---------------------------------------------------------------------- */
/* Free_anc_list frees an Anc struct */

free_anc_list(a)
Anc a;
{
  rb_free_tree(a->elts);
  rb_free_tree(a->order);
  free(a);
}

/* ---------------------------------------------------------------------- */
/* Update_anc_list takes a node's Anc list a, and an ancestor node n, and
 * merges n's Anc list with a.  All the distances of ancestors of n are 
 * incremented by 1 in a only if n is a normal node. If it is a 
 * special-purpose node, then the ancestors in a are set to the same 
 * distance as in n->anc. 
 */ 

update_anc_list(a, n)  
Anc a;
Node n;
{
  Rb_node rn, ro, r;
  int fnd, level, inc;
  Node anc;
  
  /* Try to find a in n's Anc list.  If it's there, then abort this */

  if (n->node_type == NORMAL) {
    inc = 1;
    rn = rb_find_ikey_n(a->elts, (int) n, &fnd);
    if (fnd) return;  /* Parent is already there -- it doesn't need updating */
    ro = rb_inserti_n(a->order, inc, n);
    (void) rb_insert_b_i_r(rn, (int) n, ro);
  } else {
    inc = 0;
  }
  
  /* Otherwise, traverse n's ancestors, and put them into a */

  rb_traverse(r, n->anc->order) {
    anc = (Node) r->v.val;
    level = r->k.ikey;
    rn = rb_find_ikey_n(a->elts, (int) anc, &fnd);
    if (!fnd) {
      ro = rb_inserti_n(a->order, level + inc, anc);
      (void) rb_insert_b_i_r(rn, (int) anc, ro);
    } else {
      ro = (Rb_node) rn->v.val; 
      if (ro->k.ikey <= level) {
        rb_delete_node(ro);
        ro = rb_inserti_n(a->order, level + inc, anc);
        rn->v.val = (char *) ro;
      }
    }
  }
}


/* ---------------------------------------------------------------------- */
/* Dealing with Array_anc's */

/* ---------------------------------------------------------------------- */
/* Make_new_aa allocates a new array_anc list. */

static Array_anc make_new_aa(p, a, ap)
Node a;
Param ap, p;
{
  Array_anc aa;

  aa = talloc(array_anc, 1);
  aa->anc = (void *) a;
  aa->sub_name = a->sub_name;
  aa->pvminum = a->pvminum;
  aa->apnum = ap->num;
  aa->mypnum = p->num;
  return aa;
}

/* ---------------------------------------------------------------------- */ 
/* Put_into_aa_list takes a node and its main array parameter, and an 
 * ancestor and its array parameter, and puts the proper entries into 
 * n->aa so that when n is invoked, it will query the correct nodes for 
 * the correct initialization values. There are basically four cases that
 * can happen for a given set of arguments: 
 * 
 * Case 1:  Both n and a contain the entire array. In this case, an 
 * array_anc struct is created and put into n->aa telling n to get all of
 * the array from a.  Moreover, p->state is set to 1, signifying that n 
 * doesn't need to get parameter p from any other ancestors, as all of it
 * comes from a.  
 * 
 * Case 2:  N uses the entire array, but a only uses pieces of it (these 
 * are the pieces found in ap->a->p). In that case, make an array_anc for
 * each piece in ap->a->p. 
 * 
 * Case 3:  N only uses pieces of the array, but a uses all of it.  In 
 * that case, for every piece of the array in p->a->p, create an 
 * array_anc for it, and set its state to 1.  
 * 
 * Case 4:  N and a both just use pieces of the array. In this case, 
 * consider every pair of pieces from p->a->p and ap->a->p, and calculate
 * whether they have any elements in common. If so, then make an 
 * array_anc for it.  
 * 
 * The array_ancs are always pushed in the front of n->aa. This is so 
 * that n will traverse the most distant ancestors first, and thus 
 * perform the correct overwriting of initialization values. */ 

static put_into_aa_list(n, p, a, ap)
Node n, a;
Param p, ap;
{
  int i, intersect, cover;
  Rb_node r, ar;
  Param p2, ap2;
  Exp ed, aed;

  if (p->state == 1) return;

  /* Get rid of the initializing expression -- since this will be 
     read from ancestors */

  if (p->val != ENULL) {
    free_exp(p->val);
    p->val = ENULL;
  }

  /* First check for consistency with ancestors */

  for (i = 0; i < p->a->ndims; i++) {
    if (p->a->dims[i]->args[1]->val.i != ap->a->dims[i]->args[1]->val.i) {
      error_non_matching_dim(n, p, a, ap, i);
    }
  }
  
  /* First case: The node and its ancestor uses the entire array.  
     In this case, have the ancestor send everything it has, and 
     and mark as done if the ancestor has the whole array */

  if (p->io.used && ap->io.used) {
    dl_insert_a_aa(n->aa, make_new_aa(p, a, ap));
    p->state = 1;
    return;
  }
    
  /* Second case: Node uses everything, but the ancestor doesn't.
     Then, traverse all the ancestor's pieces and send them. */

  if (p->io.used) {
    rb_traverse(r, ap->a->p) {
      ap2 = (Param) r->v.val;
      dl_insert_a_aa(n->aa, make_new_aa(p, a, ap2));
    }
    return;
  }

  /* Third case: Ancestor contains the entire array, but the node doesn't
     use the entire array.  In this case, traverse the 2nd level, and 
     put have each parameter there get its value from the ancestor's 
     first level. */

  if (ap->io.used) {
    rb_traverse(r, p->a->p) {
      p2 = (Param) r->v.val;
      if (p2->state != 1) {
        dl_insert_a_aa(n->aa, make_new_aa(p2, a, ap));
        p2->state = 1;
      } 
    }
    return;
  }

  /* Last case: Neither the node nor the ancestor contains the entire 
     array.  Traverse the 2nd level of both arrays, and put into aa 
     whatever can be put there. */

  rb_traverse(r, p->a->p) {
    p2 = (Param) r->v.val;
    if (p2->state != 1) {
      rb_traverse(ar, ap->a->p) {
        ap2 = (Param) ar->v.val;
        intersect = (p2->state != 1);
        cover = intersect;
        for (i = 0; intersect && i < ap->a->ndims; i++) {
          ed = p2->a->dims[i];
          aed = ap2->a->dims[i];
          intersect = ((h_l_bnd(ed) <= h_h_bnd(aed)) &&
                       (h_l_bnd(aed) <= h_h_bnd(ed))) ;
          cover = ((cover) && 
                   (h_l_bnd(aed) <= h_l_bnd(ed)) &&
                   (h_h_bnd(aed) >= h_h_bnd(ed))) ;
        }
        if (intersect) {
          dl_insert_a_aa(n->aa, make_new_aa(p2, a, ap2));
        }
        if (cover) p2->state = 1;
      }
    }
  }
  return;
}


/* ---------------------------------------------------------------------- */
/* Dealing with tmp_array_anc's */

typedef struct {
  Node a;
  Param ap;
  Param p;
} tmp_array_anc;

dl_insert_b_taa(d, taa)
Dlist d;
tmp_array_anc *taa;
{
  dl_insert_b(d, (char *) taa);
}

/* ---------------------------------------------------------------------- */
/* Add_array_anc adds an ancestor/array pair to the tmp_array_anc list in
 * list */

add_array_anc(list, anc, pm, anc_param)
Dlist list;
Node anc;
Param pm;
Param anc_param;
{
  tmp_array_anc *taa;

  taa = talloc(tmp_array_anc, 1);
  taa->a = anc;
  taa->ap = anc_param;
  taa->p = pm;
  dl_insert_b_taa(list, taa);
}

/* Make_aa_list() initializes n->aa from the ancestor/array pairs in 
 * tmp_aa */

make_aa_list(n, tmp_aa)
Node n;
Dlist tmp_aa;
{
  Dlist d;
  tmp_array_anc *taa;

  dl_traverse(d, tmp_aa) {
    taa = (tmp_array_anc *) d->val;
    put_into_aa_list(n, taa->p, taa->a, taa->ap);
    free(taa);
  }
  dl_delete_list(tmp_aa);
}


/* ---------------------------------------------------------------------- */
/* Error flagging */
/* ---------------------------------------------------------------------- */

static error_non_matching_dim(n, p, a, ap, i)
Node n, a;
Param p, ap;
int i;
{
  param_error_header(n, p);
  fprintf(stderr, "  Dimension %d doesn't match its ancestor's:\n    ", i);
  fprint_node_id(stderr, n);
  fprintf(stderr, ": ");
  fprint_exp(stderr, p->a->dims[i]);
  fprintf(stderr, "\n    ");
  fprint_node_id(stderr, a);
  fprintf(stderr, ": ");
  fprint_exp(stderr, ap->a->dims[i]);
  fprintf(stderr, "\n");
  bail(CNULL);
}

