/*
 *	HeNCE Tool
 *
 *	trace.c - Graphical execution tracing.
 *		This is where everything goes that happens in the trace panel.
 *
 *	Sep 1991  Robert Manchek  manchek@CS.UTK.EDU.
 *
 *	Revision Log
 *
$Log: trace.c,v $
 * Revision 1.9  1992/12/09  05:06:12  moore
 * nuke gratuitous printfs.
 * tweak some type declarations, add some casts, to make alpha happy.
 *
 * Revision 1.8  1992/09/18  01:53:13  moore
 * add label widget displaying elapsed time or current trace event
 * trace is now time-scaled.
 * 'elapsed' is now 'print_time"
 * 'elapsed_time' and 'after' copied from histo.c; need to move them
 *     elsewhere at some future date.
 * changed 'serial' to 'millisecs' in trace events
 * 'usageIsVisible' changed to 'histogramExists'
 *
 * Revision 1.7  1992/08/24  22:05:57  moore
 * take out debugging printfs
 *
 * Revision 1.6  1992/08/05  21:59:23  moore
 * made mods to handle fractional (msec) time.  Uses new trace file format.
 *
 * Revision 1.5  1992/07/10  21:56:03  moore
 * hacked-in support for histogram window
 *
 * Revision 1.4  1992/07/10  21:53:23  moore
 * fix nasty memory trashing bug in elapsed()
 *
 * Revision 1.3  1992/05/19  05:39:16  moore
 * add casts to make MetaWare hc happy.
 *
 * Revision 1.2  1992/04/08  06:50:30  moore
 * incorporates changes up to 8 April 1992
 *
 *
 */

/*
 * TODO:
 * - handle special nodes
 * - usage graph
 * - trace in either direction
 */

#define USAGE_GRAPH
#define TIME_LABEL
#define TIME_SCALE

#include <math.h>
#include <sys/types.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include "xincl.h"
#include "rb.h"
#include "param.h"
#include "exp.h"				/* XXX Yuk! */
#include "graph.h"
#include "graphP.h"
#include "xcomn.h"
#include "comn.h"
#include "costmat.h"
#include "hostmap.h"
#include "tf.h"
#include "histo.h"

#include "xbm/legend.xbm"
#include "xbm/fwd.xbm"
#include "xbm/fwdstep.xbm"
#include "xbm/hostmap.xbm"
#include "xbm/monitor.xbm"
#include "xbm/rewind.xbm"
#include "xbm/stop.xbm"
#ifdef USAGE_GRAPH
#include "xbm/usage.xbm"
#endif

#ifndef errno
extern int errno;				/* 4.3 BSD needs this */
#endif

/*******************
*  from elsewhere  *
*                  *
*******************/

extern Widget traceButton;		/* from widmain.c */
extern Graph scratchgraph;		/* from widmain.c */
extern struct costm *costmatrix; /* from widmain.c */



/*************
*  exported  *
*            *
*************/

void start_trace();
int stop_trace();
Widget hostsWin = NULL;			/* used by hostmap.c */
Widget hostsPop = NULL;			/* used by hostmap.c */
Widget usagePop = NULL;

/***********
*  global  *
*          *
***********/

static void rewind_cb(), fwd_cb(), stop_cb(), speed_cb();
#if 0
static void redraw_cb();
#endif
static void fwdstep_cb();
static void monitor_cb();
static Widget cre_cmdpanel(), cre_nodecanvas();
static void graph_ev();
static void trace_refresh();
static void hosts_cb();
#ifdef USAGE_GRAPH
static void usage_cb();
#endif

static Widget traceCmd = 0;
static Widget traceGraph = 0;
static Widget traceHost = 0;
static Widget speedKnob = 0;
static Widget rewindButton = 0;
static Widget stopButton = 0;
static Widget fwdButton = 0;
static Widget fwdstepButton = 0;
#ifdef TIME_LABEL
static Widget timeLabel = 0;
#endif

static Arg args[16];
#if 0
static char crud[1024];
#endif
static Window graphWin;
#ifdef TIME_SCALE
static float time_scale = 1.0;
#define TIMER_ALARM 0
#define READ_AGAIN 1
#define FIRST_EVENT 2
#else
static int animat_delay = 0;
#endif
static XtCallbackRec callback[2] = { { 0, 0 }, { 0, 0 } };
static Graph tracegraph = 0;
static int monitorMode = 0;
Boolean histogramExists = False;


/*********************
 *                   *
 *  widget creation  *
 *                   *
 *********************/

/*
 *	start_trace()
 *
 *	Start method for trace mode.  Create any widgets and manage em.
 *	Set default graph file name from main module.
 */

static int checkThings ();

void
start_trace()
{
	static int once = 1;
	Widget w;
#if 0
    register int h;
#endif

	if (once) {
		once = 0;
		tracegraph = gr_New ();
		/* XXX what if there's no trace file name? */
		/* tf_SetFile (traceFileName); */
	}
	/*	gr_Free (tracegraph); */
	/* tracegraph = gr_CopyGraph (scratchgraph); */

	XawFormDoLayout (mainForm, False);
	w = cre_cmdpanel (mainCmd);
	XtManageChild (traceCmd);
	w = cre_nodecanvas (w);
	XtManageChild (traceGraph);
	XawFormDoLayout (mainForm, True);
	set_togg (traceButton, 1);

	checkThings ();
}

/*
 *	stop_trace()
 *
 *	Stop method for trace mode.  Ask if want to save graph file or
 *	cancel stop.
 *	Returns 0 if successfully stopped, else 1.
 */

static void stopTrace ();

int
stop_trace()
{
#if 0
	int i;
#endif

	/* This serves to delete the trace event timer */
	stopTrace ();

	XawFormDoLayout (mainForm, False);
	XtUnmanageChild (traceCmd);
	XtUnmanageChild (traceGraph);
	XawFormDoLayout (mainForm, True);
	set_togg (traceButton, 0);

	return 0;
}

static Widget
cre_nodecanvas(below)
Widget below;
{
	Widget retw;
	Widget w;
	int n;

	if (traceGraph)
		return traceGraph;
	n = 0;
	XtSetArg(args[n], XtNfromVert, (XtArgVal)below); n++;
	XtSetArg(args[n], XtNallowHoriz, (XtArgVal)True); n++;
	XtSetArg(args[n], XtNallowVert, (XtArgVal)True); n++;
	retw = traceGraph = XtCreateWidget("traceNodeVp", viewportWidgetClass,
			mainForm, args, n);

	n = 0;
	XtSetArg(args[n], XtNbackground, (XtArgVal)(mycolor[COLBGND])); n++;
	w = XtCreateManagedWidget("traceNodeCan", widgetClass,
		traceGraph, args, n);

/* XXX should use something other than base widget class here so we
   could set more nice resources... */

	XtRealizeWidget(traceGraph);
	graphWin = XtWindow(w);
	XtAddEventHandler(w,
		StructureNotifyMask|ExposureMask, False,
		graph_ev, (XtPointer)0);

	return traceGraph;
}

static Widget
cre_cmdpanel(below)
	Widget below;
{
	Widget retw;
	Widget w;
#if 0
	Widget w2, w3;
#endif
	Widget w4;
	int n, saved_n;

	/*
	 * translation table for radio group
	 */

	XtTranslations radio_ttab = XtParseTranslationTable("\
<EnterWindow>:highlight(Always)\n\
<LeaveWindow>:unhighlight()\n\
<Btn1Down>,<Btn1Up>:set()notify()");

	if (traceCmd)
		return traceCmd;

	n = 0;
	XtSetArg(args[n], XtNfromVert, (XtArgVal)below); n++;
	XtSetArg(args[n], XtNborderWidth, (XtArgVal)0); n++;
	XtSetArg(args[n], XtNhorizDistance, (XtArgVal)0); n++;
	XtSetArg(args[n], XtNresizable, (XtArgVal)True); n++;
	retw = traceCmd = XtCreateWidget("traceCmd", formWidgetClass,
			mainForm, args, n);

	n = 0;
	XtSetArg(args[n], XtNleft, XtChainLeft); n++;
	XtSetArg(args[n], XtNright, XtChainLeft); n++;
	XtSetArg(args[n], XtNtop, XtChainTop); n++;
	XtSetArg(args[n], XtNbottom, XtChainTop); n++;

	/* BEGIN RADIO GROUP */
	saved_n = n;
	XtSetArg (args[n], XtNtranslations, radio_ttab); n++;

	w = cre_tog_but ("rewind", traceCmd, args, n,
					 (Widget) NULL, (Widget) NULL, (char *) NULL,
					 rewind_bits, rewind_width, rewind_height,
					 rewind_cb, 0, 0);
	rewindButton = w;
	w4 = w;
	XtSetArg (args[n], XtNradioGroup, w4); n++;

	w = cre_tog_but ("stop", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 stop_bits, stop_width, stop_height, stop_cb, 0, 1);
	stopButton = w;

	w = cre_tog_but ("fwdstep", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 fwdstep_bits, fwdstep_width, fwdstep_height,
					 fwdstep_cb, 0, 0);
	fwdstepButton = w;

	w = cre_tog_but ("fwd", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 fwd_bits, fwd_width, fwd_height, fwd_cb, 0, 0);
	fwdButton = w;

	n = saved_n;
	/* END RADIO GROUP */

	/* SPEED SLIDER */
	saved_n = n;
	XtSetArg (args[n], XtNfromHoriz, w); n++;
	XtSetArg(args[n], XtNorientation, XtorientHorizontal); n++;
	XtSetArg(args[n], XtNwidth, 200); n++;
	XtSetArg(args[n], XtNheight, 20); n++;
	{
#ifdef TIME_SCALE
		float f = 0.5;
#else
		float f = 1.0;
#endif

		if (sizeof(f) > sizeof(args[0].value)) {
			XtSetArg(args[n], XtNtopOfThumb, &f); n++;
			XtSetArg(args[n], XtNshown, &f); n++;
		} else {
			args[n].name = XtNtopOfThumb;
			*(float*)&args[n].value = f; n++;
			args[n].name = XtNshown;
			*(float*)&args[n].value = f; n++;
		}
	}
	callback[0].callback = speed_cb;
	XtSetArg(args[n], XtNjumpProc, (XtArgVal)callback); n++;
	w = speedKnob = XtCreateManagedWidget("execSpeed",
		scrollbarWidgetClass, traceCmd, args, n);
	n = saved_n;
	/* END SPEED SLIDER */

#if 0
	/*
	 * XXX there is no bitmap for redraw button.  Do we need a
	 * redraw button anyway?
	 */
	w = cre_cmd_but("redraw", traceCmd, args, n,
					(Widget) NULL, w, "redraw",
					(char *) NULL, 0, 0, redraw_cb, 0, 0);
#endif

	w = cre_tog_but ("monitor", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 monitor_bits, monitor_width, monitor_height,
					 monitor_cb, 0, 0);
	w = cre_tog_but ("hosts", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 hostmap_bits, hostmap_width, hostmap_height,
					 hosts_cb, 0, 0);
#ifdef USAGE_GRAPH
	w = cre_tog_but ("usage", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 usage_bits, usage_width, usage_height, usage_cb, 0, 0);
#endif /* USAGE_GRAPH */

#ifdef TIME_LABEL
	timeLabel = XtVaCreateManagedWidget ("timeLabel", labelWidgetClass,
										 traceCmd,
										 XtNleft, XtChainLeft,
										 XtNright, XtChainRight,
										 XtNtop, XtChainTop,
										 XtNbottom, XtChainBottom,
										 XtNresizable, False,
										 XtNlabel, "00:00:00.000",
										 XtNheight, 20,
										 XtNfromHoriz, (XtArgVal) w,
										 NULL);
	w = timeLabel;
#endif

	XtRealizeWidget(traceCmd);
	return traceCmd;
}

/****************************
 *  trace widget callbacks  *
 *                          *
 ***************************/

static enum { REVERSE, FORWARD } traceDirection = FORWARD;
static int traceStopped = 1;
static XtIntervalId timerId;
static int traceFileHasChanged = 1;
static int graphFileHasChanged = 1;

/*
 * do any checks that are necessary before we start tracing
 * return 0 if okay, nonzero otherwise.
 */

static int
checkThings ()
{
	if (traceFileHasChanged) {
		hmap_NukeHosts ();
		if (traceFileName == NULL || *traceFileName == '\0') {
			msg_Format ("No trace file.\n");
			return 1;
		}
		if (tf_SetFile (traceFileName) == EOF) {
			msg_Format ("Can't read trace file \"%s\"\n", traceFileName);
			return 1;
		}
		traceFileHasChanged = 0;
	}
	if (graphFileHasChanged) {
		/*
		 * unlike trace files, graph files are loaded by the routine
		 * that handles "graph" button events.  So we just check for
		 * empty graphs here.
		 */
		gr_Free (tracegraph);
		tracegraph = gr_CopyGraph (scratchgraph);
		if (gr_Empty (tracegraph)) {
			msg_Format ("Graph is empty.\n");
			return 1;
		}
		trace_refresh ();
		/*
		 * In addition to checking for errors in the graph,
		 * gr_Critic() also marks pairs of special nodes
		 */
		if (gr_Critic (tracegraph) != 0) {
			msg_Format ("Graph has errors.\n");
			return 1;
		}
		graphFileHasChanged = 0;
	}
	return 0;
}

/*
 * This function is called whenever we change a trace file.  It is
 * called with x == 0 if the caller wants to see if its okay to change the
 * file, and with x == 1 if the caller wants to let us know it's been
 * changed.  If x == 0 and we will not allow the file to be changed,
 * print out an appropriate message.
 */

int
trace_ChangeTraceFile (x)
int x;
{
	switch (x) {
	case 0:
		if (!traceStopped) {
			msg_Format ("Can't change trace file while trace is running.\n");
			return 0;
		}
		return 1;
	case 1:
		traceFileHasChanged = 1;
		break;
	}
}

int
trace_ChangeGraphFile (x)
int x;
{
	switch (x) {
	case 0:
		if (!traceStopped) {
			msg_Format ("Can't change graph file while trace is running.\n");
			return 0;
		}
		return 1;
	case 1:
		graphFileHasChanged = 1;
		break;
	}
}


static void
stopTrace ()
{
	extern Widget traceFileButton; /* in widmain.c */
	extern Widget grfButton;	/* in widmain.c */

	if (!traceStopped) {
#ifndef TIME_SCALE
		XtRemoveTimeOut (timerId);
#endif
		traceStopped = 1;
#if 1
		XtVaSetValues (traceFileButton, XtNsensitive, True, NULL);
		XtVaSetValues (grfButton, XtNsensitive, True, NULL);
#endif
	}
}


static void Tick ();

static void
startTrace ()
{
	extern Widget traceFileButton; /* in widmain.c */
	extern Widget grfButton;	/* in widmain.c */

	if (traceStopped) {
#ifdef TIME_SCALE
		/*
		 * do the first tick manually.  After that, Tick() will schedule
		 * the next tick according to the value of time_scale.
		 */
		traceStopped = 0;
		XtVaSetValues (traceFileButton, XtNsensitive, False, NULL);
		XtVaSetValues (grfButton, XtNsensitive, False, NULL);
		Tick ((XtPointer) FIRST_EVENT, 0);
#else
		timerId = XtAppAddTimeOut (context, animat_delay, Tick, NULL);
		traceStopped = 0;
		XtVaSetValues (traceFileButton, XtNsensitive, False, NULL);
		XtVaSetValues (grfButton, XtNsensitive, False, NULL);
#endif
	}
}

static char *
print_time (t)
struct EventTime *t;
{
	static char buf[30];
	char *ptr;
	int msec = t->millisecs;
	long secs = t->secs;

#if 0
	sprintf (buf, "%0d.%03d",
			 secs, msec);
	return buf;
#else
	ptr = buf + sizeof(buf);
	
	*--ptr = '\0';
	*--ptr = msec % 10 + '0';
	msec /= 10;
	*--ptr = msec % 10 + '0';
	msec /= 10;
	*--ptr = msec % 10 + '0';
	msec /= 10;
	*--ptr = '.';
	*--ptr = (secs % 10) + '0';
	secs /= 10;
	*--ptr = (secs % 6) + '0';
	secs /= 6;
	*--ptr = ':';
	*--ptr = (secs % 10) + '0';
	secs /= 10;
	*--ptr = (secs % 6) + '0';
	secs /= 6;
	*--ptr = ':';
	*--ptr = (secs % 10) + '0';
	secs /= 10;
	*--ptr = (secs % 10) + '0';
	while ((secs /= 10) > 0)
		*--ptr = (secs % 10) + '0';
		   
	return ptr;
#endif
}


/*
 * XXX this is copied from histo.c.  Eventually put all of the event/
 * time management stuff in a separate file.
 */

static struct EventTime *
elapsed_time (t1, t2)
struct EventTime *t1, *t2;
{
    static struct EventTime result;

	result.secs = 0;
    result.millisecs = t2->millisecs - t1->millisecs;
    if (result.millisecs < 0) {
		result.millisecs += 1000;
		result.secs = -1;
    }
    result.secs = result.secs + t2->secs - t1->secs;
#ifdef DEBUG
	if (result.millisecs < 0) {
		fprintf (stderr, "elapsed_time (%d.%03d %d.%03d) -> %d.%03d\n",
				 t1->secs, t1->millisecs,
				 t2->secs, t2->millisecs,
				 result.secs, result.millisecs);
	}
#endif
    return &result;
}

/*
 * XXX also stolen from histo.c
 */

static int 
after (t1, t2)
struct EventTime *t1, *t2;
{
    if (t1->secs > t2->secs)
		return 1;
    else if (t1->secs < t2->secs)
		return 0;
    else
		return (t1->millisecs > t2->millisecs);
}

/*
 * Come here to process a trace event.
 *
 * XXX this works fine for dags as is, but doesn't understand how
 * graphs are expanded when special nodes occur.  Currently, it
 * only displays the first instance of a node.
 *
 */

static struct EventTime epoch;

static void
traceEvent (tev)
struct trace_event *tev;
{
	Node n;
	struct EventTime now;
	struct EventTime *delta;

	if (tev->id == -1 && tev->inst == -1)
		;
	else {
		n = gr_GetNodeByIdInst (tracegraph, tev->id, 0);
		if (n == (Node) NULL)
			return;
	}

	if (tev->event_type == TEV_START) {
		epoch.secs = tev->timestamp.secs;
		epoch.millisecs = tev->timestamp.millisecs;
	}
	now.secs = tev->timestamp.secs;
	now.millisecs = tev->timestamp.millisecs;
		
	delta = elapsed_time (&epoch, &now);
	msg_Format ("%s %d/%d %s %c %d %d\n",
				print_time (delta),
				tev->id, tev->inst,
				tf_GetEventName (tev->event_type),
				tev->tf ? 'T' : 'F',
				tev->lo, tev->hi);

#ifdef TIME_LABEL
	set_label (timeLabel, print_time (delta));
#endif
	if (monitorMode == 1 && histogramExists)
		hist_SetEndOfTime ((struct EventTime *) &(tev->timestamp));
#ifdef HAIRLINE
	if (histogramExists)
		hist_SetHairLine ((struct EventTime *) &(tev->timestamp), 1);
#endif

	switch (tev->event_type) {
	case TEV_START:
		hmap_ResetHosts ();
		trace_refresh ();
		break;
	case TEV_WAITING:
		break;
	case TEV_READY:
		/*
		 * Grrr.  We don't get events when right special nodes "happen",
		 * so we check for them here.  If a parent of this node is a
		 * right special node (but not an endpipe), we mark it and its parent
		 * as done.
		 */
		if (n->nparents != 0 && !rb_Empty (n->parents)) {
			TreeNode tn;
			for (tn = rb_First (n->parents); tn != n->parents;
				 tn = rb_Next (tn)) {
				Node n2 = (Node) rb_Value (tn);
				switch (n2->node_type) {
				case NODE_ENDCOND:
				case NODE_ENDLOOP:
				case NODE_FANIN:
					if (n2->state != ST_DONE) {
						n2->state = ST_DONE;
						n2->pair->state = ST_DONE;
						xgr_DrawNode (graphWin, n2, -1);
						xgr_DrawNode (graphWin, n2->pair, -1);
					}
					break;
				case NODE_ENDPIPE:
					/*
					 * Horrible hack for pipes.  We don't really
					 * know if a pipe is "done" until all of its
					 * children are ready to run.  So we count
					 * the number of children that become ready,
					 * and when we reach the number of children
					 * that the endpipe node has, we know we are
					 * done.  (Actually, this would probably work
					 * for other node types also.)
					 */
					if (n2->numRunning++ == n2->nchildren) {
						n2->state = ST_DONE;
						n2->pair->state = ST_DONE;
						xgr_DrawNode (graphWin, n2, -1);
						xgr_DrawNode (graphWin, n2->pair, -1);
						n2->numRunning = 0;
					}
				}
			}
		}
		n->state = ST_READY;
		goto drawnode;
	case TEV_RUNNING:
		hmap_HostEvent (tev->host, tev->id, tev->inst, n->state, ST_RUNNING);
		n->numRunning++;
		goto drawnode;
	case TEV_DONE:
		hmap_HostEvent (tev->host, tev->id, tev->inst, n->state, ST_DONE);
		n->numRunning--;
		n->numIdle++;
		goto drawnode;
	case TEV_DEAD:
		hmap_HostEvent (tev->host, tev->id, tev->inst, n->state, ST_DEAD);
		n->state = ST_DEAD;
		n->numIdle--;
		goto drawnode;
	case TEV_FINISH:
		hmap_ResetHosts ();
		gr_ResetGraph (tracegraph, ST_DEAD);
		stopTrace ();
		trace_refresh ();
		msg_Format ("program finished.");
		set_togg (fwdButton, 0);
		set_togg (stopButton, 1);
		break;
	case TEV_MACHINE:
		break;
	case TEV_VNODE:
		switch (n->node_type) {
		case NODE_COND:
		case NODE_LOOP:
		case NODE_FANOUT:
			if (n->node_type == NODE_FANOUT ? (tev->hi > tev->lo) : tev->tf) {
				/*
				 * zero all process counters within this subgraph.
				 * set all nodes within the subgraph to ST_NOT_BEGUN,
				 * but mark this node and its pair as running
				 */
				gr_ResetSubGraph (n, ST_NOT_BEGUN);
				n->state = ST_RUNNING;
				n->pair->state = ST_RUNNING;
				trace_refresh ();
			}
			else {
				/*
				 * like above, but set all nodes to ST_DONE, and
				 * mark this node and its pair as done
				 */
				gr_ResetSubGraph (n, ST_DONE);
				n->state = ST_DONE;
				n->pair->state = ST_DONE;
				trace_refresh ();
			}
			break;
		case NODE_PIPE:
			n->state = ST_RUNNING;
			n->pair->state = ST_RUNNING;
			break;
		case NODE_ENDCOND:
		case NODE_ENDLOOP:
		case NODE_FANIN:
			/*
			 * mark this node and its pair as done
			 * (currently, this never happens because the executioner
			 * doesn't generate these events, but this is here anyway.)
			 */
			n->state = ST_DONE;
			n->pair->state = ST_DONE;
			xgr_DrawNode (graphWin, n, -1);
			xgr_DrawNode (graphWin, n->pair, -1);
			/*
			 * XXX mark all nodes within this subgraph as idle?
			 */
			break;
		case NODE_ENDPIPE:
			/* not sure what to do with these */
			break;
		default:
			break;
		}
		break;
	case TEV_UNKNOWN:
	default:
		break;
	}

	return;

	/*
	 * come here when we want to redraw a node.
	 *
	 * If we have any instances of this node that are either
	 * running or idle, we will display that as the "state"
	 * of the node on the screen.  Otherwise, we will display
	 * the "state" of the node that was there already.
	 */

 drawnode:
	if (n->numIdle < 0)
		n->numIdle = 0;
	if (n->numRunning < 0)
		n->numRunning = 0;

	if (n->numRunning > 0)
		n->state = ST_RUNNING;
	else if (n->numIdle > 0)
		n->state = ST_DONE;

	xgr_DrawNode (graphWin, n, -1);
}

/*
 * Come here when we get an X timer event.  If we are still
 * tracing, read another event and deal with it.
 */

#ifdef TIME_SCALE
static void
Tick (mode, y)
XtPointer mode;
XtIntervalId *y;
{
	static struct trace_event thisEvent;	/* this event */
	static struct EventTime thisEventTime;	/* time of this event */
	static struct EventTime pseudoNow;		/* simulated time */

	switch ((int) mode) {
	case TIMER_ALARM:			/* timer went off */
		/*
		 * NB: when the timer goes off, we don't check traceStopped
		 * until we've processed at least one event.  This is because
		 * we've already read an event into thisEvent, but we haven't
		 * yet processed it.
		 */
		break;
	case READ_AGAIN:			/* file read failed last time, retry */
		if (traceStopped)
			return;
		if (tf_NextEvent (&thisEvent) != 0)
			goto readfail;
		thisEventTime.secs = thisEvent.timestamp.secs;
		thisEventTime.millisecs = thisEvent.timestamp.millisecs;
		break;
	case FIRST_EVENT:			/* called from startTrace to prime things */
		if (tf_NextEvent (&thisEvent) != 0)
			goto readfail;
		thisEventTime.millisecs = thisEvent.timestamp.millisecs;
		thisEventTime.secs = thisEvent.timestamp.secs;
		pseudoNow = thisEventTime;
		break;
	}

	/*
	 * handle all events up to and including the trace time
	 */
	do {
		struct EventTime *delta;
		long delay_time;
		long msec;

		/*
		 * if it's too early to trace this event, update hairline
		 * and elapsed time indicator, and wait a while.
		 */
		if (after (&thisEventTime, &pseudoNow)) {
#ifdef TIME_LABEL
			delta = elapsed_time (&epoch, &pseudoNow);
			set_label (timeLabel, print_time (delta));
#endif /* TIME_LABEL */
			if (histogramExists)
				hist_SetHairLine (&pseudoNow, 1);

			/*
			 * calculate time to delay until next event is traced.
			 * if the histogram window is visible, this wants to
			 * be the amount of scaled time equal to a pixel; 
			 * otherwise, this should be the scaled time until another 
			 * event takes place.
			 */
			if (histogramExists) {
				if (hist_MsecPerPixel (&msec) < 0)
					msec = 1000;
			}
			else {
				delta = elapsed_time (&pseudoNow, &thisEventTime);
				msec = delta->secs * 1000 + delta->millisecs;
			}
			delay_time = msec * time_scale;

			/*
			 * if delay_time is longer than one second, reset msec and
			 * delay_time so that we will wake up within a second.
			 * This is so the "stop" button will work within a
			 * reasonable time.  If possible, make the clock stop on an
			 * even second boundary.  This is because it looks wierd
			 * to watch the middle digits of the clock tick by but always
			 * with the same lower order digits.
			 */
			if (delay_time > 1000) {
				delta = elapsed_time (&epoch, &pseudoNow);
				msec = 1000 - delta->millisecs;
				if (msec == 0)
					msec = 1000;
				delay_time = msec * time_scale;
				if (delay_time > 1000) {
					msec = 1000 / time_scale;
					delay_time = msec * time_scale;
				}
			}
			/*
			 * on the other hand, if delay_time is too short (< 10msec),
			 * increase the amount of clock time to be elapsed until
			 * delay_time is long enough for a minimum delay.
			 */
			else if (delay_time < 10) {
				msec = (int) ((10.0 / time_scale) + 0.5);
				delay_time = msec * time_scale;
			}

			/*
			 * figure out what time it will be when we wake up.
			 */
			pseudoNow.millisecs += msec;
			pseudoNow.secs += pseudoNow.millisecs / 1000;
			pseudoNow.millisecs = pseudoNow.millisecs % 1000;

			/*
			 * set the timer to go off then.
			 */
			timerId = XtAppAddTimeOut (context, delay_time, Tick,
									   (XtPointer) TIMER_ALARM);
			return;
		}
		traceEvent (&thisEvent);
		if (traceStopped)
			return;
		if (tf_NextEvent (&thisEvent) != 0)
			goto readfail;
		thisEventTime.secs = thisEvent.timestamp.secs;
		thisEventTime.millisecs = thisEvent.timestamp.millisecs;
	} while (1);

readfail:
	if (monitorMode) {
		/* wait 500 msec and try again */
		timerId = XtAppAddTimeOut (context, 500, Tick, (XtPointer) READ_AGAIN);
	}
	else {
		msg_Format ("%s: premature end of file\n", traceFileName);
		stopTrace ();
		set_togg (fwdButton, 0);
		set_togg (stopButton, 1);
	}
}
#else
static void
Tick (x, y)
XtPointer x;
XtIntervalId *y;
{
	struct trace_event tev;

	if (traceStopped)
		return;

	if (traceDirection == FORWARD) {
		if (tf_NextEvent (&tev) == 0) {
			traceEvent (&tev);
		}
		else if (monitorMode) {
			/*
			 * file read failed, perhaps because we're still waiting on
			 * data.  Wait at least .5 sec before trying to read file again.
			 *
			 * XXX also need to see if executioner and pvm are still running.
			 */
			int delay = 500;
			if (animat_delay > 500)
				delay = animat_delay;
			timerId = XtAppAddTimeOut (context, delay, Tick, x);
			return;
		}
		else {
			msg_Format ("%s: premature end of file\n", traceFileName);
			stopTrace ();
			set_togg (fwdButton, 0);
			set_togg (stopButton, 1);
		}
	}
	else {
		if (tf_PrevEvent (&tev) == 0)
			traceEvent (&tev);
	}

	/* reset time out */
	timerId = XtAppAddTimeOut (context, animat_delay, Tick, x);
}
#endif

#if 0
/*	redraw_cb()
 *
 *	Redraw command
 */

static void
redraw_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	XClearWindow (xDisp, graphWin);
	trace_refresh ();
}
#endif

/*	stop_cb()
 *
 *	Stop command
 */

static void
stop_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	stopTrace ();
}

/*	fwd_cb()
 *
 *	Forward command
 */

static void
fwd_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	struct trace_event tev;

	if (checkThings ())
		return;

	traceDirection = FORWARD;
	stopTrace ();

	/*
	 * Hack to detect end of tracefile and do something reasonable.
	 * Go ahead and read one event.  If it succeeds, display the
	 * results; otherwise, don't go into continuous trace mode.
	 */

	if (tf_NextEvent (&tev) == 0) {
		traceEvent (&tev);
		startTrace ();
	}
	else {
		set_togg (fwdButton, 0);
		set_togg (stopButton, 1);
		msg_Format ("End of trace file.\n");
	}
}


static void
fwdstep_cb (w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	struct trace_event tev;

	/* need to make sure a graph exists */
	if (checkThings ())
		return;

	traceDirection = FORWARD;
	stopTrace ();

	if (tf_NextEvent (&tev) == 0)
		traceEvent (&tev);
	else
		msg_Format ("End of trace file.\n");

	set_togg (fwdstepButton, 0);
	set_togg (stopButton, 1);
}

/*
 *	rewind_cb()
 *
 *	Rewind command
 */

static void
rewind_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	stopTrace ();
	traceDirection = FORWARD;
	if (tf_Rewind () < 0)
		msg_Format ("rewind failed: error %d\n", errno);
	hmap_ResetHosts ();

	/*
	 * set graph back to initial state
	 *
	 * XXX this is a hack - fix when we get generalized save/restore
	 * state - which we need for usage graph anyway.
	 */
	gr_ResetGraph (tracegraph, ST_NOT_BEGUN);
	trace_refresh ();
#ifdef HAIRLINE
	if (histogramExists)
		hist_ClearHairLine ();
#endif

	set_togg (rewindButton, 0);
	set_togg (stopButton, 1);
#ifdef TIME_LABEL
	set_label (timeLabel, "00:00:00.000");
#endif
	msg_Format ("Rewound trace file.\n");
}

/*	speed_cb()
 *
 *	Speed knob change
 */

static void
speed_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	float f;

	f = *(float*)cd;
#ifdef TIME_SCALE
	/*
	 * need a log scale
	 * if slider is at the far left, time_scale = 100
	 * if slider is in the center, time_scale = 1.0
	 * if slider is at the far right, time_scale = .01
	 * total magnitude difference = 10 ** 4;
	 */

	time_scale = pow (10.0, (1.0 - f) * 4.0 - 2.0);
#ifdef DEBUG
	fprintf (stderr, "time scale = %f\n", time_scale);
#endif
#else
	animat_delay = (1.0 - f) * 2;
#endif
}


Boolean hostsAreVisible = False;

/*
 * come here when the host map window gets an expose event
 */

void
hosts_expose (w, cli, xev, rem)
Widget w;
XtPointer cli;
XEvent *xev;
Boolean *rem;
{
	if (hostsWin == NULL)
		return;
	if (xev->type == Expose && xev->xexpose.count == 0)
		hmap_Refresh (hostsWin);
}

/*
 * "hosts" command button - pop up/down hosts map
 */

static void
hosts_cb (w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	Arg args[20];
	register int n;
	Position x, y, curr_x, curr_y;
	Widget form;
#if 0
	Widget button;
#endif

#if 0
	/* fprintf(stderr, "hosts_cb\n"); */

	/*
	 * If a cost matrix has been defined, make sure the host map
	 * has been initialized (it won't hurt to do it twice), and
	 * read in the icons file.  If the cost matrix has not been
	 * defined, we will go ahead and pop up the window, but the
	 * user will have to define a cost matrix before tracing will
	 * actually do anything.
	 */

	if (costmatrix) {
	    int h;

	    for (h = 0; h < costmatrix->nhost; h++)
		if (costmatrix->hosts[h] && *costmatrix->hosts[h])
		    hmap_AddHost(costmatrix->hosts[h]);
	    hmap_ReadIconList (XtWindow(topLevel), iconList);
	}
#endif

	if (hostsAreVisible = !hostsAreVisible) {

		curr_x = curr_y = 400;

		XtTranslateCoords (w, curr_x, curr_y,
						   (Position *) &x, (Position *) &y);

		n = 0;
		XtSetArg(args[n], XtNx, x);                         n++;
		XtSetArg(args[n], XtNy, y);                         n++;
		XtSetArg(args[n], XtNallowShellResize, True);       n++;
        hostsPop = XtAppCreateShell("Host Map", "HostsPopup",
									applicationShellWidgetClass,
									XtDisplay(w), args, n);


        n = 0;
        XtSetArg(args[n], XtNallowHoriz, (XtArgVal)True); n++;
        XtSetArg(args[n], XtNallowVert, (XtArgVal)True); n++;
        form = XtCreateManagedWidget("form", viewportWidgetClass, hostsPop,
									 args, n);

		n = 0;
		XtSetArg (args[n], XtNheight, (XtArgVal) 300); n++;
		XtSetArg (args[n], XtNwidth, (XtArgVal) 500); n++;
		hostsWin = XtCreateManagedWidget ("hostsMap", 
										widgetClass, form, args, n);


		XtAddEventHandler (hostsWin, StructureNotifyMask|ExposureMask,
						   False, hosts_expose, NULL);

		XtPopup (hostsPop, XtGrabNone);
	}
	else {
		XtDestroyWidget (hostsPop);
		hostsWin = hostsPop = NULL;
	}
}

#ifdef USAGE_GRAPH
static void ScanTraceFile ();

static void
usage_cb (w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	if (histogramExists == 0) {
		if (checkThings () != 0) {
			set_togg (w, 0);
			return;
		}
		usagePop = XtAppCreateShell ("Utilization graph", "UsagePopup",
									 applicationShellWidgetClass,
									 XtDisplay (w),
									 NULL, 0);
		setUpHistogram ();
		XtPopup (usagePop, XtGrabNone);
		histogramExists = 1;
	}
	else {
		XtDestroyWidget (usagePop);
		usagePop = NULL;
		histogramExists = 0;
	}
}
#endif

static void
monitor_cb (w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	monitorMode = !monitorMode;
}

/********************************
*  graph window event handling  *
*                               *
********************************/

/*	graph_ev()
 *
 *	Event in graph window
 */

static void
graph_ev(w, cli, xev, rem)
Widget w;
XtPointer cli;
XEvent *xev;
Boolean *rem;
{
	XButtonEvent *bev = (XButtonEvent*)xev;
	XConfigureEvent *cev = (XConfigureEvent*)xev;
	XExposeEvent *eev = (XExposeEvent*)xev;
	XMotionEvent *mev = (XMotionEvent*)xev;
	XCrossingEvent *rev = (XCrossingEvent*)xev;
#if 0
	int x, y;
#endif

	switch (xev->type) {

	case Expose:
		if (eev->count == 0)
			trace_refresh();
		break;

	case ConfigureNotify:
		break;

	}
}

static void
trace_refresh()
{
	xgr_DrawGrf(graphWin, tracegraph, 1);
}

#ifdef USAGE_GRAPH
#ifdef FAKE_DATA
struct Event lastEventRead;

int
getNextEvent (nextEvent)
struct Event *nextEvent;
{
    int hostNum = lastEventRead.hostno;
    int curTime = lastEventRead.time;
    int oldHostState, newHostState;

#define HOST_STATE(host,time) ((((time)%(2*(host+1)+2)) / (host+2) == 0) + 1)

    do {
		++hostNum;
		if (hostNum >= hmap_NumHosts) {
			hostNum = 0;
			curTime++;
		}
		oldHostState = HOST_STATE(hostNum,curTime-1);
		newHostState = HOST_STATE(hostNum,curTime);
    } while (oldHostState == newHostState);

    nextEvent->hostno = hostNum;
    nextEvent->time = curTime;
    nextEvent->state = newHostState;
    lastEventRead = *nextEvent;
    return 0;
}

void
getState (stateArray, numHosts, timep)
struct Event *stateArray;
int numHosts;
long *timep;
{
    int i;


    for (i = 0; i < hmap_NumHosts; ++i) {
		stateArray[i].state = HOST_STATE(i, *timep - 1);
		stateArray[i].hostno = i;
		stateArray[i].time = *timep - 1;
		stateArray[i].fractime = 0;
    }
    lastEventRead = stateArray[hmap_NumHosts-1];
    return 0;
}
#else

static char *hostNames[200];
static int numRunning[200];
static int numIdle[200];
static int numHosts = 0;
#ifdef FRACTIME
struct EventTime hEpoch;
struct EventTime hEndOfTime;
#else
static long hEpoch;
static long hEndOfTime;
#endif

static Tree stateTree;
static TreeNode lastEvent = NULL;

int
findHost (machine)
char *machine;
{
	int i;
	if (strcmp (machine, "x") == 0)
		return 199;
	for (i = 0; i < numHosts; ++i)
		if (strcmp (machine, hostNames[i]) == 0)
			return i;
	return 199;
}

static int
compareEvents (a, b)
Key *a, *b;
{
	struct Event *ax, *bx;
	ax = (struct Event *) a->pkey;
	bx = (struct Event *) b->pkey;
#ifdef FRACTIME
	if (ax->time.secs == bx->time.secs)
		return (ax->time.millisecs - bx->time.millisecs);
	return (ax->time.secs - bx->time.secs);
#else
	if (ax->time == bx->time)
		return (ax->fractime - bx->fractime);
	return (ax->time - bx->time);
#endif
}

static int
computeState (hostNumber)
{
	if (numRunning[hostNumber] > 0)
		return HISTO_RUNNING;
	else if (numIdle[hostNumber] > 0)
		return HISTO_IDLE;
	else
		return HISTO_OFF;
}

static void
saveState (time, fractime, hostNumber)
long time, fractime;
int hostNumber;
{
	struct Event *foo = (struct Event *) malloc (sizeof (struct Event));
	Key k;

#ifdef FRACTIME
	foo->time.secs = time;
	foo->time.millisecs = fractime;
#else
	foo->time = time;
	foo->fractime = fractime;
#endif
	foo->hostno = hostNumber;
	foo->state = computeState (hostNumber);
	k.pkey = (void *) foo;
	rb_Insert (stateTree, &k, foo, compareEvents);
}


static void
ScanTraceFile (filename)
char *filename;
{
    FILE *fp;
	char buf[256];
	char event_type[50];
	char nodetype[50];
	char machine[50];
	long timestamp, serial;
	int x;
	int oldState;

	numHosts = 0;

	stateTree = rb_MakeTree ();
    if ((fp = fopen (filename, "r")) == NULL) {
#ifdef FRACTIME
		hEpoch.secs = 0L;
		hEpoch.millisecs = 0;
		hEndOfTime = hEpoch;
#else
		hEpoch = 0L;
		hEndOfTime = 0L;
#endif
		return;
    }
	while (1) {
		if (fgets (buf, sizeof (buf), fp) == NULL) {
			msg_Format ("%s: premature end of file\n", filename);
			return;
		}
		if (*buf != '#') {
			msg_Format ("%s: bad trace file format\n", filename);
			return;
		}
		sscanf(buf + 1, "%ld %ld %*d/%*d %s %s %s %*c %*d %*d",
			   &timestamp, &serial, nodetype, machine, event_type);

		if (strcmp (event_type, "MACHINE") == 0) {
			hostNames[numHosts++] = strsave (machine);
		}
		else if (strcmp (event_type, "START") == 0) {
#ifdef FRACTIME
			hEpoch.secs = timestamp;
			hEpoch.millisecs = serial;
#else
			hEpoch = timestamp;
#endif
			if (monitorMode) {
#ifdef FRACTIME
				hEndOfTime.secs = timestamp;
				hEndOfTime.millisecs = serial;
#else
				hEndOfTime = timestamp;
#endif
			}
		}
		else if (strcmp (event_type, "RUNNING") == 0) {
			if ((x = findHost (machine)) != 199) {
				oldState = computeState (x);
				numRunning[x]++;
				if (oldState != computeState (x))
					saveState (timestamp, serial, x);
			}
		}
		else if (strcmp (event_type, "DONE") == 0) {
			if ((x = findHost (machine)) != 199){
				oldState = computeState (x);
				--numRunning[x];
				++numIdle[x];
				if (oldState != computeState (x))
					saveState (timestamp, serial, x);
			}
		}
		else if (strcmp (event_type, "DEAD") == 0) {
			if ((x = findHost (machine)) != 199) {
				oldState = computeState (x);
				--numIdle[x];
				if (oldState != computeState (x))
					saveState (timestamp, serial, x);
			}
		}
		else if (strcmp (event_type, "FINISH") == 0) {
			if (!monitorMode) {
#ifdef FRACTIME
				hEndOfTime.secs = timestamp;
				hEndOfTime.millisecs = serial;
#else
				hEndOfTime = timestamp;
#endif
			}
			break;
		}
	}
	fclose (fp);
	return;
}

int
getNextEvent (nextEvent)
struct Event *nextEvent;
{
	TreeNode x;
	struct Event *ptr;

	if (lastEvent == NULL) {
		abort ();
	}
	x = rb_Next (lastEvent);
	if (x == stateTree)
		return -1;
	ptr = (struct Event *) rb_Value (x);
	*nextEvent = *ptr;
	lastEvent = x;
}

void
getState (stateArray, nhosts, timep)
struct Event *stateArray;
int nhosts;
#ifdef FRACTIME
struct EventTime *timep;
#else
long *timep;
#endif
{
	Key k;
	struct Event x, *p;
	TreeNode tn;
	int fnd = 0;
	int hostCount = nhosts;
	int i;

#ifdef FRACTIME
	x.time = *timep;
	if (--x.time.millisecs < 0) {
		x.time.secs--;
		x.time.millisecs = 999;
	}
#else
	x.time = *timep - 1;
	x.fractime = 0x7fffffff;
#endif
	x.hostno = 0;
	k.pkey = &x;
	tn = rb_Find (stateTree, &k, compareEvents, &fnd);
	lastEvent = rb_Prev (tn);

	for (i = 0; i < nhosts; ++i) {
		stateArray[i].state = -1;
		stateArray[i].hostno = i;
#ifdef FRACTIME
		stateArray[i].time.secs = 0;
		stateArray[i].time.millisecs = 0;
#else
		stateArray[i].time = 0;
		stateArray[i].fractime = 0;
#endif
	}
	do {
		if (tn == stateTree)
			break;
		if ((tn = rb_Prev (tn)) == stateTree)
			break;
		p = (struct Event *) rb_Value (tn);
		if (p->hostno < nhosts) {
			if (stateArray[p->hostno].state == -1) {
				stateArray[p->hostno].state = p->state;
				stateArray[p->hostno].time = p->time;
#ifndef FRACTIME
				stateArray[p->hostno].fractime = 0;
#endif
				--hostCount;
			}
		}
	} while (hostCount > 0);

	for (i = 0; i < nhosts; ++i)
		if (stateArray[i].state == -1)
			stateArray[i].state = HISTO_OFF;
}

setUpHistogram ()
{
	ScanTraceFile (traceFileName);
	hist_Create (usagePop, hostNames, numHosts);
	hist_SetEpoch (&hEpoch);
	hist_SetEndOfTime (&hEndOfTime);
}

#endif /* FAKE_DATA  */
#endif /* USAGE_GRAPH */

/*
 * Local variables:
 * tab-width:4
 * End:
 */
