/*
 * functions for drawing a pseudo-map of hosts
 *
 * derived from manchek's host.c for xnt
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "comn.h"
#include "xincl.h"
#include "xcomn.h"
#include "rb.h"
#include "param.h"
#include "exp.h"				/* XXX Yuk! */
#include "graph.h"
#include "trace.h"
#include "hostmap.h"
#include "xdraw.h"
#include "tilde.h"
#include "xbm/genhost.xbm"

/*
 * XXX "hosts" and "hostCount" are used by popups.c - make them "static"
 * once popups.c is merged back in
 */

static Tree hosts = NULL;
static int hostCount = 0;

extern Widget hostsWin;			/* declared in trace.c */
extern Widget hostsPop;			/* declared in trace.c */

struct icon hmapDefaultHostIcon;
GC gcErase = NULL;
GC gcNormal = NULL;
GC gcRunning = NULL;
GC gcQuiescent = NULL;

/*
 * utility function to compare two HeNCE processes in a per-host
 * process list
 */

static int
hmap_ProcCompare (k1, k2)
Key *k1, *k2;
{
	struct proc_info *p1, *p2;

	p1 = (struct proc_info *) k1->pkey;
	p2 = (struct proc_info *) k2->pkey;

	if (p1->id == p2->id)
		return p1->inst - p2->inst;
	return p1->id - p2->id;
}

/*
 * find a HeNCE process descriptor in a per-host process list.  If its not
 * there, create one
 */

static struct proc_info *
hmap_FindProc (tree, id, inst)
Tree tree;
int id;
int inst;
{
	Key k;
	struct proc_info p, *ptr;
	int fnd;
	TreeNode tn;

	p.id = id;
	p.inst = inst;
	k.pkey = (void *) &p;
	tn = rb_Find (tree, &k, hmap_ProcCompare, &fnd);
	if (fnd) {
#if 0
		fprintf (stderr, "found %d/%d\n", p.id, p.inst);
#endif
		return (struct proc_info *) rb_Value (tn);
	}

	ptr = talloc (1, struct proc_info);
	ptr->id = id;
	ptr->inst = inst;
	k.pkey = (void *) ptr;
	rb_InsertBefore (tn, &k, ptr);
#if 0
	fprintf (stderr, "created %d/%d\n", ptr->id, ptr->inst);
#endif
	return ptr;
}

/*
 * free up a process list.  This is called from hmap_ResetHosts(),
 * and also from hmap_NukeHosts().
 */

static void
hmap_FreeProcList (pl)
Tree pl;
{
	TreeNode tn2, tn3;

	if (pl == NULL || rb_Empty (pl))
		return;

	for (tn2 = rb_First (pl); tn2 != pl; tn2 = tn3) {
		tn3 = rb_Next (tn2);
		free (rb_Value (tn2));
		rb_DeleteNode (tn2);
	}
}


/* ConvertColor is from the Xaw example program popup.c from MIT */

/* XXX this should probably be in xdraw.c */

int
ConvertColor(w, color_name)
Widget w;
char * color_name;
{
	XrmValue from, to;

	from.size = strlen(color_name) + 1;
	from.addr = color_name;

	/*
	 * This conversion accepts a colorname from rgb.txt, or a #rrrgggbbb
	 * rgb color definition, or the special toolkit strings
	 * "XtDefaultForeground" and "XtDefaultBackground".
	 */

	XtConvert(w, XtRString, (XrmValuePtr) &from, XtRPixel, (XrmValuePtr) &to);
	if (to.addr == NULL) {
		return(-1);
	}

	return( (int) *((Pixel *) to.addr) );
}

#if 0
/*
 * currently unused
 */

GC
GetGCFromColor(w, color)
Widget w;
char *color;
{
    XGCValues       myXGCV;

    int pixel = ConvertColor (w, color);

    if (pixel == -1)
        return ((GC)NULL);

    myXGCV.line_width = 1;
    myXGCV.foreground = pixel;

    return XtGetGC((Widget) w, GCForeground | GCLineWidth , &myXGCV);
}
#endif

/*
 * init function.  called by any of several other functions in this module
 * if the gc's aren't set up.  Sets up gc's, default host icon, other stuff.
 */

void
hmap_Init ()
{
    Window window = XtWindow(topLevel);
	XGCValues x;

	/* printf ("hmap_Init\n"); */
	if (inColor) {
		x.function = GXcopy;
		x.foreground = ConvertColor(topLevel, "white");
		x.background = ConvertColor(topLevel, "white");
		gcErase = XCreateGC (xDisp, window,
							  GCFunction|GCForeground|GCBackground, &x);
		x.function = GXcopy;
		x.foreground = ConvertColor(topLevel, "black");
		x.background = ConvertColor(topLevel, "white");
		gcNormal = XCreateGC (xDisp, window,
							  GCFunction|GCForeground|GCBackground, &x);
		x.function = GXcopy;
		x.background = ConvertColor(topLevel, "green");
		x.foreground = ConvertColor(topLevel, "black");
		gcRunning = XCreateGC (xDisp, window,
							   GCFunction|GCForeground|GCBackground, &x);
		x.function = GXcopy;
		x.background = ConvertColor(topLevel, "yellow");
		x.foreground = ConvertColor(topLevel, "black");
		gcQuiescent = XCreateGC (xDisp, window,
								 GCFunction|GCForeground|GCBackground, &x);
	}
	else {
		/*
		 * monochrome GCs
		 *
		 * XXX make these use stippling or something similar so that
		 * normal hosts are very dim, quiescent hosts are dim,
		 * and running hosts are dark.
		 */

		x.function = GXcopy;
		x.foreground = COLBGND;	/* is this right? */
		x.background = COLBGND;
		gcErase = XCreateGC (xDisp, window,
							  GCFunction|GCForeground|GCBackground, &x);
		x.function = GXcopy;
		x.foreground = COLFGND;
		x.background = COLBGND;
		gcNormal = XCreateGC (xDisp, window,
							  GCFunction|GCForeground|GCBackground, &x);
		x.function = GXcopy;
		x.foreground = COLFGND;
		x.background = COLBGND;
		gcRunning = XCreateGC (xDisp, window,
							   GCFunction|GCForeground|GCBackground, &x);
		x.function = GXcopy;
		x.foreground = COLBGND;
		x.background = COLFGND;
		gcQuiescent = XCreateGC (xDisp, window,
								 GCFunction|GCForeground|GCBackground, &x);
	}
	hmapDefaultHostIcon.bits =
		XCreatePixmapFromBitmapData (xDisp, window, genhost_bits,
									 genhost_width, genhost_height, 1, 0, 1);
	hmapDefaultHostIcon.width = genhost_width;
	hmapDefaultHostIcon.height = genhost_height;

    hosts = rb_MakeTree();
}

#if 0							/* CURRENTLY UNUSED */
/*
 * Draw a host in the appropriate color, at the appropriate place on
 * the screen.
 */

void
hmap_DrawHost (window, h)
Window window;
struct host_info *h;
{
	int x, y;
	int c1, c2;

	if (gcNormal == NULL)
		hmap_Init (window);

	x = h->xpos - h->icon->width / 2;
	y = h->ypos - 2 * xdraw_TextHeight (gcNormal) - h->icon->height;

	if (h->numRunning > 0)
		xdraw_Icon (window, gcRunning, h->icon, x, y);
	else if (h->numIdle > 0)
		xdraw_Icon (window, gcQuiescent, h->icon, x, y);
	else
		xdraw_Icon (window, gcNormal, h->icon, x, y);

	x = h->xpos - xdraw_StringWidth (gcNormal, h->name) / 2;
	y = h->ypos;
	xdraw_Text (window, gcNormal, x, y, h->name);
	if (h->string[0]) {
		x = h->xpos - xdraw_StringWidth (gcNormal, h->string) / 2;
		y = h->ypos - xdraw_TextHeight (gcNormal);
		xdraw_Text (window, gcNormal, x, y, h->string);
	}
}

/* XXX this is part of the host node string hack and should die too */

/*	undrawhoststr()
 *
 *	Undraw the string that appears below the host icon
 */

void
hmap_UnDrawHostStr (window, h)
Window window;
struct host_info *h;
{
	int x, y;

	if (gcNormal == NULL)
		hmap_Init (window);

	if (h->string[0]) {
		x = h->xpos - xdraw_StringWidth (gcNormal, h->string) / 2;
		y = h->ypos - xdraw_TextHeight (gcNormal);
		xdraw_Text (window, gcNormal, x, y, h->string);
	}
}

/*
 * draw the lines between the hosts
 *
 * currently, the hosts are arranged in a grid of two (vertical) by
 * (n+1)/2 (horizontal) hosts, that looks something like this:
 *
 *
 * xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx
 * xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx
 * xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx
 * xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx
 * text     text     text     text     text     text     text
 *   |        |        |        |        |        |        |
 *   +--------+--------+--------+--------+--------+--------+
 *   |        |        |        |        |        |        |
 * xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx
 * xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx
 * xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx
 * xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx    xxxxx
 * text     text     text     text     text     text     text
 *
 * The horizontal line down the center is at y=200.  The vertical
 * lines extend up from the horizontal line to either the top
 * of the host icon (if the host icon is below the center line)
 * or to just below the text under the host icon (if the host
 * icon is above the center line).
 *
 * XXX this needs to be generalized to an arbitrary rectangular
 * shape (or maybe even let the user draw the net map himself.)
 */

void
hmap_DrawNetwork (window)
Window window;
{
	int x1, y1, x2, y2;
	TreeNode tn;

	if (hostCount <= 0)
		return;
	if (hosts == (Tree) NULL || rb_First(hosts) == hosts)
		return;
	if (gcNormal == NULL)
		hmap_Init (window);

	for (tn = rb_First (hosts); tn != hosts; tn = rb_Next (tn)) {
		struct host_info *h = rb_Value (tn);

		if (h->name == (char *) NULL)
			continue;
		x1 = h->xpos;
		y1 = 200;
		x2 = x1;

		if (h->ypos < 200)		/* below the center line */
			y2 = h->ypos;
		else					/* above the center line */
			y2 = h->ypos - 2 * xdraw_TextHeight  (gcNormal) - h->icon->height;
		xdraw_Line (window, gcNormal, x1, y1, x2, y2);
	}
	xdraw_Line (window, gcNormal, 100, y1, x1, y1);
}
#endif							/* CURRENTLY UNUSED */


static void
Center_string(w, gc, s, x, y)
Widget w;
GC gc;
char *s;
int x,y;
{
    x -= (xdraw_StringWidth(gc, s) /2);
    XDrawString (XtDisplay(w), XtWindow(w), gc, x, y, s, strlen(s));
}


#define H_SPACE		100
#define V_SPACE		100

static void
host_pos_2_xy (pos, x, y)
int pos;
int *x, *y;
{
	Arg args[3];
	Dimension wide, high;

    XtSetArg(args[0], XtNwidth, &wide);
    XtSetArg(args[1], XtNheight, &high);
    XtGetValues(hostsWin, args, 2);

    if (wide < H_SPACE*2) wide = H_SPACE*2;
    if (high <= V_SPACE*2) high = V_SPACE*2;
	*x = H_SPACE * (1 + ( pos % ((wide)/H_SPACE) ));
	*y = V_SPACE * (1 + ( pos / ((wide)/H_SPACE) ));

    *x = *x - (H_SPACE/2);
    *y = *y - (V_SPACE/2);
}

/*
 * This gets called to compute the extra information that gets
 * displayed on a line below the host name.  Currently, it displays
 * the list of HeNCE node/instance pairs that are running on that
 * machine.
 */

static void
hmap_BuildHostString (h)
struct host_info *h;
{
	char *src, *dst;
	char *enddst;
	TreeNode tn;

	dst = &h->string[0];
	enddst = &h->string[0] + (sizeof (h->string) - 1);

	for (tn = rb_First (h->proc_list); tn != h->proc_list; tn = rb_Next (tn)) {
		struct proc_info *p = (struct proc_info *) rb_Value (tn);
		char foo[30];

		if (dst >= enddst)
			break;

		if (p->state == ST_RUNNING) {
			sprintf (foo, "%d/%d", p->id, p->inst);
			if (dst != &h->string[0])
				*dst++ = ',';
			for (src = foo; *src; ) {
				if (dst >= enddst)
					break;
				*dst++ = *src++;
			}
		}
	}
	*dst++ = '\0';
}


void
hmap_DrawHost (h)
struct host_info *h;
{
    int x, y;
    GC gc;

    if (h->numRunning > 0)
		gc = gcRunning;
    else if (h->numIdle > 0)
		gc = gcQuiescent;
    else
		gc = gcNormal;

    /* printf ("Blop %s run=%d idle=%d\n",
       h->name, h->numRunning, h->numIdle); */

    if (hostsPop) {
		host_pos_2_xy (h->xpos, &x, &y);
		XCopyPlane (XtDisplay(hostsWin), h->icon->bits, XtWindow(hostsWin),
					gc,
					0, 0, h->icon->width, h->icon->height, 
					x - (h->icon->width/2), y - (h->icon->height/2), 1);
		Center_string (hostsWin, gcNormal, h->name,
					   x, y + (h->icon->height/2) +15);

		/* erase the old host info string, compute a new one, and draw it */

		if (h->string[0])
			Center_string (hostsWin, gcErase, h->string,
						   x, y + (h->icon->height/2) +15 +15);
		hmap_BuildHostString (h);
		Center_string (hostsWin, gcNormal, h->string,
					   x, y + (h->icon->height/2) +15 +15);
    }
}

/*
 * refresh the host map window.  Called whenever we get expose
 * events on that window
 */

void
hmap_Refresh ()
{
	TreeNode tn;

	if (hostCount <= 0)
		return;
	if (hosts == (Tree) NULL || rb_Empty (hosts))
		return;
#if 0
	if (gcNormal == NULL)
		hmap_Init (window);
#endif

	for (tn = rb_First (hosts); tn != hosts; tn = rb_Next (tn)) {
		struct host_info *h = (struct host_info *) rb_Value (tn);

		if (h->name)
			hmap_DrawHost (h);
	}
	/*	hmap_DrawNetwork (window); NOT IMPLEMENTED YET */
}

void
hmap_ClearWindow ()
{
	if (hostsPop)
		XClearWindow (XtDisplay (hostsWin), XtWindow (hostsWin));
}

/*
 * read an icon from an x bitmap file.
 *
 * XXX Probably should only read an icon file once, and just clone
 * it otherwise.
 */

struct icon *
hmap_ReadIcon (win, filename)
Window win;
char *filename;
{
	int i;
	struct icon *foo;

	/* printf ("hmap_ReadIcon <%s>\n", filename); */
	foo = talloc (1, struct icon);
	if (XReadBitmapFile (xDisp, win, texpand (filename),
						 &(foo->width),
						 &(foo->height),
						 &(foo->bits),
						 &i,
						 &i) == BitmapSuccess) {
		foo->refcount = 1;
		return foo;
	}
	else {
		free (foo);
		msg_Format ("Can't read icon file \"%s\"\n", filename);
		return (struct icon *) NULL;
	}
}

/*
 * free up an icon.  In case an icon has multiple references, the ref
 * count keeps us from freeing it prematurely or more than once.
 *
 * also, don't ever free the default host icon.
 */

static void
hmap_FreeIcon (icon)
struct icon *icon;
{
	if (icon && icon != &hmapDefaultHostIcon && --(icon->refcount) <= 0) {
		if (icon->bits)
			XFreePixmap (xDisp, icon->bits);
		free (icon);
	}
}

/*
 * utility function used to compare host names in tree lookups
 */

static int
hmap_HostCompare (k1, k2)
Key *k1, *k2;
{
	return strcmp ((char *)(k1->pkey), (char *)(k2->pkey));
}

/*
 * Read in a list of (machine, icon) pairs from a file, and
 * remember which icons go with which machines, and where
 * the icons go.
 *
 * XXX the placement of the icons is really lame, and will blow
 * up if we have a lot of hosts.
 */

int
hmap_ReadIconList (win, filename)
Window win;
char *filename;
{
	FILE *fp;
	char hostname[512];
	char iconfilename[512];
	int x;
	int lineno;
	int fnd;
	TreeNode tn;
	struct icon *icon;

	if (gcNormal == NULL)
		hmap_Init (win);

	if ((fp = tfopen (filename, "r")) == NULL) {
		msg_Format ("Can't open icon list file \"%s\"\n", filename);
		return -1;
	}

	lineno = 0;
	while ((x = fscanf (fp, "%s %s", hostname, iconfilename)) > 0) {
		struct host_info *h;
		Key k;
#if 0
		char *ptr;
#endif

		lineno++;

		if (*hostname == '#')	/* recognize comments */
			continue;
		if (x != 2) {
			msg_Format ("%s, line %d: syntax error\n", filename, lineno);
			continue;
		}

		k.pkey = (void *) hostname;
		tn = rb_Find (hosts, &k, hmap_HostCompare, &fnd);

		if (fnd) {
			/* printf("Found out <%s>\n", hostname); */
			h = (struct host_info *) rb_Value (tn);
			if ((icon = hmap_ReadIcon (win, iconfilename)) != NULL)
				h->icon = icon;
		}
	}
	return 0;
}

/*
 * look up the host by name in the hosts table.
 * returns a pointer to the host structure, or NULL on error
 * (which never happens at present).
 *
 * If no entry in the hosts table exists for this hostname, create one
 * and give it the default icon.
 *
 * This is called for each MACHINE entry in the trace file when the trace
 * file is opened, so that there will always be an icon for every host.
 *
 * XXX probably should change the name
 */

struct host_info *
hmap_AddHost (hostname)
char *hostname;
{
	TreeNode tn;
	Key k;
	int fnd;

    if (gcNormal == NULL)
        hmap_Init ();

	k.pkey = (void *) hostname;
	tn = rb_Find (hosts, &k, hmap_HostCompare, &fnd);
	if (!fnd) {
		struct host_info *h = talloc (1, struct host_info);

		h->name = strsave (hostname);
		h->string[0] = '\0';
		h->numRunning = 0;
		h->numIdle = 0;
		h->icon = &hmapDefaultHostIcon;
		h->proc_list = rb_MakeTree ();
/*
		h->xpos = (hostCount / 2) * 100 + 100;
		h->ypos = (hostCount % 2) ? 150 : 300;
*/
		h->xpos = h->ypos = hostCount;
		hostCount++;

		k.pkey = (void *) h->name;
		rb_InsertBefore (tn, &k, h);
		return h;
	}
	else
		return (struct host_info *) rb_Value (tn);
}

/*
 * helper function used to nuke a single host.  Free up everything
 * that the host_info structure points to, then free the struct.
 */

static void
hmap_NukeHost (h)
struct host_info *h;
{
	if (h == NULL)
		return;
	free (h->name);
	hmap_FreeIcon (h->icon);
	hmap_FreeProcList (h->proc_list);
	free (h);
	return;
}

/*
 * come here whenever we want to nuke the entire host list
 */

void
hmap_NukeHosts ()
{
	TreeNode tn, tn2;

	if (hosts == NULL || rb_Empty (hosts))
		return;

	for (tn = rb_First (hosts); tn != hosts; tn = tn2) {
		hmap_NukeHost (rb_Value (tn));
		tn2 = rb_Next (tn);
		rb_DeleteNode (tn);
	}
	hostCount = 0;
	hmap_ClearWindow ();
}

/*
 * new version of hmap_HostEvent that takes a host_info pointer
 * rather than a machine name as a parameter, since the new
 * trace file routines have already looked up the host_info
 * for us.
 */

void
hmap_HostEvent (h, id, inst, oldstate, newstate)
struct host_info *h;
int id, inst;
int oldstate;
int newstate;
{
	struct proc_info *p;

	if (h == (struct host_info *) NULL) {
		/* should happen only when null() is being executed */
		return;
	}

	/*
	 * XXX need to handle state transitions if time goes
	 * backwards
	 */

	switch (newstate) {
	case ST_RUNNING:
		h->numRunning++;
		break;
	case ST_DONE:
		h->numRunning--;
		h->numIdle++;
		break;
	case ST_DEAD:
		h->numIdle--;
		break;
	}
	if ((p = hmap_FindProc (h->proc_list, id, inst)) != NULL) 
		p->state = newstate;
    hmap_DrawHost (h);
}

void
hmap_ResetHosts()
{
    TreeNode tn;

    if (hostCount <= 0)
        return;
    if (hosts == (Tree) NULL || rb_Empty (hosts))
        return;

    for (tn = rb_First (hosts); tn != hosts; tn = rb_Next (tn)) {
        struct host_info *h = (struct host_info *) rb_Value (tn);

        if (h->name)
            h->numIdle = h->numRunning = 0;

		/* free up process list */
		hmap_FreeProcList (h->proc_list);
    }

	hmap_Refresh ();
	/* 	draw_hosts(hostsWin); */
}

char **hmap_HostList = NULL;
int hmap_NumHosts = 0;

void
hmap_MakeHostList ()
{
	TreeNode tn;
	int i;
	
	hmap_HostList = NULL;
	hmap_NumHosts = hostCount;

	if (hostCount <= 0)
		return;
	if (hosts == (Tree) NULL || rb_Empty (hosts))
		return;

	hmap_HostList = (char **) malloc (hostCount * sizeof (char *));
	for (i = 0, tn = rb_First (hosts) ; tn != hosts ; tn = rb_Next (tn)) {
		struct host_info *h = (struct host_info *) rb_Value (tn);
		h->hostNumber = i;
		hmap_HostList[i++] = h->name;
	}
}

/*
 * Local variables:
 * tab-width:4
 * End:
 */
