#include <stdio.h>
#include <sys/types.h>

#include <errno.h>
#ifndef errno
extern int errno;		/* 4.3BSD and ancestors don't declare this */
#endif


/*
 * get *_OK constants for access call.  Sometimes they're in <sys/file.h>,
 * and sometimes in <unistd.h>.  But the latter doesn't exist on many systems.
 * And even if it exists, it may not define the *_OK stuff.  If all else
 * fails, define them here.
 */

#if defined(IMA_PMAX) || defined(IMA_SUN3) || defined(IMA_SUN4)
#include <unistd.h>
#endif
#if defined(IMA_RIOS) || defined(IMA_TITN)
#include <unistd.h>
#endif
#if defined(IMA_SYMM)
#include <sys/file.h>
#endif

#ifndef R_OK
#define R_OK 4
#define W_OK 2
#define X_OK 1
#define F_OK 0
#endif /*R_OK*/


#include <sys/stat.h>
/*
 * define POSIX-style permission masks in terms of the V7-style codes,
 * if the former aren't defined in <sys/stat.h>
 */

#ifndef S_IRUSR
#define S_IRUSR S_IREAD
#define S_IWUSR S_IWRITE
#define S_IXUSR S_IEXEC
#define S_IRGRP ((S_IREAD) >> 3)
#define S_IWGRP ((S_IWRITE) >> 3)
#define S_IXGRP ((S_IEXEC) >> 3)
#define S_IROTH ((S_IREAD) >> 6)
#define S_IWOTH ((S_IWRITE) >> 6)
#define S_IXOTH ((S_IEXEC) >> 6)
#endif

#include <sys/param.h>

#ifndef NGROUPS
#ifdef _POSIX_NGROUPS_MAX
#define NGROUPS _POSIX_NGROUPS_MAX
#else
#define NGROUPS 1
int
getgroups (listsize, list)
int listsize;
int *list;
{
    if (listsize > 0)
	*list = getegid ();
    return 1;
}
#endif
#endif

/*
 * a version of access that checks using the effective uid and
 * is implemented in terms of stat()
 */

int
eaccess (name, bits)
char *name;
int bits;
{
    static struct stat stat_buf;
    static stat_errno = 0;
    static char stat_filename[1024] = "";
    static int myuid;
    static int inited = 0;
    static int grouplist[NGROUPS+1];	/* room for lots of groups */
    static int ngroups = 0;
    static enum { owner, group, world } howToCheck;
    
    /*
     * cache uid and group list.  This assumes that the uid and group
     * list never changes once we call this function.
     */
    if (inited == 0) {
	myuid = geteuid ();
	ngroups = getgroups (sizeof (grouplist) / sizeof (*grouplist),
			     grouplist);
	inited = 1;
    }
    /*
     * check for NULL name to keep us from core-dumping
     */
    if (name == NULL) {
	errno = EFAULT;
	return -1;
    }
    /*
     * if name isn't the same as the last time we were called,
     * do a stat on the file and store away the results.
     */
    if (strcmp (name, stat_filename) != 0) {
	strncpy (stat_filename, name, sizeof stat_filename);
#if 0
	printf ("stat (%s)\n", name);
#endif
	if (stat (name, &stat_buf) < 0)
	    stat_errno = errno;
	else {
	    stat_errno = 0;
	    
	    /*
	     * figure out what permission bits should be used to
	     * check for accessibility
	     */

	    howToCheck = world;
	    if (stat_buf.st_uid == myuid)
		howToCheck = owner;
	    else {
		int i;
		for (i = 0; i < ngroups; ++i) {
		    if (stat_buf.st_gid == grouplist[i]) {
			howToCheck = group;
			break;
		    }
		}
	    }
	}
    }
    if (stat_errno != 0) {
	errno = stat_errno;
	return -1;
    }
    
#define CHECK(MASK) if ((stat_buf.st_mode & MASK) == 0) return -1
    
    if (bits & R_OK) {
	switch (howToCheck) {
	case owner:
	    CHECK(S_IRUSR);
	    break;
	case group:
	    CHECK(S_IRGRP);
	    break;
	case world:
	    CHECK(S_IROTH);
	    break;
	}
    }
    if (bits & W_OK) {
	switch (howToCheck) {
	case owner:
	    CHECK(S_IWUSR);
	    break;
	case group:
	    CHECK(S_IWGRP);
	    break;
	case world:
	    CHECK(S_IWOTH);
	    break;
	}
    }
    if (bits & X_OK) {
	switch (howToCheck) {
	case owner:
	    CHECK(S_IXUSR);
	    break;
	case group:
	    CHECK(S_IXGRP);
	    break;
	case world:
	    CHECK(S_IXOTH);
	    break;
	}
    }
    return 0;
}
