/*
 *	HeNCE Tool
 *
 *	compose.c - Graph editor.
 *		This is where everything goes that happens in the compose panel.
 *
 *	Jun 1991  Robert Manchek  manchek@CS.UTK.EDU.
 *
 *	Revision Log
 *
$Log: compose.c,v $
 * Revision 1.5  1992/12/09  04:58:10  moore
 * add casts to make alpha compiler happy.
 *
 * Revision 1.4  1992/09/18  01:42:59  moore
 * improve error message when an attempt to parse a node program fails.
 *
 * Revision 1.3  1992/06/23  19:34:40  moore
 * put node # on "no errors" msg after parsing a newly-edited node.
 *
 * Revision 1.2  1992/05/29  19:18:25  moore
 * get rid of subdefs stuff unless there is a subDefs resource
 *
 * Revision 1.1  1992/04/08  05:48:30  moore
 * initial RCS version
 *
 *
 */

#define SUBPROX 1				/* enables new subprocess stuff */

#include <sys/types.h>
#include <stdio.h>
#include "xincl.h"
#include "rb.h"
#include "param.h"
#include "exp.h"
#include "graph.h"
#include "parse.h"
#include "xcomn.h"
#include "comn.h"
#include "xgraph.h"
#ifdef SUBPROX
#include "subproc.h"
#endif
#include "FileSelect.h"
#include "xbm/compose_help"
#include "xbm/bl_ic"
#include "xbm/el_ic"
#include "xbm/bs_ic"
#include "xbm/es_ic"
#include "xbm/bp_ic"
#include "xbm/ep_ic"
#include "xbm/fo_ic"
#include "xbm/fi_ic"
#include "xbm/no_ic"

/*******************
*  from elsewhere  *
*                  *
*******************/

int fgetc();

extern char graphfile[1024];	/* from widmain.c */
extern char directory[1024];	/* from widmain.c */
extern Graph scratchgraph;		/* from widmain.c */
extern Widget composeButton;	/* from widmain.c */


/*************
*  exported  *
*            *
*************/

void compose_refresh();
void start_compose();
int stop_compose();

int graphmod = 0;				/* has graph been modified */
int curnodetype = NODE_NORMAL;	/* curr creat node type */
Window graphWin;

#ifdef SUBPROX
int composeSubprocCount = 0;
#endif

/************
 *  global  *
 *          *
 ************/

static void critic_cb();
static void cleanup_cb();
static void clear_cb();
static void load_cb();
static void redraw_cb();
static void compose_help_cb();
static void store_cb();
static void subdefs_cb();
static void node_t_cb();
static void graph_ev();
static Widget cre_nodecanvas();
static Widget cre_cmdpanel();

static void compose_add_node();
static void compose_delete_node();
static void compose_move_node();
static void compose_add_arc();
static void compose_move_arc();
static void compose_delete_arc();
static void compose_edit_node();
#if 0
static void edit_node_cb();
#endif

static Widget nodePop = 0;		/* node program edit widget */
static Widget nodeProg = 0;		/* node program text */
static Widget composeCmd = 0;
static Widget composeGraph = 0;
static char filename[256];		/* for load, store */
#if 0
static char nodeprog[4096];		/* for node popup to scribble in */
#endif
static XtCallbackRec callback[2] = { { 0, 0 }, { 0, 0 } };
static Arg args[16];
static char crud[1024];
static Node selected_node;		/* selected node to edit */

static Window helpWin;
static int helpVisible = 0;
static Widget helpPop = 0;


static XtActionsRec actbl[] = {
	{"add-node", compose_add_node},
	{"del-node", compose_delete_node},
	{"move-node", compose_move_node},
	{"add-arc", compose_add_arc},
	{"del-arc", compose_delete_arc},
	{"move-arc", compose_move_arc},
	{"edit-node", compose_edit_node},
};


/********************
*  widget creation  *
*                   *
********************/

/*	start_compose()
*
*	Start method for compose mode.  Create any widgets and manage em.
*	Set default graph file name from main module.
*/

void
start_compose()
{
	static int once = 1;
	Widget w;
	extern Widget grfButton;	/* in widmain.c */

	if (once) {
		once = 0;
		XtAppAddActions(context, actbl, XtNumber(actbl));
	}
	if (graphfile[0]) {
		int n;

		if (directory[0]
		&& !strncmp(graphfile, directory, (n = strlen(directory)))
		&& graphfile[n] == '/') {
			strncpy(filename, graphfile + n + 1, sizeof(filename)-1);
		} else
			strncpy(filename, graphfile, sizeof(filename)-1);

	} else {
		filename[0] = 0;
	}
	XawFormDoLayout(mainForm, False);
	w = cre_cmdpanel(mainCmd);
	XtManageChild(composeCmd);
	w = cre_nodecanvas(w);
	XtManageChild(composeGraph);
	XawFormDoLayout(mainForm, True);
	set_togg(composeButton, 1);
#if 1
	XtVaSetValues (grfButton,
				   XtNsensitive, False,
				   NULL);
#endif
}

int
compose_SaveLuser ()
{
	static char msg[] =
		"graph is not saved.  quit anyway?";

	if (graphmod) {
		switch (verify (msg)) {
		case 0:					/* yes => okay to quit*/
			msg_Format ("\
Use the \"store\" button in compose mode to save the graph\n");
			return 0;
		case 1:					/* no => don't quit */
		case 2:					/* cancel => don't quit */
		default:
			return 1;
		}
	}
	else
		return 0;				/* no graph => okay to quit */
}


/*	stop_compose()
*
*	Stop method for compose mode.  Ask if want to save graph file or
*	cancel stop.
*	Returns 0 if successfully stopped, else 1.
*/

int
stop_compose()
{
	int i;
	extern Widget grfButton;	/* in widmain.c */

#ifdef SUBPROX
	/*
	 * XXX should instead count the number of nodes with NF_EDITING bit
	 * set
	 */

	if (composeSubprocCount != 0) {
		msg_Format ("You must finish editing node programs before leaving compose mode\n");
		return 1;
	}
#endif

	if (graphmod) {
		i = verify("graph modified.  save?");
		if (i == 2)
			return 1;
		if (i == 0) {
			store_cb ();
		}
	}
	XawFormDoLayout(mainForm, False);
	XtUnmanageChild(composeCmd);
	XtUnmanageChild(composeGraph);
	XawFormDoLayout(mainForm, True);
	set_togg(composeButton, 0);
#if 1
	XtVaSetValues (grfButton,
				   XtNsensitive, True,
				   NULL);
#endif
	return 0;
}

static Widget
cre_nodecanvas(below)
	Widget below;
{
	Widget retw;
	Widget w;
	int n;

	if (composeGraph)
		return composeGraph;
	n = 0;
	XtSetArg(args[n], XtNfromVert, (XtArgVal)below); n++;
	XtSetArg(args[n], XtNallowHoriz, (XtArgVal)True); n++;
	XtSetArg(args[n], XtNallowVert, (XtArgVal)True); n++;
	retw = composeGraph = XtCreateWidget("composeNodeVp", viewportWidgetClass,
			mainForm, args, n);

	n = 0;
	XtSetArg(args[n], XtNbackground, (XtArgVal)(mycolor[COLBGND])); n++;
	w = XtCreateManagedWidget("composeNodeCan", widgetClass,
		composeGraph, args, n);

/* XXX should use something other than base widget class here so we
   could set more nice resources... */

	XtRealizeWidget(composeGraph);
	graphWin = XtWindow(w);
	XtAddEventHandler(w,
		StructureNotifyMask|ExposureMask, False,
		graph_ev, (XtPointer)0);

	return composeGraph;
}

static Widget
cre_cmdpanel(below)
	Widget below;
{
	Widget retw;
	Widget w, w2, w3, w4;
	int n;

	XtTranslations translations;

	if (composeCmd)
		return composeCmd;

	n = 0;
	XtSetArg(args[n], XtNfromVert, (XtArgVal)below); n++;
	XtSetArg(args[n], XtNborderWidth, (XtArgVal)0); n++;
	XtSetArg(args[n], XtNhorizDistance, (XtArgVal)0); n++;
	retw = composeCmd = XtCreateWidget("composeCmd", formWidgetClass,
									   mainForm, args, n);

	n = 0;
	translations = XtParseTranslationTable("\
<EnterWindow>: highlight(Always)\n\
<LeaveWindow>: unhighlight()\n\
<Btn1Down>,<Btn1Up>: set()notify()\n");

	if (composeCmd)	XtSetArg(args[n], XtNleft, XtChainLeft); n++;
	XtSetArg(args[n], XtNright, XtChainLeft); n++;
	XtSetArg(args[n], XtNtop, XtChainTop); n++;
	XtSetArg(args[n], XtNbottom, XtChainTop); n++;

	w4 = w = cre_cmd_but("load", composeCmd, args, n,
		(Widget)0, (Widget)0, "load", (char*)0, 0, 0, load_cb, 0);
	w = cre_cmd_but("store", composeCmd, args, n,
		(Widget)0, w, "store", (char*)0, 0, 0, store_cb, 0);
	w = cre_cmd_but("clear", composeCmd, args, n,
		(Widget)0, w, "clear", (char*)0, 0, 0, clear_cb, 0);
	w = cre_cmd_but("critic", composeCmd, args, n,
		(Widget)0, w, "critic", (char*)0, 0, 0, critic_cb, 0);
	if (subDefsFile != NULL) {
		w = cre_cmd_but("subdefs", composeCmd, args, n,
						(Widget)0, w, "edit subdefs", (char*)0, 0, 0,
						subdefs_cb, 0);
	}
	w = cre_cmd_but("cleanup", composeCmd, args, n,
		(Widget)0, w, "cleanup", (char*)0, 0, 0, cleanup_cb, 0);
	w = cre_cmd_but("redraw", composeCmd, args, n,
		(Widget)0, w, "redraw", (char*)0, 0, 0, redraw_cb, 0);
	w = cre_tog_but("help", composeCmd, args, n,
		(Widget)0, w, "help", (char*)0, 0, 0, compose_help_cb, 0, False);


	n = 0;
	XtSetArg(args[n], XtNleft, XtChainLeft); n++;
	XtSetArg(args[n], XtNright, XtChainLeft); n++;
	XtSetArg(args[n], XtNtop, XtChainTop); n++;
	XtSetArg(args[n], XtNbottom, XtChainTop); n++;
	XtSetArg(args[n], XtNtranslations, translations); n++;
	/*
	 * XXX buttons should not "toggle", i.e. change from black->white
	 * or vice versa.  If you press a button that is already black
	 * it should stay black.  Basically this means we need to
	 * supply some other defaults for the button translations.
	 */

	w2 =
	w = cre_tog_but("", composeCmd, args, n, w4, (Widget)0, (char*)0,
		no_ic_bits, no_ic_width, no_ic_height, node_t_cb, NODE_NORMAL, 0);
	XawToggleChangeRadioGroup(w2, w2);
	w3 = cre_tog_but("", composeCmd, args, n, w4, w, (char*)0,
		bp_ic_bits, bp_ic_width, bp_ic_height, node_t_cb, NODE_PIPE, 0);
	XawToggleChangeRadioGroup(w3, w2); w = w3;
	w3 = cre_tog_but("", composeCmd, args, n, w4, w, (char*)0,
		ep_ic_bits, ep_ic_width, ep_ic_height, node_t_cb, NODE_ENDPIPE, 0);
	XawToggleChangeRadioGroup(w3, w2); w = w3;
	w3 = cre_tog_but("", composeCmd, args, n, w4, w, (char*)0,
		bl_ic_bits, bl_ic_width, bl_ic_height, node_t_cb, NODE_LOOP, 0);
	XawToggleChangeRadioGroup(w3, w2); w = w3;
	w3 = cre_tog_but("", composeCmd, args, n, w4, w, (char*)0,
		el_ic_bits, el_ic_width, el_ic_height, node_t_cb, NODE_ENDLOOP, 0);
	XawToggleChangeRadioGroup(w3, w2); w = w3;
	w3 = cre_tog_but("", composeCmd, args, n, w4, w, (char*)0,
		fo_ic_bits, fo_ic_width, fo_ic_height, node_t_cb, NODE_FANOUT, 0);
	XawToggleChangeRadioGroup(w3, w2); w = w3;
	w3 = cre_tog_but("", composeCmd, args, n, w4, w, (char*)0,
		fi_ic_bits, fi_ic_width, fi_ic_height, node_t_cb, NODE_FANIN, 0);
	XawToggleChangeRadioGroup(w3, w2); w = w3;
	w3 = cre_tog_but("", composeCmd, args, n, w4, w, (char*)0,
		bs_ic_bits, bs_ic_width, bs_ic_height, node_t_cb, NODE_COND, 0);
	XawToggleChangeRadioGroup(w3, w2); w = w3;
	w3 = cre_tog_but("", composeCmd, args, n, w4, w, (char*)0,
		es_ic_bits, es_ic_width, es_ic_height, node_t_cb, NODE_ENDCOND, 0);
	XawToggleChangeRadioGroup(w3, w2); w = w3;
	n = 0;
	XtSetArg(args[n], XtNstate, 1); n++;
	XtSetValues(w2, args, n);

	XtRealizeWidget(composeCmd);
	return composeCmd;
}

/*	popup_node_pop()
*
*	Popup node program widget.  Create it if necessary, else just relabel
*	it and move to new location.
*/

#if 0
static
popup_node_pop(node)
	Node node;
{
	Widget box;						/* for layout */
	Widget w;
	int n;
	int x, y, wd, ht;
	static Widget nodeNum = 0;		/* node# label */
	static Widget nodeLbl1 = 0;		/* label 1 */
	static Widget nodeLbl2 = 0;		/* label 2 */
	static Widget nodeConf = 0;		/* confirm button */
	char *lbl0, *lbl1, *lbl2;

	selected_node = node;
	switch (node->type) {
	case NODE_NORMAL:
		lbl0 = "Node";
		break;

	case NODE_PIPE:
		lbl0 = "Pipe";
		break;

	case NODE_LOOP:
		lbl0 = "Loop";
		break;

	case NODE_FANOUT:
		lbl0 = "Fanout";
		break;

	case NODE_COND:
		lbl0 = "Switch";
		break;

	default:
		return;
	}
	if (!nodePop) {
/*
		atbl = XtParseAcceleratorTable(
			"#override\n<Key>Return:set()notify()reset()\n");
*/

		n = 0;
		nodePop = XtCreatePopupShell("nodePop", overrideShellWidgetClass,
				topLevel, args, n);
		n = 0;
		box = XtCreateManagedWidget("", formWidgetClass,
				nodePop, args, n);

		sprintf(nodeprog, "%s %d:", lbl0, node->id);
		n = 0;
		XtSetArg(args[n], XtNleft, XtChainLeft); n++;
		XtSetArg(args[n], XtNright, XtChainLeft); n++;
		XtSetArg(args[n], XtNtop, XtChainTop); n++;
		XtSetArg(args[n], XtNbottom, XtChainTop); n++;
		XtSetArg(args[n], XtNlabel, nodeprog); n++;
		XtSetArg(args[n], XtNborderWidth, 0); n++;
		XtSetArg(args[n], XtNresizable, True); n++;
		nodeNum = XtCreateManagedWidget("", labelWidgetClass,
			box, args, n);

		nodeprog[0] = 0;
		n = 0;
		XtSetArg(args[n], XtNfromVert, nodeNum); n++;
		XtSetArg(args[n], XtNstring, nodeprog); n++;
		XtSetArg(args[n], XtNlength, (sizeof(nodeprog)-1)); n++;
		XtSetArg(args[n], XtNeditType, XawtextEdit); n++;
		XtSetArg(args[n], XtNuseStringInPlace, 1); n++;
		XtSetArg(args[n], XtNinsertPosition, 0); n++;
		XtSetArg(args[n], XtNresize, XawtextResizeBoth); n++;
		XtSetArg(args[n], XtNresizable, True); n++;
/*
		XtSetArg(args[n], XtNtranslations, ttab); n++;
*/
		nodeProg = XtCreateManagedWidget("", asciiTextWidgetClass,
				box, args, n);

		n = 0;
		XtSetArg(args[n], XtNleft, XtChainLeft); n++;
		XtSetArg(args[n], XtNright, XtChainLeft); n++;
		XtSetArg(args[n], XtNtop, XtChainTop); n++;
		XtSetArg(args[n], XtNbottom, XtChainTop); n++;
/*
		XtSetArg(args[n], XtNaccelerators, atbl); n++;
*/
		nodeConf = w = cre_cmd_but("", box, args, n,
			nodeProg, (Widget)0, "Change", (char*)0, 0, 0, edit_node_cb, 1);
		w = cre_cmd_but("", box, args, n,
			nodeProg, w, "Cancel", (char*)0, 0, 0, edit_node_cb, 0);

		XtRealizeWidget(nodePop);
/*
		XtInstallAccelerators(nodeProg, nodeConf);
*/

	} else {	/* node pop already exists */
		sprintf(nodeprog, "%s %d:", lbl0, node->id);
		n = 0;
		XtSetArg(args[n], XtNlabel, nodeprog); n++;
		XtSetValues(nodeNum, args, n);

		nodeprog[0] = 0;
		n = 0;
		XtSetArg(args[n], XtNstring, nodeprog); n++;
		XtSetArg(args[n], XtNlength, (sizeof(nodeprog)-1)); n++;
		XtSetArg(args[n], XtNinsertPosition, strlen(nodeprog)); n++;
		XtSetValues(nodeProg, args, n);
	}

	/* warp it to mouse location */

	get_mousxy(&x, &y);
	if (x) {
		get_wgtwh(nodePop, &wd, &ht);
		if (x + wd > xDispWd)
			x = xDispWd - wd;
		if (y + ht > xDispHt)
			y = xDispHt - ht;
		if (x < 0)
			x = 0;
		if (y < 0)
			y = 0;
		XtMoveWidget(nodePop, x, y);
	}
	XtPopup(nodePop, XtGrabNone);
}
#endif /* 0 */

unpop_node_pop()
{
	if (nodePop)
		XtPopdown(nodePop);
}

#if 0
/*	ready_node_pop()
*
*	Popup node property widget.  Create it if necessary, else just relabel
*	it and move to new location.
*/

static
ready_node_pop(node)
	node_t *node;
{
	Widget box;						/* for layout */
	Widget w, w2;
	int n;
	int x, y, wd, ht;
	static Widget nodeNum = 0;		/* node# label */
	static Widget nodeLbl1 = 0;		/* label 1 */
	static Widget nodeLbl2 = 0;		/* label 2 */
	static Widget nodeConf = 0;		/* confirm button */
	char *lbl0, *lbl1, *lbl2;
	XtTranslations ttab;
	XtAccelerators atbl;

	prop_node = node;
	switch (node->type) {
	case NODE_NORMAL:
		lbl0 = "Node";
		lbl1 = "Function";
		lbl2 = "Params";
		break;

	case NODE_PIPE:
		lbl0 = "Pipe";
		lbl1 = "Index";
		lbl2 = "Test";
		break;

	case NODE_LOOP:
		lbl0 = "Loop";
		lbl1 = "Index";
		lbl2 = "Test";
		break;

	case NODE_FANOUT:
		lbl0 = "Fanout";
		lbl1 = "Index";
		lbl2 = "Width";
		break;

	case NODE_COND:
		lbl0 = "Switch";
		lbl1 = "";
		lbl2 = "Test";
		break;
	}
	if (!nodePop) {
		ttab = XtParseTranslationTable("#override\n<Key>Tab:FieldAdv()\n");
		atbl = XtParseAcceleratorTable("#override\n<Key>Return:set()notify()reset()\n");

		n = 0;
/*
		nodePop = XtCreatePopupShell("nodePop", overrideShellWidgetClass,
*/
		nodePop = XtCreatePopupShell("nodePop", transientShellWidgetClass,
				topLevel, args, n);
		n = 0;
		box = XtCreateManagedWidget("", formWidgetClass,
				nodePop, args, n);

		sprintf(nodefname, "%s %d:", lbl0, node->id);
		n = 0;
		XtSetArg(args[n], XtNlabel, nodefname); n++;
		XtSetArg(args[n], XtNborderWidth, 0); n++;
		XtSetArg(args[n], XtNresizable, True); n++;
		w2 = nodeNum = XtCreateManagedWidget("", labelWidgetClass, box, args, n);

		n = 0;
		XtSetArg(args[n], XtNfromVert, w2); n++;
		XtSetArg(args[n], XtNlabel, lbl1); n++;
		XtSetArg(args[n], XtNborderWidth, 0); n++;
		XtSetArg(args[n], XtNresizable, True); n++;
		w = nodeLbl1 = XtCreateManagedWidget("", labelWidgetClass, box, args, n);

		strcpy(nodefname, node->fun ? node->fun : "");
		n = 0;
		XtSetArg(args[n], XtNfromHoriz, w); n++;
		XtSetArg(args[n], XtNfromVert, w2); n++;
		XtSetArg(args[n], XtNstring, nodefname); n++;
		XtSetArg(args[n], XtNlength, (sizeof(nodefname)-1)); n++;
		XtSetArg(args[n], XtNeditType, XawtextEdit); n++;
		XtSetArg(args[n], XtNuseStringInPlace, 1); n++;
		XtSetArg(args[n], XtNinsertPosition, 0); n++;
		XtSetArg(args[n], XtNresize, XawtextResizeBoth); n++;
		XtSetArg(args[n], XtNresizable, True); n++;
		XtSetArg(args[n], XtNtranslations, ttab); n++;
		w2 = nodeFName = XtCreateManagedWidget("", asciiTextWidgetClass,
				box, args, n);

		n = 0;
		XtSetArg(args[n], XtNfromVert, w2); n++;
		XtSetArg(args[n], XtNlabel, lbl2); n++;
		XtSetArg(args[n], XtNborderWidth, 0); n++;
		XtSetArg(args[n], XtNresizable, True); n++;
		w = nodeLbl2 = XtCreateManagedWidget("", labelWidgetClass, box, args, n);

		strcpy(nodeparam, node->par ? node->par : "");
		n = 0;
		XtSetArg(args[n], XtNfromHoriz, w); n++;
		XtSetArg(args[n], XtNfromVert, w2); n++;
		XtSetArg(args[n], XtNstring, nodeparam); n++;
		XtSetArg(args[n], XtNlength, (sizeof(nodeparam)-1)); n++;
		XtSetArg(args[n], XtNeditType, XawtextEdit); n++;
		XtSetArg(args[n], XtNuseStringInPlace, 1); n++;
		XtSetArg(args[n], XtNinsertPosition, 0); n++;
		XtSetArg(args[n], XtNresize, XawtextResizeBoth); n++;
		XtSetArg(args[n], XtNresizable, True); n++;
		XtSetArg(args[n], XtNtranslations, ttab); n++;
		w2 = nodeParam = XtCreateManagedWidget("", asciiTextWidgetClass,
				box, args, n);

		n = 0;
		XtSetArg(args[n], XtNleft, XtChainLeft); n++;
		XtSetArg(args[n], XtNright, XtChainLeft); n++;
		XtSetArg(args[n], XtNtop, XtChainTop); n++;
		XtSetArg(args[n], XtNbottom, XtChainTop); n++;
		XtSetArg(args[n], XtNaccelerators, atbl); n++;
		nodeConf = w = cre_cmd_but("", box, args, n,
			w, (Widget)0, "Change", (char*)0, 0, 0, node_prop_cb, 1);
		n--;
		w = cre_cmd_but("", box, args, n,
			w, (Widget)0, "Cancel", (char*)0, 0, 0, node_prop_cb, 0);

		XtRealizeWidget(nodePop);
		XtInstallAccelerators(nodeFName, nodeConf);
		XtInstallAccelerators(nodeParam, nodeConf);

	} else {	/* node pop already exists */
		sprintf(nodefname, "%s %d:", lbl0, node->id);
		n = 0;
		XtSetArg(args[n], XtNlabel, nodefname); n++;
		XtSetValues(nodeNum, args, n);
		n = 0;
		XtSetArg(args[n], XtNlabel, lbl1); n++;
		XtSetValues(nodeLbl1, args, n);
		strcpy(nodefname, node->fun ? node->fun : "");
		n = 0;
		XtSetArg(args[n], XtNstring, nodefname); n++;
		XtSetArg(args[n], XtNlength, (sizeof(nodefname)-1)); n++;
		XtSetValues(nodeFName, args, n);
		n = 0;
		XtSetArg(args[n], XtNinsertPosition, strlen(nodefname)); n++;
		XtSetValues(nodeFName, args, n);
		n = 0;
		XtSetArg(args[n], XtNlabel, lbl2); n++;
		XtSetValues(nodeLbl2, args, n);
		strcpy(nodeparam, node->par ? node->par : "");
		n = 0;
		XtSetArg(args[n], XtNstring, nodeparam); n++;
		XtSetArg(args[n], XtNlength, (sizeof(nodeparam)-1)); n++;
		XtSetValues(nodeParam, args, n);
		n = 0;
		XtSetArg(args[n], XtNinsertPosition, strlen(nodeparam)); n++;
		XtSetValues(nodeParam, args, n);
	}

	/* warp it to mouse location */

	get_mousxy(&x, &y);
	if (x) {
		get_wgtxy(nodePop, &wd, &ht);
		x += wd;
		y += ht;
		get_wgtxy(nodeFName, &wd, &ht);
		x -= wd;
		y -= ht;
		get_wgtwh(nodeFName, &wd, &ht);
		x -= 30;
		y -= ht / 2;
		get_wgtwh(nodePop, &wd, &ht);
		if (x + wd > xDispWd)
			x = xDispWd - wd;
		if (y + ht > xDispHt)
			y = xDispHt - ht;
		if (x < 0)
			x = 0;
		if (y < 0)
			y = 0;
		XtMoveWidget(nodePop, x, y);
	}
	XtPopup(nodePop, XtGrabNone);
}

unpop_node_pop()
{
	if (nodePop)
		XtPopdown(nodePop);
}
#endif /* 0 */

/*****************************
*  compose widget callbacks  *
*                            *
*****************************/

/*	load_cb()
*
*	Load command.
*/

static void
load_cb(w, cli, cd)
	Widget w;
	XtPointer cli;
	XtPointer cd;
{
	int can;
	FILE *ff;

#ifdef SUBPROX
	if (composeSubprocCount != 0) {
		msg_Format ("can't load a new graph while nodes are being edited\n");
		return;
	}
#endif

	if (graphmod) {
		can = verify("graph modified.  save?");
		if (can == 2)
			return;
		if (can == 0) {
			store_cb();
		}
	}
#if 1
	can = fileVerify ("load graph from file:", filename, sizeof(filename),
					  "*.gr", FS_MustBeReadable);
#else
	can = filever("load graph from file:", filename, sizeof(filename));
#endif
	if (!can) {
		if (!filename[0]) {
			message("no filename.");
			return;
		}
		if (!(ff = fopen(filename, "r"))) {
			sprintf(crud, "\"%s\": can't read.", filename);
			message(crud);
			return;
		}
		gr_Free(scratchgraph);
		scratchgraph = parse_Program((int(*)())fgetc, (void*)ff, filename);
		fclose(ff);
		graphmod = 0;
		sprintf(crud,
			(scratchgraph ? "\"%s\" loaded." : "\"%s\": errors in file."),
			filename);
		message(crud);
		if (!scratchgraph)
			scratchgraph = gr_New();
		set_graphfile(filename);
		XClearWindow(xDisp, graphWin);
		compose_refresh();
	}
}

/*	store_cb()
*
*	Store button hit or stop_compose invoked.
*/

static void
store_cb()
{
	int can;
	FILE *ff;

#ifdef SUBPROX
	if (composeSubprocCount != 0) {
		msg_Format ("You must finish editing nodes before storing graph\n");
		return;
	}
#endif

#if 1
	can = fileVerify ("store graph in file:", filename, sizeof(filename),
					  "*.gr", FS_RegularFile|FS_MustBeWritable);
#else
	can = filever("store graph in file:", filename, sizeof(filename));
#endif
	if (!can) {
		if (!filename[0]) {
			message("no filename.");
			return;
		}
		if (!(ff = fopen(filename, "w"))) {
			sprintf(crud, "\"%s\": can't write.", filename);
			message(crud);
			return;
		}
		/* XXX probably should check for errors here and handle somehow. */
		(void) unparse_Program (scratchgraph, ff);
		fclose(ff);
		graphmod = 0;
		sprintf(crud, "\"%s\": graph written.", filename);
		message(crud);
		set_graphfile(filename);
	}
}

/*	cleanup_cb()
*
*	Cleanup command.
*/

static void
cleanup_cb(w, cli, cd)
	Widget w;
	XtPointer cli;
	XtPointer cd;
{
	int wd, ht;

#ifdef SUBPROX
	if (composeSubprocCount != 0) {
		msg_Format ("Can't cleanup graph while nodes are being edited.\n");
		return;
	}
#endif

	if (gr_HasCycles (scratchgraph)) {
		msg_Format ("can't cleanup: graph has a loop.\n");
		return;
	}
	get_wgtwh(composeGraph, &wd, &ht);
	xgr_PlaceNodes(scratchgraph, wd, ht);
	XClearWindow(xDisp, graphWin);
	compose_refresh();
}

/*	clear_cb()
 *
 *	Clear command.
 */

static void
clear_cb(w, cli, cd)
	Widget w;
	XtPointer cli;
	XtPointer cd;
{
#ifdef SUBPROX
	/*
	 * XXX should instead count the number of nodes with NF_EDITING bit
	 * set
	 */

	if (composeSubprocCount != 0) {
		msg_Format ("You must quit editing node programs before clearing the graph.\n");
		return;
	}
#endif

	if (graphmod && verify("graph modified.  really clear?"))
		return;
	gr_Free(scratchgraph);
	scratchgraph = gr_New();
	graphmod = 0;
	XClearWindow(xDisp, graphWin);
	message("graph cleared.");
}

/*	redraw_cb()
 *
 *	Redraw command
 */

static void
redraw_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{

#if 0
	Node n;

	for (n = scratchgraph; n; n = n->next)
		n->flags &= ~(NFLAG_CRITE|NFLAG_CRITW);
#endif

	XClearWindow(xDisp, graphWin);
	xgr_DrawGrf(graphWin, scratchgraph, 0);
}


/*  help_expose()
*
*   Redraw help window on expose
*/

static void
help_expose(w, cli, xev, rem)
    Widget w;
    XtPointer cli;
    XEvent *xev;
    Boolean *rem;
{
    if (xev->type == Expose && xev->xexpose.count == 0)
        xgr_drawComposeHelp(helpWin);
}

static void
compose_help_cb(w, cli, cd)
    Widget w;
    XtPointer cli;
    XtPointer cd;
{
    int n;
    Widget canvas;

    if (helpVisible = !helpVisible) {
        if (!helpPop) {
            helpPop = XtCreatePopupShell ("Compose Help",
										  topLevelShellWidgetClass,
										  topLevel,
										  (ArgList) NULL, 0);

            n = 0;
            XtSetArg(args[n], XtNheight, compose_help_height); n++;
            XtSetArg(args[n], XtNwidth, compose_help_width); n++;
            canvas = XtCreateManagedWidget("helpcanvas",
                widgetClass, helpPop, args, n);

            XtPopup(helpPop, XtGrabNone);

            helpWin = XtWindow(canvas);
            XtAddEventHandler(canvas, ExposureMask, False, help_expose , NULL);

        } else {
            XtPopup(helpPop, XtGrabNone);
        }

    } else {
        XtPopdown(helpPop);
    }
}



/*	critic_cb()
*
*	Critic command.
*/

#include "graphP.h"
static void
critic_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	int e;

	e = gr_Critic(scratchgraph);
	if (e == 0)
		e = subdefs_CheckGraph (scratchgraph, subDefsFile);
	subdefs_NukeDecls ();

	if (e) {
		sprintf(crud, "%d errors found.", e);
		message(crud);
	}
	else
		message("no errors found.");
	XClearWindow(xDisp, graphWin);
	xgr_DrawGrf(graphWin, scratchgraph, 0);
}

static void
subdefs_cb (w, cli, cd)
Widget w;
XtPointer cli, cd;
{
	static void compose_EditNodeSourceProgram ();

	if (subDefsFile != NULL)
		compose_EditNodeSourceProgram (subDefsFile);
}

/*	node_t_cb()
*
*	Node type has been selected
*/

static void
node_t_cb(w, cli, cd)
	Widget w;
	XtPointer cli;	/* node type */
	XtPointer cd;
{
	curnodetype = (int)cli;
}

#if 0
/*	edit_node_cb()
 *
 *	Confirm/cancel has been selected in node property popup.
 */

static void
edit_node_cb(w, cli, cd)
Widget w;
XtPointer cli;	/* confirm/cancel */
XtPointer cd;
{
	XtPopdown(nodePop);
	if ((int)cli && selected_node) {
		draw_node(graphWin, selected_node, 0);
		draw_node(graphWin, selected_node, 1);
		graphmod = 1;
	}
	selected_node = 0;
}
#endif

#if 0
/*	node_prop_cb()
*
*	Confirm/cancel has been selected in node property popup.
*/

static void
node_prop_cb(w, cli, cd)
	Widget w;
	XtPointer cli;	/* confirm/cancel */
	XtPointer cd;
{
	XtPopdown(nodePop);
	if ((int)cli && prop_node) {
		draw_node(graphWin, prop_node, 0);
		if (prop_node->type != NODE_COND) {
			if (prop_node->fun)
				free(prop_node->fun);
			prop_node->fun = strsave(nodefname);
		}
		if (prop_node->par)
			free(prop_node->par);
		prop_node->par = strsave(nodeparam);
		draw_node(graphWin, prop_node, 1);
		graphmod = 1;
	}
	prop_node = 0;
}
#endif /* 0 */

/********************************
*  graph window event handling  *
*                               *
********************************/

/*	graph_ev()
*
*	Event in graph window
*/

static void
graph_ev(w, cli, xev, rem)
	Widget w;
	XtPointer cli;
	XEvent *xev;
	Boolean *rem;
{
	XButtonEvent *bev = (XButtonEvent*)xev;
	XConfigureEvent *cev = (XConfigureEvent*)xev;
	XExposeEvent *eev = (XExposeEvent*)xev;
	XMotionEvent *mev = (XMotionEvent*)xev;
	XCrossingEvent *rev = (XCrossingEvent*)xev;
	/* int x, y; */

	switch (xev->type) {

	case Expose:
		if (eev->count == 0)
			compose_refresh();
		break;

	case ConfigureNotify:
		break;

	}
}

/*************************
*  graph window actions  *
*                        *
*************************/

static int
get_event_xy(xev, xp, yp)
	XEvent *xev;
	int *xp, *yp;
{
	switch (xev->type) {
	case KeyPress:
	case KeyRelease:
		*xp = xev->xkey.x;
		*yp = xev->xkey.y;
		return 1;
		break;

	case ButtonPress:
	case ButtonRelease:
		*xp = xev->xbutton.x;
		*yp = xev->xbutton.y;
		return 1;
		break;

	case MotionNotify:
		*xp = xev->xmotion.x;
		*yp = xev->xmotion.y;
		return 1;
		break;

	case EnterNotify:
	case LeaveNotify:
		*xp = xev->xcrossing.x;
		*yp = xev->xcrossing.y;
		return 1;
		break;

	default:
		return 0;
	}
}

/*	compose_add_node()
*
*	Action - add node to graph at location
*/

static void
compose_add_node(wgt, xev, par, nump)
	Widget wgt;
	XEvent *xev;
	String *par;
	int *nump;
{
	Node n;
	int x, y;

	if (get_event_xy(xev, &x, &y)) {
		n = gr_NewNode(gr_HiNodeId(scratchgraph) + 1);
		n->node_type = curnodetype;
		n->xy.x = x;
		n->xy.y = y;
		n->xy.set = 1;
		gr_AddNode(scratchgraph, n);
		xgr_DrawNode(graphWin, n, 1);
		graphmod = 1;
	}
}

/*	compose_delete_node()
*
*	Action - delete nearest node from graph
*/

static void
compose_delete_node(wgt, xev, par, nump)
	Widget wgt;
	XEvent *xev;
	String *par;
	int *nump;
{
	Node n;
	int x, y;

	if (get_event_xy(xev, &x, &y) && (n = xgr_NearestNode(scratchgraph, x, y))) {

#if 0
/* XXX for debugging */
if (*nump == 1 && par[0][0] == 'c') {
	gr_CopySubgraph(scratchgraph, n, 0, 100, 20);
	XClearWindow(xDisp, graphWin);
	compose_refresh();
	return;
}
if (*nump == 1 && par[0][0] == 'C') {
	gr_CopySubgraph(scratchgraph, n, 1, 100, 20);
	XClearWindow(xDisp, graphWin);
	compose_refresh();
	return;
}
#endif
		gr_RemNode(scratchgraph, n);
		gr_FreeNode(n);
		XClearWindow(xDisp, graphWin);
		xgr_DrawGrf(graphWin, scratchgraph, 0);
		graphmod = 1;
		if (selected_node == n) {
			unpop_node_pop();
			selected_node = 0;
		}
	}
}

/*	compose_move_node()
*
*	Action - move existing node
*		param = "begin", "drag", "cancel", "finish"
*/

static void
compose_move_node(wgt, xev, par, nump)
	Widget wgt;
	XEvent *xev;
	String *par;
	int *nump;
{
	static Node curnode = 0;	/* node we are moving */
	static int ox, oy;			/* mouse-node offset */
	int x, y;

	if (*nump != 1 || !get_event_xy(xev, &x, &y))
		return;
	if (par[0][0] == 'b') {	/* begin */
		if (!(curnode = xgr_NearestNode(scratchgraph, x, y)))
			return;
		xgr_DrawNode(graphWin, curnode, 0);
		xgr_DrawNode(graphWin, curnode, 2);
		ox = curnode->xy.x - x;
		oy = curnode->xy.y - y;

	} else {
		if (!curnode)
			return;
		switch (par[0][0]) {
		case 'd':	/* drag */
			xgr_DrawNode(graphWin, curnode, 2);
			curnode->xy.x = x + ox;
			curnode->xy.y = y + oy;
			xgr_DrawNode(graphWin, curnode, 2);
			break;

		case 'c':	/* cancel */
			xgr_DrawNode(graphWin, curnode, 2);
			xgr_DrawNode(graphWin, curnode, 1);
			curnode = 0;
			XClearWindow(xDisp, graphWin);
			xgr_DrawGrf(graphWin, scratchgraph, 0);
			graphmod = 1;
			break;

		case 'f':	/* finish */
			xgr_DrawNode(graphWin, curnode, 2);
			curnode->xy.x = x + ox;
			curnode->xy.y = y + oy;
			xgr_DrawNode(graphWin, curnode, 1);
			curnode = 0;
			XClearWindow(xDisp, graphWin);
			xgr_DrawGrf(graphWin, scratchgraph, 0);
			graphmod = 1;
			break;
		}
	}
}

#define	DRAWHEADTOTAIL 0	/* which way compose_add_arc() draws */

/*	compose_add_arc()
*
*	Action - add new arc to graph
*		param = "begin", "drag", "cancel", "finish"
*/

static void
compose_add_arc(wgt, xev, par, nump)
	Widget wgt;
	XEvent *xev;
	String *par;
	int *nump;
{
	static Node startnode = 0;	/* for drawing arcs */
	Node n;
	static Node fake = 0;		/* this isn't a real node; need it to draw */
	int x, y;

	if (!fake)
		fake = gr_NewNode(0);
	if (*nump != 1 || !get_event_xy(xev, &x, &y))
		return;
	if (par[0][0] == 'b') {	/* begin */
		if (!(startnode = xgr_NearestNode(scratchgraph, x, y)))
			return;
		fake->xy.x = x;
		fake->xy.y = y;
		xgr_DrawArc(graphWin, startnode, fake, 2);
		return;

	} else {

		if (!startnode)
			return;
		if (fake->xy.x != -1)
			if (DRAWHEADTOTAIL)
				xgr_DrawArc(graphWin, fake, startnode, 2);
			else
				xgr_DrawArc(graphWin, startnode, fake, 2);

		switch (par[0][0]) {
		case 'd':	/* drag */
			fake->xy.x = x;
			fake->xy.y = y;
			if (DRAWHEADTOTAIL)
				xgr_DrawArc(graphWin, fake, startnode, 2);
			else
				xgr_DrawArc(graphWin, startnode, fake, 2);
			break;

		case 'c':	/* cancel */
			startnode = 0;
			break;

#if 0
/* XXX for debugging */
case 'p':	/* pair */
	if ((n = xgr_NearestNode(scratchgraph, x, y)) && n != startnode) {
		n->pair = startnode;
		startnode->pair = n;
		XClearWindow(xDisp, graphWin);
		compose_refresh();
	}
	startnode = 0;
	break;
#endif

		case 'f':	/* finish */
			if ((n = xgr_NearestNode(scratchgraph, x, y)) && n != startnode)
				switch (gr_BuildArc(scratchgraph,
						(DRAWHEADTOTAIL ? n->nk.id : startnode->nk.id),
						(DRAWHEADTOTAIL ? startnode->nk.id : n->nk.id))) {
				case 0:
					if (DRAWHEADTOTAIL)
						xgr_DrawArc(graphWin, n, startnode, 1);
					else
						xgr_DrawArc(graphWin, startnode, n, 1);
					graphmod = 1;
					break;

				case 1:
					message("arc already exists.");
					break;

				case 3:
					message("arc would cause a loop.");
					break;
				}
			startnode = 0;
			break;
		}
	}
}

/*	compose_delete_arc()
*
*	Action - delete nearest arc from drawing.
*/

static void
compose_delete_arc(wgt, xev, par, nump)
	Widget wgt;
	XEvent *xev;
	String *par;
	int *nump;
{
	Node n1, n2;
	int x, y;

	if (get_event_xy(xev, &x, &y)) {
		xgr_NearestArc(scratchgraph, x, y, &n1, &n2);
		if (n1) {
			xgr_DrawArc(graphWin, n1, n2, 0);
			xgr_DrawArc(graphWin, n2, n1, 0);
			gr_RemAnArc(scratchgraph, n1, n2);
			gr_RemAnArc(scratchgraph, n2, n1);
			graphmod = 1;
		}
	}
}

/*	compose_move_arc()
*
*	Action - move existing arc
*		param = "begin", "drag", "cancel", "finish"
*/

static void
compose_move_arc(wgt, xev, par, nump)
	Widget wgt;
	XEvent *xev;
	String *par;
	int *nump;
{
	static Node n1 = 0, n2 = 0;	/* closest and other node of nearest arc */
	static hd1;					/* n1 is head? */
	static Node fake = 0;		/* this isn't a real node; need it to draw */
	Node n;
#if 0
	int noa;
#endif
	int x, y;

	if (!fake)
		fake = gr_NewNode(0);
	if (*nump != 1 || !get_event_xy(xev, &x, &y))
		return;
	if (par[0][0] == 'b') {	/* begin */
		xgr_NearestArc(scratchgraph, x, y, &n2, &n1);
		if (n1) {
			hd1 = gr_IsArc(n2, n1);
			xgr_DrawArc(graphWin, (hd1 ? n2 : n1), (hd1 ? n1 : n2), 0);
			fake->xy.x = x;
			fake->xy.y = y;
			xgr_DrawArc(graphWin, (hd1 ? fake : n1), (hd1 ? n1 : fake), 2);
		}
		return;

	} else {
		if (!n1)
			return;
		if (fake->xy.x != -1)
			xgr_DrawArc(graphWin, (hd1 ? fake : n1), (hd1 ? n1 : fake), 2);
		switch (par[0][0]) {
		case 'd':	/* drag */
			fake->xy.x = x;
			fake->xy.y = y;
			xgr_DrawArc(graphWin, (hd1 ? fake : n1), (hd1 ? n1 : fake), 2);
			break;

		case 'c':	/* cancel */
			xgr_DrawArc(graphWin, (hd1 ? n2 : n1), (hd1 ? n1 : n2), 1);
			n1 = 0;
			break;

		case 'f':	/* finish */
			if (n = xgr_NearestNode(scratchgraph, x, y)) {
				gr_RemAnArc(scratchgraph, (hd1 ? n2 : n1), (hd1 ? n1: n2));
				switch (gr_BuildArc(scratchgraph,
							(hd1 ? n : n1)->nk.id,
							(hd1 ? n1 : n)->nk.id)) {
				case 0:
					xgr_DrawArc(graphWin, (hd1 ? n : n1), (hd1 ? n1 : n), 1);
					graphmod = 1;
					break;

				case 1:
					message("arc already exists.");
					break;
				}
			} else
				xgr_DrawArc(graphWin, (hd1 ? n2 : n1), (hd1 ? n1 : n2), 1);
			n1 = 0;
			break;
		}
	}
}


#ifdef SUBPROX
struct nodeprogram {
	Node node;
	char *filename;
};

/*
 * callback for when the editor for a node program (the HeNCE glue,
 * not the C or FORTRAN source), is killed.
 */

static void
compose_NodeGlueEditorKilled (pid, status, p)
int pid;
int status;
struct nodeprogram *p;
{
	msg_Format ("The editor for node %d was killed with signal %d.\n",
				p->node->nk.id, status);
	p->node->flags &= ~NF_EDITING;
	(void) unlink (p->filename);
	free (p->filename);
	free (p);
	--composeSubprocCount;
}

/*
 * Callback for when a node glue program exits.  If editor exits normally,
 * and node program parses correctly, splice it back into the graph.
 * If the parse was incorrect, give user a chance to edit again.
 * If we aren't editing again, free up our data structures.
 */

static void
compose_NodeGlueEditDone (pid, status, p)
int pid;
int status;
struct nodeprogram *p;
{
	Node newnode;

	if (status == 0) {
		FILE *fp;

		if ((fp = fopen (p->filename, "r")) == NULL) {
			msg_Format ("can't read tmp file %s\n", p->filename);
			goto cleanup;
		}
		newnode = parse_Node ((int (*)()) fgetc, (void *) fp, NULL);
		fclose (fp);
		if (newnode != (Node) NULL) {
			/*
			 * The new node program parses without errors.  Replace
			 * the old node in the graph with the new one.
			 */
			msg_Format ("Node %d: no errors.", newnode->nk.id);
			xgr_DrawNode (graphWin, p->node, 0);
			gr_PatchNode (p->node, newnode);
			gr_FreeNode (newnode);
			p->node->state = ST_NOT_BEGUN;
			xgr_DrawNode (graphWin, p->node, 2);
			compose_refresh ();
			goto cleanup;
		}
		else {
			/*
			 * parser detected node program errors.  Offer to edit
			 * again, otherwise throw away this node.
			 */
			char msgbuf[100];

			sprintf (msgbuf, "Error in program for node %d.  Edit again?",
					 p->node->nk.id);
			xgr_DrawNode (graphWin, p->node, 0);
			p->node->state = ST_WARNING;
			xgr_DrawNode (graphWin, p->node, 2);
			if (verify (msgbuf) == 0) {
				compose_refresh ();
				sprintf (crud, editorCommand, p->filename, p->filename,
						 p->filename);
				if (subproc_Spawn (crud, compose_NodeGlueEditDone,
								   compose_NodeGlueEditorKilled, p) > 0) {
					return;
				}
				msg_Format ("couldn't rerun editor\n");
				goto cleanup;
			}
		}
	}
 cleanup:
	--composeSubprocCount;
	p->node->flags &= ~NF_EDITING;
	(void) unlink (p->filename);
	free (p->filename);
	free (p);
}

/*
 * Come here to edit a node glue program.
 */

static void
compose_EditNodeGlue (n)
Node n;
{
	struct nodeprogram *p = talloc (1, struct nodeprogram);
	FILE *fp;

	if (n->flags & NF_EDITING) {
		msg_Format ("Already editing node %d\n", n->nk.id);
		return;
	}
	p->node = n;
	p->filename = strsave (tmpnam ((char *) NULL));
	if ((fp = fopen (p->filename, "w")) == NULL) {
		msg_Format ("can't write tmp file %s\n", p->filename);
		goto fail;
	}
	unparse_Node (n, fp);
	(void) fclose (fp);

	p->node->flags |= NF_EDITING;
	if (editorCommand && *editorCommand) {
		sprintf (crud, editorCommand, p->filename, p->filename, p->filename);
		if (subproc_Spawn (crud, compose_NodeGlueEditDone,
						   compose_NodeGlueEditorKilled, p) > 0) {
			++composeSubprocCount;
			return;
		}
	}
	else
		msg_Format ("no editorCommand resource defined -- can't edit\n");

 fail:
	p->node->flags &= ~NF_EDITING;
	(void) unlink (p->filename);
	free (p->filename);
	free (p);
}

/*
 * Callback for when a node source (C or FORTRAN) editor dies, either
 * via exit() or by being killed.   Prints a message if killed and
 * frees up the filename.
 */

static void
compose_EditSourceDone (pid, status, filename, howDied)
int pid, status;
char *filename;
enum causeOfDeath howDied;
{
	if (howDied == SUBPROC_KILLED) {
		msg_Format ("editor pid %d was killed with signal %d\n",
					pid, status);
	}
	free (filename);
	--composeSubprocCount;
	return;
}

/*
 * Come here to edit a C or FORTRAN node implementation.
 * save the filename in malloc'ed space (to be freed by
 * compose_EditSourceDone), and spawn a subproc to do the
 * editing.
 *
 * XXX need to make sure user doesn't edit the same file
 * twice.
 */

static void
compose_EditNodeSourceProgram (filename)
char *filename;
{
	char *ptr = strsave (filename);

	sprintf (crud, editorCommand, ptr, ptr, ptr);
	if (subproc_Spawn (crud, compose_EditSourceDone,
					   compose_EditSourceDone, ptr) > 0) {
		++composeSubprocCount;
		return;
	}
	free (filename);
}

/*	compose_edit_node()
 *
 *	Action - popup window to edit node characteristics.
 *		param = "source", "program"
 */

static void
compose_edit_node(wgt, xev, par, nump)
	Widget wgt;
	XEvent *xev;
	String *par;
	int *nump;
{
	Node enode;
#if 0
	Node newnode;
#endif
	int x, y;
	char filename[100];

	if (*nump != 1 || !get_event_xy(xev, &x, &y))
		return;
	if (!(enode = xgr_NearestNode(scratchgraph, x, y)))
		return;

	switch (par[0][0]) {
	case 'p':	/* edit node program in graph */
		compose_EditNodeGlue (enode);
		break;

	case 's':	/* edit source code for node implementation */
		if (enode->node_type != NODE_NORMAL) {
			msg_Format ("%d: not a regular node\n", enode->nk.id);
			break;
		}
		if (editorCommand == NULL || *editorCommand == '\0') {
			msg_Format ("No editor command defined - can't edit\n");
			break;
		}
		if (enode->sub_name == NULL || enode->sub_name[0] == '\0') {
			msg_Format ("Node %d has no function name\n", enode->nk.id);
			break;
		}
		sprintf (filename, "%s.%s", enode->sub_name,
				 Language == LANG_C ? "c" : "f");
		compose_EditNodeSourceProgram (filename);
		break;

	default:
		break;
	}
}
#else /* not SUBPROX */
/*	compose_edit_node()
*
*	Action - popup window to edit node characteristics.
*		param = "source", "program"
*/

static void
compose_edit_node(wgt, xev, par, nump)
	Widget wgt;
	XEvent *xev;
	String *par;
	int *nump;
{
	FILE *ff;
	Node enode, newnode;
	int x, y;
	char *fname;

	if (*nump != 1 || !get_event_xy(xev, &x, &y))
		return;
	if (!(enode = xgr_NearestNode(scratchgraph, x, y)))
		return;

	switch (par[0][0]) {
	case 'p':	/* edit node program in graph */

		/* write node program to a file so we can edit it */
		fname = tmpnam((char*)0);
		if (!(ff = fopen(fname, "w"))) {
			sprintf(crud, "%s: can't write.", fname);
			message(crud);
			break;
		}
		unparse_Node(enode, ff);
		(void)fclose(ff);

		while (1) {
			/* edit that sucker */
			if (editorCommand && *editorCommand) {
				sprintf(crud, editorCommand, fname, fname, fname);
				system(crud);
			} else {
				msg_Format ("No editorCommand resource defined - cannot edit\n");
				break;
			}

			/* read node program back in  */
			if (!(ff = fopen(fname, "r"))) {
				sprintf(crud, "%s: can't read.", fname);
				message(crud);
				break;
			}
			newnode = parse_Node((int(*)())fgetc, (void*)ff, NULL);
			(void)fclose(ff);

			if (newnode) {
				msg_Format ("no errors.");
				xgr_DrawNode(graphWin, enode, 0);
				gr_PatchNode(enode, newnode);
				gr_FreeNode(newnode);
				enode->state = ST_NOT_BEGUN;
				xgr_DrawNode(graphWin, enode, 2);
				compose_refresh ();
				break;
			}
			else {
				xgr_DrawNode(graphWin, enode, 0);
				enode->state = ST_WARNING;
				xgr_DrawNode(graphWin, enode, 2);
				if (verify ("Error in node program. Edit again?") == 0) {
					compose_refresh ();
					continue;
				}
				break;
			}
		}

		(void)unlink(fname);
		break;

	case 's':	/* edit source code for node implementation */
		if (enode->node_type == NODE_NORMAL) {
			if (enode->sub_name && *enode->sub_name) {
				if (editorCommand && *editorCommand) {
					sprintf(crud, editorCommand,
						enode->sub_name, enode->sub_name, enode->sub_name);
					system(crud);
				} else {
					msg_Format ("No editor program defined - can't edit.\n");
					break;
				}

			} else {
				sprintf(crud, "%d: node has no function.", enode->nk.id);
				message(crud);
			}

		} else {
			sprintf(crud, "%d: not a regular node.", enode->nk.id);
			message(crud);
		}
		break;

	default:
		break;
	}
}
#endif /* SUBPROX */

void
compose_refresh()
{
	xgr_DrawGrf(graphWin, scratchgraph, 0);
}

/*
 * Local variables:
 * tab-width:4
 * End:
 */
