/* This program studies the evolution of the radiation field. The method
   used is the same as in nlin.c. The following set of equations are
   integrated thru the interaction region:

        d(gamma)
        -------- = -E sin[phi+phi0(z)]
           dz

         d(phi)           beta_p
         ------  = kL(1 - ------)
           dz              beta
*/

#include <stdio.h>
#include <math.h>
#define INITENG     2.0     /* gamma */
#define CAVILEN     1000.0  /* cavity length in unit of radiation wavelength */
#define NUMPART     2000    /* the number of test particles */
#define	STEP        0.005
#define NUMPOINTS   100     /* the number of data points */
#define OFFSET      1.303   /* the detuning factor */
#define TWOPI       6.28318530717959
#define gam(beta)   (1.0/sqrt(1.0 - beta*beta))
#define	beta(gam)   sqrt(1.0-1.0/(gam*gam))
#define sepx(b,bp)  (kL*((1.0-bp*b)/sqrt(1.0-b*b)-sqrt(1.0-bp*bp)))

main()
{
    register int  i;
    double        beta0, betap, g0, kL, maxE, step, trap;
    double        dgam[NUMPOINTS], engLost();

    /* constants */
    beta0 = sqrt(1.0 - 1.0/(INITENG*INITENG));
    kL = TWOPI*CAVILEN;
    betap = beta0*(1.0 - 2.0*OFFSET/kL);
    g0 = kL/(INITENG*INITENG*INITENG*beta0*beta0);
    trap = 0.5*sepx(beta0, betap);
    maxE = 80.0/g0;
    step = maxE/(double)NUMPOINTS;
    for (i = 0;  i < NUMPOINTS; i++) {
	dgam[i] = engLost(trap+(double)i*step, beta0, betap, kL);
	printf("%0.4g  %0.4g\n", trap+(double)i*step, dgam[i]/(INITENG-1.0));
    }
    fprintf(stderr, "mission completed\007\n");
}

/* average energy loss per electron */
double engLost(field, beta0, betap, kL)
double  field, beta0, betap, kL;
{
    register int  i;
    double        gamma[NUMPART], phi[NUMPART];
    double        dphi, gamsum, z, dz;

    /* initialization */
    dphi = TWOPI/(double)NUMPART;
    for (i = 0; i < NUMPART; i++) {
        gamma[i] = gam(beta0);
        phi[i] = i*dphi;
    }

    /* now do the integration */
    dz = STEP;
    for (z = 0.0; z < 1.0; z += dz) {
        for (i = 0; i < NUMPART; i++) {
            gamma[i] -= field*sin(phi[i])*dz;
            phi[i] += kL*(1.0-betap/beta(gamma[i]))*dz;
        }
    }
    gamsum = 0.0;
    for (i = 0; i < NUMPART; i++)
        gamsum += gamma[i];
    return (INITENG-gamsum/(double)NUMPART);
}
