/*
 * $Source$
 * $Revision$
 * $Date$
 * $Author$
 */

/* Eval.c defines routines for evaluating expressions. Expression 
 * evaluation is basically a lot of tedium and keeping track of types and
 * error cases, but isn't very complex otherwise. There are really only 
 * two main routines here: hence_eval_exp() and hence_eval_param(). The 
 * former evals an Exp and returns when the Exp has been reduced to it's 
 * lowest form -- a constant (or in some cases an address of a 
 * Parameter). The latter evals a parameter's "val" field, and reduces it
 * to its lowest form. When eval_exp runs into a parameter, it evaluates 
 * the parameter, and then substitues the parameter's value for the 
 * parameter in the expression. 
 * 
 * Hence_eval_exp and Hence_eval_param are both called with binary flags.
 * These tell the routines whether or not certain things are illegal in 
 * the evaluation -- for example, sometimes arrays and addresses cannot 
 * be dealt with. 
 * 
 * Routines which call hence_eval_exp() and hence_eval_param() should 
 * include eval.h. 
 */

#include <stdio.h>
#include "std.h"
#include "rb.h"
#include "dlist.h"
#include "hence.h"
#include "htypes.h"
#include "eval.h"
#include "yucky.h"

#define arrays_ok(flags) (flags & EVAL_ARRAY_OK)
#define address_ok(flags) (flags & EVAL_ADDRESS_OK)

/* ---------------------------------------------------------------------- */
/* Error codes */

#define FOUND_NULL 	100
#define FOUND_ARRAY	101
#define FOUND_NOT_BIT	102
#define GENERAL_ADDRESS	103
#define FOUND_ADDRESS	104
#define BAD_INT_EXP	105
#define INDIRECTION	106
#define NOT_IMP		107
#define USES_ITSELF	108
#define UNK_TYPE	109

/* ---------------------------------------------------------------------- */
/* The main routines */

hence_eval_exp(e, flags)
Exp e;
int flags;
{
  eval_exp(e, flags, e);
}

hence_eval_param(p, flags)
Param p;
int flags;
{
  push_exp('P', (char *) p, p->type);
  eval_param(p, flags, last_exp());
}

/* ---------------------------------------------------------------------- */
/* Eval_param and eval_exp are to be called recursively.  Orig is maintained
 * in these calls for error flagging */

static eval_param(p, flags, orig)
Param p;
int flags;
Exp orig;
{
  if (p->type == -1) eval_param_error(UNK_TYPE, p, flags, orig);
  if (p->val == ENULL) eval_param_error(FOUND_NULL, p, flags, orig);
  if (p->type == ARRAY && !arrays_ok(flags)) {
    eval_param_error(FOUND_ARRAY, p, flags, orig);
  }
  if (p->state == 2) return;
  if (p->state == 3) eval_param_error(USES_ITSELF, p, flags, orig);

  p->state = 3;

  eval_exp(p->val, flags, orig);
  if (p->val->type != p->type) {
    if (p->type == ARRAY) {
      if (p->a->nadims == 0) {
        if (p->a->type != p->val->type) {
          convert_exp_type(p->val, p->a->type);
        }
      } else {
        fprintf(stderr, "ERROR: Array parameter: ");
        fprint_param_brief(stderr, p);
        fprintf(stderr, "\n  cannot be set to scalar value: ");
        fprint_exp(stderr, p->val);
        nice_bail("\n");
      }
    } else {
      convert_exp_type(p->val, p->type);
    }
  }
  p->state = 2;
  return;
}

static eval_param_error(error, p, flags, orig)
int error;
Param p;
int flags;
Exp orig;
{
  fprintf(stderr, "ERROR evaluating parameter: ");
  fprint_param(stderr, p);
  fprintf(stderr, "\n  in expression: ");
  fprint_exp(stderr, orig);
  fprintf(stderr, "\n  ");
  
  switch(error) {
    case USES_ITSELF:
      fprintf(stderr, "Parameter uses itself for initialization\n");
      break;
    case UNK_TYPE:
      fprintf(stderr, "Parameter has an unknown type\n");
      break;
    case FOUND_ARRAY:
      fprintf(stderr, "Array parameters are illegal in this expression\n");
      break;
    case FOUND_NULL:
      fprintf(stderr, "Parameter's initial value is null\n");
      break;
    default:
      fprintf(stderr, "UNKNOWN ERROR CODE: %d\n", error);
      break;
  }
  flags = flags; /* to shut lint up */
  nice_bail(CNULL);
}

/* ---------------------------------------------------------------------- */
/* Set_exp is used to set e1 to e2.  It is a convenience as it keeps the
 * types straight */
  
static set_exp(e1, e2)  /* Sets e1 to e2 -- e2 is assumed to be a constant */
Exp e1, e2;
{
  if (e2->elt_type != 'c') bail("INTE: set_exp: e2->elt_type != 'c'\n");
  e1->elt_type = 'c';
  e1->type = e2->type;
  switch(e1->type) {
    case INT: e1->val.i = e2->val.i; break;
    case FLOAT: e1->val.f = e2->val.f; break;
    case DOUBLE: e1->val.d = e2->val.d; break;
    case CHAR: e1->val.c = e2->val.c; break;
    case ARRAY: e1->val.a = e2->val.a; break;
    default: fprintf(stderr, "Set_exp(): Unknown c type: %d\n", e1->type);
             bail(CNULL); break;
  }
}
  

/* ---------------------------------------------------------------------- */
/* Eval_exp evaluates expressions in a destructive fashion. If the 
 * expression is a parameter, then it calls eval_exp on the parameter's 
 * val, and substitutes the parameters val for its val. If it is a 
 * constant, then it is considered to be evaluated. Otherwise, it is a 
 * true expression: eval_exp is called on its arguments, and then the 
 * expression is reduced to a constant. The argument expression is freed 
 * (Hence the destructiveness). */ 
  
static eval_exp(e, flags, orig)
Exp e, orig;
int flags;
{
  Param p;
  int i;
  Exp nexte;

  if (e == ENULL) eval_exp_error(FOUND_NULL, e, flags, orig);

  switch(e->elt_type) {
    case 'c': 
      if (e->type == ARRAY && !arrays_ok(flags))
        eval_exp_error(FOUND_ARRAY, e, flags, orig);
      return;
      break;
    case 'P': 
      p = e->val.p;
      if (p->type == ARRAY && !arrays_ok(flags)) 
        eval_exp_error(FOUND_ARRAY, e, flags, orig);
      eval_param(p, flags, orig);
      set_exp(e, p->val);
      return;
      break;
    case 'A':
      if (e->args[0] == ENULL || e->args[0]->elt_type != 'P') {
        eval_exp_error(GENERAL_ADDRESS, e, flags, orig);
      } else if (!address_ok(flags)) {
        eval_exp_error(FOUND_ADDRESS, e, flags, orig);
      } else return;
      break;
    case '[':
    case ']':
    case '.':
    case 'T':
    case 'p':
      eval_exp_error(BAD_INT_EXP, e, flags, orig);
      break;
    case '?':
      eval_exp(e->args[0], flags, orig);
      convert_exp_type(e->args[0], INT);
      if (e->args[0]) nexte = e->args[1]; else nexte = e->args[2];
      eval_exp(nexte, flags, orig);
      set_exp(e, nexte);
      for (i = 0; i < e->nargs; i++) free_exp(e->args[i]);
      e->nargs = 0;
      return;
    default: break;
  } 

  for (i = 0; i < e->nargs; i++) eval_exp(e->args[i], flags, orig);
    
  /* These macros are defined in yucky.h */
  switch(e->elt_type) {
    case 'M': yucky_unop_macro(e, -, flags, orig); break;
    case 'I': eval_exp_error(INDIRECTION, e, flags, orig); break;
    case 'C': yucky_bitunop_macro(e, ~, flags, orig); break;
    case 'N': yucky_compunop_macro(e, !, flags, orig); break;
    case '+': yucky_binop_macro(e, +, flags, orig); break;
    case '-': yucky_binop_macro(e, -, flags, orig); break;
    case '*': yucky_binop_macro(e, *, flags, orig); break;
    case '/': yucky_binop_macro(e, /, flags, orig); break;
    case '%': yucky_bitbinop_macro(e, %, flags, orig); break;
    case 'L': yucky_bitbinop_macro(e, <<, flags, orig); break;
    case 'R': yucky_bitbinop_macro(e, >>, flags, orig); break;
    case '>': yucky_compbinop_macro(e, >, flags, orig); break;
    case '<': yucky_compbinop_macro(e, <, flags, orig); break;
    case 'l': yucky_compbinop_macro(e, <=, flags, orig); break;
    case 'g': yucky_compbinop_macro(e, >=, flags, orig); break;
    case '!': yucky_compbinop_macro(e, !=, flags, orig); break;
    case '=': yucky_compbinop_macro(e, ==, flags, orig); break;
    case 'a': yucky_compbinop_macro(e, &&, flags, orig); break;
    case 'o': yucky_compbinop_macro(e, ||, flags, orig); break;
    case '&': yucky_bitbinop_macro(e, &, flags, orig); break;
    case '^': yucky_bitbinop_macro(e, ^, flags, orig); break;
    case '|': yucky_bitbinop_macro(e, |, flags, orig); break;
    default:
      eval_exp_error(NOT_IMP, e, flags, orig);
      break;
  }
  for (i = 0; i < e->nargs; i++) free_exp(e->args[i]);
  e->elt_type = 'c';
  e->nargs = 0;
  return;
}

static eval_exp_error(error, e, flags, orig)
int error;
Exp e;
int flags;
Exp orig;
{
  fprintf(stderr, "ERROR evaluating ");
  fprint_exp(stderr, e);
  fprintf(stderr, " in expression: ");
  fprint_exp(stderr, orig);
  fprintf(stderr, "\n  ");

  switch (error) {
    case FOUND_NULL:
      fprintf(stderr, "Can't evaluate a null expression\n");
      break;
    case FOUND_ARRAY:
      fprintf(stderr, "Arrays are illegal in this expression\n");
      break;
    case FOUND_NOT_BIT:
      fprintf(stderr, "Need int or char for bit operations\n");
      break;
    case GENERAL_ADDRESS:
      fprintf(stderr, "Addresses of general expressions are illegal\n");
      break;
    case FOUND_ADDRESS:
      fprintf(stderr, "Addresses are illegal in this expression\n");
      break;
    case BAD_INT_EXP:
      fprintf(stderr, "INTERNAL ERROR.  Bad elt->type %c\n", e->elt_type);
      break;
    case INDIRECTION:
      fprintf(stderr, "Indirections are illegal\n", e->elt_type);
      break;
    case NOT_IMP:
      fprintf(stderr, "Operation %c not implemented yet\n", e->elt_type);
      break;
    default:
      fprintf(stderr, "UNKNOWN ERROR CODE: %d\n", error);
      break;
  }
  flags = flags;  /* To keep lint quiet */
  nice_bail(CNULL);
}


/* ---------------------------------------------------------------------- */
/* This evaluates the dimensions of an array parmeter. If the dimension 
 * is NULL or [], then get it from p->mirror, which should be set to an 
 * ancestor. Otherwise, if the dimension is of the form [y] or [ 0 ..  y ],
 * then y should be evaluated, and checked for consistency with regard
 * to p->mirror. If it is of the form [y], then set arg[1] to arg[0], 
 * just to make future life easier. If the parameter is the main array 
 * parameter, then call this recursively on all the parameters in its 
 * rb_tree, with itself as the mirror. */ 

eval_array_dims(n, p)
Node n;
Param p;
{
  int i;
  Parray a, aa;
  Param rp;
  Rb_node r;

  a = p->a;
  p->state = -1;
  if (p->mirror != PNULL) aa = p->mirror->a; else aa = PANULL;
  for (i = 0; i < a->ndims; i++) {
    if (a->dims[i] == ENULL) {
      if (aa == PANULL) error_unk_dim_no_anc(n, p, i);
      a->dims[i] = copy_exp(aa->dims[i], 2);
    } else if (a->dims[i]->elt_type == '[') {
      if (aa == PANULL) error_unk_dim_no_anc(n, p, i);
      free_exp(a->dims[i]);
      a->dims[i] = copy_exp(aa->dims[i], 2);
    } else if (a->dims[i]->elt_type == ']') {
      if (p->io.main) 
        bail("INT ERROR: eval_array_dims: p->io.main && elt_type == '['\n");
      if (aa == PANULL) bail("INT ERROR: eval_array_dims: aa != PANULL1\n");
      hence_eval_exp(a->dims[i]->args[0], 0);
      convert_exp_type(a->dims[i]->args[0], INT);
      if (a->dims[i]->args[0]->val.i > aa->dims[i]->args[1]->val.i ||
           a->dims[i]->args[0]->val.i < 0) error_bad_dim(n, p, i);
    } else if (a->dims[i]->elt_type == '.') {
      hence_eval_exp(a->dims[i]->args[0], 0);
      convert_exp_type(a->dims[i]->args[0], INT);
      hence_eval_exp(a->dims[i]->args[1], 0);
      convert_exp_type(a->dims[i]->args[1], INT);
      if (p->io.main) {
        if (a->dims[i]->args[0]->val.i != 0)
          bail("INT ERROR: eval_array_dims: a->dims[i]->args[0]->val.i != 0\n");
        if (aa != PANULL &&
            a->dims[i]->args[1]->val.i != aa->dims[i]->args[1]->val.i) ;
          /* This is an error which will be flagged by put_into_aa_list */
      } else if (aa == PANULL) {
        bail("INT ERROR: eval_array_dims: aa != PANULL 2\n");
      } else {
        if ( a->dims[i]->args[0]->val.i > aa->dims[i]->args[1]->val.i ||
            a->dims[i]->args[1]->val.i > aa->dims[i]->args[1]->val.i ||
            a->dims[i]->args[0]->val.i < 0 || 
            a->dims[i]->args[1]->val.i < 0) error_bad_dim(n, p, i);
        if (a->dims[i]->args[0]->val.i > a->dims[i]->args[1]->val.i) {
          error_crossed_dims(n, p, i);
        }
      }
    } else bail("INT ERROR: eval_array_dims: non-dim expression\n");
  }
  if (p->io.main) {
    rb_traverse(r, a->p) {
      rp = (Param) r->v.val;
      rp->mirror = p;
      eval_array_dims(n, rp);
    }
  }
}
      
/* ---------------------------------------------------------------------- */
/* Error flagging code */

static error_unk_dim_no_anc(n, p, i) 
Node n;
Param p;
int i;
{
  param_error_header(n, p);
  fprintf(stderr, "  Dimension %d is unspecified, and %s has no ancestors\n",
    i, p->name);
  nice_bail(CNULL);
}  

static error_bad_dim(n, p, i)
Node n;
Param p;
int i;
{
  param_error_header(n, p);
  fprintf(stderr, "  Dimension %d is not within the proper bounds:", i);
  fprint_exp(stderr, p->mirror->a->dims[i]);
  nice_bail("\n");
}
  
static error_crossed_dims(n, p, i)
Node n;
Param p;
int i;
{
  param_error_header(n, p);
  fprintf(stderr, "  Dimension %d: Max val is less than min val", i);
  nice_bail("\n");
}
  
  
