/*
 * $Source$
 * $Revision$
 * $Date$
 * $Author$
 */

/* Check.c defines check_graph, which checks the graph for correctness --
 * non-circularity and matching begin and end nodes. It does the latter 
 * by using the scope field of nodes. N->scope is defined to be the 
 * closest special-purpose node which is a parent of N.  If the graph is 
 * correct, then if N is non-special, all children of N should have the 
 * same scops as n.  If in is special, all children of N should have a 
 * scope of n.  Any clashes mean that there is an error somewhere. Scope 
 * isn't used in any other routines, so if you feel like coelescing it 
 * with some other field, that should be ok.  
*/


#include <stdio.h>
#include "std.h"
#include "list.h"
#include "rb.h"
#include "dlist.h"
#include "hence.h"
#include "htypes.h"

check_graph(G)
Graph G;
{
  Rb_node h;
  Node n;
  Node init_scope;

  if (G->numnodes == 0) return;

  init_scope = (Node) -1;
  for (h = rb_first(G->heads); h != nil(G->heads); h = rb_next(h)) {
    n = (Node) h->v.val;
    cg_traverse(n, init_scope);
  }

  /* Check for unrooted circles */
  for (h = rb_first(G->nlist); h != nil(G->nlist); h = rb_next(h)) {
    n = (Node) h->v.val;
    if (n->scope == NNULL) {
      fprintf(stderr, "ERROR: Circular graph.\n");
      fprintf(stderr, "       Node %d is involved in a cycle.\n", n->id);
      nice_bail(CNULL);
    } else if (n->scope == (Node) -1) n->scope = NNULL;
  }
}

static cg_traverse(n, scope)
Node n;
Node scope;
{
  Node newscope;
  Rb_node r;
  
  if (n->scope != NNULL) {
    if (n->scope != scope) {
      fprintf(stderr, "ERROR: Node %d belongs to two subgraphs:\n", n->id);
      fprintf(stderr, "       One defined by node %d,\n", scope->id);
      fprintf(stderr, "   and One defined by node %d,\n", n->scope->id);
      bail(CNULL);
    } else if (n->flags) {
      fprintf(stderr, "ERROR: Circular graph.\n");
      fprintf(stderr, "       Node %d is involved in a cycle.\n", n->id);
      bail(CNULL);
    } else return;
  }

  n->flags = 1;
  n->scope = scope;
    
  if (n->node_type == NORMAL) {
    newscope = scope;
  } else if (n->node_type % 2 == 1) {  /* Beginning of special purpose node */
    newscope = n;
  } else {                             /* Ending of special-purpose node */
    if (scope == (Node) -1) {
      fprintf(stderr, "ERROR: Unmatching special node pair:\n");
      fprintf(stderr, "       Beginning of the graph\n");
      fprintf(stderr, "   and %s node %d\n", node_types[n->node_type], n->id);
      bail(CNULL);
    } else if (scope->node_type != n->node_type -1) {
      fprintf(stderr, "ERROR: Unmatching special node pair:\n");
      fprintf(stderr, "       %s node %d\n", node_types[scope->node_type],
              scope->id);
      fprintf(stderr, "   and %s node %d\n", node_types[n->node_type], n->id);
      bail(CNULL);
    } else if (scope->pair != NNULL) {
      fprintf(stderr, "ERROR: %s node %d has two %s nodes: %d and %d.\n",
              node_types[scope->node_type], scope->id,
              node_types[n->node_type], scope->pair->id, n->id);
      bail(CNULL);
    }
    rb_traverse(r, n->parents) {
      if (((Node) r->v.val) == scope) {
        fprintf(stderr, 
          "ERROR: Cannot have an arc connecting a special pair:\n");
        fprintf(stderr, "       ARC %d %d shouldn't exist\n", scope->id,
                n->id);
      bail(CNULL);
      }
    }
    scope->pair = n;
    n->pair = scope;
    newscope = scope->scope;
  }

  rb_traverse(r, n->children) {
    cg_traverse((Node) r->v.val, newscope);
  }

  n->flags = 0;
}
    
