/*
 * $Source$
 * $Revision$
 * $Date$
 * $Author$
 * 
 * Array.c contains routines for dealing with array parameters and array 
 * constants. Array constants are defined in hence.h. The important thing
 * to note about them is that they can be referenced in two ways: By the 
 * "v" field, if they are to be used as a big 1-d chunk of bytes, and by 
 * the "ptr" field, if they are to be used as a multi-dimensional array 
 * of pointers. In almost all cases, both the "v" and "ptr" fields are 
 * valid (the last level of the ptr indirections is the v field). The 
 * only exceptions to this are just before the wrapper call to the end of
 * slave.c. Then, refd.ptr will be 1 if ptr is valid, and refd.v is 1 if 
 * v is valid. If refd.init == 0, then neither v nor p is valid -- there 
 * is no chunk of bytes associated with this array. 
 */

#include <stdio.h>
#if defined(IMA_SYMM) || defined(IMA_NEXT)
void *malloc ();
void free ();
void *realloc ();
void *calloc ();
#else
#include <malloc.h>
#endif /* IMA_SYMM */
#include "std.h"
#include "rb.h"
#include "dlist.h"
#include "hence.h"
#include "htypes.h"
#include "eval.h"

/* ---------------------------------------------------------------------- */
/* Get_eltsize returns the size of a type */

static int get_eltsize(type)
int type;
{
  switch (type) {
    case INT:    return sizeof(int); break;
    case CHAR:   return sizeof(char); break;
    case FLOAT:  return sizeof(float); break;
    case DOUBLE: return sizeof(double); break;
    default:     bail("INT ERROR: get_eltsize: bad type\n");
                 break;
  }
  bail("INT ERROR: get_eltsize: This statement should not be reached\n");
  return 0;  /* For lint */
}

/* ---------------------------------------------------------------------- */
/* Make_null_array() creates a constant from an array parameter. If the 
 * parameter is just one element of an array, then it returns a scalar. 
 * If the parameter is output-only, then it returns a new array constant 
 * without allocating any bytes. Otherwise, it allocates a chunk of bytes
 * for the array, and builds the pointer tree so that it can be 
 * referenced by both v and ptr. */ 

Exp make_null_array(p)
Param p;
{
  int eltsize;
  int i;
  Exp toret;
  void *last_chunk;
  Array a;
  
  /* If nadims == 0, then the array parameter is a constant -- make it
   * with make_null_exp */

  if (p->a->nadims == 0) {
    return make_null_exp(p->a->type);
  }

  /* Otherwise, allocate the array struct */

  eltsize = get_eltsize(p->a->type);

  toret = talloc(struct expression, 1);
  toret->elt_type = 'c';
  toret->nargs = 0;

  toret->type = ARRAY;
  
  a = brand_new_array(p->a->type, p->a->nadims);

  /* If the parameter is output only, then set its pointers to null,
     and return */

  if (!p->io.in && p->io.out) {
    switch(a->type) {
      case INT:    a->v.i =    (int *) 0; break;
      case CHAR:   a->v.c =   (char *) 0; break;
      case FLOAT:  a->v.f =  (float *) 0; break;
      case DOUBLE: a->v.d = (double *) 0; break;
      default: bail("INTE: make_null_array: bad a->type\n");
    }
    a->ptr = (void *) 0;
    a->tsize = 0;
    for (i = 0; i < a->ndims; i++) {
      a->size[i] = 0;
      a->indsize[i] = 0;
    }
    toret->val.a = a;
    return toret;
  }
 
  /* Otherwise, initialize size & indsize & tsize, allocate the 
     array, and build the pointer tree */

  build_size_and_indsize(p, a);

  last_chunk = (void *) calloc ((unsigned) a->tsize, (unsigned) eltsize);

  switch (p->a->type) {
    case INT:    a->v.i = (int *) last_chunk; break;
    case CHAR:   a->v.c = (char *) last_chunk; break;
    case FLOAT:  a->v.f = (float *) last_chunk; break;
    case DOUBLE: a->v.d = (double *) last_chunk; break;
    default:     bail("INT ERROR: make_null_array: bad type\n");
  }

  build_ptr_tree(a);
  a->refd.init = 1;
  toret->val.a = a;
  return toret;
}

/* ---------------------------------------------------------------------- */
/* Build_size_and_indsize initializes the size and indsize fields of an 
 * array constant a from the information in a and in the parameter p. 
 */

static build_size_and_indsize(p, a)
Param p;
Array a;
{
  int i, j, tsize, indsize;
  Exp e;

  tsize = 1;
  j = p->a->nadims - 1;
  for (i = p->a->ndims - 1; i >= 0; i--) {
    e = p->a->dims[i];
    if (e->elt_type == '.') {
      a->size[j] = e->args[1]->val.i - e->args[0]->val.i + 1;
      tsize *= a->size[j];
      j--;
    } else if (e->elt_type != ']') {
        bail("INT ERROR: build_size...: e->elt_type != '.' || ']'\n");	
    }
  }
    
  indsize = tsize;
  a->tsize = tsize;
  if (j >= 0) bail("INT ERROR: build_size...: j >= 0\n");

  for (i = 0; i < p->a->nadims; i++) {
    indsize /= a->size[i];
    a->indsize[i] = indsize;
  }
}

/* ---------------------------------------------------------------------- */
/* Build_ptr_tree takes an array. It expects all the fields except for 
 * ptr to be initialized. Then, it builds the tree of pointers on top of 
 * the chunk of bytes in a->v.x, and sets the topmost one to ptr. */

build_ptr_tree(a)
Array a;
{
  int i, j;
  int tsize, eltsize, ibase, iloc;
  void **this_chunk, *last_chunk;

  tsize = a->tsize;
  eltsize = get_eltsize(a->type);
  switch (a->type) {
    case INT:    last_chunk = (void *) a->v.i; break;
    case CHAR:   last_chunk = (void *) a->v.c; break;
    case FLOAT:  last_chunk = (void *) a->v.f; break;
    case DOUBLE: last_chunk = (void *) a->v.d; break;
    default:     bail("INT ERROR: make_null_array: bad type\n");
  }

  for (i = a->ndims - 1; i > 0; i--) {
    tsize /= a->size[i];
    eltsize *= a->size[i];
    ibase = (int) last_chunk;
    this_chunk = talloc(void *, tsize);
    for (j = 0; j < tsize; j++) {
      iloc = ibase + j * eltsize;
      this_chunk[j] = (void *) iloc;
    }
    last_chunk = (void *) this_chunk;
  }
  a->ptr = last_chunk;
}

/* ---------------------------------------------------------------------- */
/* Free_ptr_tree free's the storage of the pointer tree in a->ptr. It 
 * also free's the chunk of bytes at the bottom level of the pointer 
 * tree. This should be a->v.x, but might not if the user did something 
 * screwy in his subroutine. */ 

free_ptr_tree(a)
Array a;
{
  int i;
  void **this, *nxt;

  if (a->ptr == (void *) 0) return;
  
  this = (void **) a->ptr;
  for (i = 1; i < a->ndims; i++) {
    nxt = this[0];
    free(this);
    this = (void **) nxt;
  }
  free(this);
  a->v.c = CNULL;
}
    
/* ---------------------------------------------------------------------- */
/* Free_array frees up the storage taken by an array constant. Free_array
 * really can't work perfectly, because the user can allocate arrays in 
 * bizarre ways, but if the user doesn't do anything screwy, free_array 
 * will do ok.  Free_array does work with arrays allocated with 
 * make_null_array, etc. */ 

free_array(a)
Array a;
{
  if (a->refd.init) {
    if (a->ptr != (void *) 0) {
      free_ptr_tree(a);
    } else {
      switch(a->type) {
        case INT:    if (a->v.i !=    (int *) 0) free(a->v.i); break;
        case CHAR:   if (a->v.c !=   (char *) 0) free(a->v.c); break;
        case FLOAT:  if (a->v.f !=  (float *) 0) free(a->v.f); break;
        case DOUBLE: if (a->v.d != (double *) 0) free(a->v.d); break;
        default:     bail("INT ERROR: make_null_array: bad type\n");
      }
      a->v.c = CNULL;
    }
  }
  free(a->size);
  free(a->indsize);
  free(a);
}

/* ---------------------------------------------------------------------- */
/* Copy_array_vals takes two expressions which should be array constants.
 * It then copies the overlapping portions of e2 and e1 from e2 to e1.  
 * It assumes that the a->v.x pointer of the arrays are valid. */ 

static copy_array_vals(e1, e2)
Exp e1, e2;
{
  int *ct, ctdims, use_ct1, use_ct2;
  int i;  
  int ok, ind1, ind2;
  Array a1, a2;

  if (e1->elt_type != 'c' || e1->type != ARRAY) 
    bail("INT ERROR: copy_array_vals: Bad e1\n");
  if (e2->elt_type != 'c' || e2->type != ARRAY) 
    bail("INT ERROR: copy_array_vals: Bad e2");

  a1 = e1->val.a;
  a2 = e2->val.a;

  if (a1->ndims != a2->ndims && a1->ndims != 1 && a2->ndims != 1) 
    bail("INT ERROR: copy_array_vals: Unresolvable ndims\n");

  use_ct1 = (a1->ndims != 1);
  use_ct2 = (a2->ndims != 1);

  if (use_ct1) ctdims = a1->ndims; 
  else if (use_ct2) ctdims = a2->ndims;
  else ctdims = 0;

  if (ctdims != 0) {
    ct = talloc(int, ctdims);
    for (i = 0; i < ctdims; i++) ct[i]= 0;
  }

  ind1 = 0;
  ind2 = 0;
  ok = 1;

  while(ok) {
    set_array(a1, a2, ind1, ind2);
    ind1++;
    ind2++;
    if (ctdims != 0) {
      ct[ctdims-1] += 1;
      for (i = ctdims - 1; 
           i > 0 && ((use_ct1 && ct[i] == a1->size[i]) || 
                     (use_ct2 && ct[i] == a2->size[i]));
           i--) {
        if (use_ct1) ind1 -= ct[i] * a1->indsize[i];
        if (use_ct2) ind2 -= ct[i] * a2->indsize[i];
        ct[i] = 0;
        ct[i-1] += 1;
        if (use_ct1) ind1 += a1->indsize[i-1];
        if (use_ct2) ind2 += a2->indsize[i-1];
      }
      if (use_ct1) ok = (ok && ct[0] != a1->size[0]);
      if (use_ct2) ok = (ok && ct[0] != a2->size[0]);
    }
    if (!use_ct1) ok = (ok && ind1 != a1->size[0]);
    if (!use_ct2) ok = (ok && ind2 != a2->size[0]);
  }

  if (ctdims != 0) free(ct);
}      
      
/* ---------------------------------------------------------------------- */
/* Copy_sub_array is kind of like copy_array_vals, except that it takes 
 * two parameters -- a main array parameter p1 and a subarray parameter 
 * p2 (i.e. p2 is one of the elts in p2->a->p). It then finds which 
 * values of p1 are in p2, and copies them to p2's array constant (or to 
 * p2's scalar constant if p2 is a single array element. If you 
 * understand how copy_array_vals works, then copy_sub_array shouldn't be
 * too difficult */ 

copy_sub_array(e2, e1, p2, p1)
Param p1, p2;    /* Main param and subarray param, respectively */
Exp e1, e2;      /* Main array and subarray expressions respectively.  
                   The main array is copied to the subarray */
{
  int *ct, *adims;
  int i, j;
  int ok, ind1, ind2;
  Array a1, a2;

  if (p1->a->nadims == 0)
    bail("INT ERROR: copy_sub_array: p1->nadims == 0\n");
  if (p1->a->ndims != p2->a->ndims) 
    bail("INT ERROR: copy_sub_array: Nonequivalent ndims\n");
  if (e1->type != ARRAY)
    bail("INT ERROR: copy_sub_array: e1->type != ARRAY\n");

  a1 = e1->val.a;
  ind1 = 0;
  for (i = 0; i < p2->a->ndims; i++) {
    ind1 += a1->indsize[i] * p2->a->dims[i]->args[0]->val.i;
  }

  if (p2->a->nadims == 0) {
    set_elt_from_array(e2, a1, ind1);
    return;
  }

  a2 = e2->val.a;

  ct = talloc(int, a2->ndims);
  for (j = 0; j < a2->ndims; j++) ct[j] = 0;

  adims = talloc(int, a2->ndims);
  j = 0;
  for (i = 0; i < p2->a->ndims; i++) {
    if (p2->a->dims[i]->elt_type == '.') {
      adims[j] = i;
      j++;
    }
  }

  ind2 = 0;
  ok = 1;

  while(ok) {
    set_array(a2, a1, ind2, ind1);
    ind1 += a1->indsize[adims[a2->ndims-1]];
    ind2++;
    ct[a2->ndims-1] += 1;
    for (i = a2->ndims - 1; i > 0 && ct[i] == a2->size[i]; i--) {
      ind1 -= ct[i] * a1->indsize[adims[i]];
      ind2 -= ct[i] * a2->indsize[i];
      ct[i] = 0;
      ct[i-1] += 1;
      ind1 += a1->indsize[adims[i-1]];
      ind2 += a2->indsize[i-1];
    }
    ok = (ct[0] != a2->size[0]);
  }
  free(ct);
  free(adims);
}      

/* ---------------------------------------------------------------------- */
/* Init_array_param is called from slave.c. It takes two arguments: p, 
 * which is the parameter to initialize, and pva, which is an array of 
 * expressions, which hold not-yet-totally-initialized values. Basically,
 * init_array_param looks p->val. If it is null, and p has no fancy 
 * initialization criteria, then it initializes p->val from pva[p->num]. 
 * If p is a subarray, then if p's main array parameter is being used, 
 * then p's value is copied from the main parameter. Otherwise, it too is
 * initialized from pva[p->num]. If p->val isn't null, it is an 
 * expression which states how p should be initialized. */ 
  
init_array_param(p, pva)
Param p;
Exp *pva;
{
  if (p->state == 1) return;   /* Already done */
  if (p->state == -1) {
    fprintf(stderr, "ERROR: %sParameter: ", sl_id());
    fprint_param(stderr, p);
    nice_bail("\n  is involved in a circular initialization.\n");
  }
  p->state = -1;
  if (p->val == ENULL) {
    /* If p is a subarray, then either p has received values or it hasn't.
       The way to tell is if p->a->main has been initialized.  If it has, 
       then p hasn't received values, and should be initialized from it.
       If it hasn't, then p has. */ 

    if (!p->io.main) { 
      init_array_param(p->a->main, pva);
      if (p->a->main->val != ENULL) {
        if (p->a->main->val->elt_type != 'c' ||
            p->a->main->val->type != ARRAY) bail("INTE: init_array_param: 1");
        copy_sub_array(pva[p->num], p->a->main->val, p, p->a->main);
      }
    }
  } else if (p->val->elt_type == 'c') {
    copy_array_vals(pva[p->num], p->val);
  } else if (p->val->elt_type == 'P') {
    if (!p->val->val.p->io.in && p->val->val.p->io.out) {
      nice_bail("Parameter tries to set itself to an output parameter\n");
    }
    init_array_param(p->val->val.p, pva);
    copy_array_vals(pva[p->num], p->val->val.p->val);
  }
  free_exp(p->val);
  p->val = pva[p->num];
  pva[p->num] = ENULL;
  p->state = 1;
  return;
}

/* ---------------------------------------------------------------------- */
/* Complete_array_initialization is called on output array parameters 
 * after the subroutine has been called. It initializes the parameter's 
 * dimensions and sets their constants' ptr or v.x pointers as necessary 
 * so that the node can properly send its values to other nodes. */ 

complete_array_initialization(p)
Param p;
{
  Exp e;
  Array a;
  int i, ok;

  for(i = 0; i < p->a->nadims; i++) {
    hence_eval_exp(p->a->dims[i]->args[1], 0);
  }

  e = p->val;
  a = e->val.a;
  a->refd.init = 1;
  if (a->ptr != (void *) 0) {
    build_size_and_indsize(p, a);
    return;
  } else {
    switch(a->type) {
      case INT:    ok = (a->v.i != (int *) 0);    break;
      case CHAR:   ok = (a->v.c != (char *) 0);   break;
      case FLOAT:  ok = (a->v.f != (float *) 0);  break;
      case DOUBLE: ok = (a->v.d != (double *) 0); break;
      default: bail("INTE: complete_a_i bad a->type\n"); break;
    }
    if (ok) {
      build_size_and_indsize(p, a);
      return;
    } else {
      free_exp(e);
      p->val = ENULL;
    }
  }
}

/* ---------------------------------------------------------------------- */
/* Set_array and set_elt_from array are to set array values -- they're 
 * simply a convenience that keeps the types straight. */ 

set_array(a2, a1, ind2, ind1)
Array a2, a1;
int ind2, ind1;
{
  void *rv1, *rv2;

  switch(a2->type) {
    case INT:    rv2 = (void *) (&(a2->v.i[ind2])); break;
    case CHAR:   rv2 = (void *) (&(a2->v.c[ind2])); break;
    case FLOAT:  rv2 = (void *) (&(a2->v.f[ind2])); break;
    case DOUBLE: rv2 = (void *) (&(a2->v.d[ind2])); break;
    default: bail("INT ERROR: set_array: Bad e2 type\n"); break;
  }
  switch(a1->type) {
    case INT:    rv1 = (void *) (&(a1->v.i[ind1])); break;
    case CHAR:   rv1 = (void *) (&(a1->v.c[ind1])); break;
    case FLOAT:  rv1 = (void *) (&(a1->v.f[ind1])); break;
    case DOUBLE: rv1 = (void *) (&(a1->v.d[ind1])); break;
    default: bail("INT ERROR: set_array: Bad e1 type\n"); break;
  }
  type_coerce(rv2, rv1, a2->type, a1->type);
}
    
set_elt_from_array(e2, a1, ind1)
Exp e2;
Array a1;
int ind1;
{
  void *rv1, *rv2;

  switch(e2->type) {
    case INT:    rv2 = (void *) (&(e2->val.i)); break;
    case CHAR:   rv2 = (void *) (&(e2->val.c)); break;
    case FLOAT:  rv2 = (void *) (&(e2->val.f)); break;
    case DOUBLE: rv2 = (void *) (&(e2->val.d)); break;
    default: bail("INT ERROR: set_elt_from_array: Bad e2 type\n"); break;
  }
  switch(a1->type) {
    case INT:    rv1 = (void *) (&(a1->v.i[ind1])); break;
    case CHAR:   rv1 = (void *) (&(a1->v.c[ind1])); break;
    case FLOAT:  rv1 = (void *) (&(a1->v.f[ind1])); break;
    case DOUBLE: rv1 = (void *) (&(a1->v.d[ind1])); break;
    default: bail("INT ERROR: set_elt_from_array: Bad e1 type\n"); break;
  }
  type_coerce(rv2, rv1, e2->type, a1->type);
}

    
/* ---------------------------------------------------------------------- */
/* Brand_new_array allocates a new array constant struct */

Array brand_new_array(type, ndims)
int type, ndims;
{
  Array a;
  a = talloc(struct array, 1);
  a->type = type;
  a->ndims = ndims;
  a->size = talloc(int, a->ndims);
  a->indsize = talloc(int, a->ndims);
  a->refd.ptr = 0;
  a->refd.v = 0;
  a->refd.init = 0;
  return a;
}

/* ---------------------------------------------------------------------- */
/* Copy_array creates a copy of array struct a, and returns it */

Array copy_array(a)
Array a;
{
  Array newa;
  void *chunk;
  int eltsize, i;

  eltsize = get_eltsize(a->type);
  newa = brand_new_array(a->type, a->ndims);
  if (a->refd.ptr)
    bail("INTE: Don't know how to copy pointer-ref'd arrays yet\n");
  newa->tsize = a->tsize;
  for (i = 0; i < a->ndims; i++) {
    newa->size[i] = a->size[i];
    newa->indsize[i] = a->indsize[i];
  }
  chunk = (void *) calloc ((unsigned) newa->tsize, (unsigned) eltsize);

  switch (newa->type) {
    case INT:    
      newa->v.i = (int *) chunk; 
      for (i = 0; i < newa->tsize; i++) newa->v.i[i] = a->v.i[i];
      break;
    case CHAR:   
      newa->v.c = (char *) chunk; 
      for (i = 0; i < newa->tsize; i++) newa->v.c[i] = a->v.c[i];
      break;
    case FLOAT:  
      newa->v.f = (float *) chunk; 
      for (i = 0; i < newa->tsize; i++) newa->v.f[i] = a->v.f[i];
      break;
    case DOUBLE: 
      newa->v.d = (double *) chunk; 
      for (i = 0; i < newa->tsize; i++) newa->v.d[i] = a->v.d[i];
      break;
    default:     bail("INT ERROR: copy_array: bad type\n");
  }
  
  build_ptr_tree(newa);
  newa->refd.init = 1;
  return newa;
}
