#ifndef FuncPr
#if defined(__STDC__)
#define FuncPr 1
#else
#define FuncPr 0
#endif
#endif

#include <stdio.h>
#include "rb.h"
#include "param.h"
#include "exp.h"
#include "expP.h"
#include "graph.h"
#include "graphP.h"
#include "misc.h"
#include "unparse.h"

static FILE *output_file;
static int unparse_errors;

/*
 * print a single arc.	always returns 0.
 * called from within rb_TraverseX, which is called from unparse_child_arcs
 */

static int
unparse_an_arc (
#if FuncPr
	TreeNode tn, void *parent)
#else
	tn, parent)
	TreeNode tn;
	void *parent;
#endif
{
	Node child = (Node) rb_Value (tn);
	fprintf (output_file, "ARC %d %d\n", ((Node)parent)->nk.id, child->nk.id);
	return 0;
}

/*
 * print arcs for all of the children of a node
 * each one is printed as a separate ARC decl.
 * called from rb_Traverse, which is called from unparse_arcs
 *
 * XXX should print only one ARC decl per parent node, e.g:
 * ARC parent child1,child2,...,childn ;
 */

static int
unparse_child_arcs (
#if FuncPr
	TreeNode tn)
#else
	tn)
	TreeNode tn;
#endif
{
	Node n = (Node) rb_Value (tn);
	rb_TraverseX (n->children, unparse_an_arc, (void *) n);
	return 0;
}

/*
 * Print all of the arcs in the graph.
 * This is done after all node decls are printed, because an arc
 * decl must appear after all of the nodes it references.  It is
 * too hard to place them close to their corresponding node decls.
 */

static void
unparse_arcs (
#if FuncPr
	Graph g)
#else
	g)
	Graph g;
#endif
{
	rb_Traverse (g->nlist, unparse_child_arcs);
}

/*
 * print out the name of a parameter, and any associated array
 * dimensions
 */

static int unparse_expr (
#if FuncPr
Expr e, int need_parens
#endif
	);

static void
unparse_param (
#if FuncPr
	Param p)
#else
	p)
	Param p;
#endif
{
	fprintf (output_file, "%s", p->name);
	if (p->type == TYPE_ARRAY) {
		int i;
		/*
		 * XXX should this be p->a.ndims or p->a.nadims?
		 */
		for ( i = 0; i < p->a->ndims ; ++i) {
			switch (p->a->dims[i]->elt_type) {
			case OP_DIM:
			case OP_EMPTY_DIM:
			case OP_ARR_SECTION:
				unparse_expr (p->a->dims[i], 0);
				break;
			default:
				msg_Format ("unparse_param: bogus type %d for dimension expr",
							p->a->dims[i]->elt_type);
				unparse_errors++;
				break;
			}
		}
	}
}

/*
 * print out the literal value of a constant
 */

static void
unparse_const (
#if FuncPr
	Expr e)
#else
	e)
	Expr e;
#endif
{
	switch (e->type) {
	case TYPE_INT:
		fprintf (output_file, "%d", e->val.i);
		break;
	case TYPE_CHAR:
		/* XXX really not sure about this... */
		fprintf (output_file, "'%c'", e->val.c);
		break;
	case TYPE_FLOAT:
		fprintf (output_file, "%f", (double) e->val.f);
		break;
	case TYPE_DOUBLE:
		fprintf (output_file, "%f", e->val.d);
		break;
	case TYPE_STRING:
		/*
		 * XXX doesn't handle '"' inside a string, nor does
		 * it handle characters that need escapes (like \n)
		 */
		fprintf (output_file, "\"%s\"", e->val.s);
		break;
	case TYPE_ARRAY:
		/* XXX not sure how to handle this... */
		break;
	default:
		msg_Format ("unknown expr type %d in unparse_const\n", e->type);
		unparse_errors++;
		break;
	}
}

static struct {
	int op_code;
	int nargs;
	char *op_text;
	int precedence;
} op_table[] = {
	{ OP_COND_EXPR,		3,	"",		1,	},
	{ OP_LOGICAL_OR,	2,	"||",	2, 	},
	{ OP_LOGICAL_AND,	2,	"&&",	3,	},
	{ OP_LOGICAL_CMPL,	1,	"!",	12,	},
	{ OP_BIT_OR,		2,	"|",	4,	},
	{ OP_BIT_AND,		2,	"&",	6,	},
	{ OP_BIT_XOR,		2,	"^",	5,	},
	{ OP_BIT_CMPL,		1,	"~",	12,	},
	{ OP_COMP_NE,		2,	"!=",	7,	},
	{ OP_COMP_EQ,		2,	"==",	7,	},
	{ OP_COMP_LE,		2,	"<=",	8,	},
	{ OP_COMP_GE,		2,	">=",	8,	},
	{ OP_COMP_LT,		2,	"<",	8,	},
	{ OP_COMP_GT,		2,	">",	8,	},
	{ OP_LSHIFT,		2,	"<<",	9,	},
	{ OP_RSHIFT,		2,	">>",	9,	},
	{ OP_MULT,			2,	"*",	11,	},
	{ OP_DIV,			2,	"/",	11,	},
	{ OP_MOD,			2,	"%",	11,	},
	{ OP_ADD,			2,	"+",	10,	},
	{ OP_SUBTRACT,		2,	"-",	10,	},
	{ OP_NEG,			1,	"-",	13,	},
	{ OP_INDIR,			1,	"*",	13,	},
	{ OP_ADDRESS,		1,	"&",	13,	},
	{ OP_EMPTY_DIM,		0,	"",		-1,	},
	{ OP_DIM,			1,	"",		-1,	},
	{ OP_ARR_SECTION,	2,	"",		-1,	},
	{ OP_PARAM,			0,	"",		-1,	},
	{ OP_CONST,			0,	"",		-1,	},
};

#define SIZE(arr) (sizeof(arr)/sizeof(*arr))

static int
parens_reqd (
#if FuncPr
	int parent_op, Expr child_expr)
#else
	parent_op, child_expr)
	int parent_op;
	Expr child_expr;
#endif
{
	int child = -1, parent = -1;
	int i;

	for (i = 0; i < SIZE(op_table); ++i) {
		if (child_expr->elt_type == op_table[i].op_code)
			child = i;
		if (parent_op == op_table[i].op_code)
			parent = i;
		if (child > 0 && parent > 0)
			break;
	}
	/*
	 * if the precedcence of the child operator is higher, than its
	 * parent, we don't need parenthesis around the child expression.
	 */
	if (op_table[child].precedence > op_table[parent].precedence)
		return 0;
	return 1;
}


/*
 * unparse an expression
 */

static int
unparse_expr (
#if FuncPr
	Expr e, int need_parens)
#else
	e, need_parens)
	Expr e;
	int need_parens;
#endif
{
	int i;

	if (e == (Expr) NULL) {
		msg_Format ("NULL expression\n");
		unparse_errors++;
		return 0;
	}

	for (i = 0; i < SIZE(op_table); ++i) {
		if (e->elt_type == op_table[i].op_code)
			break;
	}
	if (i == SIZE (op_table)) {
		msg_Format ("unknown expr elt_type %d\n", e->elt_type);
		unparse_errors++;
		return 0;
	}
	if (op_table[i].nargs != e->nargs) {
		msg_Format ("wrong # of arguments for elt_type '%c'\n",
					e->elt_type);
		unparse_errors++;
		return 0;
	}

	/*
	 * eventually we will use associativity and precedence rules
	 * to determine when to add parens.	 For the time being
	 * we add them sometimes when they are not necessary.  However,
	 * it is never necessary to parenthesize an atom, and we can
	 * detect this case easily.
	 */
	if (op_table[i].nargs == 0)
		need_parens = 0;

	if (need_parens)
		fprintf (output_file, "(");

	switch (e->elt_type) {
	case OP_EMPTY_DIM:
		fprintf (output_file, "[]");
		break;
	case OP_DIM:
		fprintf (output_file, "[");
		unparse_expr (e->args[0], 0);
		fprintf (output_file, "]");
		break;
	case OP_ARR_SECTION:
		fprintf (output_file, "[");
		unparse_expr (e->args[0], 0);
		fprintf (output_file, ":");				/* was " TO " */
		unparse_expr (e->args[1], 0);
		fprintf (output_file, "]");
		break;
	case OP_COND_EXPR:			/* ? : */
		unparse_expr (e->args[0], 1);
		fprintf (output_file, " ? ");
		unparse_expr (e->args[1], 1);
		fprintf (output_file, " : ");
		unparse_expr (e->args[2], 1);
		break;
	case OP_PARAM:
		unparse_param (e->val.p);
		break;
	case OP_CONST:
		unparse_const (e);
		break;
	default:
		if (op_table[i].nargs == 2) {
			/* infix binary ops */
			unparse_expr (e->args[0], parens_reqd (e->elt_type, e->args[0]));
			fprintf (output_file, "%s", op_table[i].op_text);
			unparse_expr (e->args[1], parens_reqd (e->elt_type, e->args[1]));
			break;
		}
		else if (op_table[i].nargs == 1) {
			/* prefix unary ops */
			fprintf (output_file, "%s", op_table[i].op_text);
			unparse_expr (e->args[0], parens_reqd (e->elt_type, e->args[0]));
			break;
		}
		else
			fprintf (output_file, "bogus: unrecognized op %d\n", e->elt_type);
	}
	if (need_parens)
		fprintf (output_file, ")");
	return 0;
}

/*
 * unparse an output declaration, i.e., print all non-input declarations
 * and ignore those with the input bit set.
 * called after printing out the sub call and its args.
 */

static int
unparse_output_decl (
#if FuncPr
	TreeNode tn)
#else
	tn)
	TreeNode tn;
#endif
{
	Param p = (Param) rb_Value (tn);
	int type;

	if (p->io.in == 1 || p->io.out == 0)
		return 0;

	if (p->io.new)
		fprintf (output_file, "NEW ");
	fprintf (output_file, "    > ");
	type = p->type == TYPE_ARRAY ? p->a->type : p->type;
	switch (type) {
	case TYPE_INHERITED:		/* leave out type name */
		break;
	case TYPE_INT:
		fprintf (output_file, "int ");
		break;
	case TYPE_FLOAT:
		fprintf (output_file, "float ");
		break;
	case TYPE_DOUBLE:
		fprintf (output_file, "double ");
		break;
	case TYPE_CHAR:
		fprintf (output_file, "char ");
		break;
	default:
		msg_Format ("output_decl: unknown parameter type %d for param %s",
					p->type, p->name ? p->name : "(null)");
		unparse_errors++;
		break;
	}
	unparse_param (p);
	fprintf (output_file, ";\n");
	return 0;
}

/*
 * unparse all output declarations for this node
 */

static void
unparse_output_decl_list (
#if FuncPr
	Node n)
#else
	n)
	Node n;
#endif
{
	rb_Traverse (n->params, unparse_output_decl);
}

/*
 * unparse an input decl (that is, one that has the input bit set)
 * (i.e. ignore output-only decls)
 */

static int
unparse_input_decl (
#if FuncPr
	TreeNode tn)
#else
	tn)
	TreeNode tn;
#endif
{
	Param p = (Param) rb_Value (tn);
	int type;

	if (p->io.in == 0)
		return 0;

	fprintf (output_file, "    ");
	if (p->io.new)
		fprintf (output_file, "NEW ");
	fprintf (output_file, p->io.out ? "<> " : "< ");
	type = p->type == TYPE_ARRAY ? p->a->type : p->type;
	switch (type) {
	case TYPE_INHERITED: 
		break;
	case TYPE_INT:
		fprintf (output_file, "int ");
		break;
	case TYPE_FLOAT:
		fprintf (output_file, "float ");
		break;
	case TYPE_DOUBLE:
		fprintf (output_file, "double ");
		break;
	case TYPE_CHAR:
		fprintf (output_file, "char ");
		break;
	default:
		msg_Format ("input_decl: unknown parameter type %d for param %s",
					p->type, p->name ? p->name : "(null)");
		unparse_errors++;
		break;
	}
	unparse_param (p);
	if (p->val != (Expr) NULL) {
		fprintf (output_file, "=");
		unparse_expr (p->val, 0);
	}
	fprintf (output_file, ";\n");
	return 0;
}

/*
 * unparse all input decls for this particular node
 */

static void
unparse_input_decl_list (
#if FuncPr
	Node n)
#else
	n)
	Node n;
#endif
{
	rb_Traverse (n->params, unparse_input_decl);
}

/*
 * unparse a sub call, its return value (if any), and its args
 */

static void
unparse_sub_call (
#if FuncPr
	Node n)
#else
	n)
	Node n;
#endif
{
	int i;
	
	fprintf (output_file, "    ");
	if (n->ret_param) {
		unparse_param (n->ret_param);
		fprintf (output_file, " = ");
	}
	fprintf (output_file, " %s(", n->sub_name);
	for (i = 0; i < n->nargs; ++i) {
		unparse_expr (n->args[i], 0);
		if (i != n->nargs - 1)
			fprintf (output_file, ", ");
	}
	fprintf (output_file, ");\n");
}

/*
 * unparse a node declaration
 */

static int
unparse_single_node (
#if FuncPr
	Node n)
#else
	n)
	Node n;
#endif
{
	/* node keyword */
	switch (n->node_type) {
	case NODE_NORMAL:  fprintf (output_file, "NODE ");	   break;
	case NODE_COND:	   fprintf (output_file, "BEGINSWITCH ");break;
	case NODE_ENDCOND: fprintf (output_file, "ENDSWITCH ");  break;
	case NODE_LOOP:	   fprintf (output_file, "BEGINLOOP ");break;
	case NODE_ENDLOOP: fprintf (output_file, "ENDLOOP ");  break;
	case NODE_FANOUT:  fprintf (output_file, "FANOUT ");   break;
	case NODE_FANIN:   fprintf (output_file, "FANIN ");	   break;
	case NODE_PIPE:	   fprintf (output_file, "BEGINPIPE ");break;
	case NODE_ENDPIPE: fprintf (output_file, "ENDPIPE ");  break;
	default:
		msg_Format ("node %d: unknown node type %d\n",
					n->nk.id, n->node_type);
		unparse_errors++;
		return 0;
	}

	if (n->xy.set)
		fprintf (output_file, "[ %d %d ] ", n->xy.x, n->xy.y);
	fprintf (output_file, "%d", n->nk.id);

	switch (n->node_type) {
	case NODE_NORMAL:
		fprintf (output_file, "\n");
		if (n->params == rb_Next (n->params) && n->sub_name == (char *) NULL)
			fprintf (output_file, "    UNDEFINED;\n");
		else {
			unparse_input_decl_list (n);
			unparse_sub_call (n);
			unparse_output_decl_list (n);
		}
		break;
	case NODE_COND:
		fprintf (output_file, "\n    ");
		if (n->nargs == 0)
			fprintf (output_file, "UNDEFINED;\n");
		else {
			fprintf (output_file, "(");
			unparse_expr (n->args[0], 0);
			fprintf (output_file, ");\n");
		}
		break;
	case NODE_ENDCOND:
	case NODE_ENDLOOP:
	case NODE_ENDPIPE:
	case NODE_FANIN:
		fprintf (output_file, "\n    ;\n");
		break;
	case NODE_FANOUT:
	case NODE_PIPE:
		fprintf (output_file, "\n    ");
		if (n->ret_param == (Param) NULL)
			fprintf (output_file, "UNDEFINED;\n");
		else {
			unparse_param (n->ret_param);
			fprintf (output_file, " = ");
			unparse_expr (n->args[0], 0);
			fprintf (output_file, " TO ");
			unparse_expr (n->args[1], 0);
			fprintf (output_file, ";\n");
		}
		break;
	case NODE_LOOP:		
		fprintf (output_file, "\n    ");
		if (n->ret_param == (Param) NULL)
			fprintf (output_file, "UNDEFINED;\n");
		else {		
			fprintf (output_file, "(");
			unparse_param (n->ret_param);
			fprintf (output_file, "=");
			unparse_expr (n->args[0], 0);
			fprintf (output_file, "; ");
			unparse_expr (n->args[1], 0);
			fprintf (output_file, "; ");
			unparse_param (n->ret_param);
			fprintf (output_file, "=");
			unparse_expr (n->args[2], 0);
			fprintf (output_file, ");\n");
		}
		break;
	}
	return 0;
	
}


static int
unparse_a_node (
#if FuncPr
	TreeNode tn)
#else
	tn)
	TreeNode tn;
#endif
{
	Node n = (Node) rb_Value (tn);
	return unparse_single_node (n);
}

static void
unparse_nodes (
#if FuncPr
	Graph g)
#else
	g)
	Graph g;
#endif
{
	rb_Traverse (g->nlist, unparse_a_node);
}

#if 0
/* XXX this really should be unparse_Program, I guess */

void
unparse (
#if FuncPr
	Graph g, FILE *output, FILE *error)
#else
	g, output, error)
	Graph g;
	FILE *output;
	FILE *error;
#endif
{
	if (output != (FILE *) NULL)
		output_file = output;
	else
		output_file = stdout;

	if (error != (FILE *) NULL)
		error_file = error;
	else
		error_file = stderr;

	fprintf (output_file, "PROGRAM\n");
	unparse_nodes (g);
	unparse_arcs (g);
}
#endif

int
unparse_Program (
#if FuncPr
	Graph g, FILE *output)
#else
	g, output)
	Graph g;
	FILE *output;
#endif
{
	if (output != (FILE *) NULL)
		output_file = output;
	else
		output_file = stdout;

	unparse_errors = 0;
	fprintf (output_file, "PROGRAM\n");
	unparse_nodes (g);
	unparse_arcs (g);
	return unparse_errors;
}

int
unparse_Node (
#if FuncPr
	Node n, FILE *output)
#else
	n, output)
	Node n;
	FILE *output;
#endif
{
	if (output != (FILE *) NULL)
		output_file = output;
	else
		output_file = stdout;

	unparse_errors = 0;
	unparse_single_node (n);
	return unparse_errors;
}

/*
 * Local Variables:
 * tab-width:4
 * comment-column:40
 * End:
 */
