/*
 *	HeNCE Tool
 *
 *	trace.c - Graphical execution tracing.
 *		This is where everything goes that happens in the trace panel.
 *
 *	Sep 1991  Robert Manchek  manchek@CS.UTK.EDU.
 *
 *	Revision Log
 *
$Log$
 *
 */

/*
 * TODO:
 * - handle special nodes
 * - usage graph
 * - trace in either direction
 */


#include <sys/types.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include "xincl.h"
#include "rb.h"
#include "param.h"
#include "exp.h"				/* XXX Yuk! */
#include "graph.h"
#include "graphP.h"
#include "xcomn.h"
#include "comn.h"
#include "costmat.h"
#include "hostmap.h"
#include "tf.h"

#include "xbm/legend.xbm"
#include "xbm/fwd.xbm"
#include "xbm/fwdstep.xbm"
#include "xbm/hostmap.xbm"
#include "xbm/monitor.xbm"
#include "xbm/rewind.xbm"
#include "xbm/stop.xbm"
#ifdef USAGE_GRAPH
#include "xbm/usage.xbm"
#endif

extern int errno;				/* 4.3 BSD needs this */

/*******************
*  from elsewhere  *
*                  *
*******************/

extern Widget traceButton;		/* from widmain.c */
extern Graph scratchgraph;		/* from widmain.c */
extern struct costm *costmatrix; /* from widmain.c */



/*************
*  exported  *
*            *
*************/

void start_trace();
int stop_trace();
Widget hostsWin = NULL;			/* used by hostmap.c */
Widget hostsPop = NULL;			/* used by hostmap.c */

/***********
*  global  *
*          *
***********/

static void rewind_cb(), fwd_cb(), stop_cb(), speed_cb();
#if 0
static void redraw_cb();
#endif
static void fwdstep_cb();
static void monitor_cb();
static Widget cre_cmdpanel(), cre_nodecanvas();
static void graph_ev();
static void trace_refresh();
static void hosts_cb();
#ifdef USAGE_GRAPH
static void usage_cb();
#endif

static Widget traceCmd = 0;
static Widget traceGraph = 0;
static Widget traceHost = 0;
static Widget speedKnob = 0;
static Widget rewindButton = 0;
static Widget stopButton = 0;
static Widget fwdButton = 0;
static Widget fwdstepButton = 0;

static Arg args[16];
#if 0
static char crud[1024];
#endif
static Window graphWin;
static int animat_delay = 0;
static XtCallbackRec callback[2] = { { 0, 0 }, { 0, 0 } };
static Graph tracegraph = 0;
static int monitorMode = 0;



/*********************
 *                   *
 *  widget creation  *
 *                   *
 *********************/

/*
 *	start_trace()
 *
 *	Start method for trace mode.  Create any widgets and manage em.
 *	Set default graph file name from main module.
 */

void
start_trace()
{
	static int once = 1;
	Widget w;
#if 0
    register int h;
#endif

	if (once) {
		once = 0;
		tracegraph = gr_New ();
		/* XXX what if there's no trace file name? */
		/* tf_SetFile (traceFileName); */
	}
	/*	gr_Free (tracegraph); */
	/* tracegraph = gr_CopyGraph (scratchgraph); */

	XawFormDoLayout (mainForm, False);
	w = cre_cmdpanel (mainCmd);
	XtManageChild (traceCmd);
	w = cre_nodecanvas (w);
	XtManageChild (traceGraph);
	XawFormDoLayout (mainForm, True);
	set_togg (traceButton, 1);

	checkThings ();
}

/*
 *	stop_trace()
 *
 *	Stop method for trace mode.  Ask if want to save graph file or
 *	cancel stop.
 *	Returns 0 if successfully stopped, else 1.
 */

static void stopTrace ();

int
stop_trace()
{
#if 0
	int i;
#endif

	/* This serves to delete the trace event timer */
	stopTrace ();

	XawFormDoLayout (mainForm, False);
	XtUnmanageChild (traceCmd);
	XtUnmanageChild (traceGraph);
	XawFormDoLayout (mainForm, True);
	set_togg (traceButton, 0);

	return 0;
}

static Widget
cre_nodecanvas(below)
Widget below;
{
	Widget retw;
	Widget w;
	int n;

	if (traceGraph)
		return traceGraph;
	n = 0;
	XtSetArg(args[n], XtNfromVert, (XtArgVal)below); n++;
	XtSetArg(args[n], XtNallowHoriz, (XtArgVal)True); n++;
	XtSetArg(args[n], XtNallowVert, (XtArgVal)True); n++;
	retw = traceGraph = XtCreateWidget("traceNodeVp", viewportWidgetClass,
			mainForm, args, n);

	n = 0;
	XtSetArg(args[n], XtNbackground, (XtArgVal)(mycolor[COLBGND])); n++;
	w = XtCreateManagedWidget("traceNodeCan", widgetClass,
		traceGraph, args, n);

/* XXX should use something other than base widget class here so we
   could set more nice resources... */

	XtRealizeWidget(traceGraph);
	graphWin = XtWindow(w);
	XtAddEventHandler(w,
		StructureNotifyMask|ExposureMask, False,
		graph_ev, (XtPointer)0);

	return traceGraph;
}

static Widget
cre_cmdpanel(below)
	Widget below;
{
	Widget retw;
	Widget w;
#if 0
	Widget w2, w3;
#endif
	Widget w4;
	int n, saved_n;

	/*
	 * translation table for radio group
	 */

	XtTranslations radio_ttab = XtParseTranslationTable("\
<EnterWindow>:highlight(Always)\n\
<LeaveWindow>:unhighlight()\n\
<Btn1Down>,<Btn1Up>:set()notify()");

	if (traceCmd)
		return traceCmd;

	n = 0;
	XtSetArg(args[n], XtNfromVert, (XtArgVal)below); n++;
	XtSetArg(args[n], XtNborderWidth, (XtArgVal)0); n++;
	XtSetArg(args[n], XtNhorizDistance, (XtArgVal)0); n++;
	retw = traceCmd = XtCreateWidget("traceCmd", formWidgetClass,
			mainForm, args, n);

	n = 0;
	XtSetArg(args[n], XtNleft, XtChainLeft); n++;
	XtSetArg(args[n], XtNright, XtChainLeft); n++;
	XtSetArg(args[n], XtNtop, XtChainTop); n++;
	XtSetArg(args[n], XtNbottom, XtChainTop); n++;

	/* BEGIN RADIO GROUP */
	saved_n = n;
	XtSetArg (args[n], XtNtranslations, radio_ttab); n++;

	w = cre_tog_but ("rewind", traceCmd, args, n,
					 (Widget) NULL, (Widget) NULL, (char *) NULL,
					 rewind_bits, rewind_width, rewind_height,
					 rewind_cb, 0, 0);
	rewindButton = w;
	w4 = w;
	XtSetArg (args[n], XtNradioGroup, w4); n++;

	w = cre_tog_but ("stop", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 stop_bits, stop_width, stop_height, stop_cb, 0, 1);
	stopButton = w;

	w = cre_tog_but ("fwdstep", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 fwdstep_bits, fwdstep_width, fwdstep_height,
					 fwdstep_cb, 0, 0);
	fwdstepButton = w;

	w = cre_tog_but ("fwd", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 fwd_bits, fwd_width, fwd_height, fwd_cb, 0, 0);
	fwdButton = w;

	n = saved_n;
	/* END RADIO GROUP */

	/* SPEED SLIDER */
	saved_n = n;
	XtSetArg (args[n], XtNfromHoriz, w); n++;
	XtSetArg(args[n], XtNorientation, XtorientHorizontal); n++;
	XtSetArg(args[n], XtNwidth, 200); n++;
	XtSetArg(args[n], XtNheight, 20); n++;
	{
		float f = 1.0;

		if (sizeof(f) > sizeof(args[0].value)) {
			XtSetArg(args[n], XtNtopOfThumb, &f); n++;
			XtSetArg(args[n], XtNshown, &f); n++;
		} else {
			args[n].name = XtNtopOfThumb;
			*(float*)&args[n].value = f; n++;
			args[n].name = XtNshown;
			*(float*)&args[n].value = f; n++;
		}
	}
	callback[0].callback = speed_cb;
	XtSetArg(args[n], XtNjumpProc, (XtArgVal)callback); n++;
	w = speedKnob = XtCreateManagedWidget("execSpeed",
		scrollbarWidgetClass, traceCmd, args, n);
	n = saved_n;
	/* SPEED SLIDER */

#if 0
	/*
	 * XXX there is no bitmap for redraw button.  Do we need a
	 * redraw button anyway?
	 */
	w = cre_cmd_but("redraw", traceCmd, args, n,
					(Widget) NULL, w, "redraw",
					(char *) NULL, 0, 0, redraw_cb, 0, 0);
#endif

	w = cre_tog_but ("monitor", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 monitor_bits, monitor_width, monitor_height,
					 monitor_cb, 0, 0);
	w = cre_tog_but ("hosts", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 hostmap_bits, hostmap_width, hostmap_height,
					 hosts_cb, 0, 0);
#ifdef USAGE_GRAPH
	w = cre_tog_but ("usage", traceCmd, args, n,
					 (Widget) NULL, w, (char *) NULL,
					 usage_bits, usage_width, usage_height, usage_cb, 0, 0);
#endif /* USAGE_GRAPH */



	XtRealizeWidget(traceCmd);
	return traceCmd;
}

/****************************
 *  trace widget callbacks  *
 *                          *
 ***************************/

static enum { REVERSE, FORWARD } traceDirection = FORWARD;
static int traceStopped = 1;
static XtIntervalId timerId;
static int traceFileHasChanged = 1;
static int graphFileHasChanged = 1;

/*
 * do any checks that are necessary before we start tracing
 * return 0 if okay, nonzero otherwise.
 */

static int
checkThings ()
{
	if (traceFileHasChanged) {
		hmap_NukeHosts ();
		if (traceFileName == NULL || *traceFileName == '\0') {
			msg_Format ("No trace file.\n");
			return 1;
		}
		if (tf_SetFile (traceFileName) == EOF) {
			msg_Format ("Can't read trace file \"%s\"\n", traceFileName);
			return 1;
		}
		traceFileHasChanged = 0;
	}
	if (graphFileHasChanged) {
		/*
		 * unlike trace files, graph files are loaded by the routine
		 * that handles "graph" button events.  So we just check for
		 * empty graphs here.
		 */
		gr_Free (tracegraph);
		tracegraph = gr_CopyGraph (scratchgraph);
		if (gr_Empty (tracegraph)) {
			msg_Format ("Graph is empty.\n");
			return 1;
		}
		trace_refresh ();
		/*
		 * In addition to checking for errors in the graph,
		 * gr_Critic() also marks pairs of special nodes
		 */
		if (gr_Critic (tracegraph) != 0) {
			msg_Format ("Graph has errors.\n");
			return 1;
		}
		graphFileHasChanged = 0;
	}
	return 0;
}

/*
 * This function is called whenever we change a trace file.  It is
 * called with x == 0 if the caller wants to see if its okay to change the
 * file, and with x == 1 if the caller wants to let us know it's been
 * changed.  If x == 0 and we will not allow the file to be changed,
 * print out an appropriate message.
 */

int
trace_ChangeTraceFile (x)
int x;
{
	switch (x) {
	case 0:
		if (!traceStopped) {
			msg_Format ("Can't change trace file while trace is running.\n");
			return 0;
		}
		return 1;
	case 1:
		traceFileHasChanged = 1;
		break;
	}
}

int
trace_ChangeGraphFile (x)
int x;
{
	switch (x) {
	case 0:
		if (!traceStopped) {
			msg_Format ("Can't change graph file while trace is running.\n");
			return 0;
		}
		return 1;
	case 1:
		graphFileHasChanged = 1;
		break;
	}
}


static void
stopTrace ()
{
	extern Widget traceFileButton; /* in widmain.c */
	extern Widget grfButton;	/* in widmain.c */

	if (!traceStopped) {
		XtRemoveTimeOut (timerId);
		traceStopped = 1;
#if 1
		XtVaSetValues (traceFileButton, XtNsensitive, True, NULL);
		XtVaSetValues (grfButton, XtNsensitive, True, NULL);
#endif
	}
}


static void
startTrace ()
{
	extern Widget traceFileButton; /* in widmain.c */
	extern Widget grfButton;	/* in widmain.c */
	void Tick ();

	if (traceStopped) {
		timerId = XtAppAddTimeOut (context, animat_delay, Tick, NULL);
		traceStopped = 0;
#if 1
		XtVaSetValues (traceFileButton, XtNsensitive, False, NULL);
		XtVaSetValues (grfButton, XtNsensitive, False, NULL);
#endif
	}
}

/*
 * Come here to process a trace event.
 *
 * XXX this works fine for dags as is, but doesn't understand how
 * graphs are expanded when special nodes occur.  Currently, it
 * only displays the first instance of a node.
 *
 */

static void
traceEvent (tev)
struct trace_event *tev;
{
	Node n;

	if (tev->id == -1 && tev->inst == -1)
		;
	else {
		n = gr_GetNodeByIdInst (tracegraph, tev->id, 0);
		if (n == (Node) NULL)
			return;
	}

#if 1
	msg_Format ("%ld %ld %d/%d %s %c %d %d\n",
				tev->timestamp.secs, tev->timestamp.serial,
				tev->id, tev->inst,
				tf_GetEventName (tev->event_type),
				tev->tf ? 'T' : 'F',
				tev->lo, tev->hi);
#endif

	switch (tev->event_type) {
	case TEV_START:
		hmap_ResetHosts ();
		trace_refresh ();
		break;
	case TEV_WAITING:
		break;
	case TEV_READY:
		/*
		 * Grrr.  We don't get events when right special nodes "happen",
		 * so we check for them here.  If a parent of this node is a
		 * right special node (but not an endpipe), we mark it and its parent
		 * as done.
		 */
		if (n->nparents != 0 && !rb_Empty (n->parents)) {
			TreeNode tn;
			for (tn = rb_First (n->parents); tn != n->parents;
				 tn = rb_Next (tn)) {
				Node n2 = (Node) rb_Value (tn);
				switch (n2->node_type) {
				case NODE_ENDCOND:
				case NODE_ENDLOOP:
				case NODE_FANIN:
					if (n2->state != ST_DONE) {
						n2->state = ST_DONE;
						n2->pair->state = ST_DONE;
						xgr_DrawNode (graphWin, n2, -1);
						xgr_DrawNode (graphWin, n2->pair, -1);
					}
					break;
				case NODE_ENDPIPE:
					/*
					 * Horrible hack for pipes.  We don't really
					 * know if a pipe is "done" until all of its
					 * children are ready to run.  So we count
					 * the number of children that become ready,
					 * and when we reach the number of children
					 * that the endpipe node has, we know we are
					 * done.  (Actually, this would probably work
					 * for other node types also.)
					 */
					if (n2->numRunning++ == n2->nchildren) {
						n2->state = ST_DONE;
						n2->pair->state = ST_DONE;
						xgr_DrawNode (graphWin, n2, -1);
						xgr_DrawNode (graphWin, n2->pair, -1);
						n2->numRunning = 0;
					}
				}
			}
		}
		n->state = ST_READY;
		goto drawnode;
	case TEV_RUNNING:
		hmap_HostEvent (tev->host, tev->id, tev->inst, n->state, ST_RUNNING);
		n->numRunning++;
		goto drawnode;
	case TEV_DONE:
		hmap_HostEvent (tev->host, tev->id, tev->inst, n->state, ST_DONE);
		n->numRunning--;
		n->numIdle++;
		goto drawnode;
	case TEV_DEAD:
		hmap_HostEvent (tev->host, tev->id, tev->inst, n->state, ST_DEAD);
		n->state = ST_DEAD;
		n->numIdle--;
		goto drawnode;
	case TEV_FINISH:
		hmap_ResetHosts ();
		gr_ResetGraph (tracegraph, ST_DEAD);
		stopTrace ();
		trace_refresh ();
		msg_Format ("program finished.");
		set_togg (fwdButton, 0);
		set_togg (stopButton, 1);
		break;
	case TEV_MACHINE:
		break;
	case TEV_VNODE:
		switch (n->node_type) {
		case NODE_COND:
		case NODE_LOOP:
		case NODE_FANOUT:
			if (n->node_type == NODE_FANOUT ? (tev->hi > tev->lo) : tev->tf) {
				/*
				 * zero all process counters within this subgraph.
				 * set all nodes within the subgraph to ST_NOT_BEGUN,
				 * but mark this node and its pair as running
				 */
				gr_ResetSubGraph (n, ST_NOT_BEGUN);
				n->state = ST_RUNNING;
				n->pair->state = ST_RUNNING;
				trace_refresh ();
			}
			else {
				/*
				 * like above, but set all nodes to ST_DONE, and
				 * mark this node and its pair as done
				 */
				gr_ResetSubGraph (n, ST_DONE);
				n->state = ST_DONE;
				n->pair->state = ST_DONE;
				trace_refresh ();
			}
			break;
		case NODE_PIPE:
			n->state = ST_RUNNING;
			n->pair->state = ST_RUNNING;
			break;
		case NODE_ENDCOND:
		case NODE_ENDLOOP:
		case NODE_FANIN:
			/*
			 * mark this node and its pair as done
			 * (currently, this never happens because the executioner
			 * doesn't generate these events, but this is here anyway.)
			 */
			n->state = ST_DONE;
			n->pair->state = ST_DONE;
			xgr_DrawNode (graphWin, n, -1);
			xgr_DrawNode (graphWin, n->pair, -1);
			/*
			 * XXX mark all nodes within this subgraph as idle?
			 */
			break;
		case NODE_ENDPIPE:
			/* not sure what to do with these */
			break;
		default:
			break;
		}
		break;
	case TEV_UNKNOWN:
	default:
		break;
	}

	return;

	/*
	 * come here when we want to redraw a node.
	 *
	 * If we have any instances of this node that are either
	 * running or idle, we will display that as the "state"
	 * of the node on the screen.  Otherwise, we will display
	 * the "state" of the node that was there already.
	 */

 drawnode:
	if (n->numIdle < 0)
		n->numIdle = 0;
	if (n->numRunning < 0)
		n->numRunning = 0;

	if (n->numRunning > 0)
		n->state = ST_RUNNING;
	else if (n->numIdle > 0)
		n->state = ST_DONE;

	xgr_DrawNode (graphWin, n, -1);
}

/*
 * Come here when we get an X timer event.  If we are still
 * tracing, read another event and deal with it.
 */

static void
Tick (x, y)
XtPointer x;
XtIntervalId *y;
{
	struct trace_event tev;

	if (traceStopped)
		return;

	if (traceDirection == FORWARD) {
		if (tf_NextEvent (&tev) == 0) {
			traceEvent (&tev);
		}
		else if (monitorMode) {
			/*
			 * file read failed, perhaps because we're still waiting on
			 * data.  Wait at least .5 sec before trying to read file again.
			 *
			 * XXX also need to see if executioner and pvm are still running.
			 */
			int delay = 500;
			if (animat_delay > 500)
				delay = animat_delay;
			timerId = XtAppAddTimeOut (context, delay, Tick, x);
			return;
		}
		else {
			msg_Format ("%s: premature end of file\n", traceFileName);
			stopTrace ();
			set_togg (fwdButton, 0);
			set_togg (stopButton, 1);
		}
	}
	else {
		if (tf_PrevEvent (&tev) == 0)
			traceEvent (&tev);
	}

	/* reset time out */
	timerId = XtAppAddTimeOut (context, animat_delay, Tick, x);
}

#if 0
/*	redraw_cb()
 *
 *	Redraw command
 */

static void
redraw_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	XClearWindow (xDisp, graphWin);
	trace_refresh ();
}
#endif

/*	stop_cb()
 *
 *	Stop command
 */

static void
stop_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	stopTrace ();
}

/*	fwd_cb()
 *
 *	Forward command
 */

static void
fwd_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	struct trace_event tev;

	if (checkThings ())
		return;

	traceDirection = FORWARD;
	stopTrace ();

	/*
	 * Hack to detect end of tracefile and do something reasonable.
	 * Go ahead and read one event.  If it succeeds, display the
	 * results; otherwise, don't go into continuous trace mode.
	 */

	if (tf_NextEvent (&tev) == 0) {
		traceEvent (&tev);
		startTrace ();
	}
	else {
		set_togg (fwdButton, 0);
		set_togg (stopButton, 1);
		msg_Format ("End of trace file.\n");
	}
}


static void
fwdstep_cb (w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	struct trace_event tev;

	/* need to make sure a graph exists */
	if (checkThings ())
		return;

	traceDirection = FORWARD;
	stopTrace ();

	if (tf_NextEvent (&tev) == 0)
		traceEvent (&tev);
	else
		msg_Format ("End of trace file.\n");

	set_togg (fwdstepButton, 0);
	set_togg (stopButton, 1);
}

/*	rewind_cb()
 *
 *	Rewind command
 */

static void
rewind_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	stopTrace ();
	traceDirection = FORWARD;
	if (tf_Rewind () < 0)
		msg_Format ("rewind failed: error %d\n", errno);
	hmap_ResetHosts ();

	/*
	 * set graph back to initial state
	 *
	 * XXX this is a hack - fix when we get generalized save/restore
	 * state - which we need for usage graph anyway.
	 */
	gr_ResetGraph (tracegraph, ST_NOT_BEGUN);
	trace_refresh ();

	set_togg (rewindButton, 0);
	set_togg (stopButton, 1);
	msg_Format ("Rewound trace file.\n");
}

/*	speed_cb()
 *
 *	Speed knob change
 */

static void
speed_cb(w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	float f;

	f = *(float*)cd;
	animat_delay = (1.0 - f) * 2000;
}


Boolean hostsAreVisible = False;

/*
 * come here when the host map window gets an expose event
 */

void
hosts_expose (w, cli, xev, rem)
Widget w;
XtPointer cli;
XEvent *xev;
Boolean *rem;
{
	if (hostsWin == NULL)
		return;
	if (xev->type == Expose && xev->xexpose.count == 0)
		hmap_Refresh (hostsWin);
}

/*
 * "hosts" command button - pop up/down hosts map
 */

static void
hosts_cb (w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	Arg args[20];
	register int n;
	int x, y, curr_x, curr_y;
	Widget form;
#if 0
	Widget button;
#endif

#if 0
	/* fprintf(stderr, "hosts_cb\n"); */

	/*
	 * If a cost matrix has been defined, make sure the host map
	 * has been initialized (it won't hurt to do it twice), and
	 * read in the icons file.  If the cost matrix has not been
	 * defined, we will go ahead and pop up the window, but the
	 * user will have to define a cost matrix before tracing will
	 * actually do anything.
	 */

	if (costmatrix) {
	    int h;

	    for (h = 0; h < costmatrix->nhost; h++)
		if (costmatrix->hosts[h] && *costmatrix->hosts[h])
		    hmap_AddHost(costmatrix->hosts[h]);
	    hmap_ReadIconList (XtWindow(topLevel), iconList);
	}
#endif

	if (hostsAreVisible = !hostsAreVisible) {

		curr_x = curr_y = 400;

		XtTranslateCoords (w, curr_x, curr_y, &x, &y);

		n = 0;
		XtSetArg(args[n], XtNx, x);                         n++;
		XtSetArg(args[n], XtNy, y);                         n++;
		XtSetArg(args[n], XtNallowShellResize, True);       n++;
        hostsPop = XtAppCreateShell("Host Map", "HostsPopup",
									applicationShellWidgetClass,
									XtDisplay(w), args, n);


        n = 0;
        XtSetArg(args[n], XtNallowHoriz, (XtArgVal)True); n++;
        XtSetArg(args[n], XtNallowVert, (XtArgVal)True); n++;
        form = XtCreateManagedWidget("form", viewportWidgetClass, hostsPop,
									 args, n);

		n = 0;
		XtSetArg (args[n], XtNheight, (XtArgVal) 300); n++;
		XtSetArg (args[n], XtNwidth, (XtArgVal) 500); n++;
		hostsWin = XtCreateManagedWidget ("hostsMap", 
										widgetClass, form, args, n);


		XtAddEventHandler (hostsWin, StructureNotifyMask|ExposureMask,
						   False, hosts_expose, NULL);

		XtPopup (hostsPop, XtGrabNone);
	}
	else {
		XtDestroyWidget (hostsPop);
		hostsWin = hostsPop = NULL;
	}
}

#ifdef USAGE_GRAPH
static void
usage_cb (w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
}
#endif

static void
monitor_cb (w, cli, cd)
Widget w;
XtPointer cli;
XtPointer cd;
{
	monitorMode = !monitorMode;
}

/********************************
*  graph window event handling  *
*                               *
********************************/

/*	graph_ev()
 *
 *	Event in graph window
 */

static void
graph_ev(w, cli, xev, rem)
Widget w;
XtPointer cli;
XEvent *xev;
Boolean *rem;
{
	XButtonEvent *bev = (XButtonEvent*)xev;
	XConfigureEvent *cev = (XConfigureEvent*)xev;
	XExposeEvent *eev = (XExposeEvent*)xev;
	XMotionEvent *mev = (XMotionEvent*)xev;
	XCrossingEvent *rev = (XCrossingEvent*)xev;
#if 0
	int x, y;
#endif

	switch (xev->type) {

	case Expose:
		if (eev->count == 0)
			trace_refresh();
		break;

	case ConfigureNotify:
		break;

	}
}

static void
trace_refresh()
{
	xgr_DrawGrf(graphWin, tracegraph, 1);
}

#if 0
/**************************
*  trace animation stuff  *
*                         *
**************************/


read_tracefile()
{
	FILE *ff;
	char buf[256];
	struct tracev tev;
	char nodetype[32];
	char machine[128];
	char evtypnm[128];
	char tf;
	int i;
	char **pp;

	if ((ff = fopen(traceFileName, "r")) == NULL) {
		msg_Format ("can't open trace file \"%s\"\n", traceFileName);
		return;
	}
	while (xfgets(buf, sizeof(buf), ff)) {
		if (buf[0] != '#')
			continue;
		if (sscanf(buf + 1, "%ld %d/%d %s %s %s %c %d %d",
				&tev.time, &tev.id, &tev.inst, nodetype, machine,
				evtypnm, &tf, &tev.lo, &tev.hi) != 9)
			continue;
		for (pp = evtypenames; *pp && strcmp(evtypnm, *pp); pp++);
		tev.evtype = pp - evtypenames;
		tev.tf = tf == 'T';

		trace_event(machine, tracegraph, &tev);
		checkevents();
		if (whoa)
			break;
		delay (animat_delay);
	}
	fclose(ff);
}

trace_event(machine, grf, tvp)
char *machine;
Graph grf;
struct tracev *tvp;
{
	Node n;

	if (tvp->id == -1 && tvp->inst == -1)
		;
	else if ((n = gr_GetNodeByIdInst (grf, tvp->id, tvp->inst)) == (Node) NULL)
		return;


	switch (tvp->evtype) {

	case TEV_START:
		/* printf ("tev_start\n"); */
		hmap_ResetHosts ();
		break;

	case TEV_WAITING:
		break;

	case TEV_READY:
		n->state = ST_READY;
		xgr_DrawNode(graphWin, n, 1);
		break;

	case TEV_RUNNING:
		hmap_HostEvent (machine, n->state, ST_RUNNING);
		n->state = ST_RUNNING;
		xgr_DrawNode(graphWin, n, 1);
		break;

	case TEV_DONE:
		hmap_HostEvent (machine, n->state, ST_DONE);
		n->state = ST_DONE;
		xgr_DrawNode(graphWin, n, 1);
		break;

	case TEV_DEAD:
		hmap_HostEvent (machine, n->state, ST_DEAD);
		n->state = ST_DEAD;
		xgr_DrawNode(graphWin, n, 1);
		break;

	case TEV_FINISH:
		/* printf ("tev_finish\n"); */
		hmap_ResetHosts ();
		gr_ResetGraph (tracegraph, ST_DEAD);
		break;

	case TEV_MACHINE:
		/*
		 * The tracefile spits out a list of machines before doing
		 * any computation.  That way we don't have to define a cost
		 * matrix file to look at a trace, and the hosts map and
		 * (eventually) usage graph only reflect the hosts that were
		 * used in the computation.
		 */
		hmap_AddHost (machine);
		break;

	case TEV_VNODE:
	case TEV_UNKNOWN:	/* XXX a hack that probably means virt node exp */
		switch (n->node_type) {

		case NODE_LOOP:
			if (tvp->tf) {	/* loop happens */
				Node pairn = n->pair;
				Node nn;
				Tree ctree = pairn->children;
				TreeNode ctn;
				Node cn;

	/* make copy of loop graph */
				nn = gr_CopySubgraph(grf, n, 1,
					pairn->xy.x - n->xy.x,
					-(n->xy.y - pairn->xy.y + 20)); /* XXX 20 is a hack */
				nn = nn->pair;

	/* move arcs from end of old loop to end of new */
				while ((ctn = rb_First(ctree)) != ctree) {
					cn = (Node)rb_Value(ctn);
					gr_RemAnArc(grf, pairn, cn);
					gr_BuildAnArc(grf, nn, cn);
				}
	/* link old loop end to new loop begin */
				nn = nn->pair;
				gr_BuildAnArc(grf, pairn, nn);
	/* remove old loop virtual nodes */
				gr_NullifyNode(grf, n);
				gr_NullifyNode(grf, pairn);
				XClearWindow(xDisp, graphWin);
				trace_refresh();

			} else {	/* loop doesn't happen */
	/* short ckt loop body */
				gr_NullifySubgraph(grf, n);
				XClearWindow(xDisp, graphWin);
				trace_refresh();
			}
			break;

		case NODE_FANOUT:
			{
				int cnt = tvp->hi - tvp->lo;
				int fv;
				Node nn;
				Node pairn = n->pair;

				for (fv = 1; fv < cnt; fv++) {
					gr_CopySubgraph(grf, n, 0, fv * 50, 0);
				}
				gr_NullifyNode(grf, n);
				gr_NullifyNode(grf, pairn);
				XClearWindow(xDisp, graphWin);
				trace_refresh();
			}
			break;

		default:
			break;
		}
		break;

	default:
		break;
	}
}
#endif

/*
 * Local variables:
 * tab-width:4
 * End:
 */
