#include <stdio.h>

#ifdef IMA_PMAX
#include <sys/types.h>
#include <unistd.h>
#endif

#include "xincl.h"				/* XXX get rid of this once struct icon
								 * is removed from hostmap.h */
#include "tf.h"
#include "hostmap.h"
#include "xcomn.h"
#include "comn.h"

#ifndef SEEK_SET
#define SEEK_SET 0
#endif
#ifndef SEEK_CUR
#define SEEK_CUR 1
#endif
#ifndef SEEK_END
#define SEEK_END 2
#endif

/*
 * routines for manipulating HeNCE trace files
 */

static struct event_time tf_Epoch; /* time of START event */
static struct event_time tf_LastEventSeen; /* last event in the file
											  that we've seen */
static FILE *traceFilePointer;


/*
 * compare two event timestamps.  return
 * < 0 if t1 is before t2,
 * = 0 if t1 == t2, and
 * > 0 if t1 is after t2.
 */

int
tf_TimeCompare (t1, t2)
struct event_time *t1, *t2;
{
    if (t1->secs == t2->secs)
		return t1->serial - t2->serial;
    return t1->secs - t2->secs;
}

static struct {
    char *name;
    int event_type;
} event_names[] = {
    { "START", 	TEV_START },
    { "WAITING",TEV_WAITING },
    { "READY", 	TEV_READY },
    { "RUNNING",TEV_RUNNING },
    { "DONE", 	TEV_DONE },
    { "DEAD", 	TEV_DEAD },
    { "FINISH", TEV_FINISH },
    { "MACHINE",TEV_MACHINE },
    { "VNODE", 	TEV_VNODE },
    { NULL, 	TEV_UNKNOWN },
};

/*
 * given an event name, look up its type
 */

int
tf_GetEventCode (event_name)
char *event_name;
{
    int i;
    for (i = 0; event_names[i].name != NULL ; ++i) {
		if (*event_name != *event_names[i].name)
			continue;
		if (strcmp (event_name, event_names[i].name) ==0 )
			return event_names[i].event_type;
    }
    return TEV_UNKNOWN;
}

/*
 * given an event code, return its name
 */

char *
tf_GetEventName (event_code)
int event_code;
{
	int i;
	for (i = 0; event_names[i].name != NULL; ++i) {
		if (event_code == event_names[i].event_type)
			return event_names[i].name;
	}
	return NULL;
}

/*
 * utility function to do an fgets() on a file that might be
 * being appended to by another process.  If there's not enough 
 * bytes to complete a line, clear the EOF flag before returning.
 * Our caller (tf_ReadEvent) will seek back to the point just before
 * this record, and we can try again next time to read it.
 */

static char *
xfgets (buf, size, fp)
char *buf;
int size;
FILE *fp;
{
	int c;
	char *bufp = buf;
	
	if (size <= 0)
		return buf;

	while (--size > 0) {
		if ((c = getc (fp)) == EOF) {
			clearerr (fp);
			return NULL;
		}
		if ((*bufp++ = c) == '\n')
			break;
	}
	*bufp = '\0';
	return buf;
}


static int
tf_ReadEvent (tev)
TraceEvent tev;
{
    char buf[256];
    char event_type[50];
    char tf;
    char nodetype[50];
    char machine[50];
	long fileaddr;
	
	if (traceFilePointer == NULL)
		return EOF;

	fileaddr = ftell (traceFilePointer);
    if (xfgets (buf, sizeof (buf), traceFilePointer) == NULL) {
		/*
		 * Maybe really the end, or perhaps more data to come later.
		 * seek to the start of the record again just in case.
		 */
		fseek (traceFilePointer, fileaddr, SEEK_SET);
		return EOF;
	}

	if (*buf != '#') {
		msg_Format ("trace file @ (%ldL): doesn't start with '#'\n",
					fileaddr);
		return EOF;
	}

	if (sscanf(buf + 1, "%ld %ld %d/%d %s %s %s %c %d %d",
			   &(tev->timestamp.secs), &(tev->timestamp.serial),
			   &(tev->id), &(tev->inst),
			   nodetype, machine, event_type,
			   &tf, &(tev->lo), &(tev->hi)) != 10) {
		msg_Format ("trace file @ (%ld): syntax error\n",
					fileaddr);
		return EOF;
	}

	tev->timestamp.fileaddr = fileaddr;
	tev->event_type = tf_GetEventCode (event_type);
	tev->tf = tf == 'T';

	if (strcmp (machine, "x") == 0 || strcmp (machine, "*") == 0)
		tev->host = (struct host_info *) NULL;
	else
		tev->host = hmap_AddHost (machine);

	if (tf_TimeCompare (&(tev->timestamp), &tf_LastEventSeen) > 0)
		tf_LastEventSeen = tev->timestamp;

	return 0;
}

/*
 * read the next event from the trace file
 * return 0 on success, EOF on error
 */

int
tf_NextEvent (tev)
struct trace_event *tev;
{
	if (tf_ReadEvent (tev) < 0)
		return EOF;
	return 0;
}

/*
 * read the previous event from the trace file
 * return 0 on success, EOF on error
 */

int
tf_PrevEvent (tev)
struct trace_event *tev; 
{
	static struct trace_event prev_record, curr_record;
	long fileaddr;
	int c;

	if (traceFilePointer == NULL)
		return EOF;

	/* can't backspace past beginning of file */
	if ((fileaddr = ftell (traceFilePointer)) == tf_Epoch.fileaddr)
		return EOF;

	/*
	 * Go back 200 characters or so (more than 2 times the largest
	 * trace file record should ever be), but never before the
	 * start of the file.  Skip forward until the start of a record,
	 * then read records until the trace file position *after* the
	 * read is at the same position as we started.
	 * The record we want is the one *before* that one.
	 */

	fileaddr -= 200L;
	if (fileaddr < tf_Epoch.fileaddr)
		fileaddr = tf_Epoch.fileaddr;
	else {
		if (fseek (traceFilePointer, -200L, SEEK_CUR) < 0) {
			msg_Format ("prevEvent: fseek() error\n");
			return EOF;
		}
		while ((c = getc (traceFilePointer)) != EOF && c != '\n');
	}

	if (tf_ReadEvent (&curr_record) < 0)
		return EOF;
	do {
		prev_record = curr_record;
		if (tf_ReadEvent (&curr_record) < 0)
			return EOF;
	} while (ftell (traceFilePointer) < fileaddr);

	if (fseek (traceFilePointer, prev_record.timestamp.fileaddr, SEEK_SET) < 0)
		msg_Format ("PrevEvent: fseek() error\n");

	*tev = prev_record;
	return 0;
}

/*
 * open a trace file named filename, and read any MACHINE events that
 * appear at the beginning of the file.
 *
 * Return 0 on success, EOF on error.
 */

int
tf_SetFile (filename)
char *filename;
{
    struct trace_event event;
    int lineno = 0;
	
    if (traceFilePointer)
		fclose (traceFilePointer);
    if ((traceFilePointer = fopen (filename, "r")) == NULL)
		return EOF;
	
    /* XXX need to clear out list of host names here */

    while (1) {
		if (tf_ReadEvent (&event) < 0) {
			/* XXX premature EOF, or perhaps just not written yet? */
			return EOF;
		}
		++lineno;
		if (event.event_type == TEV_MACHINE) {
			/*
			 * MACHINE events precede everything else in the host file.  One
			 * MACHINE event is generated for each machine to be used by
			 * the executioner.  Machine events serve to cause us to allocate
			 * an entry in the host table.
			 *
			 * By the time we get here, the allocation is already done by
			 * hmap_AddHost() when called from tf_ReadEvent().
			 */
		}
		else if (event.event_type == TEV_START) {
			/*
			 * now that we know what all of the host names are,
			 * read in the icon list and associate them with
			 * their preferred icons.
			 */
			hmap_ReadIconList (XtWindow (topLevel), iconList);
			/*
			 * remember the time and file location of the first
			 * event, and break.
			 */
			tf_Epoch = event.timestamp;
			tf_LastEventSeen = event.timestamp;
			tf_Seek (&tf_Epoch);
			return 0;
		}
		else {
			msg_Format ("file %s, line %d: unexpected event type %d\n",
						filename, lineno, event.event_type);
			fclose (traceFilePointer);
			traceFilePointer = (FILE *) NULL;
			return EOF;
		}
    }
}

/*
 * find the trace event that happened at time t, or the next event
 * thereafter.
 */

int
tf_Seek (t)
struct event_time *t;
{
	long first, last;
	int c;
	struct trace_event ev;

	if (traceFilePointer == NULL)
		return EOF;

	if (tf_TimeCompare (t, &tf_Epoch) <= 0) {
		/*
		 * t is before the beginning of time.
		 * Rather than report an error, just return the first event
		 */
		return fseek (traceFilePointer, tf_Epoch.fileaddr, SEEK_SET);
	}
	else if (tf_TimeCompare (t, &tf_LastEventSeen) > 0) {
		/*
		 * Event desired is later than the last event seen.
		 */
		int comp;

		/*
		 * Seek to the end of file less a few records, but don't
		 * try to seek before the beginning of the file.
		 * Then skip to the next newline.
		 */
		fseek (traceFilePointer, -tf_Epoch.fileaddr, SEEK_END);
		while ((c = getc (traceFilePointer)) != EOF && c != '\n');

		/*
		 * read an event so we know where to look relative to
		 * the current file position
		 */
		if (tf_ReadEvent (&ev) < 0) {
			/* error - this should not happen */
			return EOF;
		}
		comp = tf_TimeCompare (t, &ev.timestamp);
		if (comp == 0) {
			/* found it on first try (really unlikely) */
			fseek (traceFilePointer, ev.timestamp.fileaddr, SEEK_SET);
			return 0;
		}
		else if (comp < 0) {
			/* desired event preceeds this one in file */
			first = tf_Epoch.fileaddr;
			last = ev.timestamp.fileaddr;
			goto binsearch;
		}
		else {
			/*
			 * Desired event is after this position in the trace file.
			 * Since we should be near EOF, do a linear search from
			 * here.
			 */
			goto linsearch;
		}
	}
	else {
		/*
		 * Event desired is somewhere in the portion of the trace file
		 * that we've seen before.
		 */
		first = tf_Epoch.fileaddr;
		last = tf_LastEventSeen.fileaddr;
		goto binsearch;
	}

/*
 * come here when we know that the event is in the trace file, and that it
 * is somewhere after the one that starts at "first" (it could be exactly
 * there) and *before* the one that starts at "last".
 *
 * "first" and "last" are seek addresses in the file, but must point to
 * the first byte in a record (immediately after a newline).
 */

 binsearch:
	while (1) {
		long try;
		int comp;

		if (first > last)		/* shouldn't happen */
			return EOF;
		/*
		 * if the distance to be searched isn't too large, just
		 * do a linear search from the beginning of that area.
		 *
		 * XXX The gap width of 2 * BUFSIZ is just a guess.
		 * it needs to be much larger than the size of any single
		 * event.
		 */
		if (last - first < 2 * BUFSIZ) {
			fseek (traceFilePointer, first, SEEK_SET);
			goto linsearch;
		}

		/*
		 * otherwise, seek about halfway between the first and last
		 * events, and scan forward until the start of the next record.
		 */
		try = (first + last) / 2;
		fseek (traceFilePointer, try, SEEK_SET);
		while ((c = getc (traceFilePointer)) != EOF && c != '\n');

		if (tf_ReadEvent (&ev) < 0) {
			/* shouldn't happen */
			return EOF;
		}
		if ((comp = tf_TimeCompare (t, &ev.timestamp)) == 0) {
			/* found it */
			fseek (traceFilePointer, ev.timestamp.fileaddr, SEEK_SET);
			return 0;
		}
		else if (comp < 0) {
			/* somewhere before this one */
			last = ev.timestamp.fileaddr;
		}
		else if (comp > 0) {
			/* somewhere after this one */
			first = ftell (traceFilePointer);
		}
	}

/*
 * Come here if we think that the file pointer is before, but nearby
 * the record we want.  We read sequentially from here until we either
 * find a record equal to or later than the one we want, or we hit
 * end of file, which means there's no record later than the one we
 * want.
 *
 * file pointer must be on a record boundary, prior to the end of the file.
 */
	
 linsearch:
	while (1) {
		int comp;

		if (tf_ReadEvent (&ev) < 0)
			return EOF;
		if ((comp = tf_TimeCompare (t, &ev.timestamp)) >= 0) {
			fseek (traceFilePointer, ev.timestamp.fileaddr, SEEK_SET);
			return 0;
		}
	}
}

int
tf_Rewind ()
{
	return tf_Seek (&tf_Epoch);
}

/*
 * Local variables:
 * tab-width:4
 * End:
 */
