/*
 * subprocess module for htool
 * Keith Moore
 */

#if defined(IMA_RIOS) && defined(_BSD)
#undef _BSD						/* fix for IBM brain-damage */
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <signal.h>
#include <X11/Intrinsic.h>
#ifdef IMA_RIOS
#include <sys/select.h>
#endif
#include "xcomn.h"
#include "comn.h"
#include "rb.h"
#include "subproc.h"

/*
 * magic stuff for machines that don't define WTERMSIG or WEXITSTATUS
 * in <sys/wait.h>.  So far, only the Sequent has needed these, but
 * others might.
 */

#ifndef WTERMSIG
#ifdef w_termsig				/* has union wait */
#define WTERMSIG(status) (((union wait *) &(status))->w_termsig)
#else							/* V7 style */
#define WTERMSIG(status) ((status) & 0x7f)
#endif
#endif

#ifndef WEXITSTATUS
#ifdef w_retcode				/* has union wait */
#define WEXITSTATUS(status) (((union wait *) &(status))->w_retcode)
#else							/* V7 style */
#define WEXITSTATUS(status) (((status) >> 8) & 0xff)
#endif
#endif

#ifdef IMA_RIOS
/*
 * fix for RIOS signal() emulation brain-damage
 * might be useful on other machines also
 */

#define signal broken_signal

void (*signal (sig, func))
int sig;
void (*func)();
{
	struct sigaction old, new;
	sigset_t newsigmask;

	sigfillset (&newsigmask);	/* mask out all other signals when
								 * servicing this signal */
	new.sa_handler = func;
	new.sa_mask = newsigmask;
	new.sa_flags = SA_RESTART;	/* restart system calls after return */
	sigaction (sig, &new, &old);
	return old.sa_handler;
}
#endif

static Tree subprocTree;

/*
 * per-subprocess information, stored in subprocTree.
 */

struct subproc {
    int pid;					/* child's pid */
    void (*whenExited)();		/* called when child exits */
    void (*whenKilled)();		/* called when child dies of a signal */
	void (*whenData)();			/* (popen) called when child produces output */
	int childOutputFile;		/* pipe fd where we read child's data */
	XtInputId xInputId;			/* returned from XtAppAddInput */
	void *clientData;			/* data to be passed to client callbacks */
};


/*
 * A pipe is used to send process completion events to clients via the
 * X toolkit, which has registered a callback function for the other
 * end of the pipe via XtAppAddInput ();  This is ugly but relatively
 * safe and simple.
 */

int subprocPipe[2];

/*
 * private information, passed to ourselves (from a signal handler) over the
 * pipe.
 */

struct subproc_event {
	int pid;
	enum causeOfDeath howDied;	/* 1 = killed, 0 = exited */
	int exitArg;				/* status if exited, signo if killed */
};

/*
 * utility function used to compare two child pids for tree lookup
 */

static int
subproc_Compare (a, b)
Key *a, *b;
{
    return a->ikey - b->ikey;
}

/*
 * send a process completion event.  Called from within a SIGCHLD
 * signal handler
 */

static void
subproc_SendEvent (pid, howDied, arg)
int pid;
enum causeOfDeath howDied;
int arg;
{
	struct subproc_event event;

	event.pid = pid;
	event.howDied = howDied;
	event.exitArg = arg;
	write (subprocPipe[1], (char *) &event, sizeof event);
}

/*
 * receive a process completion event.  Called from the X toolkit
 * when its select() loop realizes that there's data available on
 * subprocPipe.
 */

static void
subproc_EventHandler ()
{
	struct subproc_event event;
    Key key;
    TreeNode tn;
    int found;
    struct subproc *ptr;

	if (read (subprocPipe[0], (char *) &event, sizeof event) != sizeof event)
		return;

	/*
	 * find child_pid in subprocess list.  If it has a callback,
	 * call it and pass it the child's exit status/signal #
	 */

	key.ikey = event.pid;
	tn = rb_Find (subprocTree, &key, subproc_Compare, &found);
	if (!found)	{
		/* child not registered with subproc pkg */
		return;
	}
	if ((ptr = (struct subproc *) rb_Value (tn)) == NULL) { /* paranoia */
		rb_DeleteNode (tn);
		return;
	}

#if 1
	/*
	 * if client is getting output from this process, there may still
	 * be some in the pipe.  So give the client a chance to grab it
	 * before we remove the callback.  If there's an EOF or data on
	 * the child's  output fd, select() *should* return immediately;
	 * else, we will wait up to two seconds.
	 *
	 * it's up to the client data sink to read everything the child
	 * has to offer...he only gets one last chance.
	 */
	if (ptr->childOutputFile >= 0) {
		fd_set readfds, exceptfds;
		struct timeval timeout;
		
		timeout.tv_sec = 2;		/* wait up to two seconds */
		timeout.tv_usec = 0;
		FD_ZERO (&readfds);
		FD_ZERO (&exceptfds);
		FD_SET (ptr->childOutputFile, &readfds);
		FD_SET (ptr->childOutputFile, &exceptfds);

		if (select (ptr->childOutputFile + 1, &readfds, 0, &exceptfds,
					&timeout) == 1) {
			(*(ptr->whenData))(ptr->childOutputFile, ptr->clientData);
		}
	}
#endif

	if (event.howDied == SUBPROC_EXITED)
		if (ptr->whenExited)
			(*ptr->whenExited)(event.pid, event.exitArg,
							   ptr->clientData, event.howDied);
	else if (event.howDied == SUBPROC_KILLED)
		if (ptr->whenKilled)
			(*ptr->whenKilled)(event.pid, event.exitArg,
							   ptr->clientData, event.howDied);

	if (ptr->childOutputFile >= 0) {
		close (ptr->childOutputFile);
		XtRemoveInput (ptr->xInputId);
	}

	free (ptr);
	rb_DeleteNode (tn);
	return;
}

/*
 * this function gets called whenever a SIGCHLD happens.
 */

static void
subproc_SigHandler (signo)
int signo;
{
#if defined(IMA_SYMM) || defined(IMA_PMAX) || defined(IMA_UVAX) || \
	defined(IMA_RT) || defined(IMA_NEXT)
    union wait status;
#else
    int status;
#endif
    int child_pid;
	
    if (signo != SIGCHLD)	/* wrong signal! */
		return;
	if (subprocTree == NULL)	/* should not happen */
		return;
	
#ifdef HAVE_WAITPID
	while ((child_pid = waitpid (-1, &status, WNOHANG)) > 0) {
#else
    while ((child_pid = wait3 (&status, WNOHANG, NULL)) > 0) {
#endif
		/*
		 * write the exit data to a pipe, so we can tickle the
		 * X toolkit's i/o wait loop and handle these events
		 * at "normal" level, instead of within a signal handler.
		 */
		if (WIFSIGNALED (status)) {
			/* XXX might want to filter out some signals */
			subproc_SendEvent (child_pid, SUBPROC_KILLED, WTERMSIG(status));
		}
		else if (WIFEXITED (status)) {
			subproc_SendEvent (child_pid, SUBPROC_EXITED, WEXITSTATUS(status));
		}
	}

#ifdef SYSVSIGNAL
	signal (SIGCHLD, subproc_SigHandler);
#endif
}

/*
 * function called to make sure our local data structures are initialized
 */

static void
subproc_Init ()
{
    if (subprocTree != NULL)
		return;
	
    subprocTree = rb_MakeTree ();
    (void) signal (SIGCHLD, subproc_SigHandler);
	pipe (subprocPipe);
	XtAppAddInput (context, subprocPipe[0], XtInputReadMask,
				   subproc_EventHandler, 0); 
}

/*
 * Helper function called to start a managed subprocess
 *
 */

static struct subproc *
subproc_RunInChild (argv, fd0, fd1, fd2)
char **argv;
int fd0, fd1, fd2;
{
	int child_pid;
	struct subproc *x;

	subproc_Init ();
	x = talloc (1, struct subproc);

	if ((child_pid = fork ()) == 0) {
		/* child */
		int i;
		int maxdesc;

		if (fd0 >= 0)
			dup2 (fd0, 0);
		if (fd1 >= 0)
			dup2 (fd1, 1);
		if (fd2 >= 0)
			dup2 (fd2, 2);

		maxdesc = getdtablesize ();
		for (i = 3; i < maxdesc; ++i)
			(void) close (i);

		execvp (argv[0], argv);
		_exit (127);
	}
	else if (child_pid < 0) {
		free (x);
		return (struct subproc *) NULL;
	}
	else {
		/* parent */
		TreeNode tn;
		int found;
		Key key;

		/*
		 * Note: it is possible for the child to exit before we get
		 * this stuff set up.  However, this shouldn't cause a problem
		 * because the SIGCHLD handler just stuffs a message in a pipe
		 * saying that the child exited, which won't get read until
		 * we return to the Xt select() loop.
		 */

		x->pid = child_pid;
		x->whenExited = (void (*)()) NULL;
		x->whenKilled = (void (*)()) NULL;
		x->whenData = (void (*)()) NULL;
		x->childOutputFile = -1;
		x->xInputId = NULL;
		x->clientData = (void *) NULL;

		key.ikey = child_pid;
		tn = rb_Find (subprocTree, &key, subproc_Compare, &found);
		rb_InsertBefore (tn, &key, x);
		return x;
	}
}

int
subproc_Run (argv, whenExited, whenKilled, clientData)
char **argv;
void (*whenExited)();
void (*whenKilled)();
void *clientData;
{
	struct subproc *x;
	int fd;
	
	fd = open ("/dev/null", 0);
	x = subproc_RunInChild (argv, fd, 1, 2);
	close (fd);
	if (x) {
		x->whenExited = whenExited;
		x->whenKilled = whenKilled;
		x->clientData = clientData;
		return x->pid;
	}
	return -1;
}

/*
 * run a shell command in a managed subprocess
 */

int
subproc_Spawn (command, whenExited, whenKilled, clientData)
char *command;
void (*whenExited)();
void (*whenKilled)();
void *clientData;
{
	struct subproc *x;
	int fd;
	char *argv[4];
	
	argv[0] = "/bin/sh";
	argv[1] = "-c";
	argv[2] = command;
	argv[3] = NULL;

	fd = open ("/dev/null", 0);
	x = subproc_RunInChild (argv, fd, 1, 2);
	close (fd);
	if (x) {
		x->whenExited = whenExited;
		x->whenKilled = whenKilled;
		x->clientData = clientData;
		return x->pid;
	}
	return -1;
}

#if 0
int
subproc_Spawn (command, whenExited, whenKilled, clientData)
char *command;
void (*whenExited)();
void (*whenKilled)();
void *clientData;
{
    int child_pid;
	
    subproc_Init ();

    if ((child_pid = fork ()) == 0) {
		/* child */
		int i;
		int maxdesc;
		
		maxdesc = getdtablesize();
		for (i = 3; i < maxdesc; ++i)
			(void) close (i);
		
		(void) execl ("/bin/sh", "sh", "-c", command, NULL);
		_exit (127);
    }
    else if (child_pid < 0) {
		/* fork() failed */
		return -1;
    }
    else {
		/* parent */
		struct subproc *x;
		TreeNode tn;
		int found;
		Key key;
		
		x = talloc (1, struct subproc);
		x->pid = child_pid;
		x->whenExited = whenExited;
		x->whenKilled = whenKilled;
		x->whenData = (void (*)()) NULL;
		x->clientData = clientData;
		x->childOutputFile = -1;
		key.ikey = child_pid;
		
		tn = rb_Find (subprocTree, &key, subproc_Compare, &found);
		rb_InsertBefore (tn, &key, x);
		return child_pid;
    }
}
#endif


/*
 * return the number of managed subprocesses.
 */
int
subproc_Count ()
{
	int count = 0;
	TreeNode tn;

	if (subprocTree == (Tree) NULL)
		return 0;

	for (tn = rb_First (subprocTree); tn != subprocTree; tn = rb_Next (tn))
		count++;
	return count;
}

/*****************************************************************************
 *                                                                           *
 * subproc_System() and friends                                              *
 *                                                                           *
 * These routines implement a single-subprocess facility similar to system() *
 * that allows X events to be dealt with while the subprocess is running.    *
 *                                                                           *
 *****************************************************************************/

/*
 * this callback is used by subproc_System() so it will know when
 * its child completes
 */

static int subprocChildDone = 0;
static int subprocChildStatus = 0;
static int subprocChildPid = 0;

static void
subproc_SystemDone (pid, status, clientData, howDied)
int pid, status;
void *clientData;
enum causeOfDeath howDied;
{
	if (pid == subprocChildPid) {
		subprocChildDone = 1;
		if (howDied == SUBPROC_EXITED)
			subprocChildStatus = status;
		else
			subprocChildStatus = -status;
		subprocChildPid = 0;
	}
}

/*
 * a version of system() that handles X events while the subprogram
 * is being executed
 */

int
subproc_System (command)
char *command;
{
	XEvent event;

	if (subprocChildPid != 0) {
		msg_Format ("can't have more than one subprocess\n");
		return -1;
	}
	subprocChildDone = 0;
	if ((subprocChildPid =
		 subproc_Spawn (command, subproc_SystemDone,
						subproc_SystemDone, NULL)) < 0)
		return -1;
	do {
		XtAppNextEvent (context, &event);
		XtDispatchEvent (&event);
	} while (!subprocChildDone);
	return subprocChildStatus;
}

/*****************************************************************************
 *                                                                           *
 * subproc_Popen () and friends                                              *
 *                                                                           *
 * this is a substitute for popen().  Basically it builds a pipe, starts up  *
 * a managed subprocess, and establishes a callback in the parent for        *
 * whenever data appears on the child's output pipe.   The caller supplies   *
 * a callback that gets called when the data appears, passing the child's    *
 * output file descriptor and the client data.                               *
 *                                                                           *
 *****************************************************************************/

/*
 * Callback from Xt that happens whenever we get input available on
 * a particular child's output fd.  What we do is call our client's
 * callback instead.
 */

static void
subproc_ChildData (ptr, source_fd, inputid)
XtPointer ptr;
int *source_fd;
XtInputId inputid;
{
	struct subproc *x = (struct subproc *) ptr;
	if (x->whenData)
		(*(x->whenData))(*source_fd, x->clientData);
}


/*
 * run a subprocess with its output connected to a pipe.  Set things
 * up so that we will get callbacks whenever the child exits or is
 * killed, or whenever it writes data to its output file.
 *
 * returns the pid of the child, or -1 on error.
 */

int
subproc_Popen (command, whenExited, whenKilled, whenData, clientData)
char *command;
void (*whenExited)();
void (*whenKilled)();
void (*whenData)();
void *clientData;
{
	struct subproc *x;
	int child_pipe [2];
	char *argv[4];
	int fd;

	argv[0] = "/bin/sh";
	argv[1] = "-c";
	argv[2] = command;
	argv[3] = NULL;

	fd = open ("/dev/null", 0);
	pipe (child_pipe);
	x = subproc_RunInChild (argv, fd, child_pipe[1], child_pipe[1]);
	close (child_pipe[1]);
	if (x) {
		x->whenExited = whenExited;
		x->whenKilled = whenKilled;
		x->whenData = whenData;
		x->clientData = clientData;
		x->childOutputFile = child_pipe[0];
		x->xInputId = XtAppAddInput (context, child_pipe[0],
									 XtInputReadMask, subproc_ChildData,
									 (XtPointer) x);
		return x->pid;
	}
	return -1;
}

#if 0
int
subproc_Popen (command, whenExited, whenKilled, whenData, clientData)
char *command;
void (*whenExited)();
void (*whenKilled)();
void (*whenData)();
void *clientData;
{
    int child_pid;
	struct subproc *x;
	int child_pipe[2];

    subproc_Init ();
	x = talloc (1, struct subproc);
	pipe (child_pipe);

    if ((child_pid = fork ()) == 0) {
		/* child */
		int i;
		int maxdesc;
		
		dup2 (child_pipe[1], 1);		/* stdout goes to pipe */
		dup2 (child_pipe[1], 2);		/* stderr goes to pipe also */
		/* close (child_pipe[0]); */	/* redundant close */
		/* close (child_pipe[1]); */	/* redundant close */

		maxdesc = getdtablesize();
		for (i = 3; i < maxdesc; ++i)
			(void) close (i);
		
		(void) execl ("/bin/sh", "sh", "-c", command, NULL);
		_exit (127);
    }
    else if (child_pid < 0) {
		/* fork() failed.  cleanup and return an error */
		close (child_pipe[0]);
		close (child_pipe[1]);
		free (x);
		return -1;
    }
    else {
		/* parent */
		TreeNode tn;
		int found;
		Key key;
		
		close (child_pipe[1]);
		x->pid = child_pid;
		x->whenExited = whenExited;
		x->whenKilled = whenKilled;
		x->whenData = whenData;
		x->clientData = clientData;
		x->childOutputFile = child_pipe[0];
		x->xInputId = XtAppAddInput (context, child_pipe[0],
									 XtInputReadMask, subproc_ChildData,
									 (XtPointer) x);
		key.ikey = child_pid;
		
		tn = rb_Find (subprocTree, &key, subproc_Compare, &found);
		rb_InsertBefore (tn, &key, x);
		return child_pid;
    }
}
#endif

/*
 * close up the pipe associated with a child process, and tell Xt
 * to stop looking for data on that pipe.  Does not actually kill
 * the child process, though it will die with signal EPIPE if
 * it tries to write more data to the pipe.
 */

int
subproc_Pclose (pid)
int pid;
{
	TreeNode tn;
	int found = 0;
	struct subproc *x;
	Key key;

	key.ikey = pid;
	tn = rb_Find (subprocTree, &key, subproc_Compare, &found);
	if (!found)
		return -1;
	x = (struct subproc *) rb_Value (tn);
	if (x->childOutputFile > 0) {
		XtRemoveInput (x->xInputId);
		close (x->childOutputFile);
		x->childOutputFile = -1;
		return 0;
	}
	return -1;
}

/*
 * Local variables:
 * tab-width:4
 * End:
 */
