/*
 * red/black tree lib
 *
 * 19 Sep 1991
 */

#include <stdio.h>
#include "rb.h"
#include "rbP.h"

#define	isred(n)	(n->s.red)
#define	isblack(n)	(!isred(n))
#define	isleft(n)	(n->s.left)
#define	isright(n)	(!isleft(n))
#define	isint(n)	(n->s.internal)
#define	isext(n)	(!isint(n))
#define	ishead(n)	(n->s.head)
#define	isroot(n)	(n->s.root)
#define	setred(n)	(n->s.red = 1)
#define	setblack(n)	(n->s.red = 0)
#define	setleft(n)	(n->s.left = 1)
#define	setright(n)	(n->s.left = 0)
#define	sethead(n)	(n->s.head = 1)
#define	setroot(n)	(n->s.root = 1)
#define	setint(n)	(n->s.internal = 1)
#define	setext(n)	(n->s.internal = 0)
#define	setnormal(n)	{ n->s.root = 0; n ->s.head = 0; }
#define	sibling(n)	((isleft(n)) ? n->p.parent->c.child.right \
						: n->p.parent->c.child.left)

#define mk_new_ext(new, kkkey, vvval) { \
  new = (TreeNode) malloc(sizeof(struct rb_node)); \
  new->v.val = vvval; \
  bcopy(kkkey, &(new->k), sizeof(Key)); \
  setext(new); \
  setblack(new); \
  setnormal(new); \
}

/***********************
*  internal functions  *
*                      *
***********************/

static void
insert (item, list)
TreeNode item, list;
{
	TreeNode last_node;

	last_node = list->c.list.blink;

	list->c.list.blink = item;
	last_node->c.list.flink = item;
	item->c.list.blink = last_node;
	item->c.list.flink = list;
}

static void
single_rotate(y, l)
TreeNode y;
int l;
{
	int rl, ir;
	TreeNode x, yp;

	ir = isroot(y);
	yp = y->p.parent;
	if (!ir) {
		rl = isleft(y);
	}

	if (l) {
		x = y->c.child.left;
		y->c.child.left = x->c.child.right;
		setleft(y->c.child.left);
		y->c.child.left->p.parent = y;
		x->c.child.right = y;
		setright(y);
	} else {
		x = y->c.child.right;
		y->c.child.right = x->c.child.left;
		setright(y->c.child.right);
		y->c.child.right->p.parent = y;
		x->c.child.left = y;
		setleft(y);
	}

	x->p.parent = yp;
	y->p.parent = x;
	if (ir) {
		yp->p.root = x;
		setnormal(y);
		setroot(x);
	} else {
		if (rl) {
			yp->c.child.left = x;
			setleft(x);
		} else {
			yp->c.child.right = x;
			setright(x);
		}
	}
}

static
recolor(n)
TreeNode n;
{
	TreeNode p, gp, s;
	int done = 0;

	while(!done) {
		if (isroot(n)) {
			setblack(n);
			return;
		}

		p = n->p.parent;

		if (isblack(p)) return;

		if (isroot(p)) {
			setblack(p);
			return;
		}

		gp = p->p.parent;
		s = sibling(p);
		if (isred(s)) {
			setblack(p);
			setred(gp);
			setblack(s);
			n = gp;
		} else {
			done = 1;
		}
	}
	/* p's sibling is black, p is red, gp is black */

	if ((isleft(n) == 0) == (isleft(p) == 0)) {
		single_rotate(gp, isleft(n));
		setblack(p);
		setred(gp);
	} else {
		single_rotate(p, isleft(n));
		single_rotate(gp, isleft(n));
		setblack(n);
		setred(gp);
	}
}

static
mk_new_int(l, r, p, il)
TreeNode l, r, p;
int il;
{
	TreeNode new;

	new = (TreeNode) malloc(sizeof(struct rb_node));
	setint(new);
	setred(new);
	setnormal(new);
	new->c.child.left = l;
	new->c.child.right = r;
	new->p.parent = p;
	new->lext = l;
	new->v.rext = r;
	l->p.parent = new;
	r->p.parent = new;
	setleft(l);
	setright(r);
	if (ishead(p)) {
		p->p.root = new;
		setroot(new);
	} else if (il) {
		setleft(new);
		p->c.child.left = new;
	} else {
		setright(new);
		p->c.child.right = new;
	}
	recolor(new);
}

static TreeNode
lprev(n)
TreeNode n;
{
	if (ishead(n))
		return n;
	while (!isroot(n) && isleft(n))
		n = n->p.parent;
	return n->p.parent;
}

static TreeNode
rprev(n)
TreeNode n;
{
	if (ishead(n))
		return n;
	while (!isroot(n) && isright(n))
		n = n->p.parent;
	return n->p.parent;
}


/***********************
*  exported functions  *
*                      *
***********************/

Tree
rb_MakeTree()
{
	Tree head;

	head = (Tree)malloc(sizeof(struct rb_node));
	head->c.list.flink = head;
	head->c.list.blink = head;
	head->p.root = head;
	head->k.pkey = 0;
	sethead(head);
	return head;
}

void
rb_FreeTree(n)
Tree n;
{
	if (!ishead(n)) {
		fprintf(stderr, "ERROR: rb_FreeTree called on a non-head node\n");
		exit(1);
	}
	while (rb_first(n) != n) {
		rb_DeleteNode(rb_first(n));
	}
	free(n);
}

TreeNode
rb_Find(n, sk, fxn, fnd)
Tree n;
Key *sk;
int (*fxn)();
int *fnd;
{
	int cmp;

	if (fnd)
		*fnd = 0;
	if (!ishead(n)) {
		fprintf(stderr, "rb_Find called on non-head 0x%x\n", n);
		exit(1);
	}
	if (n->p.root == n)
		return n;
	cmp = (*fxn)(sk, &(n->c.list.blink->k));
	if (cmp == 0) {
		if (fnd)
			*fnd = 1;
		return n->c.list.blink;
	}
	if (cmp > 0)
		return n;
	n = n->p.root;
	while (1) {
		if (isext(n))
			return n;
		cmp = (*fxn)(sk, &(n->lext->k));
		if (!cmp) {
			if (fnd)
				*fnd = 1;
			return n->lext;
		}
		n = (cmp < 0) ? n->c.child.left : n->c.child.right;
	}
}

void
rb_DeleteNode(n)
TreeNode n;
{
	TreeNode s, p, gp;
	char ir;

	if (isint(n)) {
		fprintf(stderr, "Cannot delete an internal node: 0x%x\n", n);
		exit(1);
	}
	if (ishead(n)) {
		fprintf(stderr, "Cannot delete the head of an rb_tree: 0x%x\n", n);
		exit(1);
	}

	/* Delete it from the leaves */
	n->c.list.flink->c.list.blink = n->c.list.blink;
	n->c.list.blink->c.list.flink = n->c.list.flink;

	p = n->p.parent;	/* The only node */
	if (isroot(n)) {
		p->p.root = p;
		free(n);
		return;
	}
	s = sibling(n);		/* The only node after deletion */
	if (isroot(p)) {
		s->p.parent = p->p.parent;
		s->p.parent->p.root = s;
		setroot(s);
		free(p);
		free(n);
		return;
	}
	gp = p->p.parent;	/* Set parent to sibling */
	s->p.parent = gp;
	if (isleft(p)) {
		gp->c.child.left = s;
		setleft(s);
	} else {
		gp->c.child.right = s;
		setright(s);
	}
	ir = isred(p);
	free(p);
	free(n);

	if (isext(s)) {			/* Update proper rext and lext values */
		p = lprev(s);
		if (!ishead(p)) p->v.rext = s;
		p = rprev(s);
		if (!ishead(p)) p->lext = s;
	} else if (isblack(s)) {
		fprintf(stderr, "DELETION PROB -- sib is black, internal\n");
		exit(1);
	} else {
		p = lprev(s);
		if (!ishead(p)) p->v.rext = s->c.child.left;
		p = rprev(s);
		if (!ishead(p)) p->lext = s->c.child.right;
		setblack(s);
		return;
	}

	if (ir) return;

	/* Recolor */

	n = s;
	p = n->p.parent;
	s = sibling(n);
	while(isblack(p) && isblack(s) && isint(s) &&
				isblack(s->c.child.left) && isblack(s->c.child.right)) {
		setred(s);
		n = p;
		if (isroot(n)) return;
		p = n->p.parent;
		s = sibling(n);
	}

	if (isblack(p) && isred(s)) {		/* Rotation 2.3b */
		single_rotate(p, isright(n));
		setred(p);
		setblack(s);
		s = sibling(n);
	}

	{
		TreeNode x, z; char il;

		if (isext(s)) {
			fprintf(stderr, "DELETION ERROR: sibling not internal\n");
			exit(1);
		}

		il = isleft(n);
		x = il ? s->c.child.left : s->c.child.right ;
		z = sibling(x);

		if (isred(z)) {					/* Rotation 2.3f */
			single_rotate(p, !il);
			setblack(z);
			if (isred(p)) setred(s); else setblack(s);
			setblack(p);
		} else if (isblack(x)) {		/* Recoloring only (2.3c) */
			if (isred(s) || isblack(p)) {
				fprintf(stderr, "DELETION ERROR: 2.3c not quite right\n");
				exit(1);
			}
			setblack(p);
			setred(s);
			return;
		} else if (isred(p)) {			/* 2.3d */
			single_rotate(s, il);
			single_rotate(p, !il);
			setblack(x);
			setred(s);
			return;
		} else {						/* 2.3e */
			single_rotate(s, il);
			single_rotate(p, !il);
			setblack(x);
			return;
		}
	}
}

TreeNode
rb_InsertBefore(n, key, val)
TreeNode n;
Key *key;
void *val;
{
	TreeNode newleft, newright, newnode, list, p;

	if (ishead(n)) {
		if (n->p.root == n) {		/* Tree is empty */
			mk_new_ext(newnode, key, val);
			insert(newnode, n);
			n->p.root = newnode;
			newnode->p.parent = n;
			setroot(newnode);
			return newnode;

		} else {
			mk_new_ext(newright, key, val);
			insert(newright, n);
			newleft = newright->c.list.blink;
			setnormal(newleft);
			mk_new_int(newleft, newright, newleft->p.parent, isleft(newleft));
			p = rprev(newright);
			if (!ishead(p)) p->lext = newright;
			return newright;
		}

	} else {
		mk_new_ext(newleft, key, val);
		insert(newleft, n);
		setnormal(n);
		mk_new_int(newleft, n, n->p.parent, isleft(n));
		p = lprev(newleft);
		if (!ishead(p)) p->v.rext = newleft;
		return newleft;
	}
}

TreeNode
rb_InsertAfter(	n, key, val)
TreeNode n;
Key *key;
void *val;
{
	return rb_InsertBefore(n->c.list.flink, key, val);
}

void
rb_Insert(t, key, v, cfn)
Tree t;
Key *key;
void *v;
int (*cfn)();
{
	rb_InsertBefore(rb_Find(t, key, cfn, (int*)0), key, v);
}

int
rb_NBlack(n)
TreeNode n;
{
	int nb = 0;

	if (ishead(n) || isint(n)) {
		fprintf(stderr,
			"ERROR: rb_NBlack called on a non-external node 0x%x\n", n);
		exit(1);
	}
	while(!ishead(n)) {
		if (isblack(n))
			nb++;
		n = n->p.parent;
	}
	return nb;
}

int
rb_PathLen(n)
TreeNode n;
{
	int pl = 0;

	if (ishead(n) || isint(n)) {
		fprintf(stderr,
			"ERROR: rb_PathLen called on a non-external node 0x%x\n", n);
		exit(1);
	}
	while(!ishead(n)) {
		pl++;
		n = n->p.parent;
	}
	return pl;
}

TreeNode
rb_First(n)
TreeNode n;
{
	if (n == NULL)
		return NULL;
	return n->c.list.flink;
}

TreeNode
rb_Last(n)
TreeNode n;
{
	if (n == NULL)
		return NULL;
	return n->c.list.blink;
}

TreeNode
rb_Next(n)
TreeNode n;
{
	if (n == NULL)
		return NULL;
	return n->c.list.flink;
}

TreeNode
rb_Prev(n)
TreeNode n;
{
	if (n == NULL)
		return NULL;
	return n->c.list.blink;
}

int
rb_Empty(t)
Tree t;
{
	if (t == NULL)
		return 1;
	return (t->c.list.flink == t) ? 1 : 0;
}

Key*
rb_Key(n)
TreeNode n;
{
	return &(n->k);
}

void*
rb_Value(n)
TreeNode n;
{
	return n->v.val;
}

void
rb_SetValue (tn, v)
TreeNode tn;
void *v;
{
	tn->v.val = v;
}

int
rb_Traverse (t, fn)
Tree t;
int (*fn)();
{
	TreeNode tn;
	int x;

	for (tn = rb_First (t); tn != t; tn = rb_Next (tn))
		if (x = (*fn)(tn))
			return x;
	return 0;
}

int
rb_TraverseX (t, fn, arg)
Tree t;
int (*fn)();
void *arg;
{
	TreeNode tn;
	int x;

	for (tn = rb_First (t); tn != t; tn = rb_Next (tn))
		if (x = (*fn)(tn, arg))
			return x;
	return 0;
}


/**************
*  purgatory  *
*             *
**************/

#if 0

rb_print_tree(t, level)
Rb_node t;
int level;
{
  int i;
  if (ishead(t) && t->p.parent == t) {
    printf("tree 0x%x is empty\n", t);
  } else if (ishead(t)) {
    printf("Head: 0x%x.  Root = 0x%x\n", t, t->p.root);
    rb_print_tree(t->p.root, 0);
  } else {
    if (isext(t)) {
      for (i = 0; i < level; i++) putchar(' ');
      printf("Ext node 0x%x: %c,%c: p=0x%x, k=%s\n",
              t, isred(t)?'R':'B', isleft(t)?'l':'r', t->p.parent, t->k.pkey);
    } else {
      rb_print_tree(t->c.child.left, level+2);
      rb_print_tree(t->c.child.right, level+2);
      for (i = 0; i < level; i++) putchar(' ');
      printf("Int node 0x%x: %c,%c: l=0x%x, r=0x%x, p=0x%x, lr=(%s,%s)\n",
              t, isred(t)?'R':'B', isleft(t)?'l':'r', t->c.child.left,
              t->c.child.right,
              t->p.parent, t->lext->k.pkey, t->v.rext->k.pkey);
    }
  }
}

rb_iprint_tree(t, level)
Rb_node t;
int level;
{
  int i;
  if (ishead(t) && t->p.parent == t) {
    printf("tree 0x%x is empty\n", t);
  } else if (ishead(t)) {
    printf("Head: 0x%x.  Root = 0x%x, < = 0x%x, > = 0x%x\n",
            t, t->p.root, t->c.list.blink, t->c.list.flink);
    rb_iprint_tree(t->p.root, 0);
  } else {
    if (isext(t)) {
      for (i = 0; i < level; i++) putchar(' ');
      printf("Ext node 0x%x: %c,%c: p=0x%x, <=0x%x, >=0x%x k=%d\n",
              t, isred(t)?'R':'B', isleft(t)?'l':'r', t->p.parent,
              t->c.list.blink, t->c.list.flink, t->k.ikey);
    } else {
      rb_iprint_tree(t->c.child.left, level+2);
      rb_iprint_tree(t->c.child.right, level+2);
      for (i = 0; i < level; i++) putchar(' ');
      printf("Int node 0x%x: %c,%c: l=0x%x, r=0x%x, p=0x%x, lr=(%d,%d)\n",
              t, isred(t)?'R':'B', isleft(t)?'l':'r', t->c.child.left,
              t->c.child.right,
              t->p.parent, t->lext->k.ikey, t->v.rext->k.ikey);
    }
  }
}

#endif
/*
 * Local Variables:
 * tab-width:4
 * comment-column:40
 * End:
 */
