/*
 * functions to
 * (a) parse the function definitions file
 * (b) check the functions in the HeNCE graph against the declarations
 *     in that file.
 * (c) write out a func.defs file to be fed to mkwrap.
 *
 * XXX eventually, fold the code from mkwrap into this module, and build
 * a version of mkwrap that shares code with htool for standalone (non-X)
 * usage of HeNCE.
 */

%{
#include <stdio.h>
#include <errno.h>
#include "rb.h"
#include "param.h"
#include "exp.h"
#include "expP.h"
#include "graph.h"
#include "graphP.h"
#include "misc.h"

extern int errno;		/* 4.3bsd needs this */

/****************************************************************************
 * routines to manipulate the function symbol table.
 * unlike parameters, there can be only one function with a particular
 * name in a HeNCE program.
 ****************************************************************************/

static Tree funcSymTable = 0;

struct funcSym {
    char *name;			/* name of function */
    int retType;		/* type of returned parameter */
    int isUsed;			/* true if used in graph */
    int nargs;			/* number of arguments */
    struct argType {
	int baseType;		/* base type of each argument */
	int indirs;		/* number of indirections (*'s) */
    } *argTypes;		/* type of each argument */
} currFuncSym;

struct argType currArgTypes[100];
int currNumArgs;
int errorCount;
char *currFileName;

#ifndef TYPE_VOID
#define TYPE_VOID NTYPES
#endif

%}

%start decl_list

%union {
    long integer;
    char *id;
}

%type <id> name
%type <integer> indirs stars type
%token TK_TYPE TK_ID
%%

decl_list:	decl
    	 |	decl_list decl
	 ;

decl	:	type indirs name { currNumArgs = 0; } '(' arglist ')' ';' {
	    	    int i;

		    if ($2 != 0) {
			msg_Format ("%s%s%d: return type must be scalar\n",
				    currFileName, ", line ", yylineno);
			++errorCount;
		    }
		    else {
			currFuncSym.retType = $1;
			currFuncSym.name = $3;
			if (currNumArgs == 0)
			    currFuncSym.argTypes = NULL;
			else
			    currFuncSym.argTypes = talloc (currNumArgs,
							   struct argType);
			currFuncSym.nargs = currNumArgs;
			for (i = 0; i < currNumArgs; ++i)
			    currFuncSym.argTypes[i] = currArgTypes[i];
			currFuncSym.isUsed = 0;
			addFuncSym ();
		    }
		}
	;

name	:	TK_ID	{
    		    char *strsave ();
    		    $$ = strsave (yytext);
		}
	;

arglist	:	/* empty */
    	|	args
	;

args	:	arg
	|	args ',' arg
	;

arg	:	type indirs	{
    		    currArgTypes[currNumArgs].baseType = $1;
		    currArgTypes[currNumArgs].indirs = $2;
		    currNumArgs++;
		}
	;

indirs	:	/* empty */	{
		    $$ = 0;
		}
	|	stars	{
    		    $$ = $1;
		}
	;

stars	:	'*'		{
		    $$ = 1;
		}
	|	stars '*'	{
		    $$ = $1 + 1;
		}
	;

type	:	TK_TYPE	{
		    if (strcmp ("int", yytext) == 0)
			$$ = TYPE_INT;
		    else if (strcmp ("float", yytext) == 0)
			$$ = TYPE_FLOAT;
		    else if (strcmp ("double", yytext) == 0)
			$$ = TYPE_DOUBLE;
		    else if (strcmp ("char", yytext) == 0)
			$$ =  TYPE_CHAR;
		    else if (strcmp ("void", yytext) == 0)
			$$ = TYPE_VOID;
		}
	;

%%
#include "funcdefs.lex.out"

static int
nameCompare (k1, k2)
Key *k1, *k2;
{
    return (strcmp ((char *) k1->pkey, (char *) k2->pkey));
}


static void
addFuncSym ()
{
    TreeNode tn;
    Key k;
    struct funcSym *p;
    int fnd = 0;

    k.pkey = currFuncSym.name;
    tn = rb_Find (funcSymTable, &k, nameCompare, &fnd);
    if (fnd) {
	msg_Format ("%s, line %d: function %s already declared\n",
		    currFileName, yylineno, currFuncSym.name);
	++errorCount;
	return;
    }
    p = talloc (1, struct funcSym);
    *p = currFuncSym;
    k.pkey = (void *) strsave (currFuncSym.name);
#ifdef DEBUG
    fprintf (stderr, "defining function %s\n", currFuncSym.name);
#endif
    rb_InsertBefore (tn, &k, (void *) p);
#ifdef DEBUG
    fprintf (stderr, "writing decls\n");
    subdefs_WriteDecls (stderr);
    fprintf (stderr, "finished writing decls\n");
#endif
}

static struct funcSym *
lookupFuncDecl (name)
char *name;
{
    TreeNode tn;
    Key k;
    int fnd = 0;

#ifdef DEBUG
    fprintf (stderr, "LookupFuncDecl (%s)", name);
#endif
    k.pkey = name;
    tn = rb_Find (funcSymTable, &k, nameCompare, &fnd);
    if (!fnd) {
#ifdef DEBUG
	fprintf (stderr, " => (not found)\n");
#endif
	return NULL;
    }
#ifdef DEBUG
    fprintf (stderr, " => (found)\n");
#endif
    return (struct funcSym *) rb_Value (tn);
}

static char *
typeNameX (baseType, indirs)
int baseType, indirs;
{
    static char buf[40];
    char *ptr;

    switch (baseType) {
    case TYPE_PARAM:
	ptr = "[param]";
	break;
    case TYPE_UNSPECIFIED:
	ptr = "[unspecified]";
	break;
    case TYPE_INHERITED:
	ptr = "[inherited]" ;
	break;
    case TYPE_INT:
	ptr = "int";
	break;
    case TYPE_CHAR:
	ptr = "char";
	break;
    case TYPE_FLOAT:
	ptr = "float";
	break;
    case TYPE_DOUBLE:
	ptr = "double";
	break;
    case TYPE_ARRAY:
	ptr = "array";
	break;
    case TYPE_STRING:
	ptr = "string";
	break;
    case TYPE_VOID:
	ptr = "void";
	break;
    default:
	sprintf (buf, "[type = %d, indirs = %d]", baseType, indirs);
	return buf;
	break;
    }
    
    sprintf (buf, "%s%s", ptr, indirs ? " *" : "");
    return buf;
}

static int
compareFunctions (n, f1, f2)
Node n;
struct funcSym *f1, *f2;
{
    int x, i;
    int nodenum = n->nk.id;
    int errcount = 0;

    if (f1 == NULL || f2 == NULL)
	return -1;
    if (f1 == f2)
	return 0;

    if (strcmp (f1->name, f2->name) != 0) {
	msg_Format ("node %d: function %s != declared function %s\n",
		    nodenum, f1->name, f2->name);
	return 1;
    }
    if (f1->retType != f2->retType) {
	if (f1->retType == TYPE_VOID) {
	    msg_Format ("node %d: WARNING: return value of function %s is not used\n",
			nodenum, f1->name);
	    ++errcount;
	}
	else if (f2->retType == TYPE_VOID) {
	    msg_Format ("node %d: function %s has no return value\n",
			nodenum, f2->name);
	    ++errcount;
	}
#ifdef TYPECHECKING
	else {
	    msg_Format ("node %d: type of result variable \"%s\" (%s) does not match the result type of function \"%s\"\n",
			nodenum,
			n->ret_param->name,
			typeNameX(param_Type (n->ret_param), 0),
			f1->name);
	    msg_Format ("[function %s was declared to return %s]\n",
			f2->name, typeNameX (f2->retType, 0));
	    return 1;
	}
#endif
    }
    if (f1->nargs != f2->nargs) {
	msg_Format ("node %d: function %s requires %d args\n",
		    nodenum, f1->name, f2->nargs);
	return 1;
    }
#ifdef TYPECHECKING
    for (i = 0; i < f1->nargs; ++i) {
	if (f1->argTypes[i].baseType != f2->argTypes[i].baseType ||
	    f1->argTypes[i].indirs != f2->argTypes[i].indirs) {
	    msg_Format ("node %d, arg #%d (type = %s) does not match declaration\n",
			nodenum, i + 1,
			typeNameX (f1->argTypes[i].baseType,
				   f1->argTypes[i].indirs));
	    msg_Format ("declared type is %s\n",
			typeNameX (f2->argTypes[i].baseType,
				   f2->argTypes[i].indirs));
	    return 1;
	}
    }
#endif
    return errcount;
}

/*
 * given a parameter p, return its type
 *
 * XXX this should be in param.c, but it can't be because param.c
 * doesn't know about parameter symbol tables.
 */

int
param_Type (p)
Param p;
{
    Param fixParam();

#ifdef DEBUG
    fprintf (stderr, "param_Type (%s)\n", p ? p->name : "NULL");
#endif

    if (p == NULL)
	return TYPE_INHERITED;
    else if (p->type == TYPE_INHERITED)
	return param_Type (fixParam (p));
    else if (p->type == TYPE_ARRAY) {
	if (p->a->type == TYPE_INHERITED) {
	    /* XXX need to make sure returned type is an array */
	    return param_Type (fixParam (p));
	}
	else
	    return p->a->type + 100; /* XXX FIX THIS! */
    }
    return (p->type);
}

/*
 * Return the type of the result that you would get if you evaluated
 * expr e.  This is used to do crude type-checking before generating
 * wrappers.
 *
 * XXX The checking is currently pretty lame.  In particular, since
 * we currently represent type as an integer, we punt all handling of
 * arrays -- all arrays are treated as if they were the same type
 * regardless of the base type of the array.  Neither do we check
 * for number of dimensions, etc.  The prefix '*' and '&' aren't
 * handled correctly, either, but these should be nuked from the whole
 * HeNCE language.
 *
 * XXX somewhere (perhaps in parse.y) we ought to check to make sure
 * that we aren't doing bogus operations (e.g. left shift on double,
 * arithmetic on strings)
 *
 * XXX this should be in exp.c
 */


int
expr_Type (e)
Expr e;
{
    int promote ();
    int x;

    switch (e->elt_type) {
    case OP_COND_EXPR:
	x = promote (e->args[1], e->args[2]);
	break;
    case OP_LSHIFT:
    case OP_RSHIFT:
    case OP_NEG:
	x = expr_Type (e->args[0]);
	break;
    case OP_LOGICAL_OR:
    case OP_LOGICAL_AND:
    case OP_LOGICAL_CMPL:
    case OP_COMP_NE:
    case OP_COMP_EQ:
    case OP_COMP_LE:
    case OP_COMP_GE:
    case OP_COMP_LT:
    case OP_COMP_GT:
	x = TYPE_INT;
	break;
    case OP_BIT_OR:
    case OP_BIT_AND:
    case OP_BIT_XOR:
    case OP_BIT_CMPL:
    case OP_MULT:
    case OP_DIV:
    case OP_MOD:
    case OP_ADD:
    case OP_SUBTRACT:
	x = promote (e->args[0], e->args[1]);
	break;
    case OP_CONST:
	x = e->type;
	break;
    case OP_PARAM:
	x = param_Type (e->val.p);
	break;
#if 0
    case OP_INDIR:
    case OP_DIM:
	if ((x = expr_Type (e->args[0])) > 100)
	    x -= 100;
	break;
    case OP_EMPTY_DIM:
    case OP_ARR_SECTION:
    case OP_ADDRESS:
	if ((x = expr_Type (e->args[0])) <= 100)
	    x += 100;
	break;
#else
    case OP_INDIR:
    case OP_DIM:
    case OP_EMPTY_DIM:
    case OP_ARR_SECTION:
    case OP_ADDRESS:
	return TYPE_UNSPECIFIED;
#endif
    default:
#ifdef DEBUG
	fprintf (stderr, "expr_Type (%x op = %c) => unknown type %d\n",
		 e, e->elt_type, x);
#endif
	return TYPE_UNSPECIFIED;
    }
#ifdef DEBUG
    fprintf (stderr, "expr_Type (%x op = %c) => %d\n", e, e->elt_type, x);
#endif
    return x;
}


int
promote (e1, e2)
Expr e1, e2;
{
    int t1, t2;

    t1 = expr_Type (e1);
    t2 = expr_Type (e2);
    if (t1 == t2)
	return t1;
    if (t1 > t2)
	return t1;
    return t2;
}

static struct funcSym *
buildFuncDecl (n)
Node n;
{
    struct funcSym *f;
    int i;

    f = talloc (1, struct funcSym);
    f->name = n->sub_name;
    f->nargs = n->nargs;
    f->argTypes = n->nargs > 0 ? talloc (n->nargs, struct argType) : NULL;
    if (n->ret_param) {
	if (n->ret_param->type == TYPE_INHERITED && n->ret_param->name)
	    f->retType = param_Type (n->ret_param);
	else
	    f->retType = n->ret_param->type;
    }
    else
	f->retType = TYPE_VOID;

#ifdef DEBUG
    fprintf (stderr, "buildFuncDecl (node %d)\n", n->nk.id);
#endif

    for (i = 0; i < n->nargs; ++i) {
	int x;

#ifdef DEBUG
	fprintf (stderr, "node %d: checking type of arg %d\n",
		 n->nk.id, i+1);
#endif
	x = expr_Type (n->args[i]);
	if (x <= 100) {		/* XXX fix this here and in param.c */
	    f->argTypes[i].indirs = 0;
	    f->argTypes[i].baseType = x;
	}
	else {
	    f->argTypes[i].indirs = 1;
	    f->argTypes[i].baseType = x - 100;
	}
#ifdef DEBUG
	fprintf (stderr, "node %d: function %s, arg %d, type %s\n",
		 n->nk.id,
		 f->name, i + 1, typeNameX (f->argTypes[i].baseType,
					    f->argTypes[i].indirs));
#endif
    }
    return f;
}

static void
nukeFuncDecls ()
{
    TreeNode tn, tn2;

    if (funcSymTable == NULL)
	return;

    for (tn = rb_First (funcSymTable); tn != funcSymTable; tn = tn2) {
	free (rb_Value (tn));
	tn2 = rb_Next (tn);
	rb_DeleteNode (tn);
    }
}

static int
yywrap ()
{
    return 1;
}

static int
subdefs_ReadDecls (filename)
char *filename;
{

    if (funcSymTable)
	nukeFuncDecls ();
    else
	funcSymTable = rb_MakeTree ();
    
    if ((yyin = fopen (filename, "r")) == NULL) {
	msg_Format ("can't open %s (%s)\n", filename, strerror (errno));
	return 1;
    }

    /*
     * this is a horrible hack to reset the lexical analyzer's state
     * back to its initial state.  This probably won't work with anything
     * except Lex Classic.  If this code ever has to run with anything
     * else we should probably rewrite the lexical analyzer in bare C.
     */

    yylineno = 1;
    yysptr = yysbuf;
    yytchar = 0;
    yyprevious = YYNEWLINE;
    BEGIN 0;

    currFileName = filename;
    errorCount = 0;
    yyparse ();
    fclose (yyin);
    return errorCount;
}

static char *
stars (n)
int n;
{
    static char buf[20];
    char *ptr = buf;

    if (n == 0)
	return "";
    else if (n > 18)
	return "???";

    *ptr++ = ' ';
    while (n > 0) {
	*ptr++ = '*';
	--n;
    }
    *ptr = '\0';
    return buf;
}


static char *
typeName (t)
int t;
{
    switch (t) {
    case TYPE_VOID: return "void";
    case TYPE_INT:  return "int";
    case TYPE_FLOAT: return "float";
    case TYPE_DOUBLE: return "double";
    case TYPE_CHAR: return "char";
    default: return "???";
    }
}

int
subdefs_WriteDecls (fp)
FILE *fp;
{
    TreeNode tn;
    int i;

    for (tn = rb_First (funcSymTable); tn != funcSymTable; tn = rb_Next (tn)) {
	struct funcSym *x = rb_Value (tn);

	if (x->isUsed == 0) {
	    if (fp == stderr)
		fprintf (stderr, "%s (%x) is not used\n", x->name, x);
	    continue;
	}

	fprintf (fp, "%s %s (", typeName (x->retType), x->name);
	for (i = 0; i < x->nargs; ++i) {
	    fprintf (fp, "%s%s%s",
		     typeName (x->argTypes[i].baseType),
		     stars(x->argTypes[i].indirs),
		     i == x->nargs - 1 ? "" : ", ");
	}
	fprintf (fp, ");\n");
    }
    return 0;
}

static int
yyerror ()
{
    extern int yylineno;

    char *foo = (yychar == TK_ID) ? "identifier" : "symbol";

    if (currFileName != (char *) NULL)
	msg_Format ("%s, line %d: unexpected %s \"%s\"\n",
		    currFileName, yylineno, foo, yytext);
    else
	msg_Format ("line %d: unexpected %s \"%s\"\n", yylineno, foo, yytext);
    ++errorCount;
}

/****************************************************************************
 * routines to manipulate parameter symbol table:
 *
 * since there can be more than one parameter with the same name, the
 * symbol table entry is actually a stack with the most recent binding
 * to that name on top.
 *
 * the symbol table is built from the top of the tree down through
 * each possible path to each node.
 ****************************************************************************/

static Tree paramSymTable;

struct paramSym {
    Param p;
    int nodeWhereDeclared;	/* if of node where param was declared */
    struct paramSym *next;
};

/*
 * declare a parameter
 *
 * XXX should this push a new stack entry if a parameter by that
 * name already exists?
 */

static void
declareParameter (p, n)
Param p;
Node n;
{
    Key k;
    int found;
    TreeNode tn;
    struct paramSym *ps;

#ifdef DEBUG
    fprintf (stderr, "declareParameter (%x \"%s\", %d)\n", p, p->name,
	     n->nk.id);
#endif

    ps = talloc (1, struct paramSym);
    ps->p = p;
    ps->nodeWhereDeclared = n->nk.id;
    ps->next = NULL;

    k.pkey = (void *) p->name;
    tn = rb_Find (paramSymTable, &k, nameCompare, &found);
    if (found) {
	ps->next = (struct paramSym *) rb_Value (tn);
	rb_SetValue (tn, ps);
    }
    else {
	rb_InsertBefore (tn, &k, (void *) ps);
    }
}

/*
 * remove a parameter's (most recent) declaration, if declared at
 * this node.
 */

static void
undeclareParameter (p, n)
Param p;
Node n;
{
    Key k;
    TreeNode tn;
    int found = 0;
    struct paramSym *ps;

#ifdef DEBUG
    fprintf (stderr, "undeclareParameter (%s, %d)\n", p->name, n->nk.id);
#endif

    k.pkey = (void *) p->name;
    tn = rb_Find (paramSymTable, &k, nameCompare, &found);
    if (found) {

	/*
	 * if there are multiple instances of this parameter,
	 * pop the most recently declared one.  Otherwise,
	 * delete the symbol table entry.
	 */
	ps = rb_Value (tn);

	if (ps->nodeWhereDeclared != n->nk.id)
	    return;

	if (ps->next) {
	    rb_SetValue (tn, ps->next);
	    ps->next = NULL;
	}
	else
	    rb_DeleteNode (tn);

	free (ps);
    }
}

/*
 * find the most recent declaration of a parameter
 */

static struct paramSym *
lookupParameter (name)
char *name;
{
    TreeNode tn;
    Key k;
    int found = 0;

#ifdef DEBUG
    fprintf (stderr, "lookupParameter (%s)", name);
#endif

    if (name == NULL)
	return NULL;
    k.pkey = name;
    tn = rb_Find (paramSymTable, &k, nameCompare, &found);
    if (found) {
#ifdef DEBUG
	fprintf (stderr, " [found]\n");
#endif
	return (struct paramSym *) rb_Value (tn);
    }
#ifdef DEBUG
    fprintf (stderr, " [not found]\n");
#endif
    return NULL;
}

static void
nukeParamDecls ()
{
    TreeNode tn, tn2;

    if (paramSymTable == NULL)
	return;

    for (tn = rb_First (paramSymTable); tn != paramSymTable; tn = tn2) {
	free (rb_Value (tn));
	tn2 = rb_Next (tn);
	rb_DeleteNode (tn);
    }
}

/*
 * find and return the original definition of parameter p
 */

Param
fixParam (p)
Param p;
{
    struct paramSym *ps;

#ifdef DEBUG
    fprintf (stderr, "fixParam (%s)\n", p->name);
#endif

    if ((ps = lookupParameter (p->name)) == NULL) {
	msg_Format ("parameter %s used in expr without declaration\n",
		    p->name);
	return NULL;
    }
#ifdef DEBUG
    fprintf (stderr, "original declaration of %s found at node %d\n",
	     p->name, ps->nodeWhereDeclared);
#endif
    return ps->p;
}

#if 0
static void
fixExpression (e)
Expr e;
{
    int i;

    if (e->elt_type == OP_PARAM) {
	e->val.p = fixParam (e->val.p);
	return;
    }
    for (i = 0; i < e->nargs; ++i)
	fixExpression (e->args[i]);
}
#endif


/*
 * traverse a HeNCE subgraph.  Make sure parameter and function declarations
 * are consistent, and collect definitions for functions.
 */

static int
scanSubGraph (n)
Node n;
{
    TreeNode tn;
    int errors = 0;
    struct funcSym *sc0, *sc1;
    struct paramSym *ps;
    int i;

    /*
     * special case for non-normal nodes, since we don't have to check
     * parameters or function names (is this true?  how about loop vars?)
     */

#ifdef DEBUG
    fprintf (stderr, "ScanSubGraph (node %d)\n", n->nk.id);
#endif

    switch (n->node_type) {

    case NODE_NORMAL:
	/* handle below */
	break;
    case NODE_PIPE:
    case NODE_FANOUT:
	/* declare control parameter as a variable */
	declareParameter (n->ret_param, n);
	for (tn = rb_First (n->children); tn != n->children;
	     tn = rb_Next (tn)){
	    errors += scanSubGraph ((Node) rb_Value (tn));
	}
	undeclareParameter (n->ret_param, n);
	return errors;
    default:
	if (n->nchildren == 0 || rb_Empty (n->children))
	    return 0;
	for (tn = rb_First (n->children); tn != n->children;
	     tn = rb_Next (tn)){
	    errors += scanSubGraph ((Node) rb_Value (tn));
	}
	return errors;
    }

    /*
     * 1.  for each named parameter:
     * if it's new, declare it; else, find its declaration
     */
    for (tn = rb_First (n->params); tn != n->params; tn = rb_Next (tn)) {
	struct paramSym *ps;
	Param p0, p1;

	p1 = rb_Value (tn);
	if (p1->io.new) {
	    /*
	     * new parameter.  ignore any previous declarations, and
	     * push this declaration on the stack
	     */
	    p1->io.used = 1;	/* XXX this is probably bogus */
	    p0 = p1;
#if 0
	    p1->nodeWhereDeclared = n;
#endif
	    declareParameter (p1, n);
	}
	else {
	    /*
	     * not declared with NEW keyword; find in symbol table
	     */
	    ps = lookupParameter (p1->name);
	    p0 = ps ? ps->p : (Param) NULL;

	    /*
	     * Not declared as new, and no previous declaration
	     */
	    if (p0 == (Param) NULL) {
		msg_Format ("Node %d: no NEW declaration for parameter %s.\n",
				n->nk.id, p1->name);
#ifdef DEBUG
		fprintf (stderr,
			 "Node %d: no NEW declaration for parameter %s.\n",
			 n->nk.id, p1->name);
#endif
		++errors;
		/* if a type was specified, pretend we had a declaration */
		if (p1->type != TYPE_INHERITED)
		    declareParameter (p1, n);
		else
		    continue;
	    }		
	    /*
	     * (simple) check for type mismatch
	     *
	     * XXX should check for equivalence of array types
	     */
	    else if (p1->type != TYPE_INHERITED && p1->type != p0->type) {
		msg_Format ("Node %d: type mismatch for parameter %s.\n",
			    n->nk.id, p1->name);
		msg_Format ("(originally declared @ node %d)\n",
			    ps->nodeWhereDeclared);	
		++errors;
		continue;
	    }
	    /*
	     * if we've been here before, check to make sure that there
	     * isn't a declaration through this path of ancestors that
	     * conflicts with another declaration through another path.
	     *
	     * XXX the use of p1->io.used is questionable
	     */
	    else if (p1->io.used && p1 != p0) {
		msg_Format ("Node %d: multiple declarations for %s %s\n",
			    n->nk.id, "parameter", p1->name);
		msg_Format ("(originally declared @ node %d)\n",
			    ps->nodeWhereDeclared);
		++errors;
		continue;
	    }
#if 0
	    /*
	     * okay.  we've found the declaration for this parameter.
	     * replace this parameter with its original declaration.
	     */
	    if (p0 != p1) {
		rb_SetValue (tn, (void *) p0);
		/* free (p1); */
	    }
#endif
	}
    }

#if 0
    /*
     * 1a. We may have parameters in the expression trees that are still
     * of unknown type.  Fix these.
     */

    for (i = 0; i < n->nargs; ++i) {
	fprintf (stderr, "fixExpression [arg %d]\n", i + 1);
	fixExpression (n->args[i]);
    }
#endif

    /*
     * 2. Now we should know the type of each param.  Build a descriptor
     * for this function call.  If there are no previously declared
     * instances of this function call, declare one; else, check
     * to make sure this use of the function matches all others.
     */
    if (n->node_type == NODE_NORMAL) {
	sc1 = buildFuncDecl (n);
	sc0 = lookupFuncDecl (n->sub_name);
	if (sc0 == NULL) {
	    msg_Format ("Node %d: no declaration for function %s\n",
			n->nk.id, n->sub_name);
	    ++errors;
	}
	else if (compareFunctions (n, sc1, sc0) != 0) {
	    msg_Format ("Node %d: use of function %s does not match declaration\n",
			n->nk.id, n->sub_name);
	    ++errors;
	}
	else {
#ifdef DEBUG
	    fprintf (stderr, "marking %s (%x) as used\n", sc0->name, sc0);
#endif
	    sc0->isUsed = 1;		/* mark function as being used */
	}

	if (sc1->argTypes)
	    free (sc1->argTypes);
	free (sc1);
    }

    /*
     * 3. scan all of our children looking for similar declaration conflicts
     */
    if (n->nchildren != 0 && n->children != NULL) {
	for (tn = rb_First (n->children); tn != n->children; tn = rb_Next (tn))
	    errors += scanSubGraph ((Node) rb_Value (tn));
    }

    /*
     * 4. for any parameters that we declared here, undeclare them now
     */

    for (tn = rb_First (n->params); tn != n->params; tn = rb_Next (tn)) {
	Param p1;

	p1 = rb_Value (tn);
#if 0
	if (p1->nodeWhereDeclared == n)
#endif
	undeclareParameter (p1, n);
    }
    return errors;
}

/*
 * do any initialization that needs to be done before collecting
 * parameters and function names
 */

static int
zapNodes (n)
Node n;
{
    n->flags &= ~NF_BEEN_HERE;
    return 0;
}

/*
 * do any cleanup that needs to be done after collecting parameters
 * and function names
 */

static int
unzapNodes (n)
Node n;
{
    n->flags &= ~NF_BEEN_HERE;
    return 0;
}

int
subdefs_CheckGraph (g, filename)
Graph g;
char *filename;
{
    TreeNode tn;
    int errors = 0;

#ifdef DEBUG
    fprintf (stderr, "checking graph\n");
#endif
    if (subdefs_ReadDecls (filename) != 0)
	return 1;
#ifdef DEBUG
    subdefs_WriteDecls (stderr);
#endif
    paramSymTable = rb_MakeTree ();

    if (!rb_Empty (g->heads)) {
	for (tn = rb_First (g->heads); tn != g->heads; tn = rb_Next (tn))
	    errors += scanSubGraph (rb_Value (tn));
    }
#if 0
    nukeFuncDecls ();
    nukeParamDecls ();
#endif

#ifdef DEBUG
    fprintf (stderr, "%d error%s\n", errors, errors == 1 ? "" : "s");
#endif

    return errors;
}

void
subdefs_NukeDecls ()
{
    nukeFuncDecls ();
    nukeParamDecls ();
}



/*
 * Local variables:
 * mode:c
 * End:
 */
