/*
 *	HeNCE Tool
 *
 *	costmat.c - Main widget stuff.
 *		This is stuff for dealing with cost matrices.
 *
 *	Jun 1991  Robert Manchek  manchek@CS.UTK.EDU.
 *
 *	Revision Log
 *
$Log$
 *
 */

#include <stdio.h>
#include <ctype.h>
#include "xincl.h"
#include "xcomn.h"
#include "rb.h"
#include "param.h"
#include "exp.h"				/* XXX Yuk! */
#include "graph.h"
#include "graphP.h"				/* XXX Yuk! */
#include "comn.h"
#include "costmat.h"

static char crud[1024];

/*	cm_New ()
 *
 *	Make a new cost matrix.  Add a single entry with null hostname
 *	and function name.
 */

struct costm*
cm_New ()
{
	struct costm *cm;

	if (cm = talloc(1, struct costm)) {
		cm->nhost = 0;
		cm->nsub = 0;
		cm->hosts = 0;
		cm->subs = 0;
		cm->mat = 0;
		cm_AddEntry (cm, (char*)0, (char*)0, 0);
		cm->isModified = 0;
	}
	return cm;
}

/*	cm_Free ()
 *
 *	Free any storage associated with a cost matrix.
 */

void
cm_Free (cm)
struct costm *cm;
{
	int i;

	if (cm) {
		if (cm->hosts) {
			for (i = cm->nhost; i-- > 0; )
				if (cm->hosts[i])
					free(cm->hosts[i]);
			free(cm->hosts);
		}
		if (cm->subs) {
			for (i = cm->nsub; i-- > 0; )
				if (cm->subs[i])
					free(cm->subs[i]);
			free(cm->subs);
		}
		if (cm->mat)
			free(cm->mat);
		free(cm);
	}
}

/*	cm_AddEntry ()
 *
 *	Add (or change) the cost of a host/subroutine pair in a cost matrix.
 *	Creates a new row or column if necessary.
 *
 *	Returns 0 if ok
 *		1 if can't get memory.
 */

int
cm_AddEntry (cm, host, sub, val)
struct costm *cm;
char *host;
char *sub;
int val;
{
	int h, s;
	char **pp;
	int *ip;
	int i, j, k;
	int n;

	cm->isModified = 1;
	if (host) {
		for (h = cm->nhost; h-- > 0; )
			if (cm->hosts[h] && !strcmp(cm->hosts[h], host))
				break;
	}
	else {
		for (h = cm->nhost; h-- > 0; )
			if (!cm->hosts[h])
				break;
	}
	if (sub) {
		for (s = cm->nsub; s-- > 0; )
			if (cm->subs[s] && !strcmp(cm->subs[s], sub))
				break;
	}
	else {
		for (s = cm->nsub; s-- > 0; )
			if (!cm->subs[s])
				break;
	}

	if (h < 0 || s < 0) {
		n = (k = ((s < 0) ? cm->nsub + 1 : cm->nsub))
		* ((h < 0) ? cm->nhost + 1 : cm->nhost);
		if (!(ip = talloc(n, int)))
			return 1;
		bzero(ip, n * sizeof(int));
		for (i = cm->nsub; i-- > 0; )
			for (j = cm->nhost; j-- > 0; )
				ip[i + j * k] = cm->mat[i + j * cm->nsub];
		free(cm->mat);
		cm->mat = ip;
	}
	if (h < 0) {
		h = cm->nhost++;
		if (!(pp = talloc(h + 1, char*))
		|| !(ip = talloc(h + 1, int)))
			return 1;
		if (h) {
			bcopy(cm->hosts, pp, h * sizeof(char*));
			free(cm->hosts);
			bcopy(cm->hflag, ip, h * sizeof(int));
			free(cm->hflag);
		}
		cm->hosts = pp;
		cm->hflag = ip;
		cm->hosts[h] = host ? strsave(host) : 0;
		cm->hflag[h] = SUBHOST_EN;
	}
	if (s < 0) {
		s = cm->nsub++;
		if (!(pp = talloc(s + 1, char*))
		|| !(ip = talloc(s + 1, int)))
			return 1;
		if (s) {
			bcopy(cm->subs, pp, s * sizeof(char*));
			free(cm->subs);
			bcopy(cm->sflag, ip, s * sizeof(int));
			free(cm->sflag);
		}
		cm->subs = pp;
		cm->sflag = ip;
		cm->subs[s] = sub ? strsave(sub) : 0;
		cm->sflag[s] = 0;
	}
	cm->mat[s + h * cm->nsub] = val;
	return 0;
}

/*	cm_ReadFile ()
 *
 *	Read a file into a new cost matrix and return it.
 *
 *	Returns 0 if ok
 *		1 if can't read file
 *		2 if errors in file
 *		3 if can't get memory
 */

int
cm_ReadFile (fn, cmp)
char *fn;
struct costm **cmp;
{
	struct costm *cm;
	FILE *ff;
	char buf[1024];
	int ac;
	char *av[3];
	char *p;
	int err = 0;

	if (!(cm = cm_New ()))
		return 3;
	if (!(ff = fopen(fn, "r")))
		return 1;

	while (fgets(buf, sizeof(buf), ff)) {
		for (p = buf; *p && isspace(*p); p++);
		if (!*p || *p == '#')	/* skip comments */
			continue;
		ac = 3;
		if (acav(p, &ac, av) || ac != 3) {
			err++;
			continue;
		}
		cm_AddEntry (cm, av[0], av[1], atoi(av[2]));
	}
	fclose(ff);
	cm->isModified = 0;

	if (err) {
		cm_Free (cm);
		return 2;
	}

	if (cmp)
		*cmp = cm;
	else
		cm_Free (cm);
	return 0;
}

/*	cm_WriteFile ()
 *
 *	Write a cost matrix into a file.
 *
 *	Returns 0 if ok
 *		1 if can't write file.
 *		2 if null cost matrix.
 */

int
cm_WriteFile (fn, cmp)
char *fn;
struct costm *cmp;
{
	FILE *ff;
	int h, s;

	if (!cmp)
		return 2;
	if (!(ff = fopen(fn, "w")))
		return 1;

	for (h = 0; h < cmp->nhost; h++)
		if (cmp->hosts[h] && *cmp->hosts[h])
			for (s = 0; s < cmp->nsub; s++)
				if (cmp->subs[s] && *cmp->subs[s])
					fprintf(ff, "%s %s %d\n", cmp->hosts[h], cmp->subs[s],
						cmp->mat[s + h * cmp->nsub]);

	(void)fclose(ff);
	cmp->isModified = 0;
	return 0;
}


/*	cm_CheckGraph ()
 *
 *	Check that every node in a graph has is mapped to at least one
 *	host (SUBHOST_EN is true and the cost is > 0).
 *	Mark all needed hosts with SUBHOST_HOST and needed subs with
 *	SUBHOST_SUB.
 *
 *	Returns 0 if ok, else 1;
 */

int
cm_CheckGraph (grf, cm)
Graph grf;
struct costm *cm;
{
	Node n;
	TreeNode tn;
	int err = 0;
	int i, j, k;

	if (grf == NULL) 
		return 0;

	if (grf->heads == NULL || gr_Empty (grf->heads))
		return 0;

	/* clear all 'needed' flags */

	for (i = cm->nsub; i-- > 0; )
		cm->sflag[i] &= ~SUBHOST_SUB;
	for (i = cm->nhost; i-- > 0; )
		cm->hflag[i] &= ~SUBHOST_HOST;

	/*	for each node, mark function used and also
		all enabled hosts with positive cost */

	for (tn = rb_First(grf->nlist); tn != grf->nlist; tn = rb_Next (tn)) {
		n = (Node) rb_Value (tn);

		if (n == NULL || n->node_type != NODE_NORMAL)
			continue;

		for (i = cm->nsub; i-- > 0; )
			if (cm->subs[i] != NULL && n->sub_name != NULL &&
				strcmp(n->sub_name, cm->subs[i]) == 0) {
				cm->sflag[i] |= SUBHOST_SUB;
				k = 0;
				for (j = cm->nhost; j-- > 0; )
					if ((cm->hflag[j] & SUBHOST_EN)
					&& cm->hosts[j]
					&& cm->mat[i + j * cm->nsub] > 0) {
						cm->hflag[j] |= SUBHOST_HOST;
						k++;
					}
				if (!k)
					i = -1;
				break;
			}
		if (i < 0) {
			if (n->sub_name == NULL)
				msg_Format ("node %d: missing function name\n", n->nk.id);
			else
				msg_Format ("node %d: no mapping for function %s\n",
							n->nk.id, n->sub_name);
			err++;
		}
	}
	return err ? 1 : 0;
}

#if 0
/*	cm_MakePvmHostFile ()
 *
 *	Convert a cost matrix into a pvm host file.
 *	Makes sure the localhost has an entry and is first.
 *
 *	XXX domain names would be a problem here... what to do...
 *
 *	Write an entry for each needed host.
 *	Returns:
 *		0 for ok.
 *		1 if can't write host file.
 *
 *	XXX This function should go away.  The pvm hosts file should
 *  be supplied by the user, and the default cost matrix should
 *  be derived from the pvm hosts file instead of the other way
 *  'round.
 */

int
cm_MakePvmHostFile (cm, fn)
	struct costm *cm;	/* cost matrix */
	char *fn;			/* filename */
{
	FILE *ff;
	int h;
	char buf[256];

	gethostname(buf, sizeof(buf)-1);
	buf[sizeof(buf)-1] = 0;

	if (!(ff = fopen(fn, "w")))
		return 1;

	fputs("#PVM host file written by HeNCE tool\n", ff);
	fprintf(ff, "%s\n", buf);
	for (h = cm->nhost; h-- > 0; )
		if (cm->hflag[h] & SUBHOST_HOST
		&& strcmp(cm->hosts[h], buf))
			fprintf(ff, "%s\n", cm->hosts[h]);

	fclose(ff);
	return 0;
}
#endif

/*
 * return 1 if hostname is mentioned in cost matrix, else 0
 * This is called by cm_BuildPvmHostFile().
 *
 * Also, set the INPVM bit as a side effect.  cm_BuildPvmHostFile()
 * will call cm_ClearPvmFlag() before starting, and will call
 * cm_CheckPvmFlag() after building the host file.  If there is
 * a host that is defined in the cost matrix, but not in the pvm
 * hosts file, cm_CheckPvmFlag() will complain about it.
 */

int
cm_HostIsPresent (cm, hostname)
struct costm *cm;
char *hostname;
{
	int h;
	for (h = cm->nhost; h-- > 0; ) {
		if ((cm->hflag[h] & SUBHOST_HOST) &&
			strcmp (cm->hosts[h], hostname) == 0) {
			cm->hflag[h] |= SUBHOST_INPVM;
			return 1;
		}
	}
	return 0;
}

void
cm_ClearPvmFlag (cm)
struct costm *cm;
{
	int h;
	for (h = cm->nhost; h-- > 0; )
		cm->hflag[h] &= ~SUBHOST_INPVM;
}

int
cm_CheckPvmFlag (cm)
struct costm *cm;
{
	int h;
	int error = 0;

	for (h = cm->nhost; h-- > 0; ) {
		if (cm->hosts[h] && (cm->hflag[h] & SUBHOST_INPVM) == 0) {
			msg_Format ("\
%s appears in the cost matrix, but not in the pvm hosts file.\n",
						cm->hosts[h]);
			error = -1;
		}
	}
	return error;
}

int
cm_SubIsPresent (cm, sub)
struct costm *cm;
char *sub;
{
	int s;
	for (s = cm->nsub; s-- > 0; ) {
		if ((cm->sflag[s] & SUBHOST_SUB) &&
			strcmp (cm->subs[s], sub) == 0) {
			return 1;
		}
	}
	return 0;
}


#if 0
/*	defcostnames()
 *
 *	Make sure that each node function in a graph has a cost matrix column.
 */

defcostnames(grf, cm)
	Node grf;
	struct costm *cm;
{
	node_t *n;

	if (cm) {
		for (n = grf; n; n = n->next)
			if (n->type == NTYPE_NO && n->fun && *n->fun)
				cm_AddEntry (cm, (char*)0, n->fun, 0);
	}
}
#endif

/*
 * make sure there is a "sub" entry in the cost matrix for every
 * function called by the HeNCE graph.  Don't call cm_AddEntry()
 * for any functions that area already there, since this has the
 * side effect of setting the isModified bit.
 *
 * returns 1 if any functions were added, else 0.
 */

int
cm_AddNamesFromGraph (g, cm)
Graph g;
struct costm *cm;
{
	TreeNode tn;
	int modified = 0;

	if (g->nlist == NULL || rb_Empty (g->nlist))
		return 0;
	for (tn = rb_First (g->nlist); tn != g->nlist; tn = rb_Next (tn)) {
		Node n = (Node) rb_Value (tn);

		if (n->node_type == NODE_NORMAL && n->sub_name && *n->sub_name) {
			int s;

			for (s = cm->nsub; s-- > 0; )
				if (cm->subs[s] && strcmp (cm->subs[s], n->sub_name) == 0)
					break;
			if (s <= 0) {
				cm_AddEntry (cm, (char *) NULL, n->sub_name, 0);
				modified = 1;
			}
		}
	}
	return modified;
}

/*
 * Given a "master" pvm hosts file, emit a replica of that host file
 * containing only those hosts we need for this HeNCE program, as
 * determined by the cost matrix.
 *
 * if "in" is NULL, it means the pvm hosts file did not exist, so
 * we simply write out a host file containing everything in the
 * cost matrix.
 *
 * XXX old versions of pvm require that the local host name appear
 * first.  This tries to ensure this by always emitting the local
 * hostname (as determined by gethostbyname()) first, and refusing
 * to copy it if it appears later in the hosts file.  This doesn't
 * cause any problems with current pvm implementations, since all
 * of the per-host "options" in a pvm hosts file are only relevant
 * to remote hosts.  This might change in the future.  Also, the
 * check to see if a hostname in the pvm hosts file is "local" is
 * a simple strcmp () of the hostname from the master pvm file with
 * the one from gethostname().  It would be better done with a
 * case-insensitive string comparision, or perhaps by comparing IP
 * addresses.
 */

int
cm_BuildPvmHostFile (in, out, cm)
FILE *in, *out;
struct costm *cm;
{
	char line[1024];
	char hostname[1024];
	char myhostname[256];
	int h;

	gethostname (myhostname, sizeof myhostname);

	/*
	 * reset all "host is in pvm" flags in the cost matrix, then
	 * set that bit for the local host, since we always copy it.
	 */

	cm_ClearPvmFlag (cm);
	(void) cm_HostIsPresent (cm, myhostname);

	/*
	 * pass 1: read and parse the master pvm hosts file, setting
	 * the "host is in pvm" flag for every host that appears and
	 * is also in the cost matrix.
	 */

	if (in) {
		while (fgets (line, sizeof (line), in)) {
			int hostIsMe;

			if (*line == '\0' || *line == '#')
				continue;
			sscanf (line, "%[^# \t\n] %*[^\n]", hostname);
			hostIsMe = (strcmp (hostname, myhostname) == 0);

			if (hostIsMe)
				continue;

			(void) cm_HostIsPresent (cm, hostname);
		}

		if (ferror (in) || ferror (out))
			return -1;
	}


	/*
	 * Now, prepare to write the temp pvm hosts file.  First the
	 * local host name is written to the file, then the names of
	 * any hosts that were in the cost matrix but were not in the
	 * master pvm hosts file.
	 */

	fprintf (out, "%s\n", myhostname);
#if 0
	fprintf (stderr, "%s\n", myhostname);
#endif
	for (h = cm->nhost; h-- > 0; ) {
		if (cm->hosts[h] && (cm->hflag[h] & SUBHOST_INPVM) == 0) {
#if 0
			msg_Format ("\
warning: %s appears in the cost matrix, but not in the pvm hosts file.\n",
						cm->hosts[h]);
#endif
			fprintf (out, "%s\n", cm->hosts[h]);
#if 0
			fprintf (stderr, "%s\n", cm->hosts[h]);
#endif
		}
	}

	/*
	 * pass 2: copy the line from the pvm hosts file for any host
	 * (other than the local host) that exists in the cost matrix.
	 */

	if (in) {
		rewind (in);
		while (fgets (line, sizeof (line), in)) {
			int hostIsMe;

			if (*line == '\0' || *line == '#')
				continue;
			sscanf (line, "%[^# \t\n] %*[^\n]", hostname);
			hostIsMe = (strcmp (hostname, myhostname) == 0);

			if (hostIsMe)
				continue;

			if (strcmp (hostname, "*") == 0 ||
				cm_HostIsPresent (cm, hostname)) {
				fputs (line, out);
#if 0
				fputs (line, stderr);
#endif

			}
		}

		if (ferror (in) || ferror (out))
			return -1;
	}
	return 0;
}
/*
 * Local variables:
 * tab-width:4
 * End:
 */
