/*
 *	imal.c
 *
 *	Instrumented malloc library.
 *
 *	7 Jun 1991  Robert Manchek  manchek@CS.UTK.EDU.
 *
 *	To use, change all malloc()s to x_malloc()s and free()s to x_free()s
 *	and link in.
 *
 *	Facilities:
 *
 *	$ All errors written to fd 2 (not using stdio).
 *
 *	$ Configurable to die on error for debugger.
 *
 *	$ X_malloc():
 *		Error on [adjustable] unreasonble length or malloc() failed.
 *		Hashes descriptors of all chunks for x_free() checking.
 *		Numbers requests to indicate order received.
 *		Writes [adjustable] pseudo-random head and tail pads to
 *			aid in checking overrun of chunk.
 *
 *	$ X_free():
 *		Error if chunk not x_malloc()d.
 *		Checks head and tail pads.
 *		[Optionally] zeros chunk to aid in detecting late use.
 *
 *	$ X_dump():
 *		Callable from user program.
 *		Dumps the hash table to see what chunks are active.
 */

#include <stdio.h>
#include <strings.h>

#define	DIEONERROR
#define	ZEROONFREE


#ifdef	DIEONERROR
#define	CHOKE	{ char *p = 0; *p = 1; }
#else
#define	CHOKE
#endif

#define	NEXTRN(x)	(x = (x) + (x) + (((x ^ (x >> 3)) & 0x2000) ? 1 : 0))
#define	HASH(p)		(((int)(p) ^ ((int)(p) >> 8) ^ ((int)(p) >> 16) ^ ((int)(p) >> 24)) & 0xff)

/* this describes a chunk of memory */

struct glob {
	char *base;			/* baseaddr of user chunk */
	int len;			/* len of user chunk */
	int id;				/* chunk id */
	int lop;			/* nbytes head padding */
	int hip;			/* nbytes tail padding */
	int rst;			/* starting random state */
	struct glob *next;	/* next glob in hash bucket */
};

/* note:  default values */

static int debfd = 2;				/* debug file descriptor */
static int lengthlimit = 1048576;	/* max length malloc allowed */
static int lopad = 16;				/* chunk head pad */
static int hipad = 16;				/* chunk tail pad */

/* globals */

static int rnstate = 1;				/* random sequence gen. */
static struct glob *hashtbl[256];	/* chunk hash table */
static char msbuf[256];				/* error message buffer */
static int firsttime = 1;
static int globid = 0;				/* chunk id counter */

char *n_malloc(len)
int len;
{
  return (char *) malloc(len);
}

n_free(p)
char *p;
{
   (void) free(p);
}

char *n_realloc(p, sz)
char *p;
int sz;
{
  return (char *) realloc(p, sz);
}
  
char*
x_malloc(len)
	int len;
{
	char *ptr;
	struct glob *ob;	/* hash tbl entry */
	struct glob **he;
	int i;				/* gp */

	if (firsttime) {
		firsttime = 0;
		bzero((char*)hashtbl, sizeof(hashtbl));
	}
	if (len < 1 || len > lengthlimit) {
		(void)sprintf(msbuf, "x_mall: bogus len=%d\n", len);
		(void)write(debfd, msbuf, strlen(msbuf));
		CHOKE;
		return (char*)0;
	}
	if (!(ptr = (char*)malloc(len + lopad + hipad))) {
		(void)sprintf(msbuf, "x_mall: malloc failed len=%d\n", len);
		(void)write(debfd, msbuf, strlen(msbuf));
		CHOKE;
		return (char*)0;
	}
	if (!(ob = (struct glob*)malloc(sizeof(struct glob)))) {
		(void)sprintf(msbuf, "x_mall: malloc failed for glob\n");
		(void)write(debfd, msbuf, strlen(msbuf));
		CHOKE;
		return (char*)0;
	}
	ob->id = ++globid;
	ob->len = len;
	ob->lop = lopad;
	ob->hip = hipad;
	ob->rst = rnstate;
	for (i = lopad; i-- > 0; *ptr++ = NEXTRN(rnstate));
	ob->base = ptr;
	he = &hashtbl[HASH(ptr)];
	for (i = hipad, ptr += len; i-- > 0; *ptr++ = NEXTRN(rnstate));
	ob->next = *he;
	*he = ob;
	return ob->base;
}

x_free(loc)
	char *loc;
{
	struct glob *ob;	/* freeing this object */
	struct glob *preob;	/* object before in chain */
	int rs;				/* reproduced random seqn */
	char *ptr = loc;	/* gp */
	struct glob **he;	/* hash tbl entry */
	int i;				/* gp */

	he = &hashtbl[HASH(loc)];
	for (preob = 0, ob = *he; ob && ob->base != loc; preob = ob, ob = ob->next);
	if (!ob) {
		(void)sprintf(msbuf, "x_free: bogus loc=0x%x\n", loc);
		(void)write(debfd, msbuf, strlen(msbuf));
		CHOKE;
		return;
	}
	rs = ob->rst;
	for (i = ob->lop, ptr -= i; i > 0; i--)
		if ((0xff & (int)(*ptr++)) != (0xff & NEXTRN(rs))) {
			(void)sprintf(msbuf, "x_free: scribbled in 0x%x[%d]\n", loc, -i);
			(void)write(debfd, msbuf, strlen(msbuf));
		CHOKE;
		}
	for (i = ob->hip, ptr += ob->len; i > 0; i--)
		if ((0xff & (int)(*ptr++)) != (0xff & NEXTRN(rs))) {
			(void)sprintf(msbuf, "x_free: scribbled in 0x%x[%d+%d]\n",
				loc, ob->len, ob->hip - i);
			(void)write(debfd, msbuf, strlen(msbuf));
		CHOKE;
		}
#ifdef	ZEROONFREE
	bzero(loc - ob->lop, ob->len + ob->lop + ob->hip);
#endif	ZEROONFREE
	free(loc - ob->lop);
	if (preob)
		preob->next = ob->next;
	else
		*he = ob->next;
	free((char*)ob);
}

x_dump()
{
	int ht;
	struct glob *ob;
	char *p;
	char c;
	char *r;
	int i;

	for (ht = 0; ht < 256; ht++) {
		for (ob = hashtbl[ht]; ob; ob = ob->next) {
			sprintf(msbuf, "%5d 0x%08x[%d]", ob->id, ob->base, ob->len);
			r = msbuf + strlen(msbuf);
			*r++ = ' ';
			if ((i = ob->len) > 16)
				i = 16;
			for (p = ob->base; i-- > 0; p++) {
				c = 0x7f & *p;
				if (c < ' ' || c == 0x7f) {
					c = (c + '@') & 0x7f;
					*r++ = '^';
				} else
					*r++ = ' ';
				*r++ = c;
			}
			*r++ = '\n';
			*r = 0;
			(void)write(debfd, msbuf, strlen(msbuf));
		}
	}
}

