#include <stdio.h>
#include <fcntl.h>		/* XXX Bondoize? */
#include <unistd.h>
#include <math.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/Xaw/Toggle.h>

#include <Errno.h>
#include <Malloc.h>

#include "config.h"
#include "costmat.h"
#include "global.h"
#include "graph.h"
#include "graph_draw.h"
#include "graph_edit.h"
#include "histo.h"
#include "tfnew.h"
#include "trace_new.h"


struct stats {
    int num_running;
    int num_idle;
};

enum trace_timer_mode {		/* mode for trace_timer callback */
    TIMER_ALARM, FIRST_EVENT
};

Widget graph_canvas;		/* where we scribble */
static int trace_fd = -1;	/* file descriptor when in trace mode */
static int histo_fd = -1;	/* file descriptof for histogram */
static enum {
    REVERSE = -1, FORWARD = 1
} trace_direction = FORWARD;	/* current trace direction when playing */
static int trace_stopped = 1;	/* true iff not playing */
static float time_scale = 1.0;	/* how trace time relates to
				   wall clock elapsed time */
static XtIntervalId trace_timer_id; /* handle for Xt timer callback */

static int num_machines = 0;	/* number of machines used */
static int num_subrs = 0;	/* number of subroutines */
static int num_nodes = 0;	/* number of nodes */
static int num_cells = 0;	/* number of (host,subr) cells */
static int num_stats = 0;	/* number of horizontally-encoded stats */

static struct machine_rec *machines = NULL;
static struct subr_rec *subrs = NULL;
static struct node_rec *nodes = NULL;
static struct cost_matrix *costs = NULL;
static struct event_rec current_event;

/*
 * XXX node_stats are redundant with the numRunning and numIdle fields
 * of the nodes themselves.  Get rid of either one or the other.
 */
static struct stats *machine_stats = NULL;
static struct stats *histo_machine_stats = NULL;
static struct stats *subr_stats = NULL;
static struct stats *node_stats = NULL;
static struct stats *per_cell_stats = NULL;

static struct event_time epoch;	/* abs time that execution started */
static struct event_time epoch_wall_time;
				/* wall clock time at execution epoch */
static struct event_time final;	/* abs time that execution started */
static long epoch_rec_addr = 0;	/* file address of first event */
static long final_rec_addr = 0;	/* file address of last event */
static unsigned int epoch_seq = 0; /* sequence number of epoch */
static unsigned int final_seq;	/* sequence number of final event */
static struct event_time trace_time; /* abs time of current event */

/*
 * XXX this is copied from histo.c.  Eventually put all of the event/
 * time management stuff in a separate file.
 *
 * return the difference between two time stamps
 */

static struct event_time *
elapsed_time (t1, t2)
struct event_time *t1, *t2;
{
    static struct event_time result;

    result.seconds = 0;
    result.msec = t2->msec - t1->msec;
    if (result.msec < 0) {
	result.msec += 1000;
	result.seconds = -1;
    }
    result.seconds = result.seconds + t2->seconds - t1->seconds;
#ifdef DEBUG
    if (result.msec < 0) {
	fprintf (stderr, "elapsed_time (%d.%03d %d.%03d) -> %d.%03d\n",
		 t1->seconds, t1->msec,
		 t2->seconds, t2->msec,
		 result.seconds, result.msec);
    }
#endif
    return &result;
}

/*
 * XXX these were also stolen from histo.c
 *
 * return true iff t1 is later than t2
 */

static int 
after (t1, t2)
struct event_time *t1, *t2;
{
    if (t1->seconds > t2->seconds)
	return 1;
    else if (t1->seconds < t2->seconds)
	return 0;
    else
	return (t1->msec > t2->msec);
}

/*
 * return true iff t1 is before t2
 */

static int
before (t1, t2)
struct event_time *t1, *t2;
{
    if (t1->seconds < t2->seconds)
	return 1;
    else if (t1->seconds > t2->seconds)
	return 0;
    else
	return (t1->msec < t2->msec);
}

static int
equal (t1, t2)
struct event_time *t1, *t2;
{
    return ((t1->seconds == t2->seconds) &&
	    (t1->msec == t2->msec));
}

/*
 * return the event with the given sequence number.
 */

static int
read_event_by_seq (fd, evp, seq)
int fd;
struct event_rec *evp;
int seq;
{
    int x;

    if (seq < 0 || seq < epoch_seq || seq > final_seq)
	return EOF;
#if 0
    fprintf (stderr, "lseek (fd, %ld, SEEK_SET)\n",
	     epoch_rec_addr + (seq - epoch_seq) * sizeof (struct event_rec));
#endif
    lseek (fd,
	   epoch_rec_addr + (seq - epoch_seq) * sizeof (struct event_rec),
	   SEEK_SET);
    x = read_event_rec (fd, evp);
#if 0
    if (evp->sequence != seq) {
	fprintf (stderr, "seq == %d but evp->sequence == %d\n",
		 seq, evp->sequence);
    }
#endif
    return x;
}

/*
 * find the event at or immediately preceeding time t
 */

static int
read_event_by_time (fd, evp, t)
int fd;
struct event_rec *evp;
struct event_time *t;
{
    int low, hi, mid;
    struct event_time evt;

#if 0
    fprintf (stderr, "read_event_by_time (%d.%d)\n",
	     t->seconds, t->msec);
#endif
    if (before (t, &epoch) || after (t, &final))
	return EOF;
    if (equal (t, &epoch)) {
	read_event_by_seq (fd, evp, epoch_seq);
	return 0;
    }
    if (equal (t, &final)) {
	read_event_by_seq (fd, evp, final_seq);
	return 0;
    }
    low = epoch_seq;
    hi = final_seq;
    while (low < hi) {
	mid = (low + hi) / 2;
	read_event_by_seq (fd, evp, mid);
#if 0
	fprintf (stderr, "lo=%d hi=%d rec=%d %d.%d\n",
		 low, hi, mid, evp->seconds, evp->msec);
#endif
	evt.seconds = evp->seconds;
	evt.msec = evp->msec;
	if (after (&evt, t)) {
	    /*
	     * this event occurs *after* t, so we can safely
	     * set hi to mid.  (NOT mid-1, since that might occur before t)
	     */
	    hi = mid;
	    /*
	     * if (low + 1 == hi), low is the sequence number of the
	     * record we want.  but since (low + 1 + low) / 2 == low,
	     * we'll automatically pick it up the next time around.
	     */
	}
	else if (equal (t, &evt))
	    return 0;
	else {
	    /*
	     * this event occurs *before* t.
	     * (but mid + 1 might occur *after* t)
	     */
	    low = mid;
	    /*
	     * since hi is always after t, if there's nothing
	     * between lo and hi, we're done
	     */
	    if (low + 1 == hi)
		return 0;
	}
    }
    return 0;
}

/*
 * given a subroutine index number, return its name
 */

static char *
find_sub_name (s)
int s;
{
    if (subrs == NULL || s < 0 || s >= num_subrs)
	return NULL;
    return subrs[s].subr_name;
}


/*
 * given a machine index number, return its name
 */

static char *
find_machine_name (m)
int m;
{
    if (machines == NULL || m < 0 || m >= num_machines)
	return NULL;
    return machines[m].machine_name;
}


/*
 * given a node number from the graph, find its index number
 */

static int
find_node_index (n)
int n;
{
    int i;

    for (i = 0; i < num_nodes; ++i)
	if (nodes[i].node_number == n)
	    return i;
    return -1;
}

/*
 * reset all private data structures having to do with trace files
 */

int
trace_reset ()
{
    if (machine_stats) {
	FREE (machine_stats);
	machine_stats = NULL;
    }
    if (histo_machine_stats) {
	FREE (histo_machine_stats);
	histo_machine_stats = NULL;
    }
    if (subr_stats) {
	FREE (subr_stats);
	subr_stats = NULL;
    }
    if (node_stats) {
	FREE (node_stats);
	node_stats = NULL;
    }
    if (per_cell_stats) {
	FREE (per_cell_stats);
	per_cell_stats = NULL;
    }
    if (machines) {
	FREE (machines);
	machines = NULL;
    }
    if (subrs) {
	FREE (subrs);
	subrs = NULL;
    }
    if (nodes) {
	FREE (nodes);
	nodes = NULL;
    }
#if 0
    /*
     * don't free costs.  cost matrices are alloc'ed and freed by 
     * enter_{run,trace}_mode because the cost matrix widget needs to
     * be able to get to this data at any time.
     */
    if (costs) {
	cm_Free (costs);
	costs = NULL;
    }
#endif
    num_machines = num_subrs = num_cells = num_stats = num_nodes = 0;
}

static void
get_wall_time (wall_time)
struct event_time *wall_time;
{
    struct timeval tp;
    struct timezone tz;

    gettimeofday (&tp, &tz);
    wall_time->seconds = tp.tv_sec;
    wall_time->msec = tp.tv_usec / 1000;
}

/*
 * read the header of the trace file.
 * return 0 on success.
 * return -1 on premature EOF
 * return -2 on a file format error
 *
 * side effects:
 * + set epoch_rec_addr to the seek address at the end of the header.
 */

int
trace_read_header (fd)
int fd;
{
    struct section_hdr section_hdr;
    struct hdr_rec hdr_rec;
    struct subr_rec subr_rec;
    struct node_rec node_rec;
    struct link_rec link_rec;
    struct cost_rec cost_rec;
    int last_section_type = 0;
    int i;
    int graph_needs_scaling;
    char *msg = "premature EOF";

    while (1) {
	if (read_section_hdr (fd, &section_hdr) < 0)
	    goto eof;
	if (section_hdr.section_type < last_section_type) {
	    msg = "illegal file format";
	    goto fail;
	}
	last_section_type = section_hdr.section_type;
	switch (section_hdr.section_type) {
	case SECT_HDR:
#if 0
	    fprintf (stderr, "reading file header\n");
#endif
	    if (read_hdr_rec (fd, &hdr_rec) < 0)
		goto eof;
	    if (strcmp (hdr_rec.version, "2.0") != 0) {
		msg = "can't read trace files with this version #";
		goto fail;
	    }
	    break;
	case SECT_MACHINES:
#if 0
	    fprintf (stderr, "reading machines\n");
#endif
	    num_machines = section_hdr.record_count;
	    machines = TALLOC (num_machines, struct machine_rec);
	    for (i = 0; i < num_machines; ++i) {
		if (read_machine_rec (fd, &(machines[i])) < 0)
		    goto eof;
#if 0
		fprintf (stderr, "[%d]%s\n", i, machines[i].machine_name);
#endif
	    }
	    break;
	case SECT_SUBRS:
#if 0
	    fprintf (stderr, "reading subrs\n");
#endif
	    num_subrs = section_hdr.record_count;
	    subrs = TALLOC (num_subrs, struct subr_rec);
	    for (i = 0; i < num_subrs; ++i) {
		if (read_subr_rec (fd, &(subrs[i])) < 0)
		    goto eof;
#if 0
		fprintf (stderr, "[%d] %s\n", i, subrs[i].subr_name);
#endif
	    }
	    break;
	case SECT_NODES:
#if 0
	    fprintf (stderr, "reading nodes\n");
#endif
	    global.traceGraph = gr_New ();
	    graph_needs_scaling = 0;
	    num_nodes = section_hdr.record_count;
	    nodes = TALLOC (num_nodes, struct node_rec);
	    for (i = 0; i < num_nodes; ++i) {
		Node n;

		if (read_node_rec (fd, &(nodes[i])) < 0)
		    goto eof;
#if 0
		fprintf (stderr, "[%d] %d %d %d\n", i, nodes[i].node_number,
			 nodes[i].node_type, nodes[i].subr_number);
#endif
		/* add node to graph */
		n = gr_NewNode (nodes[i].node_number);
		if (nodes[i].x == 0 && nodes[i].y == 0)
		    graph_needs_scaling = 1;
		gr_SetNodeLoc (n, nodes[i].x, nodes[i].y);
		n->nk.inst = 0;
		n->node_type = nodes[i].node_type;
		n->sub_name = find_sub_name (nodes[i].subr_number);
		gr_AddNode (global.traceGraph, n);
	    }
	    /*
	     * allocate global totals for horizontally-encoded stats
	     */
	    num_cells = num_machines * num_subrs;
	    num_stats = num_machines + num_subrs + num_nodes + num_cells;

	    machine_stats = TALLOC (num_machines, struct stats);
	    memset ((char *) machine_stats, '\0',
		    num_machines * sizeof (struct stats));
	    histo_machine_stats = TALLOC (num_machines, struct stats);
	    memset ((char *) histo_machine_stats, '\0',
		    num_machines * sizeof (struct stats));
	    subr_stats = TALLOC (num_subrs, struct stats);
	    memset ((char *) subr_stats, '\0',
		    num_subrs * sizeof (struct stats));
	    node_stats = TALLOC (num_nodes, struct stats);
	    memset ((char *) node_stats, '\0',
		    num_nodes * sizeof (struct stats));
	    per_cell_stats = TALLOC (num_cells, struct stats);
	    memset ((char *) per_cell_stats, '\0',
		    num_cells * sizeof (struct stats));
	    break;
	case SECT_LINKS:
#if 0
	    fprintf (stderr, "reading links\n");
#endif
	    for (i = 0; i < section_hdr.record_count; ++i) {
		if (read_link_rec (fd, &link_rec) < 0)
		    goto eof;
#if 0
		fprintf (stderr, "[%d] %d %d\n", i, link_rec.from_node,
			 link_rec.to_node);
#endif
		/* add link to graph */
		gr_BuildArc (global.traceGraph,
			     link_rec.from_node, link_rec.to_node);
	    }
	    break;
	case SECT_COSTS:
#if 0
	    fprintf (stderr, "reading costs\n");
#endif
	    for (i = 0; i < section_hdr.record_count; ++i) {
		if (read_cost_rec (fd, &cost_rec) < 0)
		    goto eof;
#if 0
		fprintf (stderr, "[%d] %s %s %d\n",
			 i,
			 find_machine_name (cost_rec.host_number),
			 find_sub_name (cost_rec.subr_number),
			 cost_rec.cost);
#endif
		/* add this to cost matrix */
		cm_AddEntry (costs,
			     find_machine_name (cost_rec.host_number),
			     find_sub_name (cost_rec.subr_number),
			     cost_rec.cost);
	    }
	    break;
	case SECT_EVENTS:
#if 0
	    fprintf (stderr, "reading first event\n");
#endif
	    epoch_rec_addr = lseek (fd, 0L, SEEK_CUR);

	    /*
	     * read the first event record to get the epoch
	     */
	    if (read_event_rec (fd, &current_event) < 0)
		goto eof;
	    
	    epoch_seq = current_event.sequence;
	    epoch.seconds = current_event.seconds;
	    epoch.msec = current_event.msec;
	    if (global.mode == MODE_EXECUTE)
		get_wall_time (&epoch_wall_time);
	    global.traceClock.seconds = 0;
	    global.traceClock.msec = 0;

	    /*
	     * if we are in trace mode (not execute mode), get
	     * the address of the end of the file.
	     */
	    if (global.mode == MODE_TRACE) {
		final_rec_addr = lseek (fd,
					-(long)(sizeof (struct event_rec)),
					SEEK_END);
		if (read_event_rec (fd, &current_event) < 0)
		    goto eof;
		if (current_event.event_type != EV_FINISH) {
		    msg = "premature EOF";
		    goto fail;
		}
		final_seq = current_event.sequence;
		final.seconds = current_event.seconds;
		final.msec = current_event.msec;
		
		lseek (fd, epoch_rec_addr, SEEK_SET);
	    }
	    else {
		/* run mode */
		final = epoch;
		final_rec_addr = epoch_rec_addr;
		final_seq = 0;
	    }

	    /*
	     * now that we have read the whole header...we can
	     * display the graph and cost matrix...
	     */

	    /*
	     * 1. if the graph doesn't have position information,
	     * figure out where to display things on the screen.
	     * check for cycles first, just in case the trace file
	     * was corrupted.
	     */
	    if (graph_needs_scaling) {
		if (graph_has_cycles (global.traceGraph)) {
		    msg = "graph from trace file has cycles";
		    goto fail;
		}
		else
		    scale_graph (graph_canvas, global.traceGraph);
	    }

	    /*
	     * 2. draw the trace graph
	     *
	     * XXX do we really need to clear window?  shouldn't
	     * this already be done for us?
	     */
	    XClearWindow (global.display, XtWindow (graph_canvas));
	    draw_graph (graph_canvas, global.traceGraph, 1);

	    /*
	     * 3. call graph_critic, which has the side-effect
	     * of initializing the pair values
	     * (do this after drawing the graph, because graph_critic
	     * will mark the errors on the screen)
	     *
	     * XXX might it be better to make graph_has_cycles() mark
	     * pair nodes?
	     */
	    if (graph_critic (global.traceGraph) != 0) {
		msg = "trace graph has errors";
		goto fail;
	    }

	    /*
	     * 4. display the cost matrix.  
	     */
	    newmatrices (costs);

	    /*
	     * 5. initialize the histogram
	     */
	    hist_SetHosts (costs->hosts, num_machines, &epoch, &final);
	    return 0;

	default:
#if 0
	    fprintf (stderr, "unknown section %x\n",
		     section_hdr.section_type);
#endif
	    /* don't know what this is ... skip this section */
	    lseek (fd, section_hdr.record_count *
		   section_hdr.record_length, SEEK_CUR);
	}
    }
 fail:
    msg_Format ("Error reading trace file: %s\n", msg);
    return -2;
 eof:
    lseek (fd, 0L, SEEK_SET);
    trace_reset ();
    return -1;
}

void
enter_run_mode ()
{
    /* 1. clear the compose graph from the graph canvas */

    XClearWindow (global.display, XtWindow (graph_canvas));

    /* 2. allocate a new cost matrix, and display that */

    costs = cm_New ();
    newmatrices (costs);

    /* 3. (XXX) initialize histogram */
}

void
exit_run_mode ()
{
    /* 1. draw the compose graph */

    XClearWindow (global.display, XtWindow (graph_canvas));
    draw_graph (graph_canvas, global.graph, 0);

    /* 2. turn off the histogram (before freeing our cost matrix!) */

    hist_SetHosts (NULL, 0, &epoch, &epoch);
    hist_ClearHairLine ();
    close (histo_fd);
    histo_fd = -1;

    /* 3. draw the compose cost matrix, and nuke the scratch one */

    newmatrices (global.costMatrix);
    cm_Free (costs);
    costs = NULL;

    /*
     * 4. go back to compose mode
     *
     * XXX should prob. go to trace mode instead here.
     */

    set_mode (MODE_COMPOSE);
}

/*
 * read and process the next trace event.
 * return 0 if we see a FINISH event,
 * -1 if we hit end of file (no more events yet),
 * and 1 if we see a normal event.
 */

int
trace_read_next_event (fd)
int fd;
{
    if (read_event_rec (fd, &current_event) < 0)
	return -1;
    /* trace_update_state (1); */
    /*
     * if in execute mode, update the idea of the final event
     * (for the benefit of the histogram)
     */
    if (global.mode == MODE_EXECUTE) {
	if (current_event.sequence > final_seq) {
	    final_seq = current_event.sequence;
	    final_rec_addr = lseek (fd, 0L, SEEK_CUR)
		- sizeof (struct event_rec);
	    final.seconds = current_event.seconds;
	    final.msec = current_event.msec;
#if 1
	    hist_SetEndOfTime (&final);
	    hist_SetHairLine (&final, 1);
#endif
	}
    }
    if (current_event.event_type == EV_FINISH)
	return 0;
    return 1;
}


/*
 * translate between the machine index number used in the trace file,
 * to the machine index used by the cost matrix
 */

static int
cm_machine_number (m)
int m;
{
    int i;

    if (m >= num_machines || costs == NULL)
	return -1;

    for (i = 0; i < costs->nhost; ++i) {
	if (costs->hosts[i] == NULL)
	    continue;
	if (strcmp (machines[m].machine_name, costs->hosts[i]) == 0)
	    return i;
    }
    return -1;
}

/*
 * translate between the subroutine index number used in the trace file,
 * to the subr index used by the cost matrix
 */

static int
cm_subr_number (s)
int s;
{
    int i;

    for (i = 0; i < costs->nsub; ++i) {
	if (costs->subs[i] == NULL)
	    continue;
	if (strcmp (subrs[s].subr_name, costs->subs[i]) == 0)
	    return i;
    }
    return -1;
}

/*
 * update the machine, node, subr, and per-cell state from the
 * current trace event, and also the display.
 * 
 * + If 'dir' is 1, we're moving forward in time.
 * + if 'dir' is -1, we're moving backward in time.
 * + if 'dir' is 0, we're scanning forward to update the machine
 *   state, having just done a seek to the middle of the trace file.
 *   (don't update the display in this case)
 */

static struct event_time *elapsed_time ();

int
trace_update_state (dir)
int dir;
{
    int m = current_event.machine_number;
    int ni = find_node_index (current_event.node_number);
    int s = nodes[ni].subr_number;
    int n;
    rbNode tn;
    int cell = m * num_subrs + s;
    int machine_is_valid = (current_event.event_type == EV_STATE_CHANGE &&
			    current_event.state >= ST_RUNNING &&
			    m < num_machines);
    int subr_is_valid = (current_event.event_type == EV_STATE_CHANGE &&
			  nodes[ni].node_type == NT_NORMAL);
    int cmmn;			/* cost matrix machine number */
    int cmsn;			/* cost matrix subr number */
    struct event_time *delta;
    struct event_time now;

    int q;
    Node np;
    int incr;

    /*
     * don't try updating state if cost matrix has already been freed
     */
    if (costs == NULL)
	return;

    if (dir >= 0)
	incr = 1;
    if (dir < 0)
	incr = -1;

    if (machine_is_valid)
	cmmn = cm_machine_number (m);
    if (subr_is_valid)
	cmsn = cm_subr_number (s);
    if (machine_is_valid && subr_is_valid)
	cell = m * num_subrs + s;

    now.seconds = current_event.seconds;
    now.msec = current_event.msec;
    delta = elapsed_time (&epoch, &now);

    switch (current_event.event_type) {
    case EV_START:
#if 1
	fprintf (stderr, "[%d] %d.%03d START\n", 
		 current_event.sequence,
		 delta->seconds, delta->msec);
#endif
	break;
    case EV_STATE_CHANGE:
	switch (current_event.state) {
	case ST_NOT_BEGUN:
	    break;
	case ST_READY:
#if 1
	    fprintf (stderr, "[%d] %d.%03d READY node=%d.%d\n",
		     current_event.sequence,
		     delta->seconds, delta->msec,
		     current_event.node_number, current_event.inst_number);
#endif
	    break;
	case ST_RUNNING:
#if 1
	    fprintf (stderr, "[%d] %d.%03d RUNNING node=%d.%d\n",
		     current_event.sequence,
		     delta->seconds, delta->msec,
		     current_event.node_number, current_event.inst_number);
#endif
	    if (m >= num_machines)
		break;
	    per_cell_stats[cell].num_running += incr;
	    machine_stats[m].num_running += incr;
	    subr_stats[s].num_running += incr;
	    node_stats[ni].num_running += incr;
	    break;
	case ST_DONE:
#if 1
	    fprintf (stderr, "[%d] %d.%03d DONE node=%d.%d\n",
		     current_event.sequence,
		     delta->seconds, delta->msec,
		     current_event.node_number, current_event.inst_number);
#endif
	    if (m >= num_machines)
		break;
	    per_cell_stats[cell].num_running -= incr;
	    per_cell_stats[cell].num_idle += incr;
	    machine_stats[m].num_running -= incr;
	    machine_stats[m].num_idle += incr;
	    subr_stats[s].num_running -= incr;
	    subr_stats[s].num_idle += incr;
	    node_stats[ni].num_running -= incr;
	    node_stats[ni].num_idle += incr;
	    break;
	case ST_DEAD:
#if 1
	    fprintf (stderr, "[%d] %d.%03d DEAD node=%d.%d\n",
		     current_event.sequence,
		     delta->seconds, delta->msec,
		     current_event.node_number, current_event.inst_number);
#endif
	    if (m >= num_machines)
		break;
	    per_cell_stats[cell].num_idle -= incr;
	    machine_stats[m].num_idle -= incr;
	    subr_stats[s].num_idle -= incr;
	    node_stats[s].num_idle -= incr;
	    break;
	    break;
	}
	break;
    case EV_FINISH:
#if 1
	fprintf (stderr, "[%d] %d.%03d FINISH\n", 
		 current_event.sequence,
		 delta->seconds, delta->msec);
#endif
#if 0
	/*
	 * reset data structures
	 *
	 * XXX this breaks tracing backwards from the last event,
	 * because we lose all memory of the machine state.
	 * leave it out for now; eventually, get the machine
	 * state from the horizontal data.
	 */
	for (m = 0; m < num_machines; ++m) {
	    machine_stats[m].num_running = 0;
	    machine_stats[m].num_idle = 0;
	}
	for (s = 0; s < num_subrs; ++s) {
	    subr_stats[s].num_running = 0;
	    subr_stats[s].num_idle = 0;
	}
	for (n = 0; n < num_nodes; ++n) {
	    node_stats[n].num_running = 0;
	    node_stats[n].num_idle = 0;
	}
	for (m = 0; m < num_machines; ++m) {
	    for (s = 0; s < num_subrs; ++s) {
		per_cell_stats[m * num_subrs + s].num_running = 0;
		per_cell_stats[m * num_subrs + s].num_idle = 0;
	    }
	}

	/*
	 * mark all nodes as dead (&redraw)
	 */
	for (tn = rb_First (global.traceGraph->nlist);
	     !rb_Done (tn, global.traceGraph->nlist);
	     tn = rb_Next (tn)) {
	    Node np = rb_Value (tn);
	    
	    np->state = ST_DEAD;
	    np->numRunning = 0;
	    np->numIdle = 0;
	}
	draw_graph (graph_canvas, global.traceGraph, 1);

	/*
	 * mark all machines as done in cost matrix (&redraw)
	 */
	for (s = 0; s < costs->nsub; ++s)
	    for (m = 0; m < costs->nhost; ++m)
		cm_SetColor (costs, s, m, COLOR_NORMAL);
	for (s = 0; s < costs->nsub; ++s)
	    cm_SetColor (costs, s, -1, COLOR_NORMAL);
	for (m = 0; m < costs->nhost; ++m)
	    cm_SetColor (costs, 1, -m, COLOR_NORMAL);
	config_repaint_cells (-1, -1);
#endif
	break;
    case EV_VNODE:
#if 1
	fprintf (stderr, "[%d] %d.%03d VNODE node=%d.%d\n",
		 current_event.sequence,
		 delta->seconds, delta->msec,
		 current_event.node_number, current_event.inst_number);
#endif
	break;
    default:
	break;
    }

#if 0
    /*
     * now update from the horizontally-encoded state
     * (but only if we're scanning.)
     */
    if (dir == 0) {
	q = current_event.sequence % num_stats;
	if (q < num_machines ) {
	    /* q is machine number */
	    machine_stats[q].num_running = current_event.num_running;
	    machine_stats[q].num_idle = current_event.num_idle;
	    goto update_display;
	}
	q -= num_machines;
	if (q < num_subrs) {
	    /* q is subr number */
	    subr_stats[q].num_running = current_event.num_running;
	    subr_stats[q].num_idle = current_event.num_idle;
	    goto update_display;
	}
	q -= num_subrs;
	if (q < num_nodes) {
	    /* q is node number */
	    node_stats[q].num_running = current_event.num_running;
	    node_stats[q].num_idle = current_event.num_idle;
	    goto update_display;
	}
	q -= num_nodes;
	if (q < num_cells) {
	    /* q is cell number */
	    per_cell_stats[q].num_running = current_event.num_running;
	    per_cell_stats[q].num_idle = current_event.num_idle;
	    goto update_display;
	}
    }
#endif

 update_display:
    /*
     * update display, if we're going forward or backward,
     * but not if we're just scanning to get the current
     * machine state
     *
     * XXX dir is never 0 anymore
     */
    if (dir != 0) {
	int color;

	/*
	 * update the following as necessary:
	 *
	 * 1. node color
	 * 2. node running/idle count
	 */
	/*
	 * If one or more machines is running this node, mark it RUNNING, else
	 * if one or more machines has this node idle, mark it IDLE, else
	 * mark it according to whatever this state change says.
	 *
	 * XXX if we have done random access stuff, we lose the ability
	 * to reliably figure out READY or DONE node state.  This is because
	 * the trace file keeps only RUNNING and IDLE process counts for each
	 * node.
	 */
	np = gr_GetNodeByIdInst (global.traceGraph, current_event.node_number, 0);
	if (np) {
	    if (node_stats[ni].num_running > 0)
		np->state = ST_RUNNING;
	    else if (node_stats[ni].num_idle > 0)
		np->state = ST_DONE;
	    else {
		if (dir > 0)
		    np->state = current_event.state;
		else {
		    /* hack for retrograde stepping */
		    if (current_event.state == ST_NOT_BEGUN)
			np->state = ST_NOT_BEGUN;
		    else if (current_event.state == ST_READY)
			np->state = ST_NOT_BEGUN;
		    else if (current_event.state == ST_RUNNING)
			np->state = ST_READY;
		    else if (current_event.state == ST_DONE)
			np->state = ST_RUNNING;
		    else if (current_event.state == ST_DEAD)
			np->state = ST_DONE;
		}
	    }

	    /* XXX */
	    np->numIdle = node_stats[ni].num_idle;
	    np->numRunning = node_stats[ni].num_running;

	    draw_node (graph_canvas, np, -1);

	    /*
	     * if this is a special node, color in the opposite
	     * node also.
	     */
	    if (np->node_type != NODE_NORMAL && np->pair != NULL) {
		np->pair->state = np->state;
		draw_node (graph_canvas, np->pair, -1);
	    }
	}
	/*
	 * 3. machine color in cost matrix
	 * 4. subr color in cost matrix
	 * 5. cell color in cost matrix
	 */
	if (machine_is_valid) {
	    if (machine_stats[m].num_running > 0)
		color = COLOR_RUNNING;
	    else if (machine_stats[m].num_idle > 0)
		color = COLOR_IDLE;
	    else
		color = COLOR_NORMAL;
	    if (cm_SetColor (costs, cmmn, -1, color))
		config_repaint_cells (cmmn, -1);
	}

	if (subr_is_valid) {
	    if (subr_stats[s].num_running > 0)
		color = COLOR_RUNNING;
	    else if (subr_stats[s].num_idle > 0)
		color = COLOR_IDLE;
	    else
		color = COLOR_NORMAL;
	    if (cm_SetColor (costs, -1, cmsn, color))
		config_repaint_cells (-1, cmsn);
	}

	if (machine_is_valid && subr_is_valid) {
	    if (per_cell_stats[cell].num_running > 0)
		color = COLOR_RUNNING;
	    else if (per_cell_stats[cell].num_idle > 0)
		color = COLOR_IDLE;
	    else
		color = COLOR_NORMAL;
	    if (cm_SetColor (costs, cmmn, cmsn, color))
		config_repaint_cells (cmmn, cmsn);
	}
	/*
	 * update clock display
	 *
	 * if we're going in reverse, we need to display the time
	 * from the *previous* trace event.
	 */
	if (dir == -1) {
	    struct event_rec foo;
	    long x;

	    /*
	     * XXX for now various pieces of code depend on having
	     * the seek address be sane, so we have to leave it
	     * like we found it.
	     */
	    x = lseek (trace_fd, 0L, SEEK_CUR);
	    if (x >= epoch_rec_addr + 2 * sizeof (struct event_rec)) {
		lseek (trace_fd, -2 * sizeof (struct event_rec), SEEK_CUR);
		read_event_rec (trace_fd, &foo);
		trace_time.seconds = foo.seconds;
		trace_time.msec = foo.msec;
		global.traceSequence = foo.sequence;
		lseek (trace_fd, x, SEEK_SET);
	    }
	    else {
		trace_time.seconds = epoch.seconds;
		trace_time.msec = epoch.msec;
		global.traceSequence = 1;
	    }
	}
	else {
	    trace_time.seconds = current_event.seconds;
	    trace_time.msec = current_event.msec;
	    global.traceSequence = current_event.sequence;
	}
	if (global.mode == MODE_EXECUTE) {
	    delta = elapsed_time (&epoch, &trace_time);
	    global.traceClock = *delta;
	    draw_time (graph_canvas, &global.traceClock, global.traceSequence);
	}
#if 0
	/*
	 * update histogram
	 */
	if (global.mode == MODE_EXECUTE) {
	    hist_SetEndOfTime (&final);
	    hist_SetHairLine (&final, 1);
	}
#endif
    }
}

/*
 * this is called when leaving trace mode
 */

void
exit_trace_mode (x)
void *x;
{
    /*
     * 1. zero out the trace time
     * (this may not really be necessary)
     */
    global.traceClock.seconds = 0;
    global.traceClock.msec = 0;
    global.traceSequence = 0;

    /*
     * 2. go back to compose mode.  do this before redrawing the
     * graph, so the graph will be drawn in compose mode rather
     * than in trace mode.
     */
    set_mode (MODE_COMPOSE);

    /* 3. draw the compose graph */

    XClearWindow (global.display, XtWindow (graph_canvas));
    draw_graph (graph_canvas, global.graph, 0);

    /* 4. draw the compose cost matrix, and nuke the old one */

    newmatrices (global.costMatrix);
    cm_Free (costs);
    costs = NULL;

    /* 5. free up the file descriptors */

    close (trace_fd);
    trace_fd = -1;
    close (histo_fd);
    histo_fd = -1;

    /* 6. cleanup/redraw histogram */

    hist_SetHosts (NULL, 0, &epoch, &epoch);
    hist_ClearHairLine ();
}

int
trace_open_file (filename)
char *filename;
{
    if ((trace_fd = open (filename, O_RDONLY, 0)) < 0) {
	msg_Format ("Error: can't open %s: %s\n",
		    filename, strerror (errno));
	return EOF;
    }
    if ((histo_fd = open (filename, O_RDONLY, 0)) < 0) {
	msg_Format ("Error: can't open %s: %s\n",
		    filename, strerror (errno));
	close (trace_fd);
	trace_fd = -1;
	return EOF;
    }
    return trace_fd;
}


void
trace_close_file (fd)
{
    close (fd);
    close (trace_fd);
    /* don't close(histo_fd) until exiting run mode! */
}

/*
 * this is called to load a trace file
 */

void
trace_load (filename)
char *filename;
{
    if ((trace_fd = open (filename, O_RDONLY, 0)) < 0) {
	msg_Format ("Error: can't open %s: %s\n",
		    filename, strerror (errno));
	return;
    }
    if ((histo_fd = open (filename, O_RDONLY, 0)) < 0) {
	msg_Format ("Error: can't open %s: %s\n",
		    filename, strerror (errno));
	return;
    }

    /* clear the compose graph from the graph canvas */

    XClearWindow (global.display, XtWindow (graph_canvas));

    /*
     * allocate a new cost matrix, and display that
     * (this must happen before reading the file)
     */

    costs = cm_New ();
    newmatrices (costs);

    set_mode (MODE_TRACE);

    if (trace_read_header (trace_fd) != 0) {
	exit_trace_mode ((void *) NULL);
	return;
    }

    msg_Format ("trace file %s loaded.\n", filename);
}

/*
 * stop tracing
 */

static void
stop_trace ()
{
    extern Widget trace_play_button;

    XtVaSetValues (trace_play_button,
		   XtNstate, False,
		   NULL);
    trace_stopped = 1;
    if (trace_timer_id)
	XtRemoveTimeOut (trace_timer_id);
}

static void
trace_timer (mode, y)
XtPointer mode;
XtIntervalId *y;
{
    static struct event_time this_event_time;
    static struct event_time pseudo_now;
    int status;

    if (trace_stopped)
	return;

    trace_timer_id = (XtIntervalId) 0;

    switch ((enum trace_timer_mode) mode) {
    case TIMER_ALARM:		/* timer went off */
	/*
	 * NB: when the timer goes off, we don't check trace_stopped
	 * until we've processed at least one event.  This is because
	 * we've already read an event into thisEvent, but we haven't
	 * yet processed it.
	 */
	break;
    case FIRST_EVENT:		/* called by start_trace ()
				   to prime things */
	if (trace_read_next_event (trace_fd) <= 0)
	    goto eof;
	this_event_time.seconds = current_event.seconds;
	this_event_time.msec = current_event.msec;
	pseudo_now = this_event_time;
	break;
    }
    /*
     * handle all events up to and including the trace time
     */
    do {
	struct event_time *delta;
	long delay_time;
	long msec;

	/*
	 * if it's too early to trace this event, update hairline
	 * and elapsed time indicator, and wait a while.
	 */
	if (after (&this_event_time, &pseudo_now)) {
	    delta = elapsed_time (&epoch, &pseudo_now);
	    global.traceClock = *delta;
	    draw_time (graph_canvas, &global.traceClock, global.traceSequence);
	    hist_SetHairLine (&pseudo_now, 1);
	    /*
	     * calculate time to delay until next event is traced.
	     * if the histogram window is visible, this wants to
	     * be the amount of scaled time equal to a pixel; 
	     * otherwise, this should be the scaled time until another 
	     * event takes place.
	     * (Now histogram is always visible -- maybe the test
	     * should be if the hairline is visible.)
	     */

	    if (hist_HairLineIsVisible ()) {
		if (hist_MsecPerPixel (&msec) < 0)
		    msec = 1000;
	    }
	    else {
		delta = elapsed_time (&pseudo_now, &this_event_time);
		msec = delta->seconds * 1000 + delta->msec;
	    }
	    delay_time = msec * time_scale;

	    /*
	     * if delay_time is longer than one second, reset msec and
	     * delay_time so that we will wake up within a second.
	     * This is so the "stop" button will work within a
	     * reasonable time.  If possible, make the clock stop on an
	     * even second boundary.  This is because it looks wierd
	     * to watch the middle digits of the clock tick by but always
	     * with the same lower order digits.
	     */
	    if (delay_time > 1000) {
		delta = elapsed_time (&epoch, &pseudo_now);
		msec = 1000 - delta->msec;
		if (msec == 0)
		    msec = 1000;
		delay_time = msec * time_scale;
		if (delay_time > 1000) {
		    msec = 1000 / time_scale;
		    delay_time = msec * time_scale;
		}
	    }
	    /*
	     * on the other hand, if delay_time is too short (< 10msec),
	     * increase the amount of clock time to be elapsed until
	     * delay_time is long enough for a minimum delay.
	     */
	    else if (delay_time < 10) {
		msec = (int) ((10.0 / time_scale) + 0.5);
		delay_time = msec * time_scale;
	    }

	    /*
	     * figure out what time it will be when we wake up.
	     */
	    pseudo_now.msec += msec;
	    pseudo_now.seconds += pseudo_now.msec / 1000;
	    pseudo_now.msec = pseudo_now.msec % 1000;
	    
	    /*
	     * set the timer to go off then.
	     */
#if 0
	    fprintf (stderr, "waiting %d msec\n", delay_time);
#endif
	    trace_timer_id = XtAppAddTimeOut (global.context,
					      delay_time, trace_timer,
					      (XtPointer) TIMER_ALARM);
	    return;
	}
	trace_update_state (1);
	delta = elapsed_time (&epoch, &trace_time);
	global.traceClock = *delta;
	draw_time (graph_canvas, &global.traceClock, global.traceSequence);
	if (trace_stopped)
	    return;
	if (trace_read_next_event (trace_fd) <= 0)
	    goto eof;
	this_event_time.seconds = current_event.seconds;
	this_event_time.msec = this_event_time.msec;
    } while (1);

eof:
    msg_Format ("end of trace file\n", global.traceFile);
    stop_trace ();
#if 0
    set_togg (fwdButton, 0);
    set_togg (stopButton, 1);
#endif
}
    
static void
start_trace ()
{
    if (trace_stopped) {
	/*
	 * do the first tick manually. After that, trace_timer()
	 * will schedule the next tick according to the value of
	 * time_scale
	 */
	trace_stopped = 0;
	trace_timer ((XtPointer) FIRST_EVENT, 0);
    }
}

void
trace_rwd_button (w, cli, cd)
Widget w;
XtPointer cli, cd;
{
    int m, s, n;
    rbNode tn;

    stop_trace ();
    trace_direction = FORWARD;

    /* reset file pointer */
    lseek (trace_fd, epoch_rec_addr, SEEK_SET);

    /* reset data structures and redraw cost matrix */
    for (m = 0; m < num_machines; ++m) {
	machine_stats[m].num_running = 0;
	machine_stats[m].num_idle = 0;
    }
    for (s = 0; s < num_subrs; ++s) {
	subr_stats[s].num_running = 0;
	subr_stats[s].num_idle = 0;
    }
    for (n = 0; n < num_nodes; ++n) {
	node_stats[n].num_running = 0;
	node_stats[n].num_idle = 0;
    }
    for (m = 0; m < num_machines; ++m) {
	for (s = 0; s < num_subrs; ++s) {
	    per_cell_stats[m * num_subrs + s].num_running = 0;
	    per_cell_stats[m * num_subrs + s].num_idle = 0;
	}
    }
    for (s = 0; s < costs->nsub; ++s)
	cm_SetColor (costs, -1, s, COLOR_NORMAL);
    for (m = 0; m < costs->nhost; ++m)
	cm_SetColor (costs, m, -1, COLOR_NORMAL);
    for (s = 0; s < costs->nsub; ++s)
	for (m = 0; m < costs->nhost; ++m)
    	    cm_SetColor (costs, m, s, COLOR_NORMAL);
    config_repaint_cells (-1, -1);

    /* reset and redraw graph */
    for (tn = rb_First (global.traceGraph->nlist);
	 !rb_Done (tn, global.traceGraph->nlist);
	 tn = rb_Next (tn)) {
	Node np = rb_Value (tn);

	np->state = ST_NOT_BEGUN;
	np->numRunning = 0;
	np->numIdle = 0;
    }
    draw_graph (graph_canvas, global.traceGraph, 1);

    /* reset and redraw clock */
    global.traceClock.seconds = 0;
    global.traceClock.msec = 0;
    global.traceSequence = epoch_seq;
    draw_time (graph_canvas, &global.traceClock, global.traceSequence);

    /* set hairline */
    hist_SetHairLine (&epoch, 0);
}

/*
 * XXX don't allow user to step before BOF
 */

void
trace_rwdstep_button (w, cli, cd)
Widget w;
XtPointer cli, cd;
{
    long x;
    struct event_time *delta;

    /*
     * if trace_direction is forward, we need to re-process
     * the last record read, but this time in the reverse
     * direction.  so backspace over just one record.
     * if we're already reversing, backspace two records.
     */
    if (trace_direction == FORWARD)
	x = lseek (trace_fd, - (sizeof (struct event_rec)), SEEK_CUR);
    else
	x = lseek (trace_fd, -2L * sizeof (struct event_rec), SEEK_CUR);
    trace_direction = REVERSE;
    stop_trace ();

    if (x < 0)
	return;
    if (x < epoch_rec_addr) {
	lseek (trace_fd, epoch_rec_addr, SEEK_SET);
	return;
    }
    
    if (read_event_rec (trace_fd, &current_event) < 0) {
	msg_Format ("can't read event\n");
	return;
    }
    trace_update_state (-1);
    delta = elapsed_time (&epoch, &trace_time);
    global.traceClock = *delta;
    draw_time (graph_canvas, &global.traceClock, global.traceSequence);

    /* set hairline */
    hist_SetHairLine (&trace_time, 0);
}

void
trace_stop_button (w, cli, cd)
Widget w;
XtPointer cli, cd;
{
    struct event_time *delta;

    stop_trace ();
    delta = elapsed_time (&epoch, &trace_time);
    global.traceClock = *delta;
    draw_time (graph_canvas, &global.traceClock, global.traceSequence);

    /* set hairline */
    hist_SetHairLine (&trace_time, 0);
}

/*
 * XXX don't allow user to step past EOF
 */

void
trace_fwdstep_button (w, cli, cd)
Widget w;
XtPointer cli, cd;
{
    struct event_time *delta;

    /*
     * if changing direction, re-read the last record read
     */
    if (trace_direction == REVERSE) {
	int x;

	x = lseek (trace_fd, -(sizeof(struct event_rec)), SEEK_CUR);
	if (x < epoch_rec_addr)
	    lseek (trace_fd, epoch_rec_addr, SEEK_SET);
    }
    trace_direction = FORWARD;
    stop_trace ();
    if (read_event_rec (trace_fd, &current_event) < 0) {
	msg_Format ("end of trace file\n");
	return;
    }
    trace_update_state (1);
    delta = elapsed_time (&epoch, &trace_time);
    global.traceClock = *delta;
    draw_time (graph_canvas, &global.traceClock, global.traceSequence);
    /* set hairline */
    hist_SetHairLine (&trace_time, 0);
}

void
trace_fwd_button (w, cli, cd)
Widget w;
XtPointer cli, cd;
{
    /*
     * turn button "on"
     */
    XtVaSetValues (w,
		   XtNstate, True,
		   NULL);
    /*
     * if changing direction, re-read the last record read
     */
    if (trace_direction == REVERSE) {
	int x;

	x = lseek (trace_fd, -(sizeof(struct event_rec)), SEEK_CUR);
	if (x < epoch_rec_addr)
	    lseek (trace_fd, epoch_rec_addr, SEEK_SET);
    }
    trace_direction = FORWARD;
    
    /*
     * Hack to detect end of tracefile and do something reasonable.
     * Go ahead and read one event.  If it succeeds, display the
     * results; otherwise, don't go into continuous trace mode.
     */

    if (trace_read_next_event (trace_fd) <= 0) {
	msg_Format ("end of trace file\n");
	return;
    }
    trace_update_state (1);
    start_trace ();
}

void
trace_speed_jump (w, cli, cd)
Widget w;
XtPointer cli, cd;
{
    float f;

    f = *(float*)cd;
    /*
     * need a log scale
     * if slider is at the far left, time_scale = 100
     * if slider is in the center, time_scale = 1.0
     * if slider is at the far right, time_scale = .01
     * total magnitude difference = 10 ** 4;
     */

#if 0
    /* this is for left-to-right */
    time_scale = pow (10.0, (1.0 - f) * 4.0 - 2.0);
#else
    /* this is for down-to-up */
    time_scale = pow (10.0, f * 4.0 - 2.0);
#endif
#if 0
    fprintf (stderr, "time scale = %f\n", time_scale);
#endif
}

void
trace_init (w)
Widget w;
{
    graph_canvas = w;
    /*
     * XXX should set up enter/exit mode hooks for run/trace mode
     */
}

static void
histo_update_state (p)
struct event_rec *p;
{
    int m = p->machine_number;
    int machine_is_valid = (p->event_type == EV_STATE_CHANGE &&
			    p->state >= ST_RUNNING &&
			    m < num_machines);
    int q;

    if (machine_is_valid) {
	switch (p->event_type) {
	case EV_STATE_CHANGE:
	    switch (p->state) {
	    case ST_RUNNING:
		histo_machine_stats[m].num_running++;
		break;
	    case ST_DONE:
		histo_machine_stats[m].num_idle++;
		histo_machine_stats[m].num_running--;
		break;
	    case ST_DEAD:
		histo_machine_stats[m].num_idle--;
		break;
	    }
	}
    }
#if 0
    if (m < num_machines) {
	fprintf (stderr, "(#%d) %s(%d) %dr,%di\n",
		 p->sequence, machines[m].machine_name, m,
		 histo_machine_stats[m].num_running,
		 histo_machine_stats[m].num_idle);
    }
#endif
    /*
     * now update from the horizontally-encoded state
     */
    q = p->sequence % num_stats;
    if (q < num_machines ) {
	/* q is machine number */
	histo_machine_stats[q].num_running = p->num_running;
	histo_machine_stats[q].num_idle = p->num_idle;
#if 0
	if (q != m) {
	    fprintf (stderr, "<#%d> %s(%d) %dr,%di\n",
		     p->sequence, machines[q].machine_name, q,
		     histo_machine_stats[q].num_running,
		     histo_machine_stats[q].num_idle);
	}
#endif
    }
}

/* called from histo.c */
void
trace_GetState (savedState, numHosts, mintime)
struct histogram_event *savedState;
int numHosts;
struct event_time *mintime;
{
    struct event_rec buf;
    struct event_time evt;
    struct event_time when;
    int seq0, seq1;
    int m;

#if 0
    {
	struct event_time *delta;
	delta = elapsed_time (&epoch, mintime);
	
	fprintf (stderr, "trace_GetState (%d.%03d)\n",
		 delta->seconds, delta->msec);
    }
#endif
    /*
     * 1. clear out all histogram machine state
     */
    for (m = 0; m < num_machines; ++m) {
	histo_machine_stats[m].num_running = 0;
	histo_machine_stats[m].num_idle = 0;
    }

    /*
     * 2. get the event at or before 'mintime'
     * (so we can find out its sequence number)
     */

    if (read_event_by_time (histo_fd, &buf, mintime) < 0)
	return;
    seq1 = buf.sequence;

    /*
     * 3. now, go num_stats records back from there
     * and read sequentially, updating the state.
     */

    seq0 = buf.sequence - num_stats;
    if (seq0 < epoch_seq)
	seq0 = epoch_seq;
    read_event_by_seq (histo_fd, &buf, seq0);
    do {
	histo_update_state (&buf);
	when.seconds = buf.seconds;
	when.msec = buf.msec;
	if (read_event_rec (histo_fd, &buf) < 0)
	    break;
	if (buf.event_type == EV_FINISH)
	    break;
#if 0
	fprintf (stderr, "buf.sequence = %d seq1 = %d\n",
		 buf.sequence, seq1);
#endif
    } while (buf.sequence <= seq1);

    /*
     * 4. set things up so that next record to be read is the
     * first event after mintime
     */
    evt.seconds = buf.seconds;
    evt.msec = buf.msec;
    while (after (&evt, mintime)) {
	lseek (histo_fd, -2 * sizeof (struct event_rec), SEEK_CUR);
	read_event_rec (histo_fd, &buf);
	evt.seconds = buf.seconds;
	evt.msec = buf.msec;
    }

    /*
     * 5. fill in the blanks
     */
    for (m = 0; m < numHosts; ++m) {
	int cmmn = cm_machine_number (m);

	savedState[cmmn].hostno = cmmn;
	if (histo_machine_stats[m].num_running > 0)
	    savedState[cmmn].state = HISTO_RUNNING;
	else if (histo_machine_stats[m].num_idle > 0)
	    savedState[cmmn].state = HISTO_IDLE;
	else
	    savedState[cmmn].state = HISTO_OFF;
	savedState[cmmn].time.seconds = when.seconds;
	savedState[cmmn].time.msec = when.msec;
#if 0
	{
	    struct event_time *delta = elapsed_time (&epoch, &when);
	    fprintf (stderr, "%d.%03d state[%d] = %s == %d\n",
		     delta->seconds, delta->msec,
		     cmmn, costs->hosts[cmmn],
		     savedState[cmmn].state);
	}
#endif
    }

}


/*
 * return -1 when done.
 */
int
trace_NextEvent (nextEvent)
struct histogram_event *nextEvent;
{
    struct event_rec buf;

    while (1) {
	if (read_event_rec (histo_fd, &buf) < 0)
	    return -1;
	if (buf.event_type == EV_FINISH)
	    return -1;
	histo_update_state (&buf);
	if (buf.event_type == EV_STATE_CHANGE && buf.state >= ST_RUNNING) {
	    int m = buf.machine_number;

	    if (m >= num_machines) /* "null" process brain damage */
		continue;

	    nextEvent->time.seconds = buf.seconds;
	    nextEvent->time.msec = buf.msec;
	    nextEvent->hostno = cm_machine_number (m);

	    if (histo_machine_stats[m].num_running > 0)
		nextEvent->state = HISTO_RUNNING;
	    else if (histo_machine_stats[m].num_idle > 0)
		nextEvent->state = HISTO_IDLE;
	    else
		nextEvent->state = HISTO_OFF;
#if 0
	    {
		struct event_time *delta =
		    elapsed_time (&epoch, &(nextEvent->time));
		fprintf (stderr, "event %d.%03d %d (%s)=>%d\n",
			 delta->seconds, delta->msec,
			 nextEvent->hostno,
			 costs->hosts[nextEvent->hostno], nextEvent->state);
	    }
#endif
	    return 0;
	}
    }
}


void
trace_set_hairline ()
{
    struct event_time now;
    struct event_time *delta;
    struct event_time simtime;

    if (global.mode != MODE_EXECUTE)
	return;

    /*
     * compute the difference between the wall time when we read
     * the epoch trace record and the current wall time.
     */
    get_wall_time (&now);
    delta = elapsed_time (&epoch_wall_time, &now);
    /*
     * add that to the epoch time and indicate the result on the
     * histogram.
     */
    simtime.msec = epoch.msec + delta->msec;
    simtime.seconds = epoch.seconds + delta->seconds;
    if (simtime.msec > 1000) {
	simtime.msec -= 1000;
	simtime.seconds++;
    }
    hist_SetEndOfTime (&simtime);
    hist_SetHairLine (&simtime, 1);
}
