/*
 * stuff to futz with text windows
 *
 * $Id: text_sink.c,v 1.2 1994/06/11 20:46:32 moore Exp $
 *
 * $Log: text_sink.c,v $
 * Revision 1.2  1994/06/11  20:46:32  moore
 * initialize buffer to start with a NUL so we don't see trash.
 *
 * Revision 1.1  1994/02/17  21:36:58  moore
 * Initial revision
 *
 */

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xos.h>

#ifdef TS_DEBUG
#include <stdio.h>
#endif


#include "text_sink.h"

#include <Malloc.h>
#include <String.h>

#ifdef USE_ASCII_TEXT_WIDGET
#include <X11/Xaw/AsciiText.h>
#else
#include <X11/IntrinsicP.h>
#include "Canvas.h"
#include "CanvasP.h"
#endif

/*
 * a text buffer consists of a chunk of heap-allocated memory.
 * 'chunkStart' points to the beginning; it is 'chunkSize' bytes
 * long.
 * 
 * A pointer 'bufStart' points to the beginning of the circular
 * buffer, whichis 'bufSize' characters long.  Initially bufSize
 * is zero, it can be no larger than chunkSize - 1.
 *
 * the "top half" of the buffer is the portion from bufStart to
 * either the end of the chunk, or bufStart + bufSize - 1, whichever
 * comes first.
 *
 * if bufSize is bigger than the "top half" of the buffer, the "bottom
 * half" of the buffer extends from chunkStart to
 * chunkStart + (the end of the chunk - bufStart) - 1.
 */

struct text_sink {
    Widget wid;
#ifdef USE_ASCII_TEXT_WIDGET
    char *buf;
    int bufsize;
    int current_length;
#else
    char *chunkStart;
    int chunkSize;
    char *bufStart;
    int bufSize;
    GC gc;
#endif
};


#ifndef USE_ASCII_TEXT_WIDGET

/*
 * pointer just past the end of the chunk (points at the
 * sentinel newline after the buffer)
 */
#define CHUNK_END(b) ((b)->chunkStart + (b)->chunkSize)
/*
 * size of the "top half" of the chunk -- the space between
 * the start of the buffer and the end of the chunk
 */
#define TOP_HALF_SIZE(b)  (CHUNK_END(b) - (b)->bufStart)
/*
 * largest number of bytes we can put in the buffer.
 */
#define MAX_BUF_SIZE(b) ((b)->chunkSize - 1)
/*
 * amount of space left in the buffer
 */
#define ROOM(b) (MAX_BUF_SIZE(b) - (b)->bufSize)


/*
 * copy 'count' bytes from src to dst.
 * return the number of newlines seen.
 */

static int
bufcpy (dst, src, count)
register char *dst;
register char *src;
register int count;
{
    register int newline_count = 0;

    if (count <= 0)
	return 0;
    do {
	if ((*dst++ = *src++) == '\n')
	    ++newline_count;
    } while (--count > 0);

    return newline_count;
}

static void
ts_make_gc (ts)
struct text_sink *ts;
{
    XGCValues foo;
    CanvasWidget cw = (CanvasWidget) ts->wid;

    foo.foreground = cw->canvas.foreground;
    foo.background = cw->core.background_pixel;
    foo.font = cw->canvas.fontinfo->fid;
    foo.function = GXcopy;
    ts->gc = XCreateGC (XtDisplay (ts->wid), XtWindow (ts->wid),
			GCFunction|GCFont|GCForeground|GCBackground, &foo);
}

#define WIN_HEIGHT(ts) height_of((ts)->wid)
#define WIN_WIDTH(ts) width_of((ts)->wid)
#define FONT_ASCENT(ts) (((CanvasWidget) (ts)->wid)->canvas.fontinfo->ascent)
#define FONT_DESCENT(ts) (((CanvasWidget) (ts)->wid)->canvas.fontinfo->descent)
#define LINE_HEIGHT(ts) (FONT_ASCENT(ts)+FONT_DESCENT(ts))
#define BASELINE(ts, line) (WIN_HEIGHT (ts) \
			    - (line) * LINE_HEIGHT(ts) \
			    - FONT_DESCENT(ts))


static void
scroll_up (ts, nlines)
struct text_sink *ts;
int nlines;
{
    Widget wid = ts->wid;
    Display *dis = XtDisplay (wid);
				/* display to draw on */
    Window win = XtWindow (wid);
				/* window to draw on */
    CanvasWidget cw = (CanvasWidget) wid;
    XFontStruct *fs = cw->canvas.fontinfo;
    int win_height = WIN_HEIGHT(ts);
    int win_width = WIN_WIDTH(ts);
    int line_height = LINE_HEIGHT(ts);

#ifdef TS_DEBUG
    fprintf (stderr, "scroll_up (ts, %d)\n", nlines);
#endif

    /*
     * create a gc to draw with
     */
    if (ts->gc == 0)
	ts_make_gc (ts);

    if (nlines <= 0)
	return;
#ifdef TS_DEBUG
    fprintf (stderr,
	     "XCopyArea (x0=%d, y0=%d, width=%d, height=%d, x1=%d, y1=%d\n",
	     0, win_height, win_width, win_height - nlines * line_height,
	     0, nlines * line_height);
#endif
    XCopyArea (dis, win, win, ts->gc,
	       0,					/* src x */
	       win_height,				/* src y */
	       win_width,				/* width */
	       win_height - nlines * line_height,	/* height */
	       0,					/* dst x */
	       nlines * line_height);			/* dst y */
}

static void
display_line (ts, line, buf, size, buf2, size2)
struct text_sink *ts;
int line;
char *buf;
int size;
char *buf2;
int size2;
{
    int x, y;			/* initial display coordinates */
    Widget wid = ts->wid;
    Display *dis = XtDisplay (wid);
				/* display to draw on */
    Window win = XtWindow (wid);
				/* window to draw on */
    CanvasWidget cw = (CanvasWidget) wid;
    XFontStruct *fs = cw->canvas.fontinfo;

#ifdef TS_DEBUG
    fprintf (stderr, "display_line (ts, %d, %.*s, %d, %.*s, %d)\n",
	     line,
	     size, buf, size,
	     size2, buf2, size2);
#endif

    if (ts->gc == 0)
	ts_make_gc (ts);

    /*
     * compute initial x, y coordinates
     * y coordiante is of the baseline;
     * x is of the left edge of the bounding box for the leftmost character
     */
    x = 0;
    y = BASELINE(ts, line);
    /*
     * draw the first chunk of text, figure out how wide it was,
     * and bump x by that amount.
     */
    if (size > 0) {
#ifdef TS_DEBUG
	fprintf (stderr,
		 "XDrawImageString (x=%d, y=%d, buf=\"%.*s\", size=%d)\n",
		 x, y, size, buf, size);
#endif
	XDrawImageString (dis, win, ts->gc, x, y, buf, size);
	x += XTextWidth (fs, buf, size);
    }
    /*
     * if there's a second chunk of text, draw it and bump x again.
     */
    if (size2 > 0) {
#ifdef TS_DEBUG
	fprintf (stderr,
		 "XDrawImageString (x=%d, y=%d, buf2=\"%.*s\", size2=%d)\n",
		 x, y, size2, buf2, size2);
#endif
	XDrawImageString (dis, win, ts->gc, x, y, buf2, size2);
	x += XTextWidth (fs, buf2, size2);
    }
    /*
     * now clear to end-of-line.  Note that the 'y' coordinates need
     * to take into account the vertical extent of the bounding box.
     * x ranges from here to end-of-line.
     */
#ifdef TS_DEBUG
    fprintf (stderr, "XClearArea (x0=%d, y0=%d, width=%d, height=%d)\n",
	     x, y - fs->ascent, width_of(wid) - x, fs->ascent + fs->descent);
#endif
    XClearArea (dis, win,
		x, y - fs->ascent,		/* upper left (x,y) */
		width_of (wid) - x, 		/* width */
		fs->ascent + fs->descent,	/* height */
		False);				/* no exposures */
}

#define CEIL(a,b) (((a) + ((b) - 1)) / (b))

/*
 * draw the contents of the buffer on the display.
 * if 'scroll_count' >= 0, scroll that many lines up before doing so,
 * and only redraw the bottom lines.  if 'scroll_count' < 0, redraw
 * the entire display
 */

static void
redisplay (ts, scroll_count)
struct text_sink *ts;
int scroll_count;
{
    /*
     * XXX eventually, wrap this whole thing into a widget
     */
    CanvasWidget cw = (CanvasWidget) ts->wid;
				/* alias pointer to the private
				 * state of a canvas widget */
    int window_height_lines;	/* height of the window in lines */
    int current_line;		/* line # of the line currently
				 * being redisplayed */
    int partial_line_length;	/* number of characters left over on
				 * the last partial line from the top
				 * half of the buffer */
    char *eolptr, *bolptr;	/* {end,beginning}-of-line pointers
				 * these should always point to newlines
				 * when redisplay is called */
    int top_line, bottom_line;	/* top and bottom lines to be
				 * re-displayed */

#ifdef TS_DEBUG
    fprintf (stderr, "redisplay (ts, scroll_count=%d)\n", scroll_count);
#endif

    window_height_lines =
	CEIL( height_of (ts->wid),
	     (cw->canvas.fontinfo)->ascent + (cw->canvas.fontinfo)->descent);

    /*
     * If scroll_count is nonnegative and less than the number of
     * lines on the screen, scroll up that many lines and redisplay
     * only the new lines.
     * 
     * otherwise, redisplay the whole window.
     *
     * XXX scroll_up may generate expose events, which we need
     * to handle in such a way that they will work even if we
     * have scrolled again since then.
     */
    if (0 <= scroll_count && scroll_count < window_height_lines) {
	scroll_up (ts->wid, scroll_count);
	bottom_line = 0;
	top_line = bottom_line + scroll_count - 1;
    }
    else {
	top_line = window_height_lines;
	bottom_line = 0;
    }

    /*
     * start drawing from the bottom of the window and go up; this
     * is easier (since the bottom line is always aligned with the
     * bottom of the window) and should be more efficient.
     */
    current_line = 0;
    partial_line_length = 0;

    /*
     * if there is any text in the bottom half of the buffer, draw
     * it.  There may also be a line split between the top half
     * and the bottom half; if so, remember how long the "bottom"
     * end of the line was.
     */
    if (ts->bufSize > TOP_HALF_SIZE(ts)) {
	/*
	 * start at the end of the buffer.
	 */
	eolptr = ts->chunkStart + (ts->bufSize - TOP_HALF_SIZE(ts));
	while (current_line < window_height_lines) {
	    /*
	     * find the beginning of this line.  This will
	     * be the character immediately following the
	     * previouly occuring newline.  Note: if the
	     * line is completely 'blank' the length of the
	     * line will be zero.
	     *
	     * XXX rather than checking for chunkStart,
	     * use a sentinel
	     */
	    for (bolptr = eolptr - 1; *bolptr != '\n'; --bolptr)
		if (bolptr <= ts->chunkStart)
		    break;
	    /*
	     * if we did not find a newline, the start of this line
	     * is before the beginning of the chunk.  Remember how long
	     * the tail was, and break out of this loop to fall into
	     * the "top half" display loop.
	     *
	     * Also set eolptr to point to the last char of the top half.
	     * (XXX is this right?)
	     */
	    if (*bolptr != '\n') {
		partial_line_length = eolptr - ts->chunkStart;
		eolptr = CHUNK_END(ts);
		break;
	    }
	    /*
	     * otherwise, display this line, update the
	     * line count and end of line pointer, and continue
	     * with the loop.
	     */
	    else {
		if (bottom_line <= current_line && current_line <= top_line)
		    display_line (ts, current_line,
				  bolptr + 1, eolptr - bolptr - 1,
				  (char *) 0, 0);
		++current_line;
		eolptr = bolptr;
	    }
	}
    }
    else {
	/*
	 * calculate end of buffer for "bottom half"
	 */
	eolptr = ts->bufStart + ts->bufSize;
    }

    /*
     * "top half" display loop.  this is much like the one above
     * except that we terminate the search for BOL at bufStart
     * rather than chunkStart, and we have to deal with the
     * possible left-over partial line from above.  Also, we
     * always display the top line of the buffer, even if
     * it is truncated.  (there's no way to tell the difference
     * anyway, since the buffer need not start with a newline).
     *
     * XXX use sentinel rather than checking for bufStart
     */
    while (current_line < window_height_lines) {
	for (bolptr = eolptr - 1; *bolptr != '\n'; --bolptr)
	    if (bolptr < ts->bufStart)
		break;
	if (bottom_line <= current_line && current_line <= top_line)
	    display_line (ts, current_line,
			  bolptr + 1, eolptr - bolptr - 1,
			  ts->chunkStart, partial_line_length);
	partial_line_length = 0;
	eolptr = bolptr;
	++current_line;
    }
}

static void
ts_redraw (w, client_data, min_x, min_y, max_x, max_y)
Widget w;
XtPointer client_data;
int min_x, min_y;
int max_x, max_y;
{
    struct text_sink *ts = (struct text_sink *) client_data;
    redisplay (ts, -1);
}

#define min(a,b) ((a) < (b) ? (a) : (b))
#define max(a,b) ((a) > (b) ? (a) : (b))

static void
ts_expose (w, client_data, xevent, continue_to_dispatch)
Widget w;
XtPointer client_data;
XEvent *xevent;
Boolean *continue_to_dispatch;
{
    XEvent xev;
    XExposeEvent xexp;
    XGraphicsExposeEvent xgrexp;
    int min_x, max_x, min_y, max_y;

    switch (xevent->type) {
    case Expose:
	xexp = xevent->xexpose;
#ifdef TS_DEBUG
	fprintf (stderr, "Expose %x %d %d %d %d count=%d\n",
		 w, xexp.x, xexp.y, xexp.width, xexp.height, xexp.count);
#endif
	min_x = xexp.x;
	max_x = xexp.x + xexp.width;
	min_y = xexp.y;
	max_y = xexp.y + xexp.height;
	while (xexp.count > 0) {
	    XNextEvent (XtDisplay (w), &xev);
	    xexp = xev.xexpose;
#ifdef TS_DEBUG
	    fprintf (stderr, "Expose %x %d %d %d %d count=%d\n",
		     w, xexp.x, xexp.y, xexp.width, xexp.height, xexp.count);
#endif
	    min_x = min (min_x, xexp.x);
	    max_x = max (max_x, xexp.x + xexp.width);
	    min_y = min (min_y, xexp.y);
	    max_y = max (max_y, xexp.y + xexp.height);
	}
	ts_redraw (w, client_data, min_x, min_y, max_x, max_y);
	return;
    case GraphicsExpose:
	xgrexp = xevent->xgraphicsexpose;
#ifdef TS_DEBUG
	fprintf (stderr, "GraphicsExpose %x %d %d %d %d count=%d\n",
		 w, xgrexp.x, xgrexp.y, xgrexp.width, xgrexp.height,
		 xgrexp.count);
#endif
	min_x = xgrexp.x;
	max_x = xgrexp.x + xgrexp.width;
	min_y = xgrexp.y;
	max_y = xgrexp.y + xgrexp.height;
	while (xgrexp.count > 0) {
	    XNextEvent (XtDisplay (w), &xev);
	    xgrexp = xev.xgraphicsexpose;
#ifdef TS_DEBUG
	    fprintf (stderr, "GraphicsExpose %x %d %d %d %d count=%d\n",
		     w, xgrexp.x, xgrexp.y, xgrexp.width, xgrexp.height,
		     xgrexp.count);
#endif
	    min_x = min (min_x, xgrexp.x);
	    max_x = max (max_x, xgrexp.x + xgrexp.width);
	    min_y = min (min_y, xgrexp.y);
	    max_y = max (max_y, xgrexp.y + xgrexp.height);
	}
	ts_redraw (w, client_data, min_x, min_y, max_x, max_y);
	return;
    case ConfigureNotify:
	XClearWindow (XtDisplay (w), XtWindow (w));
	ts_redraw (w, client_data, 0, 0, width_of(w), height_of(w));
	return;

    case MapNotify:
    case NoExpose:
    default:
	return;
    }
}


/*
 * create a text-sink, and associate it with a widget.
 * 'size' is the number of bytes in the buffer.
 */

struct text_sink *
ts_create (w, size)
Widget w;
int size;
{
    struct text_sink *ts;
    CanvasWidget cw = (CanvasWidget) w;

    ts = XMALLOC (1, struct text_sink);
    ts->chunkStart = XMALLOC (size + 1, char);
    ts->chunkSize = size;
    ts->bufStart = ts->chunkStart;
    ts->bufSize = 0;
    ts->wid = w;
    XtAddEventHandler (w, StructureNotifyMask|ExposureMask,
		       True, ts_expose, (XtPointer) ts);
    /*
     * zero out the gc now; since we haven't realized the widget
     * yet.  we will create one before drawing anything on the window.
     */
    ts->gc = 0;
    return ts;
}


/*
 * append 'size' bytes to a text-sink, and redisplay
 */

void
ts_append_l (ts, ptr, size)
struct text_sink *ts;
char *ptr;
int size;
{
    int max_buf_size;
    int room_in_top_half;
    int newline_count = 0;

    if (size <= 0)
	return;
    /*
     * if requested size is too large, truncate the request
     */
    max_buf_size = MAX_BUF_SIZE(ts);
    if (size > max_buf_size) {
	ptr += size - max_buf_size;
	size = max_buf_size;
    }

    /*
     * if there's not enough room in the buffer, delete enough stuff
     * from the beginning so that we have room.
     */
    if (ROOM(ts) < size) {
	int need = size - ROOM (ts);

	if (need < TOP_HALF_SIZE (ts)) {
	    /* XXX should delete up 'til next newline */
	    ts->bufStart += need;
	}
	else {
	    ts->bufStart = ts->chunkStart + (need - TOP_HALF_SIZE (ts));
	}
	ts->bufSize -= need;
    }

    /*
     * now append this text to the end of the buffer.
     */

    room_in_top_half = TOP_HALF_SIZE(ts) - ts->bufSize;
    if (size <= room_in_top_half) {
	/*
	 * The whole thing will fit in the top half, so just copy it.
	 */
	newline_count += bufcpy (ts->bufStart + ts->bufSize, ptr, size);
	ts->bufSize += size;
    }
    else {
	/*
	 * text won't completely fit in top half, so we have to split it.
	 * first do the top half (if there's any room at all),
	 * then copy the rest to the bottom half.
	 */
	if (room_in_top_half > 0) {
	    newline_count += bufcpy (ts->bufStart + ts->bufSize,
				     ptr, room_in_top_half); 
	    ts->bufSize += room_in_top_half;
	    ptr += room_in_top_half;
	    size -= room_in_top_half;
	}
	newline_count +=
	    bufcpy (ts->chunkStart + (ts->bufSize - TOP_HALF_SIZE(ts)),
		    ptr, size);
	ts->bufSize += size;
    }

    /*
     * display the new text
     */
    redisplay (ts, newline_count);
    return;
}

void
ts_append (ts, s)
struct text_sink *ts;
char *s;
{
    ts_append_l (ts, s, strlen (s));
}




#else
/****************************************************************************
 *	     old ascii-text-widget based code starts here                   *
 *           XXX nuke this once the stuff above works                       *
 ****************************************************************************/

struct text_sink *
ts_create (w, size)
Widget w;
int size;
{
    struct text_sink *ts;

    if ((ts = (struct text_sink *) MALLOC (sizeof (struct text_sink))) == NULL)
	return NULL;

    ts->wid = w;
    if ((ts->buf = MALLOC (size)) == NULL) {
	FREE ((char *) ts);
	return NULL;
    }
    ts->bufsize = size;
    ts->current_length = 0;
    *(ts->buf) = '\0';

    XtVaSetValues (w,
		   XtNstring, ts->buf,
		   XtNlength, size,
		   XtNtype, XawAsciiString,
		   NULL);
    return ts;
}

void
ts_append_l (ts, s, append_length)
struct text_sink *ts;
char *s;
int append_length;
{
    char *ptr;
    XawTextBlock textblock;
    int end_of_file = ts->bufsize + 1;
    char *strchr ();

    XawTextDisableRedisplay (ts->wid);
    XtVaSetValues (ts->wid, XtNeditType, XawtextEdit, NULL);

    /*
     * make sure there's enough space in the buffer before
     * doing the append.  If necessary, delete some stuff
     * from the beginning of the buffer.
     */

    if (append_length > ts->bufsize - ts->current_length) {
	ptr = ts->buf;
	do {
	    ptr = strchr (ptr, '\n');
	    if (ptr > ts->buf + append_length)
		break;
	    ++ptr;
	} while (1);
	/*
	 * now ptr points to a newline.  Trim down the buffer.
	 */
	textblock.firstPos = 0;
	textblock.format = FMT8BIT;
	textblock.length = 0;
	textblock.ptr = 0;

	XawTextReplace (ts->wid, 0, ptr - ts->buf, &textblock);
	
	ts->bufsize -= (ptr - ts->buf);
    }

    textblock.firstPos = 0;
    textblock.format = FMT8BIT;
    textblock.length = append_length;
    textblock.ptr = s;

    XawTextReplace(ts->wid, end_of_file, end_of_file, &textblock);
    XawTextSetInsertionPoint(ts->wid, end_of_file);

    XtVaSetValues (ts->wid, XtNeditType, XawtextRead, NULL);
    XawTextEnableRedisplay (ts->wid);
}

void
ts_append (ts, s)
struct text_sink *ts;
char *s;
{
    int length = strlen (s);
    ts_append_l (ts, s, length);
}
#endif
