/*
 * red-black tree routines
 *
 * written by Jim Plank, c. Summer 1991
 * hacked over by Keith Moore, October 1992
 *
 * Red-Black trees are cool because they do everything in O(log n) time,
 * where n is the number of items in the tree.
 *
 * Jim wrote the code for the first version of the HeNCE executioner.
 * I subsequently hacked over it to make the programming interface
 * simpler.  There is some loss in functionality -- the current package
 * doesn't export routines to build trees with integer or char string keys.
 * The programmer simply supplies a comparison function when calling
 * rb_Create ();
 *
 * $Id: rb.c,v 1.1 1994/02/17 21:35:28 moore Exp $
 *
 * $Log: rb.c,v $
 * Revision 1.1  1994/02/17  21:35:28  moore
 * Initial revision
 *
 *
 */

#include "rb.h"
#include <stdio.h>

#include <Malloc.h>

typedef struct {
    unsigned red : 1 ;
    unsigned internal : 1 ;
    unsigned left : 1 ;
    unsigned root : 1 ;
    unsigned head : 1 ;
} status;

struct rb_node {
    union {
	struct {
	    struct rb_node *flink;
	    struct rb_node *blink;
	} list;
	struct {
	    struct rb_node *left;
	    struct rb_node *right;
	} child;
    } c;
    union {
	struct rb_node *parent;
	struct rb_node *root;
    } p;
    status s;
    union {
	int ikey;
	char *key;
	struct rb_node *lext;
    } k;
    union {
	char *val;
	struct rb_node *rext;
    } v;
};


#define rb_traverse(ptr, lst) \
  for((ptr) = rb_first(lst); (ptr) != nil(lst); (ptr) = rb_next(ptr))

struct rb_tree {
    int (*compare)();
    struct rb_node *root;
};

#define isred(n) (n->s.red)
#define isblack(n) (!isred(n))
#define isleft(n) (n->s.left)
#define isright(n) (!isleft(n))
#define isint(n) (n->s.internal)
#define isext(n) (!isint(n))
#define ishead(n) (n->s.head)
#define isroot(n) (n->s.root)
#define setred(n) n->s.red = 1
#define setblack(n) n->s.red = 0
#define setleft(n) n->s.left = 1
#define setright(n) n->s.left = 0
#define sethead(n) n->s.head = 1
#define setroot(n) n->s.root = 1
#define setint(n) n->s.internal = 1
#define setext(n) n->s.internal = 0
#define setnormal(n) { n->s.root = 0; n ->s.head = 0; }
#define sibling(n) ((isleft(n)) ? n->p.parent->c.child.right \
                                : n->p.parent->c.child.left)

#define mk_new_ext(new, kkkey, vvval) {\
  new = (rbNode) MALLOC(sizeof(struct rb_node));\
  new->v.val = vvval;\
  new->k.key = kkkey;\
  setext(new);\
  setblack(new);\
  setnormal(new);\
}

static void recolor ();
static void single_rotate ();
static rbNode make_rb();
static rbNode rb_insert_b(/* node, char *key, char *value */);
static rbNode rb_find_key(/* tree, char *key */);
static rbNode rb_find_ikey(/* tree, int ikey */);
static rbNode rb_find_gkey(/* tree, char *key, cmpfxn */);
static rbNode rb_find_key_n(/* tree, char *key, int *found */);
static rbNode rb_find_ikey_n(/* tree, int ikey, int *found */);
static rbNode rb_find_gkey_n(/* tree, char *key, cmpfxn, int *found */);
static void rb_delete_node(/* node */);
static void rb_free_tree(/* node */);  /* Deletes and frees an entire tree */

#define rb_insert_a(n, k, v) rb_insert_b((n)->c.list.flink, (k), (v))
#define rb_insert(t, k, v) rb_insert_b(rb_find_key((t), (k)), (k), (v))
#define rb_inserti(t, k, v) rb_insert_b(rb_find_ikey((t), (k)), (char *) (k), (v))
#define rb_insertg(t, k, v, f) rb_insert_b(rb_find_gkey((t), (k), (f)), (k), (v))
#define rb_first(n) ((n)->c.list.flink)
#define rb_last(n) ((n)->c.list.blink)
#define rb_next(n) ((n)->c.list.flink)
#define rb_prev(n) ((n)->c.list.blink)
#define rb_empty(t) ((t)->c.list.flink == (t))
#ifndef nil
#define nil(t) (t)
#endif

static void
insert(item, list)      /* Inserts to the end of a list */
rbNode item;
rbNode list;
{
    rbNode last_node;

    last_node = list->c.list.blink;

    list->c.list.blink = item;
    last_node->c.list.flink = item;
    item->c.list.blink = last_node;
    item->c.list.flink = list;
}

static void
delete_item(item)               /* Deletes an arbitrary iterm */
rbNode item;
{
    item->c.list.flink->c.list.blink = item->c.list.blink;
    item->c.list.blink->c.list.flink = item->c.list.flink;
}

static void
mk_new_int(l, r, p, il)
rbNode l, r, p;
int il;
{
    rbNode new;

    new = (rbNode) MALLOC(sizeof(struct rb_node));
    setint(new);
    setred(new);
    setnormal(new);
    new->c.child.left = l;
    new->c.child.right = r;
    new->p.parent = p;
    new->k.lext = l;
    new->v.rext = r;
    l->p.parent = new;
    r->p.parent = new;
    setleft(l);
    setright(r);
    if (ishead(p)) {
	p->p.root = new;
	setroot(new);
    }
    else if (il) {
	setleft(new);
	p->c.child.left = new;
    }
    else {
	setright(new);
	p->c.child.right = new;
    }
    recolor(new);
}
  
   
static rbNode
lprev(n)
rbNode n;
{
    if (ishead(n))
	return n;
    while (!isroot(n)) {
	if (isright(n))
	    return n->p.parent;
	n = n->p.parent;
    }
    return n->p.parent;
}

static rbNode
rprev(n)
rbNode n;
{
    if (ishead(n))
	return n;
    while (!isroot(n)) {
	if (isleft(n))
	    return n->p.parent;
	n = n->p.parent;
    }
    return n->p.parent;
}

static rbNode
make_rb()
{
    rbNode head;

    head = (rbNode) MALLOC (sizeof(struct rb_node));
    head->c.list.flink = head;
    head->c.list.blink = head;
    head->p.root = head;
    head->k.key = "";
    sethead(head);
    return head;
}

static rbNode
rb_find_key_n(n, key, fnd)
rbNode n;
char *key;
int *fnd;
{
    int cmp;

    *fnd = 0;
    if (!ishead(n)) {
	fprintf(stderr, "rb_find_key_n called on non-head 0x%x\n", n);
	exit(1);
    }
    if (n->p.root == n)
	return n;
    cmp = strcmp(key, n->c.list.blink->k.key);
    if (cmp == 0) {
	*fnd = 1;
	return n->c.list.blink; 
    }
    if (cmp > 0)
	return n; 
    else
	n = n->p.root;
    while (1) {
	if (isext(n))
	    return n;
	cmp = strcmp(key, n->k.lext->k.key);
	if (cmp == 0) {
	    *fnd = 1;
	    return n->k.lext;
	}
	if (cmp < 0)
	    n = n->c.child.left ;
	else
	    n = n->c.child.right;
    }
}

static rbNode 
rb_find_key(n, key)
rbNode n;
char *key;
{
    int fnd;
    return rb_find_key_n(n, key, &fnd);
}

static rbNode
rb_find_ikey_n(n, ikey, fnd)
rbNode n;
int ikey;
int *fnd;
{
    *fnd = 0;
    if (!ishead(n)) {
	fprintf(stderr, "rb_find_ikey_n called on non-head 0x%x\n", n);
	exit(1);
    }
    if (n->p.root == n)
	return n;
    if (ikey == n->c.list.blink->k.ikey) {
	*fnd = 1;
	return n->c.list.blink; 
    }
    if (ikey > n->c.list.blink->k.ikey)
	return n; 
    else
	n = n->p.root;
    while (1) {
	if (isext(n))
	    return n;
	if (ikey == n->k.lext->k.ikey) {
	    *fnd = 1;
	    return n->k.lext;
	}
	n = (ikey < n->k.lext->k.ikey) ? n->c.child.left : n->c.child.right;
    }
}

static rbNode
rb_find_ikey(n, ikey)
rbNode n;
int ikey;
{
    int fnd;
    return rb_find_ikey_n(n, ikey, &fnd);
}

static rbNode 
rb_find_gkey_n(n, key, fxn, fnd)
rbNode n;
char *key;
int (*fxn)();
int *fnd;
{
    int cmp;

    *fnd = 0;
    if (!ishead(n)) {
	fprintf(stderr, "rb_find_key_n called on non-head 0x%x\n", n);
	exit(1);
    }
    if (n->p.root == n)
	return n;
    cmp = (*fxn)(key, n->c.list.blink->k.key);
    if (cmp == 0) {
	*fnd = 1;
	return n->c.list.blink; 
    }
    if (cmp > 0)
	return n; 
    else
	n = n->p.root;
    while (1) {
	if (isext(n))
	    return n;
	cmp = (*fxn)(key, n->k.lext->k.key);
	if (cmp == 0) {
	    *fnd = 1;
	    return n->k.lext;
	}
	if (cmp < 0)
	    n = n->c.child.left ;
	else
	    n = n->c.child.right;
    }
}

static
rbNode rb_find_gkey(n, key, fxn)
rbNode n;
char *key;
int (*fxn)();
{
    int *fnd;
    return rb_find_gkey_n(n, key, fxn, &fnd);
}

static
rbNode rb_insert_b(n, key, val)
rbNode n;
char *key;
char *val;
{
    rbNode newleft, newright, newnode, list, p;

    if (ishead(n)) {
	if (n->p.root == n) {         /* Tree is empty */
	    mk_new_ext(newnode, key, val);
	    insert(newnode, n);
	    n->p.root = newnode;
	    newnode->p.parent = n;
	    setroot(newnode);
	    return newnode;
	}
	else {
	    mk_new_ext(newright, key, val);
	    insert(newright, n);
	    newleft = newright->c.list.blink;
	    setnormal(newleft);
	    mk_new_int(newleft, newright, newleft->p.parent, isleft(newleft));
	    p = rprev(newright);
	    if (!ishead(p)) p->k.lext = newright;
	    return newright;
	}
    }
    else {
	mk_new_ext(newleft, key, val);
	insert(newleft, n);
	setnormal(n);
	mk_new_int(newleft, n, n->p.parent, isleft(n));
	p = lprev(newleft);
	if (!ishead(p)) p->v.rext = newleft;
	return newleft;    
    }
}

static void
recolor(n)
rbNode n;
{  
    rbNode p, gp, s;
    int done = 0;

    while(!done) {
	if (isroot(n)) {
	    setblack(n);
	    return;
	}

	p = n->p.parent;
	
	if (isblack(p))
	    return;
    
	if (isroot(p)) {
	    setblack(p);
	    return;
	}

	gp = p->p.parent;
	s = sibling(p);
	if (isred(s)) {
	    setblack(p);
	    setred(gp);
	    setblack(s);
	    n = gp;
	}
	else {
	    done = 1;
	}
    }
    /* p's sibling is black, p is red, gp is black */
  
    if ((isleft(n) == 0) == (isleft(p) == 0)) {
	single_rotate(gp, isleft(n));
	setblack(p);
	setred(gp);
    }
    else {
	single_rotate(p, isleft(n));
	single_rotate(gp, isleft(n));
	setblack(n);
	setred(gp);
    }
}

static void
single_rotate(y, l)
rbNode y;
int l;
{
    int rl, ir;
    rbNode x, yp;
    char *tmp;

    ir = isroot(y);
    yp = y->p.parent;
    if (!ir) {
	rl = isleft(y);
    }
  
    if (l) {
	x = y->c.child.left;
	y->c.child.left = x->c.child.right;
	setleft(y->c.child.left);
	y->c.child.left->p.parent = y;
	x->c.child.right = y;
	setright(y);  
    }
    else {
	x = y->c.child.right;
	y->c.child.right = x->c.child.left;
	setright(y->c.child.right);
	y->c.child.right->p.parent = y;
	x->c.child.left = y;
	setleft(y);  
    }

    x->p.parent = yp;
    y->p.parent = x;
    if (ir) {
	yp->p.root = x;
	setnormal(y);
	setroot(x);
    }
    else {
	if (rl) {
	    yp->c.child.left = x;
	    setleft(x);
	}
	else {
	    yp->c.child.right = x;
	    setright(x);
	}
    }
}
    
static void
rb_delete_node(n)
rbNode n;
{
    rbNode s, p, gp;
    char ir;

    if (isint(n)) {
	fprintf(stderr, "Cannot delete an internal node: 0x%x\n", n);
	exit(1);
    }
    if (ishead(n)) {
	fprintf(stderr, "Cannot delete the head of an rb_tree: 0x%x\n", n);
	exit(1);
    }
    delete_item(n); /* Delete it from the list */
    p = n->p.parent;  /* The only node */
    if (isroot(n)) {
	p->p.root = p;
	FREE((char *) n);
	return;
    } 
    s = sibling(n);    /* The only node after deletion */
    if (isroot(p)) {
	s->p.parent = p->p.parent;
	s->p.parent->p.root = s;
	setroot(s);
	FREE((char *) p);
	FREE((char *) n);
	return;
    }
    gp = p->p.parent;  /* Set parent to sibling */
    s->p.parent = gp;
    if (isleft(p)) {
	gp->c.child.left = s;
	setleft(s);
    }
    else {
	gp->c.child.right = s;
	setright(s);
    }
    ir = isred(p);
    FREE((char *) p);
    FREE((char *) n);
  
    if (isext(s)) {      /* Update proper rext and lext values */
	p = lprev(s); 
	if (!ishead(p))
	    p->v.rext = s;
	p = rprev(s);
	if (!ishead(p))
	    p->k.lext = s;
    }
    else if (isblack(s)) {
	fprintf(stderr, "DELETION PROB -- sib is black, internal\n");
	exit(1);
    }
    else {
	p = lprev(s);
	if (!ishead(p))
	    p->v.rext = s->c.child.left;
	p = rprev(s);
	if (!ishead(p))
	    p->k.lext = s->c.child.right;
	setblack(s);
	return;
    }

    if (ir)
	return;

  /* Recolor */
  
    n = s;
    p = n->p.parent;
    s = sibling(n);
    while(isblack(p) && isblack(s) && isint(s) && 
	  isblack(s->c.child.left) && isblack(s->c.child.right)) {
	setred(s);
	n = p;
	if (isroot(n))
	    return;
	p = n->p.parent;
	s = sibling(n);
    }
  
    if (isblack(p) && isred(s)) {  /* Rotation 2.3b */
	single_rotate(p, isright(n));
	setred(p);
	setblack(s);
	s = sibling(n);
    }
    
    {
	rbNode x, z; char il;
    
	if (isext(s)) {
	    fprintf(stderr, "DELETION ERROR: sibling not internal\n");
	    exit(1);
	}

	il = isleft(n);
	x = il ? s->c.child.left : s->c.child.right ;
	z = sibling(x);

	if (isred(z)) {  /* Rotation 2.3f */
	    single_rotate(p, !il);
	    setblack(z);
	    if (isred(p)) setred(s); else setblack(s);
	    setblack(p);
	}
	else if (isblack(x)) {   /* Recoloring only (2.3c) */
	    if (isred(s) || isblack(p)) {
		fprintf(stderr, "DELETION ERROR: 2.3c not quite right\n");
		exit(1);
	    }
	    setblack(p);
	    setred(s);
	    return;
	}
	else if (isred(p)) { /* 2.3d */
	    single_rotate(s, il);
	    single_rotate(p, !il);
	    setblack(x);
	    setred(s);
	    return;
	}
	else {  /* 2.3e */
	    single_rotate(s, il);
	    single_rotate(p, !il);
	    setblack(x);
	    return;
	}
    }
}


static void
rb_print_tree(t, level)
rbNode t;
int level;
{
    int i;
    if (ishead(t) && t->p.parent == t) {
	printf("tree 0x%x is empty\n", t);
    }
    else if (ishead(t)) {
	printf("Head: 0x%x.  Root = 0x%x\n", t, t->p.root);
	rb_print_tree(t->p.root, 0);
    }
    else {
	if (isext(t)) {
	    for (i = 0; i < level; i++) putchar(' ');
	    printf("Ext node 0x%x: %c,%c: p=0x%x, k=%s\n", 
		   t, isred(t)?'R':'B', isleft(t)?'l':'r',
		   t->p.parent, t->k.key);
	}
	else {
	    rb_print_tree(t->c.child.left, level+2);
	    rb_print_tree(t->c.child.right, level+2);
	    for (i = 0; i < level; i++) putchar(' ');
	    printf("Int node 0x%x: %c,%c: l=0x%x,r=0x%x,p=0x%x,lr=(%s,%s)\n", 
		   t, isred(t)?'R':'B', isleft(t)?'l':'r', t->c.child.left, 
		   t->c.child.right, 
		   t->p.parent, t->k.lext->k.key, t->v.rext->k.key);
	}
    }
}

static void
rb_iprint_tree(t, level)
rbNode t;
int level;
{
    int i;
    if (ishead(t) && t->p.parent == t) {
	printf("tree 0x%x is empty\n", t);
    }
    else if (ishead(t)) {
	printf("Head: 0x%x.  Root = 0x%x, < = 0x%x, > = 0x%x\n", 
	       t, t->p.root, t->c.list.blink, t->c.list.flink);
	rb_iprint_tree(t->p.root, 0);
    }
    else {
	if (isext(t)) {
	    for (i = 0; i < level; i++) putchar(' ');
	    printf("Ext node 0x%x: %c,%c: p=0x%x, <=0x%x, >=0x%x k=%d\n", 
		   t, isred(t)?'R':'B', isleft(t)?'l':'r', t->p.parent, 
		   t->c.list.blink, t->c.list.flink, t->k.ikey);
	}
	else {
	    rb_iprint_tree(t->c.child.left, level+2);
	    rb_iprint_tree(t->c.child.right, level+2);
	    for (i = 0; i < level; i++) putchar(' ');
	    printf("Int node 0x%x: %c,%c: l=0x%x,r=0x%x,p=0x%x,lr=(%d,%d)\n", 
		   t, isred(t)?'R':'B', isleft(t)?'l':'r', t->c.child.left, 
		   t->c.child.right, 
		   t->p.parent, t->k.lext->k.ikey, t->v.rext->k.ikey);
	}
    }
}
      
static int
rb_nblack(n)
rbNode(n);
{
    int nb;
    if (ishead(n) || isint(n)) {
	fprintf(stderr,
		"ERROR: rb_nblack called on a non-external node 0x%x\n", n);
	exit(1);
    }
    nb = 0;
    while(!ishead(n)) {
	if (isblack(n)) nb++;
	n = n->p.parent;
    }
    return nb;
}

static int
rb_plength(n)
rbNode(n);
{
    int pl;
    if (ishead(n) || isint(n)) {
	fprintf(stderr,
		"ERROR: rb_plength called on a non-external node 0x%x\n", n);
	exit(1);
    }
    pl = 0;
    while(!ishead(n)) {
	pl++;
	n = n->p.parent;
    }
    return pl;
}

static void
rb_free_tree(n)
rbNode n;
{
    if (!ishead(n)) {
	fprintf(stderr, "ERROR: Rb_free_tree called on a non-head node\n");
	exit(1);
    }

    while(rb_first(n) != nil(n)) {
	rb_delete_node(rb_first(n));
    }
    FREE((char *) n);
}

/*
 * create a new tree.  The caller supplies a comparison function (like
 * strcmp) which is used to order the elements.
 */

rbTree 
rb_Create (compare)
int (*compare) ();
{
    rbTree ptr = (rbTree) MALLOC (sizeof (*ptr));
    if (ptr == NULL)
	return NULL;
    ptr->compare = compare;
    ptr->root = make_rb ();
    if (ptr->root == NULL) {
	FREE((char *) (ptr));
	return NULL;
    }
    return ptr;
}

/*
 * Insert a new node in the tree immediately before thisNode.
 * the caller is responsible for making sure that they keys
 * are in order.  Normally this is called immediately after
 * rb_Find(), but it could also be used if the caller were
 * traversing the list for some reason.
 */

void
rb_InsertBefore (thisNode, key, value)
rbNode thisNode;
void *key;
void *value;
{
    rb_insert_b (thisNode, (char *) key, (char *) value);
}


/*
 * find a node with the given key in the tree.  If found is non-NULL,
 * (*found) will be set to zero if the key was not found, nonzero if the
 * key was found.  If the key was found, the node returned is the one
 * containing the key.  If the key was not found, the node returned will be
 * the one whose key would immediately follow the one supplied to
 * rb_Find were that key in the tree.
 */

rbNode
rb_Find (tree, key, found)
rbTree tree;
void *key;
int *found;
{
    if (found)
	return rb_find_gkey_n (tree->root, (char *) key, tree->compare, found);
    else
	return rb_find_gkey (tree->root, (char *) key, tree->compare);
}

/*
 * remove a particular node from the tree
 */

void
rb_DeleteNode (node)
rbNode node;
{
    rb_delete_node (node);
}

/*
 * delete an entire tree.  This routine DOES NOT free the contents of keys
 * and values, so this routine isn't very useful if these point to heap
 * storage.
 */

void
rb_DeleteTree (tree)
rbTree tree;
{
    rb_free_tree (tree->root);
    FREE ((char *) tree);
}

/*
 * like rb_InsertBefore, only the new node is inserted after thisNode
 */

void
rb_InsertAfter (thisNode, key, value)
rbNode thisNode;
void *key;
void *value;
{
    rb_InsertBefore (thisNode->c.list.flink, key, value);
}

/*
 * given a (key, value) pair, insert them into the tree
 */

void
rb_Insert (tree, key, value)
rbTree tree;
void *key;
void *value;
{
    rb_insert_b (rb_find_gkey (tree->root, (char *) key, tree->compare),
		 (char *) key, (char *) value);
}

/*
 * return the first node in the tree
 */

rbNode
rb_First (tree)
rbTree tree;
{
    return tree->root->c.list.flink;
}

/*
 * return the last node in the tree
 */
rbNode
rb_Last (tree)
rbTree tree;
{
    return tree->root->c.list.blink;
}

/*
 * return the successor to this node
 */

rbNode
rb_Next (node)
rbNode node;
{
    return node->c.list.flink;
}

/*
 * return the predecessor to this node
 */

rbNode
rb_Prev (node)
rbNode node;
{
    return node->c.list.blink;
}

/*
 * return 1 if the tree is empty, else 0
 */

int
rb_Empty (tree)
rbTree tree;
{
    return (tree->root->c.list.flink == tree->root);
}

/*
 * return 1 if this node is an end-of-list
 */

int
rb_Done (node, tree)
rbNode node;
rbTree tree;
{
    return node == tree->root;
}

/*
 * return the key from a particular node
 */

void *
rb_Key (node)
rbNode node;
{
    return (void *) node->k.key;
}

/*
 * return the value from a particular node
 */

void *
rb_Value (node)
rbNode node;
{
    return (void *) node->v.val;
}

void
rb_SetValue (node, value)
rbNode node;
void *value;
{
    node->v.val = value;
}


int
rb_Traverse (t, fn)
rbTree t;
int (*fn)();
{
    rbNode tn;
    int x;

    for (tn = rb_First (t); !rb_Done (tn, t); tn = rb_Next (tn))
	if (x = (*fn)(tn))
	    return x;
    return 0;
}

int
rb_TraverseX (t, fn, arg)
rbTree t;
int (*fn)();
void *arg;
{
    rbNode tn;
    int x;

    for (tn = rb_First (t); !rb_Done (tn, t); tn = rb_Next (tn))
	if (x = (*fn)(tn, arg))
	    return x;
    return 0;
}
