#include <stdio.h>
#include <sys/types.h>

#include <Errno.h>
#include <File.h>
#include <Malloc.h>
#include <Select.h>
#include <Signal.h>
#include <Stat.h>
#include <String.h>
#include <Utime.h>
#include <Wait.h>

#include "pvm3.h"
#include "proxy.h"

extern int pvm_errno;

char **child_environ;
int signal_pipe[2];

int my_tid = 0;
int parent_tid = 0;

char *
pvm_strerror (x)
int x;
{
    extern int pvm_nerr;
    extern char *pvm_errlist[];

    x = -x;
    if (x >= 0 && x < pvm_nerr)
	return pvm_errlist[x];
    return "Unknown error";
}


#if 0
char *
libs ()
{
    static char buf[1024];
    static char buf2[1024];
    static char *libnames[] = {
	"slave3", "hence3", "rb", "dl", "alloc", "pvm3",
    };
    int i;

    *buf = '\0';
    for (i = 0; i < sizeof(libnames) / sizeof (*libnames); ++i) {
	sprintf (buf2, " %s/lib/%s/lib%s.a",
		 getenv ("PVM_ROOT"),
		 getenv ("PVM_ARCH"),
		 libnames[i]);
	strcat (buf, buf2);
    }    
    return buf;
}

char *
compile_cmd (lang, output, source, wrapper)
char *lang;
char *output;
char *source;
char *wrapper;
{
    static char buf[1024];

    if (strcasecmp (lang, "c")) {
	sprintf (buf, "%s -o %s %s %s %s",
		 C_COMPILER, output, source, wrapper, libs());
	return buf;
    }
    else {
	sprintf (buf, "%s -o %s %s/lib/hence_main.f %s %s %s",
		 FORTRAN_COMPILER, output, getenv ("PVM_ROOT"),
		 source, wrapper, libs());
	return buf;
    }
}
#endif

struct death_event {
    int pid;
    int status;
    int signaled;
};

static SIG_RETURN_TYPE
child_sig (sig)
int sig;
{
    struct death_event foo;
    waitbuf w;

#ifdef HAVE_WAITPID
    if ((foo.pid = waitpid (-1, &w, WNOHANG)) > 0) {
#else
    if ((foo.pid = wait3 (&w, WNOHANG, NULL)) > 0) {
#endif
	if (WIFSIGNALED(w)) {
	    foo.signaled = 1;
	    foo.status = WTERMSIG(w);
	}
	else {
	    foo.signaled = 0;
	    foo.status = WEXITSTATUS(w);
	}
#if 0
    fprintf (stderr, "child_sig(signaled=%d,status=%d)\n",
	     foo.signaled, foo.status);
#endif

	write (signal_pipe[1], &foo, sizeof (foo));
    }
#ifdef SYSVSIGNAL
    signal (SIGCHLD, child_sig); /* SYSV brain damage */
#endif
}

/*
 * find a variable name in an environment list, and return its value
 */

char *
find_var (name, length, envp)
char *name;
int length;
char **envp;
{
    int i;
    for (i = 0; envp[i] != NULL; ++i)
	if (strncmp (envp[i], name, length) == 0 && envp[i][length] == '=')
	    return (envp[i] + length + 1);
}

/*
 * expand environment variables in a child process's arguments
 */

char *
expand_vars (arg, env)
char *arg;
char **env;
{
    char buf[1024];
    char *var_name;
    int var_length = -1;
    char *s1, *s2, *d1;

    d1 = buf;
    s1 = arg;
    while (*s1) {
	if (var_length < 0) {
	    /* not within $(...) */
	    if (s1[0] == '$' && s1[1] == '(') {
		s1 += 2;
		var_name = s1;	/* remember start of var_name */
		var_length = 0;
	    }
	    else
		*d1++ = *s1++;
	}
	else {
	    /* within $(...) */
	    if (*s1 == ')') {
		++s1;
		s2 = find_var (var_name, var_length, env);
		if (s2) {
		    while (*s2)
			*d1++ = *s2++;
		}
		var_length = -1;
	    }
	    else {
		++s1;
		var_length++;
	    }
	}
    }
    *d1++ = '\0';
    if (var_length > 0)
	return strdup (arg);
    else
	return strdup (buf);
}

/*
 * spawn a child process.  while it is running, grab its stdout
 * and stderr and send them back as msgs to the caller.
 * Also grab its termination status (when it dies) and send that.
 *
 * XXX need to be able to grab signals from pvm also, in case caller
 * wants us to go away or kill the child, whatever.
 */

static void
do_spawn (tid, child_argv)
int tid;
char **child_argv;
{
    int child_stdout[2];	/* pipe for child's stdout */
    int child_stderr[2];	/* pipe for child's stderr */
    int child_pid;		/* child's pid */
    fd_set read_fds;
    fd_set temp_fds;
    int max_fd;
    int nbytes;
    char buf[MSGBUFSIZE];
    struct death_event foo;
    extern char **environ;

    pipe (child_stdout);
    pipe (child_stderr);
    fflush (stderr);		/* make sure stderr buf is empty */
    if ((child_pid = fork ()) == 0) {
	/* child */
	int fd;
	int i;

	close (child_stderr[0]);
	dup2 (child_stderr[1], 2);
	close (child_stderr[1]);

	close (child_stdout[0]);
	dup2 (child_stdout[1], 1);
	close (child_stdout[1]);

	fd = open ("/dev/null", 0); /* stdin gets /dev/null */
	dup2 (fd, 0);
	close (fd);

	for (i = getdtablesize (); i >= 3; --i)
	    close (i);

	environ = child_environ;
	execvp (child_argv[0], child_argv);
	fprintf (stderr, "execvp(%s) failed: %s\n", child_argv[0],
		 strerror (errno));
	fclose (stderr);	/* flush stderr before exiting */
	_exit (127);
    }
    else {
	/* parent */

	close (child_stderr[1]);
	close (child_stdout[1]);
	FD_ZERO (&read_fds);
	FD_SET (child_stderr[0], &read_fds);
	FD_SET (child_stdout[0], &read_fds);
	FD_SET (signal_pipe[0], &read_fds);

#define max(a,b) ((a) > (b) ? (a) : (b))

	max_fd = max (child_stderr[0], child_stdout[0]);
	max_fd = max (max_fd, signal_pipe[0]);
    }
    while (1) {
	temp_fds = read_fds;
	if (select (max_fd + 1, &temp_fds, 0, 0, 0) > 0) {
	    if (FD_ISSET (child_stderr[0], &temp_fds)) {
		nbytes = read (child_stderr[0], buf, sizeof (buf));
		if (nbytes > 0) {
		    pvm_packf ("%+ %d %*c", PvmDataDefault,
			       nbytes, nbytes, buf);
		    pvm_send (tid, cmdStderrTag);
		}
		else if (nbytes == 0) {
		    close (child_stderr[0]);
		    FD_CLR (child_stderr[0], &read_fds);
		}
	    }
	    if (FD_ISSET (child_stdout[0], &temp_fds)) {
		nbytes = read (child_stdout[0], buf, sizeof (buf));
		if (nbytes > 0) {
		    pvm_packf ("%+ %d %*c", PvmDataDefault,
			       nbytes, nbytes, buf);
		    pvm_send (tid, cmdStdoutTag);
		}
		else if (nbytes == 0) {
		    close (child_stdout[0]);
		    FD_CLR (child_stdout[0], &read_fds);
		}		    
	    }
	    if (FD_ISSET (signal_pipe[0], &temp_fds)) {
		nbytes = read (signal_pipe[0], &foo, sizeof foo);
#if 0
		fprintf (stderr, "read obit\n");
#endif
		/*
		 * suck up all of the stdout and stderr from
		 * the child before returning an exit msg
		 */
		while ((nbytes = read (child_stdout[0], buf,
				       sizeof (buf))) > 0) {
		    pvm_packf ("%+ %d %*c", PvmDataDefault,
			       nbytes, nbytes, buf);
		    pvm_send (tid, cmdStdoutTag);
		}
		while ((nbytes = read (child_stderr[0], buf,
				       sizeof (buf))) > 0) {
		    pvm_packf ("%+ %d %*c", PvmDataDefault,
			       nbytes, nbytes, buf);
		    pvm_send (tid, cmdStderrTag);
		}
#if 0
		fprintf (stderr, "writing exit msg\n");
#endif
		if (foo.pid == child_pid) {
		    pvm_packf ("%+ %d", PvmDataDefault, foo.status);
		    pvm_send (tid, cmdExitTag);
		    break;
		}
	    }
	}
    }
    close (child_stdout[0]);
    close (child_stderr[0]);
}


static int
reply (a, b)
int a, b;
{
#if 0
    fprintf (stderr, "<<< reply (%d %d)\n", a, b);
#endif
    pvm_packf ("%+ %d %d", PvmDataDefault, a, b);
}


/*
 * receive a message of type pxyCommand from the parent, and unpack
 * the first integer into 'cmd'.
 *
 * any message of any other type, or from any other source, produces
 * a gripe to stderr but is otherwise ignored.
 *
 * returns a pvm error code if we ran into some sort of pvm error.
 * otherwise it returns zero.
 */

static int
recvcmd (cmd)
int *cmd;
{
    int x;
    int tid;
    int tag;
    int mt;

    while (1) {
	if ((x = pvm_recv (-1, -1)) < 0)
	    return x;
	if ((x = pvm_bufinfo (x, NULL, &tag, &tid)) < 0)
	    return x;
	if (tid != parent_tid) {
	    fprintf (stderr,
		     "%08x: discarding (src=%x,tag=%d): bad origin\n",
		     my_tid, tid, tag);
	    continue;
	}
	if (tag != commandTag) {
	    fprintf (stderr,
		     "%08x: discarding (src=%x,tag=%d): bad msg tag\n",
		     my_tid, tid, tag);
	    continue;
	}
	if ((x = pvm_upkint (cmd, 1, 1)) < 0)
	    return x;
	return 0;
    }
}

/*
 * return a response to a command
 */

static void
sendresp ()
{
    int x;
    
    x = pvm_send (parent_tid, responseTag);
    if (x < 0) {
	fprintf (stderr, "pvm_send (%08x, %d) failed: %s\n",
		 parent_tid, responseTag, pvm_strerror (x));
    }
}

/*
 * main loop
 */

static char dest_filename[1024];
static int fd = -1;
static enum { MODE_NULL, MODE_READ, MODE_WRITE } fdmode = MODE_NULL;

static void
write_cleanup ()
{
    close (fd);
    fd = -1;
    unlink (dest_filename);
    *dest_filename = '\0';
    sendresp ();
    fdmode = MODE_NULL;
}

static void
work (msgbuf)
char *msgbuf;
{
    int cmd;
    int nbytes;
    int x;
    char buf[PROXY_FILEBUFSIZE];
    char tempname[1024];
    int size;			/* XXX should be size_t */
    time_t mtime;
    struct stat sbuf;
    int child_argc;
    char **child_argv;
    extern char **environ;
    int i;

    while (1) {
	if (recvcmd (&cmd) < 0) {
	    fprintf (stderr, "recvcmd () failed: %s\n",
		     pvm_strerror (pvm_errno));
	    break;
	}

#if 0
	fprintf (stderr, "cmd == %d\n", cmd);
#endif

	switch (cmd) {
	case versionCmd:
#if 0
	    fprintf (stderr, ">>> version\n");
#endif
	    reply (versionCmd, PROXY_PROTOCOL_VERSION);
	    sendresp ();
	    break;

	case createFileCmd:
	    if (fd != -1) {
		fprintf (stderr, "file %s was already open\n", dest_filename);
		reply (cmd, pProtocolErr);
		write_cleanup ();
		break;
	    }

	    pvm_upkstr (dest_filename);
	    pvm_upkint (&size, 1, 1);
	    pvm_upkint ((int *) &mtime, 1, 1);

#if 0
	    fprintf (stderr, ">>> create_file (%s %d %d)\n",
		     dest_filename, size, mtime);
#endif
	    
	    if (stat (dest_filename, &sbuf) == 0 &&
		sbuf.st_size == size && sbuf.st_mtime == mtime) {
		reply (cmd, pFileExistsErr);
	    }
	    else if ((fd = open (dest_filename, O_WRONLY|O_CREAT|O_TRUNC,
				 0600)) < 0) {
		reply (cmd, unixErrno (errno));
	    }
	    else {
		fdmode = MODE_WRITE;
		reply (cmd, 0);
	    }
	    sendresp ();
	    break;

	case createExclusiveFileCmd:
	    if (fd != -1) {
		fprintf (stderr, "file %s was already open\n", dest_filename);
		reply (cmd, pProtocolErr);
		write_cleanup ();
		break;
	    }

	    pvm_upkstr (dest_filename);
	    pvm_upkint (&size, 1, 1);
	    pvm_upkint ((int *) &mtime, 1, 1);

#if 0
	    fprintf (stderr, ">>> create_file_exclusive (%s %d %d)\n",
		     dest_filename, size, mtime);
#endif
	    /*
	     * do an exclusive file create over NFS.
	     * first, create a file with a unique name.
	     * then, try to link it to the file we want.
	     */
	    sprintf (tempname, "tmp%x.%x", getpid (), pvm_mytid ());
	    
	    if ((fd = open (tempname, O_WRONLY|O_CREAT|O_EXCL,
				 0600)) < 0) {
		reply (cmd, unixErrno (errno));
	    }
	    else if (link (tempname, dest_filename) < 0) {
		unlink (tempname);
		close (fd);
		fd = -1;
		reply (cmd,
		       errno == EEXIST ? pFileExistsErr : unixErrno (errno));
	    }
	    else {
		unlink (tempname);
		fdmode = MODE_WRITE;
		reply (cmd, 0);
	    }
	    sendresp ();
	    break;

	case writeFileCmd:
	    pvm_upkint (&nbytes, 1, 1);
#if 0
	    fprintf (stderr, ">>> write_file (nbytes=%d)\n", nbytes);
#endif
	    if (nbytes < 0 || nbytes > sizeof (buf)) {
		reply (cmd, pIllegalArgErr);
		write_cleanup ();
		break;
	    }
	    if (fd < 0 || fdmode != MODE_WRITE) {
		reply (cmd, pProtocolErr);
		write_cleanup ();
		break;
	    }
	    if ((x = pvm_upkbyte (buf, nbytes, 1)) < 0) {
		reply (cmd, x);
		write_cleanup ();
		break;
	    }
	    if (nbytes != 0) {
		if (write (fd, buf, nbytes) != nbytes) {
		    reply (cmd, unixErrno(errno));
		    write_cleanup ();
		    break;
		}
		else {
		    reply (cmd, nbytes);
		    sendresp ();
		}
	    }
	    else {
		/*
		 * file is complete.  make sure size is correct
		 */
		if (fstat (fd, &sbuf) < 0) {
		    reply (cmd, unixErrno(errno));
		    write_cleanup ();
		    break;
		}
		if (sbuf.st_size != size) {
		    reply (cmd, pProtocolErr);
		    write_cleanup ();
		    break;
		}
#if 0
		fprintf (stderr, "closing %s\n", dest_filename);
#endif
		if (close (fd) < 0) {
		    reply (cmd, unixErrno (errno));
		    write_cleanup ();
		    break;
		}
		fd = -1;
		/*
		 * set time of new file
		 */
		if (setfiletime (dest_filename, mtime) < 0) {
		    reply (cmd, unixErrno (errno));
		    write_cleanup ();
		    break;
		}
		else {
		    reply (cmd, 0);
		    sendresp ();
		}
	    }
	    break;

	case abortCmd:
#if 0
	    fprintf (stderr, ">>> abort\n");
#endif
	    reply (cmd, 0);
	    if (fdmode == MODE_WRITE)
		write_cleanup ();
	    else
		sendresp ();
	    break;

	case spawnCmd:
	    pvm_upkint (&child_argc, 1, 0);
	    child_argv = TALLOC (child_argc + 1, char *);
	    for (i = 0; i < child_argc; ++i) {
		pvm_upkstr (buf);
		child_argv[i] = expand_vars (buf, environ);
	    }
	    child_argv[i] = NULL;
#if 0
	    fprintf (stderr, ">>> spawn (");
	    for (i = 0; i < child_argc; ++i)
		fprintf (stderr, " '%s'", child_argv[i]);
	    fprintf (stderr, ")\n");
#endif
	    reply (spawnCmd, 0);
	    sendresp ();

	    /*
	     * XXX the current implementation of do_spawn will
	     * wait until the process has finished.  so we
	     * we won't read any more commands until this happens.
	     * might want to change this to allow us to accept
	     * commands (e.g. to kill the task), but will then
	     * need an 'alreadyinprogress' error for spawn.
	     */
	    do_spawn (parent_tid, child_argv);
	    for (i = 0; i < child_argc; ++i) {
		FREE (child_argv[i]);
		child_argv[i] = NULL;
	    }
	    FREE ((char *) child_argv);
	    child_argv = (char **) NULL;
	    break;

	default:
	    fprintf (stderr, "illegal command %d\n", cmd);
	    reply (illegalCmd, cmd);
	    if (fdmode == MODE_WRITE)
		write_cleanup ();
	    else
		sendresp ();
	    break;
	}
    }
}

#if 0
int
r_mkdir (dir, mode)
char *dir;
int mode;
{
    int x;

    if ((x = mkdir (dir, mode)) < 0 && errno == ENOENT) {
	char *slash;

	if ((slash = strrchr (dir, '/')) == NULL || slash == dir)
	    return x;
	*slash = '\0';
	x = r_mkdir (dir, mode);
	*slash = '/';
	if (x < 0)
	    return x;
	return mkdir (dir, mode);
    }
    return x;
}
#endif

main (argc, argv, envp)
int argc;
char **argv;
char **envp;
{
    char foobar[1024];
    char msgbuf[1024];
    int i, j;
    char **new_envp;

    /*
     * set up new environment
     */
    for (i = 0; envp[i]; ++i);	/* count current number of entries  */
    i += 2;			/* leave room for more */
    new_envp = (char **) malloc (i * sizeof (char *)); /* allocate space */

    /* copy existing entries, except for ARCH */
    j = 0;
    for (i = 0, j = 0; envp[i]; ++i)
	if (strncmp (envp[i], "ARCH=", 5) != 0)
	    new_envp[j++] = envp[i];

    sprintf (foobar, "ARCH=%s", ARCHSTR);
    new_envp[j++] = strdup (foobar);
    new_envp[j] = NULL;
    child_environ = new_envp;

    if ((my_tid = pvm_mytid ()) < 0) {
	fprintf (stderr, "pvm_mytid() failed .... is pvm running?\n");
	exit (1);
    }
    if ((parent_tid = pvm_parent ()) < 0) {
	fprintf (stderr, "pvm_parent() failed .... is pvm running?\n");
	exit (1);
    }

    /*
     * set up the child sig handler stuff
     */
    pipe (signal_pipe);
    signal (SIGCHLD, child_sig);

    /*
     * cd to the right place before starting
     * create directory if necessary
     */
    sprintf (foobar, "%s/pvm3/bin/%s/tmp", getenv("HOME"), ARCHSTR);
    if (chdir (foobar) < 0) {
	mkdir (foobar, 0755);
	errno = 0;
    }
    if (chdir (foobar) < 0) {
	sprintf (msgbuf, "chdir (%s) failed (%s)\n", foobar,
		 strerror (errno));
	fprintf (stderr, "%s", msgbuf);
    }
    else
	*msgbuf = '\0';

    /*
     * okay, do the nitty-gritty
     */
    work (msgbuf);
    exit (0);
}
