#include <stdio.h>

#include <X11/Intrinsic.h>

#include <Errno.h>
#include <Malloc.h>
#include <Stat.h>
#include <String.h>

#include "pvm3.h"

#include "config.h"
#include "costmat.h"
#include "global.h"
#include "proxy.h"
#include "pvmglue.h"

extern int pvm_errno;

struct queue {
    char *str;
    int subr;
    struct queue *next;
};

static struct per_host {
    int pvmd_tid;		/* tid of server pvmd */
    int proxyd_tid;		/* tid of proxyd on that server */
    struct queue *cmd_q_head;
    struct queue *cmd_q_tail;
    int status;			/* current proxyd state */
    int exit_status;		/* exit status of last command */
    char *partial_line;
} *per_host = NULL;


static void
delete_cmd_queue (h)
int h;
{
    if (per_host[h].cmd_q_head)
	per_host[h].cmd_q_head = per_host[h].cmd_q_head->next;
    if (per_host[h].cmd_q_head == NULL)
	per_host[h].cmd_q_tail = NULL;
}


static void
nuke_cmd_queue (h)
int h;
{
    struct queue *foo = per_host[h].cmd_q_head;
    struct queue *bar;
    while (foo) {
	bar = foo->next;
	FREE (foo);
	foo = bar;
    }
    per_host[h].cmd_q_head = NULL;
    per_host[h].cmd_q_tail = NULL;
}

/*
 * per_host status values...
 *
 * negative status values mean process has died.
 * positive status values mean process is alive.
 */
#define DEAD	-1		/* proxyd died */
#define UNBORN	0		/* proxyd never started */
#define IDLE	1		/* proxyd started, not busy */
#define RUNNING	2		/* proxyd is running a program */


static int num_hosts = 0;
static int pvm_status = 0;

char *
proxy_strerror (x)
int x;
{
    static char buf[20];

    if (isPvmErr (x))
	return pvm_strerror (x);
    if (isUnixErr (x))
	return strerror (unixErrno (x));

    switch (x) {
    case pBadHostErr:
	return "bad host number";
    case pIllegalCmdErr:
	return "illegal proxy cmd code";
    case pHostDiedErr:
	return "host died";
    case pServerDiedErr:
	return "proxyd server died";
    case pVersionErr:
	return "server version mismatch";
    case pFileExistsErr:
	return "file already exists";
    case pProtocolErr:
	return "protocol error";
    case pIllegalArgErr:
	return "illegal argument";
    default:
	sprintf (buf, "error %d", -x);
	return buf;
    }
}

/*
 * return an error code if we know that we cannot reach the remote proxyd.
 * This might be because the proxyd died, host died, or pvm died.
 */

static int 
hostok (host)
int host;
{
    if (per_host == NULL)
	return pBadHostErr;
    if (host < 0 || host >= num_hosts)
	return pBadHostErr;
    if (pvm_status < 0)
	return pvm_status;
    if (per_host[host].status < 0)
	return per_host[host].status;
    return 0;
}

/*
 * Handle PvmSysErr error codes from pvm.  If 'x' is PvmSysErr,
 * set the global pvm status that inhibits future pvm commands.
 *
 * XXX for every host with a process outstanding, fake obituary
 * messages.
 */

static int
pvmsyserr (x)
int x;
{
    if (x == PvmSysErr)
	pvm_status = PvmSysErr;
    return x;
}

static void
set_cell_color (h, s, color)
int h, s, color;
{
    if (cm_SetColor (global.costMatrix, h, s, color)) {
	config_repaint_cells (h, s);
	XFlush (global.display);
    }
}

/*
 * handle "asychronous" messages, i.e. those that may occur at any
 * time (not just in response to a command).
 */

static void
async_msg (msgtag, tid)
int msgtag;
int tid;
{
    int remote_tid;
    int h;
    int length;
    char buf[20];
    static char *buf2 = NULL;
    static int buf2len; 
    int exit_status;
    int tmp;
    char *ptr;

#if 0
    fprintf (stderr, "async_msg (msgtag=%d)\n", msgtag);
#endif

    if (buf2 == NULL) {
	buf2len = 10240;
	buf2 = (char *) MALLOC (buf2len);
    }
    switch (msgtag) {

    case pvmStdoutTag:
	pvm_unpackf ("%d %d", &remote_tid, &length);
	/*
	 * figure out which host this is
	 */
	for (h = 0; h < num_hosts; ++h) {
	    if (remote_tid == per_host[h].proxyd_tid)
		break;
	}
	if (length > 0) {
	    if (length + 1024 > buf2len) {
		buf2len = length + 1024;
		buf2 = (char *) REALLOC ((void *) buf2, buf2len);
	    }
	    /*
	     * if it's not on our list (wha?)
	     * just print out the line with the tid
	     */
	    if (h == num_hosts) {
		sprintf (buf2, "[t%08x] ", remote_tid);
		pvm_upkbyte (buf2 + 12, length, 1);
		ts_append_l (global.pvm_sink, buf2, length + 12);
		return;
	    }
	    /*
	     * okay, it's one of ours.
	     * if there's a partial line left over from last
	     * time, prepend it to the buffer before reading
	     * anything else.
	     *
	     * 'tmp' keeps track of the current position in
	     * the buffer.
	     */
	    tmp = 0;
	    if (per_host[h].partial_line) {
		strcpy (buf2, per_host[h].partial_line);
		tmp = strlen (buf2);
		FREE (per_host[h].partial_line);
		per_host[h].partial_line = NULL;
	    }
	    /*
	     * now read the new data into the buffer.
	     * XXX really should do a length check before unpacking
	     */
	    pvm_upkbyte (buf2 + tmp, length, 1);
	    tmp += length;
	    buf2[tmp] = '\0';
	    /*
	     * print out each complete line, prefixed with
	     * the remote host name.  anything that's left over,
	     * save for the next time.
	     */
	    tmp = 0;
	    sprintf (buf, "[%s]: ", global.costMatrix->hosts[h]);
	    while (ptr = strchr (buf2 + tmp, '\n')) {
		int len = ptr - (buf2 + tmp) + 1;

		ts_append (global.pvm_sink, buf);
		ts_append_l (global.pvm_sink, buf2 + tmp, len);
		tmp += len;
	    }
	    if (buf2[tmp] != '\0')
		per_host[h].partial_line = STRDUP (buf2 + tmp);
	}
	/*
	 * XXX negative length codes are undocumented.
	 * this is from notes that BoB sent me.
	 */
	else if (length == 0) {
	    /* EOF from task */
	    if (h == num_hosts)
		sprintf (buf2, "[t%08x] EOF\n", remote_tid);
	    else
		sprintf (buf2, "[%s] EOF\n", global.costMatrix->hosts[h]);
	    ts_append (global.pvm_sink, buf2);
	}
	else if (length == -1) {
	    /* new task */
	}
	else if (length == -2) {
	    /* first output from task */
	}
	break;

    case hostDiedTag:
	pvm_unpackf ("%d", &remote_tid);
	for (h = 0 ; h < num_hosts; ++h) {
	    if (remote_tid == per_host[h].pvmd_tid) {
		per_host[h].status = pHostDiedErr;
		/* mark host as being broken in cost matrix */
		set_cell_color (h, -1, COLOR_BROKEN);
		msg_Format ("host %s (%08x) died.  R.I.P.\n",
			    global.costMatrix->hosts[h],
			    remote_tid);
		break;
	    }
	}
	break;

    case serverDiedTag:
	pvm_unpackf ("%d", &remote_tid);
	for (h = 0 ; h < num_hosts; ++h) {
	    if (remote_tid == per_host[h].proxyd_tid) {
		per_host[h].status = pServerDiedErr;
		/* mark host as being broken in cost matrix */
		set_cell_color (h, -1, COLOR_BROKEN);
		msg_Format ("proxyd on %s (%08x) died.  R.I.P.\n",
			    global.costMatrix->hosts[h],
			    remote_tid);
		break;
	    }
	}
	break;

    case cmdExitTag:
	pvm_unpackf ("%d", &exit_status);
	for (h = 0; h < num_hosts; ++h) {
	    if (tid == per_host[h].proxyd_tid) {
		/*
		 * if there is any output left over, print it now.
		 */		 
		if (per_host[h].partial_line) {
		    char buf[10240];

		    sprintf (buf, "[%s] %s\n", global.costMatrix->hosts[h],
			     per_host[h].partial_line);
		    ts_append (global.pvm_sink, buf);
		    FREE (per_host[h].partial_line);
		    per_host[h].partial_line = NULL;
		}
		per_host[h].exit_status = exit_status;
		per_host[h].status = IDLE;
		if (exit_status == 0) {
		    /*
		     * HACK! if next entry in cmd queue is not for the
		     * same subr, mark this subr as done.
		     */
		    if (per_host[h].cmd_q_head->subr >= 0 &&
			(per_host[h].cmd_q_head->next == NULL ||
			 per_host[h].cmd_q_head->next->subr !=
			 per_host[h].cmd_q_head->subr)) {
			set_cell_color (h, per_host[h].cmd_q_head->subr,
					COLOR_DONE);
		    }
		    delete_cmd_queue (h);
		    if (per_host[h].cmd_q_head == NULL)
			set_cell_color (h, -1, COLOR_DONE);
		}
		else {
		    msg_Format ("%s: '%.40s...' failed with status %d\n",
				global.costMatrix->hosts[h],
				per_host[h].cmd_q_head->str,
				exit_status);
		    set_cell_color (h, per_host[h].cmd_q_head->subr,
				    COLOR_BROKEN);
		    nuke_cmd_queue (h);
		}
		return;
	    }
	}
	msg_Format ("unexpected cmdExit(%d) msg from %08x\n",
		    exit_status, tid);
	break;

    case cmdStdoutTag:
    case cmdStderrTag:
	tmp = 0;
	for (h = 0; h < num_hosts; ++h) {
	    if (per_host[h].proxyd_tid == tid) 
		break;
	}

	if (h == num_hosts) {
	    sprintf (buf2, "[t%08x] ", tid);
	    tmp = strlen (buf2);
	    pvm_upkint (&length, 1, 1);
	    pvm_upkbyte (buf2 + tmp, length, 1);
	    tmp += length;
	    ts_append_l (global.pvm_sink, buf2, tmp);
	    if (buf2[tmp - 1] != '\n')
		ts_append (global.pvm_sink, "\n");
	    return;
	}

	/*
	 * if there is a partial line left over from last time,
	 * prepend it to the buffer before reading anything else.
	 */
	if (per_host[h].partial_line) {
	    strcpy (buf2, per_host[h].partial_line);
	    tmp = strlen (buf2);
	    FREE (per_host[h].partial_line);
	    per_host[h].partial_line = NULL;
	}
	/*
	 * now read the new data into the buffer.
	 */
	pvm_upkint (&length, 1, 1);
	pvm_upkbyte (buf2 + tmp, length, 1);
	tmp += length;
	buf2[tmp] = '\0';
	/*
	 * print out each complete line, prefixed with the
	 * host name.  if any partial line is left over,
	 * save it for next time.
	 */
	tmp = 0;
	sprintf (buf, "[%s] ", global.costMatrix->hosts[h]);
	while (ptr = strchr (buf2 + tmp, '\n')) {
	    int len = ptr - (buf2 + tmp) + 1;

	    ts_append (global.pvm_sink, buf);
	    ts_append_l (global.pvm_sink, buf2 + tmp, len);
	    tmp += len;
	}
	if (buf2[tmp] != '\0')
	    per_host[h].partial_line = STRDUP (buf2 + tmp);
	break;

    case pvmDiedTag:
	break;

    default:
	msg_Format ("async_msg: unexpected msg tag %d from %08x\n",
		    msgtag, tid);
	break;
    }
}

/*
 * send a command to the remote host and wait for a response.
 * when the response comes, see if it is a reply to this command.
 *
 * handle other kinds of messages while we are at it, including
 * notifications that a particular host died.
 */

static int
docmd (host, cmd)
int host;
int cmd;
{
    int msgtag, tid, resp, code;
    int x;

    if ((x = hostok (host)) < 0)
	return x;
    if ((x = pvm_send (per_host[host].proxyd_tid, commandTag)) < 0)
	return pvmsyserr (x);
    while (1) {
	/*
	 * if host or proxyd or pvm is dead, 
	 * break out of loop and return an error
	 */
	if ((x = hostok (host)) < 0)
	    return x;
	/*
	 * flush any pending output to the display
	 */
	XFlush (global.display);
	if ((x = pvmglue_recv (-1, -1)) < 0)
	    return pvmsyserr (x);
	if ((x = pvm_bufinfo (x, NULL, &msgtag, &tid)) < 0)
	    return pvmsyserr (x);

#if 0
	fprintf (stderr, "docmd: msg type %d from %x\n", msgtag, tid);
#endif

	switch (msgtag) {

	case responseTag:
	    if ((x = pvm_upkint (&resp, 1, 1)) < 0) {
		msg_Format ("docmd: error unpacking resp: %s\n",
			    pvm_strerror (pvm_errno));
		continue;
	    }
	    if (tid != per_host[host].proxyd_tid) {
		msg_Format ("docmd: unsolicited response %d from %08x\n",
			    resp, tid);
		continue;
	    }
	    if (resp == cmd) {
		if ((x = pvm_upkint (&code, 1, 1)) < 0) {
		    msg_Format ("docmd: error unpacking code: %s\n",
				pvm_strerror (pvm_errno));
		    continue;
		}
#if 0
		fprintf (stderr, "docmd: cmd=%d resp=%d code=%d\n",
			 cmd, resp, code);
#endif
		return code;
	    }
	    if (resp == illegalCmd)
		return pIllegalCmdErr;
	    else {
		msg_Format ("docmd: resp %d from %08x does not match req %d\n",
			    resp, tid, cmd);
		return pIllegalCmdErr;
	    }
	    break;

	default:
	    async_msg (msgtag, tid);
	    break;
	}
    }
}

int
proxy_init (nhosts)
int nhosts;
{
    int i;

    if (per_host)
	FREE (per_host);
    per_host = TALLOC (nhosts, struct per_host);
    for (i = 0; i < nhosts; ++i) {
	per_host[i].pvmd_tid = -1;
	per_host[i].proxyd_tid = -1;
	per_host[i].status = UNBORN;
	per_host[i].cmd_q_head = (struct queue *) NULL;
	per_host[i].cmd_q_tail = (struct queue *) NULL;
	per_host[i].partial_line = NULL;
    }
    num_hosts = nhosts;
    pvm_status = 0;
    pvm_setopt (PvmOutputTid, pvm_mytid ());
    pvm_setopt (PvmOutputCode, pvmStdoutTag);
    return 0;
}

int
proxy_start_server (host)
int host;
{
    int proxyd_tid;
    int pvmd_tid;
    char *hostname;
    int x;

    if (pvm_status != 0)
	return pvm_status;
    if (host < 0 || host >= num_hosts)
	return pBadHostErr;
    if (host >= global.costMatrix->nhost)
	return pBadHostErr;
    hostname = global.costMatrix->hosts[host];
    if (hostname == NULL || *hostname == '\0')
	return pBadHostErr;
	
    if ((proxyd_tid = pvmglue_spawn ("proxyd", hostname)) < 0)
	return proxyd_tid;
    per_host[host].proxyd_tid = proxyd_tid;
    per_host[host].status = IDLE;
    if ((x = pvmglue_notify_taskexit (proxyd_tid, serverDiedTag)) < 0)
	return x;
    if ((pvmd_tid = pvmglue_notify_hostexit (hostname, hostDiedTag)) < 0)
	return pvmd_tid;
    per_host[host].pvmd_tid = pvmd_tid;
    return 0;
}

int
proxy_shutdown_server (host)
int host;
{
    int x;

#if 0
    fprintf (stderr, "shutting down proxyd on %s\n",
	     global.costMatrix->hosts[host]);
#endif
    if (per_host[host].proxyd_tid > 0)
	return pvm_kill (per_host[host].proxyd_tid);
}


int
proxy_check_server_version (host)
int host;
{
    int x;
    int version;

    if (pvm_status != 0)
	return pvm_status;
    pvm_packf ("%+ %d", PvmDataDefault, versionCmd);
    if ((x = docmd (host, versionCmd)) < 0)
	return x;
    if (x != PROXY_PROTOCOL_VERSION)
	return pVersionErr;
    return 0;
}

int
proxy_copy_file (src_filename, dest_host, dest_filename)
char *src_filename;
int dest_host;
char *dest_filename;
{
    struct stat sbuf;
    FILE *fp;
    long bytestosend;
    char buf[PROXY_FILEBUFSIZE];
    int nbytes;
    int x;
    int errcode = 0;

    /*
     * if we can't open the file on our end, punt.
     */
    if ((fp = fopen (src_filename, "r")) == NULL)
	return unixErrno (errno);
    if (fstat (fileno (fp), &sbuf) < 0) {
	fclose (fp);
	return unixErrno (errno);
    }

    pvm_packf ("%+ %d %s %d %d",  PvmDataDefault, createFileCmd,
	       dest_filename, sbuf.st_size, sbuf.st_mtime);

    x = docmd (dest_host, createFileCmd);
#if 0
    fprintf (stderr, "create_file (%s %d %d) => %s\n",
	     dest_filename, sbuf.st_size, sbuf.st_mtime, proxy_strerror (x));
#endif
    if (x < 0) {
	fclose (fp);
	return x;
    }

    bytestosend = sbuf.st_size;
    while (bytestosend > 0) {
	nbytes = fread (buf, 1, sizeof (buf), fp);
	if (nbytes < 0) {
	    errcode = unixErrno (errno);
	    goto abort;
	}
	pvm_packf ("%+ %d %d %*c", PvmDataDefault,
		   writeFileCmd, nbytes, nbytes, buf);
	x = docmd (dest_host, writeFileCmd);
#if 0
	fprintf (stderr, "write_file (%d) => %s\n", nbytes,
		 proxy_strerror (x));
#endif
	if (x < 0) {
	    fclose (fp);
	    return x;
	}
	bytestosend -= nbytes;
    }
    fclose (fp);
    pvm_packf ("%+ %d %d", PvmDataDefault, writeFileCmd, 0);
    x = docmd (dest_host, writeFileCmd);
#if 0
	fprintf (stderr, "write_file (0) => %s\n", proxy_strerror (x));
#endif
    if (x < 0)
	return x;
    return 0;

 abort:
    pvm_packf ("%+ %d", PvmDataDefault, abortCmd);
    (void) docmd (dest_host, abortCmd);
#if 0
    fprintf (stderr, "abort()\n");
#endif
    fclose (fp);
    return errcode;
}


int
proxy_copy_lock_file (src_filename, dest_host, dest_filename)
char *src_filename;
int dest_host;
char *dest_filename;
{
    struct stat sbuf;
    FILE *fp;
    long bytestosend;
    char buf[PROXY_FILEBUFSIZE];
    int nbytes;
    int x;
    int errcode = 0;

    /*
     * if we can't open the file on our end, punt.
     */
    if ((fp = fopen (src_filename, "r")) == NULL)
	return unixErrno (errno);
    if (fstat (fileno (fp), &sbuf) < 0) {
	fclose (fp);
	return unixErrno (errno);
    }

    pvm_packf ("%+ %d %s %d %d",  PvmDataDefault, createExclusiveFileCmd,
	       dest_filename, sbuf.st_size, sbuf.st_mtime);

    x = docmd (dest_host, createExclusiveFileCmd);
#if 0
    fprintf (stderr, "create_file_exclusive (%s) => %s\n",
	     dest_filename, proxy_strerror (x));
#endif
    if (x < 0) {
	fclose (fp);
	return x;
    }

    bytestosend = sbuf.st_size;
    while (bytestosend > 0) {
	nbytes = fread (buf, 1, sizeof (buf), fp);
	if (nbytes < 0) {
	    errcode = unixErrno (errno);
	    goto abort;
	}
	pvm_packf ("%+ %d %d %*c", PvmDataDefault,
		   writeFileCmd, nbytes, nbytes, buf);
	x = docmd (dest_host, writeFileCmd);
#if 0
	fprintf (stderr, "write_file (%d) => %s\n", nbytes,
		 proxy_strerror (x));
#endif
	if (x < 0) {
	    fclose (fp);
	    return x;
	}
	bytestosend -= nbytes;
    }
    fclose (fp);
    pvm_packf ("%+ %d %d", PvmDataDefault, writeFileCmd, 0);
    x = docmd (dest_host, writeFileCmd);
#if 0
	fprintf (stderr, "write_file (0) => %s\n", proxy_strerror (x));
#endif
    if (x < 0)
	return x;
    return 0;

 abort:
    pvm_packf ("%+ %d", PvmDataDefault, abortCmd);
    (void) docmd (dest_host, abortCmd);
#if 0
    fprintf (stderr, "abort()\n");
#endif
    fclose (fp);
    return errcode;
}


/*
 * queue up a command on a particular host.
 */

int
proxy_queue_cmd (host, subr, cmd)
int host;
int subr;
char *cmd;
{
    int x;
    struct queue *q;

    if ((x = hostok (host)) < 0)
	return x;

    q = TALLOC (1, struct queue);
    q->str = STRDUP (cmd);
    q->subr = subr;
    q->next = NULL;
    if (per_host[host].cmd_q_head == NULL)
	per_host[host].cmd_q_head = q;
    if (per_host[host].cmd_q_tail != NULL)
	per_host[host].cmd_q_tail->next = q;
    per_host[host].cmd_q_tail = q;
    per_host[host].exit_status = 0;
#if 0
    fprintf (stderr,
	     "%s: cmd '%s' queued\n", global.costMatrix->hosts[host],
	     cmd);
#endif
    return 0;
}

/*
 * execute queued commands until all are done.
 */

#define READY(h) (per_host[h].status == IDLE && per_host[h].exit_status == 0 \
		  && per_host[h].cmd_q_head != NULL)
int
proxy_run_cmds ()
{
    int h;
    char *cmd;
    struct queue *tmp;
    int pending ;
    int x;
    int msgtag;
    int tid;

#if 0
    fprintf (stderr, "proxy_run_cmds ()\n");
#endif

    for (h = 0; h < num_hosts; ++h)
	per_host[h].exit_status = 0;

    do {
	pending = 0;
	for (h = 0; h < num_hosts; ++h) {
	    /*
	     * if host is ready to run, feed it.
	     */
	    if (READY (h)) {
		cmd = per_host[h].cmd_q_head->str;
#if 0
		tmp = per_host[h].cmd_q_head;
		per_host[h].cmd_q_head = per_host[h].cmd_q_head->next;
		FREE (tmp);
#endif
		pvm_packf ("%+ %d", PvmDataDefault, spawnCmd);
		pvm_packf ("%d %s %s %s", 3, "/bin/sh", "-c", cmd);
#if 0
		fprintf (stderr, "%s: running '%s'\n",
			 global.costMatrix->hosts[h], cmd);
#endif
		if ((x = docmd (h, spawnCmd)) < 0) {
		    msg_Format ("Error: can't run \"%s\" on %s: %s\n",
				cmd, global.costMatrix->hosts[h],
				proxy_strerror (x));
		    per_host[h].status = IDLE;
		    per_host[h].exit_status = -1;
		}
		per_host[h].status = RUNNING;
		/*
		 * figure out which subr we are compiling
		 */
		set_cell_color (h, per_host[h].cmd_q_head->subr,
				COLOR_RUNNING);
	    }
	    if (per_host[h].status == RUNNING)
		pending++;
	}
	if (pending) {
	    if ((x = pvmglue_recv (-1, -1)) < 0) {
		pvmsyserr (x);
		msg_Format ("pvmglue_recv: %s\n", pvm_strerror (x));
	    }
	    if ((x = pvm_bufinfo (x, NULL, &msgtag, &tid)) < 0) {
		pvmsyserr (x);
		msg_Format ("pvm_bufinfo: %s\n", pvm_strerror (x));
	    }
	    async_msg (msgtag, tid);
	}
    } while (pending > 0);

    for (h = 0; h < num_hosts; ++h) {
	if (per_host[h].exit_status != 0) {
#if 0
	    msg_Format ("last command on %s exited with status %d\n",
			global.costMatrix->hosts[h], per_host[h].exit_status);
#endif
	    return -1;
	}
    }
    return 0;
}

#if 0
char *
proxy_get_compile_cmd (host, language, output, source, wrapper)
int host;
int language;
char *output;
char *source;
char *wrapper;
{
    int x;
    static char compile_command[1024];

    if ((x = hostok (host)) < 0) {
	goto abort;
    }
    pvm_packf ("%+ %d %s %s %s %s", PvmDataDefault, getCompileCmd,
	       language == LANG_C ? "c" : "f",
	       output, source, wrapper);
    if ((x = docmd (dest_host, getCompileCmd)) < 0)
	goto abort;

    if ((x = pvm_unpackf ("%s", compile_command)) >= 0)
	return compile_command;

 abort:
    msg_Format ("proxy_get_compile_cmd: error on %s: %s\n",
		HOSTNAME(host), pvm_strerror (x));
    return NULL;
}
#endif
