#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xos.h>

#include <X11/IntrinsicP.h>
#include <X11/Xlibint.h>
#include "Canvas.h"
#include "CanvasP.h"

#include "global.h"
#include "help.h"

#include "helptext.c"

#define MARGIN 10		/* left/right/top margin */
#define HPAD 10			/* horizontal space around table entry */
#define VPAD 3			/* vertical space above/below table entry */
#define LW 1			/* line width */


static int inited = 0;
static GC gc;

struct entity {
    char *name;			/* name of the entity */
    int type;			/* entity type */
#define CHAR 0
#define PIXMAP 1
    int code;			/* character code or other cookie */
    void *bits;			/* pointer to pixmap or other data */
};

static struct entity entity_list[20] = {
    { "amp", CHAR, '&', },
    { "lt",  CHAR, '<', },
    { "gt",  CHAR, '>', },
    { NULL, }, 
};

/*
 * add an entity to the list.  used during initialization
 */

static void
add_entity (name, type, code, bits)
char *name;
int type;
int code;
void *bits;
{
    static int next_slot = -1;

    if (next_slot == -1) {
	int i;

	for (i = 0; i < sizeof (entity_list) / sizeof (*entity_list); ++i)
	    if (entity_list[i].name == NULL)
		break;
	next_slot = i;
    }
    if (next_slot < sizeof (entity_list) / sizeof (*entity_list)) {
	entity_list[next_slot].name = name;
	entity_list[next_slot].type = type;
	entity_list[next_slot].code = code;
	entity_list[next_slot].bits = bits;
	++next_slot;
    }
}


/*
 * find an entity in the list.  used by display and getsize routines
 */

static struct entity *
find_entity (name, length)
char *name;
int length;
{
    int i;

    for (i = 0; i < sizeof (entity_list) / sizeof (*entity_list); ++i) {
	if (strncmp (name, entity_list[i].name, length) == 0 &&
	    entity_list[i].name[length] == '\0')
	    return &entity_list[i];
    }
    return (struct entity *) NULL;
}
	

/*
 * given a pointer to the first character in an entity name, return
 * its size.
 */

static char *
get_entity_size (w, ptr, ascent, descent, width)
Widget w;
char *ptr;
int *ascent, *descent, *width;
{
    int length = 0;
    struct entity *foo;
    XFontStruct *fs = ((CanvasWidget) w)->canvas.fontinfo;
    XCharStruct *bar;
    Pix p;

    if (*ptr != '&') {
	*ascent = *descent = *width = 0;
	return ptr + 1;
    }
    ++ptr;

    while (ptr[length] != '\0' && ptr[length] != ';' && ptr[length] != '\n')
	++length;

    if ((foo = find_entity (ptr, length)) == (struct entity *) NULL) {
	*ascent = *descent = *width = 0;
	return ptr;
    }

    switch (foo->type) {
    case CHAR:
	CI_GET_CHAR_INFO_1D (fs, foo->code, NULL, bar);
	if (bar) {
	    *ascent = bar->ascent;
	    *descent = bar->descent;
	    *width = bar->width;
	}
	else
	    *ascent = *descent =  *width = 0;
	break;

    case PIXMAP:
	p = (Pix) foo->bits;
	*ascent = p->height;
	*descent = 0;
	*width = p->width;
	break;

    default:
	*ascent = *descent = *width = 0;
	break;
    }
    return ptr[length] == ';' ? ptr + length + 1 : ptr + length;
}

/*
 * Display an entity on widget w's window, at location (*x, *y).
 * Update *x and *y to point to the location of the next character.
 * Should be called with ptr pointing to the '&' before the entity name.
 * Returns with ptr pointing to the next character following the semicolon
 * after the entity name.
 */

static char *
display_entity (w, ptr, x, y)
Widget w;
char *ptr;
int *x;
int *y;
{
    int length = 0;
    struct entity *foo;
    XFontStruct *fs = ((CanvasWidget) w)->canvas.fontinfo;
    char splat;
    XCharStruct *bar;
    Pix pix;

    /* ptr must point to a semicolon, else skip this character */
    if (*ptr != '&')
	return ptr + 1;
    ++ptr;

    /* figure out how long the entity name is */
    while (ptr[length] != '\0' && ptr[length] != ';' && ptr[length] != '\n')
	++length;

    /*
     * find the entity name in the table.  if not there, just return
     * the next character so the entity name will print out as text.
     *
     * XXX maybe should display a '&'.
     */
    if ((foo = find_entity (ptr, length)) == (struct entity *) NULL)
	return ptr;

    switch (foo->type) {

    case CHAR:
	CI_GET_CHAR_INFO_1D (fs, foo->code, NULL, bar);
	if (bar) {
	    XDrawString (XtDisplay (w), XtWindow (w), gc, *x, *y, &splat, 1);
	    *x += bar->width;
	}
	break;

    case PIXMAP:
	pix = (Pix) foo->bits;

	XCopyPlane (XtDisplay (w),	/* display */
		    pix->pm,		/* src drawable */
		    XtWindow (w),	/* dst drawable */
		    gc,			/* gc */
		    0,			/* src x */
		    0,			/* src y */
		    pix->width,		/* width */
		    pix->height,	/* height */
		    *x,		 	/* dst x */
		    *y - pix->height,	/* dst y */
		    1);
	*x += pix->width;
	break;

    default:
	break;
    }
    return ptr[length] == ';' ? ptr + length + 1 : ptr + length;
}

/*****************************************************************
 *			    TABLE SUPPORT                        *
 *****************************************************************/
/*
 * return the width in pixels of the string
 *
 * XXX eventually hack to handle entities, multiple fonts,
 * and multi-line strings.  For now, table entries have to
 * be a single line with no entities.
 */

static int
string_width (w, s)
Widget w;
char *s;
{
    CanvasWidget cw = (CanvasWidget) w;
    return XTextWidth (cw->canvas.fontinfo, s, strlen (s));
}

/*
 * return the height of the line.  Knows about margins so it can
 * figure out the right height even if the line wraps -- it only
 * returns the height of the unwrapped portion
 */

static void
get_line_height (w, ptr, ap, dp)
Widget w;
char *ptr;
int *ap;
int *dp;
{
    register int ascent, descent, width;
    int max_width = width_of (w);
    XFontStruct *fs = ((CanvasWidget) w)->canvas.fontinfo;
    register XCharStruct *x;

    ascent = 0;
    descent = 0;
    width = 0;

    if (*ptr == '\n') {
	*ap = fs->max_bounds.ascent;
	*dp = fs->max_bounds.descent;
	return;
    }
    while (*ptr && *ptr != '\n') {
	if (*ptr == '&') {
	    int x_ascent = 0;
	    int x_descent = 0;
	    int x_width = 0;

	    ptr = get_entity_size (w, ptr, &x_ascent, &x_descent, &x_width);
	    if (width + x_width + MARGIN > max_width)
		break;
	    if (x_ascent > ascent)
		ascent = x_ascent;
	    if (x_descent > descent)
		ascent = x_descent;
	}
	else {
	    CI_GET_CHAR_INFO_1D (fs, (*ptr), NULL, x);
	    if (x) {
		if (width + x->width + MARGIN > max_width)
		    break;
		if (x->ascent > ascent)
		    ascent = x->ascent;
		if (x->descent > descent)
		    descent = x->descent;
		width += x->width;
	    }
	    ++ptr;
	}
    }
    *ap = ascent;
    *dp = descent;
}

/*
 * Function to draw the mouse binding table
 *
 * Sigh.  This seems to be the easiest way to have it look nice.
 * To be really fancy it could check the translations for mouse
 * events, so it would always be correct.
 */

/* private data */
static char *rows[3][4] = {
    { "none", "add node", "draw arc", "edit node pgm" },
    { "ctrl", "delete node", "delete arc", "undefined" },
    { "shift", "move node", "move arc", "edit source" },
};

static char *heading = "mouse button";
static int cw[4];		/* colmun widths */

/*
 * helper function to center a string in a table cell
 */

static void
center_string (w, col, x, y, str)
Widget w;
int col;			/* column number */
int x;				/* left margin of graph */
int y;				/* baseline */
char *str;
{
    int i;

    x += LW;			/* width of line at left of table*/
    for (i = 0; i < col; ++i)
	x += LW + cw[i];	/* add width of col + line at right of col */

    x += (cw[col] - string_width (w, str)) / 2;
    XDrawString (XtDisplay (w), XtWindow (w), gc, x, y, str, strlen (str));
}

static void
display_mouse_bindings (w, x0, y0)
Widget w;
int x0, y0;
{
    int x, y;
    int ascent, descent;
    int col, row;

    if (cw[0] == 0) {
	for (col = 0; col < 4; ++col) {
	    int width = 0;

	    for (row = 0; row < 3; ++row) {
		width = string_width (w, rows[row][col]);
		if (width > cw[col])
		    cw[col] = width;
	    }
	    cw[col] += 2 * HPAD; /* add margins */
	}
    }

    y = y0;
    
    /*
     * draw heading centered above the right three columns, with
     * line below stretching across the same.
     */
    get_line_height (w, heading, &ascent, &descent);
    x = x0 + LW + cw[0] + LW + ((cw[1] + LW + cw[2] + LW + cw[3]) -
	 string_width (w, heading)) / 2;
    y += ascent + VPAD;
    XDrawString (XtDisplay (w), XtWindow (w), gc, x, y, heading,
		 strlen(heading));
    y += descent + VPAD;
    XDrawLine (XtDisplay (w), XtWindow (w), gc,
	       x0 + LW + cw[0],
	       y,
	       x0 + LW + cw[0] + LW + cw[1] + LW + cw[2] + LW + cw[3] + LW,
	       y);
    y += LW;

    /*
     * now draw column headings
     */
    get_line_height (w, "left middle right", &ascent, &descent);

    /* vertical lines between them */
    XDrawLine (XtDisplay (w), XtWindow (w), gc,
	       x0 + LW + cw[0],
	       y,
	       x0 + LW + cw[0],
	       y + ascent + descent + 2 * VPAD);
    XDrawLine (XtDisplay (w), XtWindow (w), gc,
	       x0 + LW + cw[0] + LW + cw[1],
	       y,
	       x0 + LW + cw[0] + LW + cw[1],
	       y + ascent + descent + 2 * VPAD);
    XDrawLine (XtDisplay (w), XtWindow (w), gc,
	       x0 + LW + cw[0] + LW + cw[1] + LW + cw[2],
	       y,
	       x0 + LW + cw[0] + LW + cw[1] + LW + cw[2],
	       y + ascent + descent + 2 * VPAD);
    XDrawLine (XtDisplay (w), XtWindow (w), gc,
	       x0 + LW + cw[0] + LW + cw[1] + LW + cw[2] + LW + cw[3],
	       y,
	       x0 + LW + cw[0] + LW + cw[1] + LW + cw[2] + LW + cw[3],
	       y + ascent + descent + 2 * VPAD);

    /* now the text */
    y += ascent + VPAD;

    center_string (w, 1, x0, y, "left");
    center_string (w, 2, x0, y, "middle");
    center_string (w, 3, x0, y, "right");
    
    /* now the line separating columns from the main table */
    y += descent + VPAD;
    XDrawLine (XtDisplay (w), XtWindow (w), gc,
	       x0,
	       y,
	       x0 + LW + cw[0] + LW + cw[1] + LW + cw[2] + LW + cw[3] + LW,
	       y);
    y += LW;

    for (row = 0; row < 3; ++row) {
	int row_height = 0;

	/* find out how high this row is
	 *
	 * XXX probably should use the maximal height
	 * for this font, rather than the height of the
	 * text; if no entry in this column has descenders
	 * the column will not be as high as the others,
	 * even if a uniform font is used.
	 */
	for (col = 0; col < 4; ++col) {
	    get_line_height (w, rows[row][col], &ascent, &descent);
	    if (row_height < ascent + descent)
		row_height = ascent + descent;
	}
	row_height += 2 * VPAD;

	/* draw each row */
	x = x0;
	/* draw the vertical line on the left edge of the table */
	XDrawLine (XtDisplay (w), XtWindow (w), gc,
		   x, y, x, y + row_height);
	x += LW;
	/*
	 * for each column, draw the text centered in the column,
	 * followed by the line at the right edge of that column
	 */
	for (col = 0; col < 4; ++col) {
	    center_string (w, col, x0, y + VPAD + ascent, rows[row][col]);
	    x += cw[col];
	    XDrawLine (XtDisplay (w), XtWindow (w), gc,
		       x, y, x, y + row_height);
	    x += LW;
	}
	y += row_height;
	XDrawLine (XtDisplay (w), XtWindow (w), gc,
		   x0,
		   y,
		   x0 + LW + cw[0] + LW + cw[1] + LW + cw[2] + LW + cw[3] + LW,
		   y);
	y += LW;
    }
}


/*****************************************************************
 *			 TEXT DRAWING SUPPORT                    *
 *****************************************************************/

/*
 * draw text up to the point where it should wrap
 * return a ptr to the next character to be displayed
 * *wrap is set to 1 iff the text would have wrapped
 *
 * XXX do word-wrapping rather than character wrapping
 */

static char *
display_text (w, ptr, x, y, wrap)
Widget w;
char *ptr;
int *x, *y;
int *wrap;
{
    XFontStruct *fs = ((CanvasWidget) w)->canvas.fontinfo;
    register XCharStruct *cs;
    int max_width = width_of (w) - MARGIN;
    char *p0;
    XTextItem foo[1];
    int width;

    p0 = ptr;
    width = 0;
    while (*ptr && *ptr != '\n' && *ptr != '&') {
	CI_GET_CHAR_INFO_1D (fs, (*ptr), NULL, cs);
	/*
	 * if there's not enough room to display this character,
	 * stop here
	 */
	if ((cs != NULL) && (*x + width + cs->width) > max_width) {
	    *wrap = 1;
	    break;
	}
	width += cs->width;
	++ptr;
    }
    /*
     * draw everything up to this point
     */
    if (ptr > p0) {
	foo[0].chars = p0;
	foo[0].nchars = ptr - p0;
	foo[0].delta = 0;
	foo[0].font = fs->fid;
	XDrawText (XtDisplay (w), XtWindow (w), gc, *x, *y, foo, 1);
    }
    *x += width;
    return ptr;
}

/*
 * display a line up to the point where it should wrap.
 * return a ptr to the next character to be displayed.
 *
 * XXX handle multiple colors & fonts
 */

static char *
display_line (w, ptr, x, y)
Widget w;
char *ptr;
int x, y;
{
    register int width;
    int max_width = width_of (w);
    char *p0;
    int wrap = 0;

    if (*ptr == '\n')		/* blank line */
	return ++ptr;

    while (*ptr && *ptr != '\n') {
	/*
	 * scan a sequence of displayable characters
	 */
	switch (*ptr) {
	case '&':
	    /*
	     * XXX need to see if there's room on the line
	     */
	    ptr = display_entity (w, ptr, &x, &y);
	    break;
	default:
	    ptr = display_text (w, ptr, &x, &y, &wrap);
	    break;
	}
	if (wrap)
	    return ptr;
    }
    if (*ptr == '\n')
	++ptr;
    return ptr;
}

/*
 * display a particular help page
 *
 * XXX since the page counter is maintained elsewhere, the user
 * can keep displaying the last page if he presses Next multiple
 * times.  Probably should move all of the help window code here.
 */
void
display_help_page (w, p)
Widget w;
int p;
{
    int x, y;
    int line_height;
    char *ptr;

    if (!inited) {
	XGCValues values;
	CanvasWidget cw = (CanvasWidget) w;
	int i;

	values.function = GXcopy;
	values.foreground = cw->canvas.foreground;
	values.background = cw->core.background_pixel;
	values.font = cw->canvas.fontinfo->fid;

	gc = XCreateGC (XtDisplay (w), XtWindow (w),
			GCFunction|GCForeground|GCBackground|GCFont,
			&values);

	add_entity ("normal_node",       PIXMAP, 0, (void *) pix.notReady);
	add_entity ("fanout_node",       PIXMAP, 0, (void *) pix.fanOut);
	add_entity ("beginloop_node",    PIXMAP, 0, (void *) pix.beginLoop);
	add_entity ("begincond_node",    PIXMAP, 0, (void *) pix.beginSwitch);
	add_entity ("beginpipe_node",    PIXMAP, 0, (void *) pix.beginPipe);
	add_entity ("fanin_node",        PIXMAP, 0, (void *) pix.fanIn);
	add_entity ("endloop_node",      PIXMAP, 0, (void *) pix.endLoop);
	add_entity ("endcond_node",      PIXMAP, 0, (void *) pix.endSwitch);
	add_entity ("endpipe_node",      PIXMAP, 0, (void *) pix.endPipe);
	inited = 1;

    }

    XClearWindow (XtDisplay (w), XtWindow (w));
    /*
     * start out with a small left and top margin
     */
    x = MARGIN;
    y = MARGIN;
    if (p < 0)
	p = 0;
    if (p >= sizeof (pages) / sizeof (*pages))
	p = sizeof(pages) / sizeof (*pages) - 1;
    ptr = (char *) pages[p];

    while (*ptr) {
	int ascent, descent;
	get_line_height (w, ptr, &ascent, &descent);
	y += ascent;
	ptr = display_line (w, ptr, x, y);
	y += descent;
    }
    /*
     * HORRIBLE HACK
     */
    if (p == 6)
	display_mouse_bindings (w, x, y);
}
