/*
 * graph editor
 *
 * $Id: graph_edit.c,v 1.1 1994/02/17 20:22:57 moore Exp $
 *
 * $Log: graph_edit.c,v $
 * Revision 1.1  1994/02/17  20:22:57  moore
 * Initial revision
 *
 */

#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include "filever.h"
#include "global.h"
#include "graph.h"
#include "graph_draw.h"
#include "graph_edit.h"
#include "graph_parse.h"
#include "labels.h"
#include "subproc.h"

/*
 * Bondo
 */
#include <Malloc.h>
#include <String.h>

#ifdef max
#undef max
#endif
#define max(a,b) ((a) > (b) ? (a) : (b))
#ifdef min
#undef min
#endif
#define min(a,b) ((a) < (b) ? (a) : (b))

static int currentNodeType;
static int editorSubprocessCount = 0;

static Widget graphCanvas;	/*
				 * this is a copy of the widget defined in
				 * graph_panel.c
				 */

static void
set_modified (g)
Graph g;
{
    g->modified = 1;
    if (g == global.graph)
	set_graphfile_label (global.graphFile, 1);
}


/*
 * make graph fit in the window
 * 
 * XXX need to take the legends into account -- leave more room
 * on the right than the left.
 *
 * XXX also, we adjust the space between nodes when necessary to
 * make the graph smaller, but never to make the graph wider or
 * taller.  Should we?  When?
 */

void
scale_graph (w, g)
Widget w;
Graph g;
{
    Dimension height, width;
    int maxx, maxy, minx, miny;
    rbNode rbn;
    int margin = 50;
    int scale_x, scale_y;
    int offset_x, offset_y;
    int delta_x, delta_y;

    XtVaGetValues (w,
		   XtNheight, &height,
		   XtNwidth, &width,
		   NULL);

    minx = 1000000;
    maxx = 0;
    miny = 1000000;
    maxy = 0;

    for (rbn = rb_First (g->nlist); !rb_Done (rbn, g->nlist);
	 rbn = rb_Next (rbn)) {
	
	Node n = (Node) rb_Value (rbn);
	if (n->xy.set) {
	    maxx = max (maxx, n->xy.x);
	    maxy = max (maxy, n->xy.y);
	    minx = min (minx, n->xy.x);
	    miny = min (miny, n->xy.y);
	}
    }

    delta_x = maxx - minx;
    delta_y = maxy - miny;

    /*
     * if spread of coordinates is larger than window, put them
     * inside the window.  Otherwise, just center along that axis.
     */
    if (delta_x > (int) width - 2 * margin) {
	scale_x = ((int) width - 2 * margin);
	offset_x = margin;
    }
    else if (delta_x == 0) {
	scale_x = delta_x = 1;
	offset_x = (int) width / 2;
    }
    else {
	scale_x = delta_x;
	offset_x = ((int) width - delta_x) / 2;
    }

    if (delta_y > (int) height - 2 * margin) {
	scale_y = ((int) height - 2 * margin);
	offset_y = margin;
    }
    else if (delta_y == 0) {
	scale_y = delta_y = 1;
	offset_y = (int) height / 2;
    }
    else {
	scale_y = delta_y;
	offset_y = ((int) height - delta_y) / 2;
    }

    for (rbn = rb_First (g->nlist); !rb_Done (rbn, g->nlist);
	 rbn = rb_Next (rbn)) {
	
	Node n = (Node) rb_Value (rbn);
	if (n->xy.set) {
	    n->xy.x = ((n->xy.x - minx) * scale_x) / delta_x + offset_x;
	    n->xy.y = ((n->xy.y - miny) * scale_y) / delta_y + offset_y;
	    set_modified (g);
	}
    }
}



/*
 *	graph_place_nodes ()
 *
 *	Start at level 0. All nodes are initially at level 0. All
 *	nodes set the level of their parents to current level + 1.
 *	Since no leaf node has children, its level is not incremented.
 *	All other nodes have their levels set to 1 because they
 *	have children.
 *
 *	Now go to level 1. All nodes at that level set the
 *	level of their parents. Thus, nodes that have only leaf nodes
 *	as children do not have their levels set to 2, and of course,
 *	all others do.  Continue as such until only the root
 *	node is left.
 */

void
graph_place_nodes(g, wd, ht)
Graph g;
int wd, ht;
{
    rbTree ntree = g->nlist;
    rbTree atree;
    rbNode tn, atn;
    Node n1, n2;
    int j, k, n;
    int lev = 0;
    int done = 0;

    /* start all nodes at level zero */
    
    for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next(tn)) {
	n1 = (Node) rb_Value (tn);
	n1->xy.y = 0;
    }

    /* increment by levels until root node is on a level by itself */

    for (lev = 0; !done; lev++) {
	done = 1;
	for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next(tn)) {
	    n1 = (Node) rb_Value (tn);
	    if (n1->xy.y != lev)
		continue;
	    atree = n1->children;
	    for (atn = rb_First (atree); !rb_Done (atn, atree);
		 atn = rb_Next (atn)) {
		n2 = (Node) rb_Value (atn);
		n2->xy.y = lev + 1;
		done = 0;
	    }
	}
    }

    /* set x coords */

    for (j = 0; j < lev; j++) {
	n = 0;
	for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next (tn)) {
	    n1 = (Node) rb_Value (tn);
	    if (n1->xy.y == j)
		n++;
	}
	if (!n)
	    continue;
	k = 1;
	for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next (tn)) {
	    n1 = (Node) rb_Value (tn);
	    if (n1->xy.y == j) {
		n1->xy.x = (wd * k) / (n + 1);
		/*
		  n1->xy.x = (wd * k) / (n + 1) + (j & 1 ? -10 : 10);
		  */
		k++;
	    }
	}
    }
    
    /* set y coords */
    
    for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next (tn)) {
	n1 = (Node) rb_Value (tn);
	n1->xy.y = (ht * (lev - n1->xy.y)) / (lev + 1);
	n1->xy.set = 1;
    }

}



void
edit_refresh (w)
Widget w;
{
    draw_graph (w, global.graph, 0);
}


/*
 * callback for node select palette.
 */

void
edit_select_node_type (node_type)
int node_type;
{
    currentNodeType = node_type;
}


static int
get_event_xy (xev, xp, yp)
XEvent *xev;
int *xp, *yp;
{
    switch (xev->type) {
    case KeyPress:
    case KeyRelease:
	*xp = xev->xkey.x;
	*yp = xev->xkey.y;
	return 1;
	break;

    case ButtonPress:
    case ButtonRelease:
	*xp = xev->xbutton.x;
	*yp = xev->xbutton.y;
	return 1;
	break;

    case MotionNotify:
	*xp = xev->xmotion.x;
	*yp = xev->xmotion.y;
	return 1;
	break;

    case EnterNotify:
    case LeaveNotify:
	*xp = xev->xcrossing.x;
	*yp = xev->xcrossing.y;
	return 1;
	break;

    default:
	return 0;
    }
}

/*
 * edit_add_node()
 *
 * Action - add node to graph at location
 */

static void
edit_add_node (wgt, xev, par, nump)
Widget wgt;			/* should be graphCanva */
XEvent *xev;
String *par;
Cardinal *nump;
{
    Node n;
    int x, y;

    if (global.mode != MODE_COMPOSE)
	return;

    if (get_event_xy (xev, &x, &y)) {
	/*
	 * don't let user add a new node on top of an existing one
	 */
	if ((n = find_nearest_node (global.graph, x, y)) != (Node) NULL) {
	    int dx = x - n->xy.x;
	    int dy = y - n->xy.y;

	    if (dx * dx + dy * dy < NODERADIUS * NODERADIUS * 4)
		return;
	}
	n = gr_NewNode (gr_HiNodeId (global.graph) + 1);
	n->node_type = currentNodeType;
	n->xy.x = x;
	n->xy.y = y;
	n->xy.set = 1;
	gr_AddNode (global.graph, n);
	draw_node (graphCanvas, n, 1);
	set_modified (global.graph);
    }
}

/*
 * edit_delete_node()
 *
 * Action - delete nearest node from graph
 */

static void
edit_delete_node (wgt, xev, par, nump)
Widget wgt;
XEvent *xev;
String *par;
Cardinal *nump;
{
    Node n;
    int x, y;

    if (global.mode != MODE_COMPOSE)
	return;

    if (get_event_xy (xev, &x, &y) &&
	(n = find_nearest_node (global.graph, x, y))) {
#if 0
	fprintf (stderr, "deleting node %d\n", n->nk.id);
#endif
#if 0
	/* XXX for debugging */
	if (*nump == 1 && par[0][0] == 'c') {
	    gr_CopySubgraph (global.graph, n, 0, 100, 20);
	    XClearWindow (global.display, XtWindow (graphCanvas));
	    edit_refresh (graphCanvas);
	    return;
	}
	if (*nump == 1 && par[0][0] == 'C') {
	    gr_CopySubgraph (global.graph, n, 1, 100, 20);
	    XClearWindow (global.display, XtWindow (graphCanvas));
	    edit_refresh (graphCanvas);
	    return;
	}
#endif
	gr_RemNode (global.graph, n);
	gr_FreeNode (n);
	XClearWindow (global.display, XtWindow (graphCanvas));
	draw_graph (graphCanvas, global.graph, 0);
	set_modified (global.graph);
#if 0
	if (selected_node == n) {
	    unpop_node_pop ();
	    selected_node = 0;
	}
#endif
    }
}

/*
 * edit_move_node()
 *
 * Action - move existing node
 *          param = "begin", "drag", "cancel", "finish"
 */

static void
edit_move_node (wgt, xev, par, nump)
Widget wgt;
XEvent *xev;
String *par;
Cardinal *nump;
{
    static Node curnode = 0;	/* node we are moving */
    static int ox, oy;		/* mouse-node offset */
    int x, y;

    if (global.mode != MODE_COMPOSE)
	return;

    if (*nump != 1 || !get_event_xy(xev, &x, &y))
	return;
    if (par[0][0] == 'b') {	/* begin */
	if (!(curnode = find_nearest_node (global.graph, x, y)))
	    return;
	draw_node (graphCanvas, curnode, 0);
	draw_node (graphCanvas, curnode, 2);
	ox = curnode->xy.x - x;
	oy = curnode->xy.y - y;
	
    }
    else {
	if (!curnode)
	    return;
	switch (par[0][0]) {
	case 'd':	/* drag */
	    draw_node (graphCanvas, curnode, 2);
	    curnode->xy.x = x + ox;
	    curnode->xy.y = y + oy;
	    draw_node (graphCanvas, curnode, 2);
	    break;
	    
	case 'c':	/* cancel */
	    draw_node (graphCanvas, curnode, 2);
	    draw_node (graphCanvas, curnode, 1);
	    curnode = 0;
	    XClearWindow (global.display, XtWindow (graphCanvas));
	    draw_graph (graphCanvas, global.graph, 0);
	    set_modified (global.graph);/* XXX really? */
	    break;
	    
	case 'f':	/* finish */
	    draw_node (graphCanvas, curnode, 2);
	    curnode->xy.x = x + ox;
	    curnode->xy.y = y + oy;
	    draw_node (graphCanvas, curnode, 1);
	    curnode = 0;
	    XClearWindow (global.display, XtWindow (graphCanvas));
	    draw_graph (graphCanvas, global.graph, 0);
	    set_modified (global.graph);
	    break;
	}
    }
}

#define	DRAWHEADTOTAIL 0	/* which way edit_add_arc() draws */

/*
 *	edit_add_arc()
 *
 *	Action - add new arc to graph
 *		param = "begin", "drag", "cancel", "finish"
 */

static void
edit_add_arc (wgt, xev, par, nump)
Widget wgt;
XEvent *xev;
String *par;
Cardinal *nump;
{
    static Node startnode = 0;	/* for drawing arcs */
    Node n;
    static Node fake = 0;	/* this isn't a real node; need it to draw */
    int x, y;

    if (global.mode != MODE_COMPOSE)
	return;

    if (!fake)
	fake = gr_NewNode (0);
    if (*nump != 1 || !get_event_xy (xev, &x, &y))
	return;
    if (par[0][0] == 'b') {	/* begin */
	if (!(startnode = find_nearest_node (global.graph, x, y)))
	    return;
	fake->xy.x = x;
	fake->xy.y = y;
	draw_arc (graphCanvas, startnode, fake, 2);
	return;
	
    }
    else {

	if (!startnode)
	    return;
	if (fake->xy.x != -1)
	    if (DRAWHEADTOTAIL)
		draw_arc (graphCanvas, fake, startnode, 2);
	    else
		draw_arc (graphCanvas, startnode, fake, 2);
	
	switch (par[0][0]) {
	case 'd':	/* drag */
	    fake->xy.x = x;
	    fake->xy.y = y;
	    if (DRAWHEADTOTAIL)
		draw_arc (graphCanvas, fake, startnode, 2);
	    else
		draw_arc (graphCanvas, startnode, fake, 2);
	    break;
	    
	case 'c':	/* cancel */
	    startnode = 0;
	    break;

#if 0
	    /* XXX for debugging */
	case 'p':	/* pair */
	    if ((n = find_nearest_node (global.graph, x, y)) &&
		n != startnode) {
		n->pair = startnode;
		startnode->pair = n;
		XClearWindow (global.display, graphCanvas);
		edit_refresh (graphCanvas);
	    }
	    startnode = 0;
	    break;
#endif

	case 'f':	/* finish */
	    if ((n = find_nearest_node (global.graph, x, y)) && n != startnode)
		switch (gr_BuildArc (global.graph,
				     (DRAWHEADTOTAIL ?
				      n->nk.id :  startnode->nk.id),
				     (DRAWHEADTOTAIL ?
				      startnode->nk.id : n->nk.id))) {
		case 0:
		    if (DRAWHEADTOTAIL)
			draw_arc (graphCanvas, n, startnode, 1);
		    else
			draw_arc (graphCanvas, startnode, n, 1);
		    set_modified (global.graph);
		    break;
		    
		case 1:
		    msg_Format ("arc already exists.\n");
		    break;
		    
		case 3:
		    msg_Format ("arc would cause a loop.\n");
		    break;
		}
	    startnode = 0;
	    break;
	}
    }
}

/*
 *	edit_delete_arc()
 *
 *	Action - delete nearest arc from drawing.
 */

static void
edit_delete_arc (wgt, xev, par, nump)
Widget wgt;
XEvent *xev;
String *par;
Cardinal *nump;
{
    Node n1, n2;
    int x, y;

    if (global.mode != MODE_COMPOSE)
	return;

    if (get_event_xy (xev, &x, &y)) {
	find_nearest_arc (global.graph, x, y, &n1, &n2);
	if (n1) {
#if 0
	    fprintf (stderr, "deleting arc from node %d to %d\n",
		     n1->nk.id, n2->nk.id);
#endif
	    draw_arc (graphCanvas, n1, n2, 0);
	    draw_arc (graphCanvas, n2, n1, 0);
	    gr_RemAnArc (global.graph, n1, n2);
	    gr_RemAnArc (global.graph, n2, n1);
	    set_modified(global.graph);
	}
#if 0
	else
	    fprintf (stderr, "couldn't find an arc\n");
#endif
    }
}

/*	edit_move_arc()
 *
 *	Action - move existing arc
 *		param = "begin", "drag", "cancel", "finish"
 */

static void
edit_move_arc (wgt, xev, par, nump)
Widget wgt;
XEvent *xev;
String *par;
Cardinal *nump;
{
    static Node n1 = 0, n2 = 0;	/* closest and other node of nearest arc */
    static hd1;			/* n1 is head? */
    static Node fake = 0;	/* this isn't a real node; need it to draw */
    Node n;
#if 0
    int noa;
#endif
    int x, y;

    if (global.mode != MODE_COMPOSE)
	return;

    if (!fake)
	fake = gr_NewNode (0);
    if (*nump != 1 || !get_event_xy (xev, &x, &y))
	return;
    if (par[0][0] == 'b') {	/* begin */
	find_nearest_arc (global.graph, x, y, &n2, &n1);
	if (n1) {
	    hd1 = gr_IsArc (n2, n1);
	    draw_arc (graphCanvas, (hd1 ? n2 : n1), (hd1 ? n1 : n2), 0);
	    fake->xy.x = x;
	    fake->xy.y = y;
	    draw_arc (graphCanvas, (hd1 ? fake : n1), (hd1 ? n1 : fake), 2);
	}
	return;
    }
    else {
	if (!n1)
	    return;
	if (fake->xy.x != -1)
	    draw_arc (graphCanvas, (hd1 ? fake : n1), (hd1 ? n1 : fake), 2);
	switch (par[0][0]) {
	case 'd':	/* drag */
	    fake->xy.x = x;
	    fake->xy.y = y;
	    draw_arc (graphCanvas, (hd1 ? fake : n1), (hd1 ? n1 : fake), 2);
	    break;

	case 'c':	/* cancel */
	    draw_arc (graphCanvas, (hd1 ? n2 : n1), (hd1 ? n1 : n2), 1);
	    n1 = 0;
	    break;

	case 'f':	/* finish */
	    if (n = find_nearest_node (global.graph, x, y)) {
		gr_RemAnArc (global.graph, (hd1 ? n2 : n1), (hd1 ? n1: n2));
		switch (gr_BuildArc (global.graph,
				     (hd1 ? n : n1)->nk.id,
				     (hd1 ? n1 : n)->nk.id)) {
		case 0:
		    draw_arc (graphCanvas, (hd1 ? n : n1), (hd1 ? n1 : n), 1);
		    set_modified (global.graph);
		    break;
		    
		case 1:
		    msg_Format ("arc already exists.\n");
		    break;
		}
	    }
	    else
		draw_arc (graphCanvas, (hd1 ? n2 : n1), (hd1 ? n1 : n2), 1);
	    n1 = 0;
	    break;
	}
    }
}


struct nodeprogram {
    Node node;
    char *filename;
    Widget canvas;
};

/*
 * callback for when the editor for a node program (the HeNCE glue,
 * not the C or FORTRAN source), is killed.
 */

static void
edit_node_glue_killed (pid, status, p)
int pid;
int status;
struct nodeprogram *p;
{
    msg_Format ("The editor for node %d was killed with signal %d.\n",
		p->node->nk.id, status);
    p->node->flags &= ~NF_EDITING;
    (void) unlink (p->filename);
    FREE (p->filename);
    FREE ((char *) p);
    --editorSubprocessCount;
}

/*
 * Callback for when a node glue program exits.  If editor exits normally,
 * and node program parses correctly, splice it back into the graph.
 * If the parse was incorrect, give user a chance to edit again.
 * If we aren't editing again, free up our data structures.
 */

static void
edit_node_glue_done (pid, status, p)
int pid;
int status;
struct nodeprogram *p;
{
    Node newnode;

    if (status == 0) {
	FILE *fp;
	int fgetc ();

	if ((fp = fopen (p->filename, "r")) == NULL) {
	    msg_Format ("can't read tmp file %s\n", p->filename);
	    goto cleanup;
	}
	newnode = parse_Node (fgetc, (void *) fp, NULL);
	fclose (fp);
	if (newnode != (Node) NULL) {
	    /*
	     * The new node program parses without errors.  Replace
	     * the old node in the graph with the new one.
	     */
	    msg_Format ("Node %d: no errors.\n", newnode->nk.id);
	    draw_node (p->canvas, p->node, 0);
	    gr_PatchNode (p->node, newnode);
	    gr_FreeNode (newnode);
	    p->node->state = ST_NOT_BEGUN;
	    draw_node (p->canvas, p->node, 2);
	    edit_refresh (p->canvas);
	    goto cleanup;
	}
	else {
	    /*
	     * parser detected node program errors.  Offer to edit
	     * again, otherwise throw away this node.
	     */
	    char msgbuf[100];
	    char buf[1000];
	    
	    sprintf (msgbuf, "Error in program for node %d.  Edit again?",
		     p->node->nk.id);
	    draw_node (p->canvas, p->node, 0);
	    p->node->state = ST_WARNING;
	    draw_node (p->canvas, p->node, 2);
	    if (verify (msgbuf) == 0) {
		edit_refresh (p->canvas);
		sprintf (buf, defaults.editorCommand, p->filename,
			 p->filename, p->filename);
		if (subproc_Spawn (buf, edit_node_glue_done,
				   edit_node_glue_killed, p) > 0) {
		    return;
		}
		msg_Format ("couldn't rerun editor\n");
		goto cleanup;
	    }
	}
    }
 cleanup:
    --editorSubprocessCount;
    p->node->flags &= ~NF_EDITING;
    (void) unlink (p->filename);
    FREE (p->filename);
    FREE ((char *) p);
}

/*
 * Come here to edit a node glue program.
 */

static void
edit_node_glue (w, n)
Widget w;
Node n;
{
    struct nodeprogram *p = TALLOC (1, struct nodeprogram);
    FILE *fp;
    char buf[1024];

    if (global.mode != MODE_COMPOSE)
	return;

    if (n->flags & NF_EDITING) {
	msg_Format ("Already editing node %d\n", n->nk.id);
	return;
    }
    p->node = n;
    p->canvas = w;
    p->filename = STRDUP (tmpnam ((char *) NULL));
    if ((fp = fopen (p->filename, "w")) == NULL) {
	msg_Format ("can't write tmp file %s\n", p->filename);
	goto fail;
    }
    graph_unparse_node (n, fp);
    (void) fclose (fp);

    p->node->flags |= NF_EDITING;
    if (*defaults.editorCommand) {
	sprintf (buf, defaults.editorCommand,
		 p->filename, p->filename, p->filename);
	if (subproc_Spawn (buf, edit_node_glue_done,
			   edit_node_glue_killed, p) > 0) {
	    ++editorSubprocessCount;
	    return;
	}
    }
    else
	msg_Format ("no htool.editorCommand resource defined -- can't edit\n");

 fail:
    p->node->flags &= ~NF_EDITING;
    (void) unlink (p->filename);
    FREE (p->filename);
    FREE ((char *) p);
}

/*
 * Callback for when a node source (C or FORTRAN) editor dies, either
 * via exit() or by being killed.   Prints a message if killed and
 * frees up the filename.
 */

static void
edit_source_done (pid, status, filename, howDied)
int pid, status;
char *filename;
enum causeOfDeath howDied;
{
    if (howDied == SUBPROC_KILLED) {
	msg_Format ("editor pid %d was killed with signal %d\n",
		    pid, status);
    }
    FREE (filename);
    --editorSubprocessCount;
    return;
}

/*
 * Come here to edit a C or FORTRAN node implementation.
 * save the filename in malloc'ed space (to be freed by
 * edit_EditSourceDone), and spawn a subproc to do the
 * editing.
 *
 * XXX need to make sure user doesn't edit the same file
 * twice.
 */

static void
edit_source_program (filename)
char *filename;
{
    char *ptr = STRDUP (filename);
    char buf[1024];

    if (global.mode != MODE_COMPOSE)
	return;

    sprintf (buf, defaults.editorCommand, ptr, ptr, ptr);
    if (subproc_Spawn (buf, edit_source_done,
		       edit_source_done, ptr) > 0) {
	++editorSubprocessCount;
	return;
    }
    FREE (filename);
}

/*
 * edit_node()
 *
 * Action - popup window to edit node characteristics.
 *	    param = "source", "program"
 */

static void
edit_node (wgt, xev, par, nump)
Widget wgt;
XEvent *xev;
String *par;
Cardinal *nump;
{
    Node enode;
#if 0
    Node newnode;
#endif
    int x, y;
    char filename[100];

    if (global.mode != MODE_COMPOSE)
	return;
    if (*nump != 1 || !get_event_xy (xev, &x, &y))
	return;
    if (!(enode = find_nearest_node (global.graph, x, y)))
	return;

#if 0
    fprintf (stderr, "editing node %d\n", enode->nk.id);
#endif

    switch (par[0][0]) {
    case 'p':	/* edit node program in graph */
	edit_node_glue (graphCanvas, enode);
	break;

    case 's':	/* edit source code for node implementation */
	if (enode->node_type != NODE_NORMAL) {
	    msg_Format ("%d: not a regular node\n", enode->nk.id);
	    break;
	}
	if (*defaults.editorCommand == '\0') {
	    msg_Format ("No htool.editorCommand resource - can't edit\n");
	    break;
	}
	if (enode->sub_name == NULL || enode->sub_name[0] == '\0') {
	    msg_Format ("Node %d has no function name\n", enode->nk.id);
	    break;
	}
	sprintf (filename, "%s.%s", enode->sub_name,
		 global.language == LANG_C ? "c" : "f");
	edit_source_program (filename);
	break;

    default:
	break;
    }
}

#if 0
/*
 * edit sub.defs file
 * should this go in labels?  If not, where?
 */

static void
edit_subdefs (w, cli, cd)
Widget w;
XtPointer cli, cd;
{
    static void edit_EditNodeSourceProgram ();

    if (*defaults.subDefsFile != '\0')
	edit_source_program (defaults.subDefsFile);
}
#endif

#ifdef DEMO
static void
change_color (wgt, xev, par, nump)
Widget wgt;
XEvent *xev;
String *par;
int *nump;
{
    Node enode;
    int x, y;
    char filename[100];

    if (!get_event_xy (xev, &x, &y))
	return;
    if (!(enode = find_nearest_node (global.graph, x, y)))
	return;

#if 0
    fprintf (stderr, "editing node %d\n", enode->nk.id);
#endif

    if (++enode->state >= ST_NUM_STATES)
	enode->state = ST_NOT_BEGUN;
    draw_node (graphCanvas, enode, 1);
}
#endif /* DEMO */

static XtActionsRec actbl[] = {
    { "add-node", edit_add_node },
    { "del-node", edit_delete_node },
    { "move-node", edit_move_node },
    { "add-arc", edit_add_arc },
    { "del-arc", edit_delete_arc },
    { "move-arc", edit_move_arc },
    { "edit-node", edit_node },
#ifdef DEMO
    { "change-color", change_color },
#endif /* DEMO */
};

void
edit_init (w)
Widget w;
{
    graphCanvas = w;
    XtAppAddActions (global.context, actbl, XtNumber (actbl));
    global.graph = gr_New ();
}
