/*
 * functions for doing consistency checks on HeNCE graphs
 *
 * $Id: graph_critic.c,v 1.1 1994/02/17 20:22:26 moore Exp $
 *
 * $Log: graph_critic.c,v $
 * Revision 1.1  1994/02/17  20:22:26  moore
 * Initial revision
 *
 */


#include <stdio.h>
#include "rb.h"
#include "param.h"
#include "exp.h"
#include "graph.h"

/*
 * Bondo
 */
#include <Malloc.h>

static int error_count = 0;		/* # of nodes in error */

static char *
node_type_name (n)
int n;
{
    switch (n) {
    case NODE_NORMAL:
	return "NORMAL";
    case NODE_COND:
	return "BEGINSWITCH";
    case NODE_ENDCOND:
	return "ENDSWITCH";
    case NODE_LOOP:
	return "BEGINLOOP";
    case NODE_ENDLOOP:
	return "ENDLOOP";
    case NODE_FANOUT:
	return "FANOUT";
    case NODE_FANIN:
	return "FANIN";
    case NODE_PIPE:
	return "BEGINPIPE";
    case NODE_ENDPIPE:
	return "ENDPIPE";
    }
}

static int
is_left_special (n)
Node n;
{
    if (n == (Node) NULL)
	return 0;
    switch (n->node_type) {
    case NODE_COND:
    case NODE_LOOP:
    case NODE_FANOUT:
    case NODE_PIPE:
	return 1;
    }
    return 0;
}

static int
is_right_special (n)
Node n;
{
    if (n == (Node) NULL)
	return 0;
    switch (n->node_type) {
    case NODE_ENDCOND:
    case NODE_ENDLOOP:
    case NODE_FANIN:
    case NODE_ENDPIPE:
	return 1;
    }
    return 0;
}

static int
matching_node_type (n)
Node n;
{
    if (n == (Node) NULL)
	return -1;
    switch (n->node_type) {
    case NODE_NORMAL:
	return -1;
    case NODE_ENDCOND:
	return NODE_COND;
    case NODE_ENDLOOP:
	return NODE_LOOP;
    case NODE_FANIN:
	return NODE_FANOUT;
    case NODE_ENDPIPE:
	return NODE_PIPE;
    case NODE_COND:
	return NODE_ENDCOND;
    case NODE_LOOP:
	return NODE_ENDLOOP;
    case NODE_FANOUT:
	return NODE_FANIN;
    case NODE_PIPE:
	return NODE_ENDPIPE;
    }
}

/*
 * this routine marks a node so that it will show up on-screen
 * as an error, and bumps the error count.  If the node already
 * indicates an error, it doesn't bump the count, so that error_count
 * always indicates the number of nodes in error.
 *
 * returns 1 if the node didn't have the error flag set.
 */

static int
mark_error (n)
Node n;
{
    if (n->state == ST_ERROR)
	return 0;
    n->state = ST_ERROR;
    ++error_count;
    return 1;
}

/*
 * find the set of terminal nodes of each of a node's children
 * if the cardinality of the set != 1, figure out which is
 * "most likely" to be the correct terminal node.  Normally this
 * will be the one that occurs most frequently, but if the start
 * node is a special node then the end node must be a matching pair.
 *
 * - if a node has no children, its terminal node is NULL.
 * - if a node has one child which is a right special, its terminal node
 *   is that right special node.
 * - if a node is a left special node, its terminal node is the terminal
 *   node of the matching right special node.
 * - otherwise, the terminal node of a node is the terminal node of each
 *   of its children, which should all be the same.
 */

#if defined(__STDC__) || defined(FUNCPROTO)
static Node find_terminal_node (Node n, Node left);
#else
static Node find_terminal_node ();
#endif

static Node
really_find_terminal_node (n, left)
Node n;
Node left;
{
    struct nf {
	Node node;
	int count;
    } *nf;
    rbNode tn;
    int i;
    int result;
    int nfsize;
    Node result_node;
    int max;
    
    /*
     * special case: 0 children
     */
    
    if (n->nchildren == 0) {
	if (is_left_special (n)) {
	    if (mark_error (n))
		msg_Format ("node %d: special node has no children\n",
			    n->nk.id);
	}
	return (Node) NULL;
    }
    
    nf = TALLOC (n->nchildren, struct nf);
    nfsize = 0;
    
    /*
     * find each child's terminal node and keep track of how many times
     * we've seen each one.
     */
    for (tn = rb_First (n->children); !rb_Done(tn, n->children);
	 tn = rb_Next (tn)) {
	Node start = (Node) rb_Value (tn);
	Node end; 
	
	if (is_right_special (start)) {
	    if (n->nchildren > 1) {
		/*
		 * if node has more than one child, none of them can be a
		 * right special
		 */
		if (mark_error (n))
		    msg_Format ("node %d: only one child can be a %s\n",
				n->nk.id, node_type_name (start->node_type));
	    }
	    if (left == (Node) NULL) {
		if (mark_error (start))
		    msg_Format ("node %d: no path to a matching %s node through node %d\n",
				start->nk.id,
				node_type_name (matching_node_type(start)),
				n->nk.id);
	    }
	    else {
		FREE ((char *) nf);
		return start;
	    }
	}
	if (is_left_special (n))
	    end = find_terminal_node (start, n);
	else
	    end = find_terminal_node (start, left);
	
	for (i = 0 ; i < nfsize; ++i)
	    if (end == nf[i].node)
		++nf[i].count;
	if (i == nfsize) {
	    /* haven't seen this node yet - add it to the list */
	    nf[nfsize].node = end;
	    nf[nfsize].count = 1;
	    ++nfsize;
	}
    }
    
    /*
     * now scan each node in the list and find the most frequently
     * occuring matching node that qualifies.
     */
    result = -1;
    max = 0;
    for (i = 0; i < nfsize; ++i) {
	if (is_left_special (n)) {
	    if (nf[i].node != (Node) NULL &&
		nf[i].node->node_type == matching_node_type (n) &&
		nf[i].count > max)
		result = i;
	}
	else {
	    if (nf[i].count > max)
		result = i;
	}
    }
    
    /*
     * if there was no qualifying match, just use the first node
     * in the list.
     */
    if (result == -1)
	result = 0;
    result_node = nf[result].node;
    
    /*
     * for all of the problem children, mark their terminal nodes
     * as being in error
     */
    
    for (tn = rb_First (n->children); !rb_Done(tn, n->children);
	 tn = rb_Next (tn)) {
	Node child = (Node) rb_Value (tn);
	Node end = child->scope ? child->scope : child;
	
	/*
	 * find_terminal_node(), already called above for each child,
	 * guarantees that the child's scope field is valid
	 */
	if (child->scope != result_node) {
	    if (left) {
		if (mark_error (end)) {
		    msg_Format ("node %d: no path to %s matching %s node %d\n",
				end->nk.id,
				node_type_name (matching_node_type (left)),
				node_type_name (left->node_type),
				left->nk.id);
		}
	    }
	    else {
		if (mark_error (end))
		    msg_Format ("node %d: branch out of subgraph\n",
				end->nk.id);
	    }
	}
    }
    
    /*
     * for special node pairs, set their pair fields.
     */
    if (is_left_special (n)) {
	if (is_right_special (result_node) &&
	    n->node_type == matching_node_type (result_node)) {
	    n->pair = result_node;
	    n->pair->pair = n;
	    result_node = find_terminal_node (result_node, left);
	}
	else {
	    if (mark_error (n))
		msg_Format ("node %d: no match for special node\n",
			    n->nk.id);
	}
    }
    
    /*
     * return the most likely terminal node
     */
    FREE ((char *) nf);
    return result_node;
}

/*
 * this is a wrapper for the really_find_terminal_node function that
 * does loop detection, makes sure the scope field is set, and
 * takes advantage of previously computed terminal nodes.
 */

static Node
find_terminal_node (n, left)
Node n;
Node left;
{
    Node result;
    
    if (n->flags & NF_BEEN_HERE) {
	if (mark_error (n)) {
	    /* XXX need to mark all nodes in the loop to make
	       it obvious to the user. */
	    msg_Format ("node %d: loop detected\n", n->nk.id);
	}
	result = n;
    }
    else {
	n->flags |= NF_BEEN_HERE;
	
	if (n->flags & NF_SCOPE_VALID)
	    result = n->scope;
	else {
	    result = really_find_terminal_node (n, left);
	    n->scope = result;
	    n->flags |= NF_SCOPE_VALID;
	}
	
	n->flags &= ~NF_BEEN_HERE;
    }
    return result;
}


static int
ensure_terminal_node_is_null (tn)
rbNode tn;
{
    Node n = (Node) rb_Value (tn);
    
    if (find_terminal_node (n, NULL) != (Node) NULL) {
	if (mark_error (n)) {
	    if (is_right_special (n))
		msg_Format ("Node %d: %s has no ancestors.\n",
			    n->nk.id, node_type_name (n->node_type));
	    else {
		msg_Format ("Node %d: path from head ends in loop\n",
			    n->nk.id);
	    }
	}
    }
    return 0;
}

static int
clear_flags (tn)
rbNode tn;
{
    Node n = (Node) rb_Value (tn);
    n->flags &= ~(NF_SCOPE_VALID|NF_BEEN_HERE);
    n->state = ST_NOT_BEGUN;
    n->pair = (Node) NULL;
    n->scope = (Node) NULL;
    return 0;
}

/*
 * scan a graph.
 * check for loops, unmatched special node pairs, branches out of
 * subgraphs, etc.
 *
 * return 0 if all is well.
 * return nonzero if there were any errors
 *
 * side effects:
 * - for every node n of the graph, sets n->scope to the terminal node
 *   of node n, and sets the NF_SCOPE_VALID flag.
 * - for special nodes, sets n->pair to the matching special node
 * - sets state of every node to either ST_NOT_BEGUN or ST_ERROR, so
 *   they will show up right on the screen.
 * - prints error messages
 */

int
graph_critic (g)
Graph g;
{
    if (g == (Graph) NULL ||
	g->nlist == (rbTree) NULL || g->heads == (rbTree) NULL) {
	msg_Format ("Graph is NULL.\n");
	return 1;
    }
    (void) rb_Traverse (g->nlist, clear_flags);
    error_count = 0;
    if (rb_Empty (g->nlist)) {
	msg_Format ("graph is empty.\n");
	return 0;
    }
    else if (rb_Empty (g->heads)) {
	/*
	 * this is lame.  there's no one node that is incorrect,
	 * but we have to mark something else the user won't
	 * understand why his program doesn't run.
	 */
	if (mark_error (rb_Value (rb_First (g->nlist))))
	    msg_Format ("graph has no root.\n");
	return 1;
    }
    (void) rb_Traverse (g->heads, ensure_terminal_node_is_null);
    return error_count;
}

static int
check_graph_for_cycles (tn)
rbNode tn;
{
    Node n = (Node) rb_Value (tn);
    if (n->flags & NF_BEEN_HERE)
	return 1;
    if (n->nchildren == 0)
	return 0;
    n->flags |= NF_BEEN_HERE;
    if (rb_Traverse (n->children, check_graph_for_cycles))
	return 1;
    n->flags &= ~NF_BEEN_HERE;
    return 0;
}


static int
clear_been_here_flag (tn)
rbNode tn;
{
    Node n = (Node) rb_Value (tn);
    n->flags &= ~NF_BEEN_HERE;
    return 0;
}

int
graph_has_cycles (g)
Graph g;
{
    int x;
    
    if (rb_Empty (g->nlist))
	return 0;
    else if (rb_Empty (g->heads))
	return 1;
    (void) rb_Traverse (g->nlist, clear_been_here_flag);
    x = rb_Traverse (g->heads, check_graph_for_cycles);
    (void) rb_Traverse (g->nlist, clear_been_here_flag);
    return x;
}

#if 0
static int
printApair (tn)
rbNode tn;
{
    Node n = (Node) rb_Value (tn);
    if (is_left_special (n) && n->pair != (Node) NULL) {
	msg_Format ("pair: %d <-> %d\n", n->nk.id, n->pair->nk.id);
	if (n->pair->pair == (Node) NULL || n->pair->pair != n) {
	    msg_Format ("non symmetrical\n");
	}
    }
    return 0;
}

static int
printMatchingPairs (g)
Graph g;
{
    (void) rb_Traverse (g->nlist, printApair);
}
#endif /* 0 */
