#include <stdio.h>
#include "rb.h"
#include "param.h"
#include "exp.h"
#include "graph.h"
#include "msg.h"

/*
 * Bondo
 */

#include <Malloc.h>
#include <String.h>

#define BEEN_HERE 1


/*
 * compare two nodes by id and instance number.
 * this is used to order various rb-trees of nodes
 */

static int
compare_nodes (a, b)
void *a, *b;
{
    struct nodekey *ak = (struct nodekey *) a;
    struct nodekey *bk = (struct nodekey *) b;

    if (ak->id == bk->id)
	return (ak->inst - bk->inst);
    return (ak->id - bk->id);
}


/*
 * build a new empty graph structure
 */

Graph
gr_New ()
{
    Graph g = TALLOC (1, struct graph);

    g->numnodes = 0;
    g->nlist = rb_Create (compare_nodes);
    g->heads = rb_Create (compare_nodes);
    g->modified = 0;

    return g;
}

/*
 * Free up a graph structure
 */

void
gr_Free (g)
Graph g;
{
#if 0
    /*
     * XXX don't even bother until we implement a reasonable memory
     * management strategy throughout htool.
     */

    rbNode tn;

    if (g == NULL || g->nlist == NULL)
	return;
    for (tn = rb_First (g->nlist); !rb_Done(tn, g->nlist);
	 tn = rb_Next (g->nlist))
	gr_FreeNode (rb_Value (tn));

    rb_FreeTree (g->nlist);
    rb_FreeTree (g->heads);
    g->numnodes = 0;
    g->nlist = NULL;
    g->heads = NULL;
#endif
}

/*
 * Create a new node
 */

Node
gr_NewNode (node_id)
int node_id;
{
    Node n;
    int strcmp ();

    n = TALLOC (1, struct node);
    n->node_type = NODE_NORMAL;
    n->xy.x = 0;
    n->xy.y = 0;
    n->xy.set = 0;
    n->nk.id = node_id;
    n->nk.inst = 0;
    n->instgen = TALLOC (1, int);
    n->instgen[0] = 1;
    n->nparams = 0;
    n->params = rb_Create (strcmp);
    n->sub_name = 0;
    n->ret_param = 0;
    n->arrays = 0;
    n->nargs = 0;
    n->args = 0;
    n->parents = rb_Create (compare_nodes);
    n->nparents = 0;
    n->children = rb_Create (compare_nodes);
    n->nchildren = 0;
    n->pair = 0;
    n->scope = 0;
    n->flags = 0;
    n->pvminum = 0;
    n->state = 0;
    n->mirror = 0;
    n->numRunning = 0;
    n->numIdle = 0;
    return n;
}

/*
 * Free up a node
 */

void
gr_FreeNode(n)
Node n;
{
    /* XXX Better write this!!! */
}

/*
 * Given a node id #, return a pointer to that node
 *
 * This is used by the graph editor.
 */

Node
gr_GetNodeById (g, id)
Graph g;
int id;
{
    rbNode r;
    int fnd = 0;
    struct nodekey nk;

    nk.id = id;
    nk.inst = 0;
    r = rb_Find (g->nlist, (void *) &nk, &fnd);
    return fnd ? (Node) rb_Value (r) : (Node) NULL;
}

/*
 * Given a node id #, return a pointer to that node.
 *
 * This is used by trace.
 */

Node
gr_GetNodeByIdInst (g, id, inst)
Graph g;
int id;
int inst;
{
    rbNode r;
    int fnd = 0;
    struct nodekey nk;

    nk.id = id;
    nk.inst = inst;
    r = rb_Find (g->nlist, (void *) &nk, &fnd);
    return fnd ? (Node) rb_Value (r) : (Node) NULL;
}

/*
 * Add a node to a graph.
 * There shouldn't be any arcs to or from this node when we call.
 */

void
gr_AddNode (g, n)
Graph g;
Node n;
{
    rbNode x;
    int fnd = 0;
    
    x = rb_Find (g->nlist, (void *) &(n->nk), &fnd);
    if (fnd) {
	msg_Format ("error: two definitions of node %d/%d\n", n->nk.id,
		    n->nk.inst);
	return;
    }
    /*
     * insert node in list of all nodes
     */
    rb_InsertBefore (x, (void *) &(n->nk), (void *) n);

    /*
     * insert in list of head nodes.  If we later build an arc
     * pointing to it we will delete it from the list of head
     * nodes.
     */
    rb_Insert (g->heads, (void *) &(n->nk), (void *) n);
    g->numnodes++;
}

/*
 *  Remove a node from a graph.
 *  Removes arcs to/from the node.
 *
 *  Return 0 if ok, else 1 if node wasn't in graph.
 *
 *  XXX This is less efficient than it could be -- calling
 *  gr_RemAnArch causes an unnecessary tree search.
 */

int
gr_RemNode (g, n)
Graph g;
Node n;
{
#if 1
    rbNode x;
#else
    rbNode x, tna;
    rbTree tra;
#endif
    int fnd;

    x = rb_Find (g->nlist, (void *) &(n->nk), &fnd);
    if (!fnd)
	return 1;
    
    /* remove all out arcs */
    
#if 1
    while (!rb_Empty (n->children)) {
	Node n2 = (Node) rb_Value (rb_First (n->children));
	gr_RemAnArc (g, n, n2);
    }
#else
    tra = n->children;
    while (!rb_Done (tna = rb_First (tra), tra))
	gr_RemAnArc (g, n, (Node) rb_Value (tna));
#endif

    /* remove all in arcs */

#if 1
    while (!rb_Empty (n->parents)) {
	Node n2 = (Node) rb_Value (rb_First (n->parents));
	gr_RemAnArc (g, n2, n);
    }
#else
    tra = n->parents;
    while (!rb_Done (tna = rb_First (tra), tra))
	gr_RemAnArc (g, (Node) rb_Value (tna), n);
#endif
	
    /* remove from graph */

    rb_DeleteNode (x);

    /* remove from graph head list */

    x = rb_Find (g->heads, (void *) &(n->nk), &fnd);
    if (fnd)
	rb_DeleteNode (x);
    
    return 0;
}


/*
 * set the number of arguments to a node, making sure there
 * is at least enough space in its args array there to store
 * that many Exp's.
 */

void
gr_ResizeNodeArgs (node, nargs)
Node node;
int nargs;
{
    /*
     * Instead of reallocing every time, just round the # args we need
     * up to some multiple of 8.  Don't realloc if we already have
     * enough space.
     */
    int cur_args_space = (node->nargs | 0x7) + 1;
    int need_args_space = (nargs | 0x7) + 1;

    if (node->nargs == 0 || node->args == (Expr*) NULL)
	node->args = TALLOC (need_args_space, Expr);
    if (need_args_space > cur_args_space)
	node->args = (Expr *) REALLOC ((char *) node->args,
				       need_args_space * sizeof(Expr));
    node->nargs = nargs;
}

/*
 * Add an argument to a node
 *
 * this is used by the graph parser
 */

void
gr_AddNodeArg (node, exp)
Node node;
Expr exp;
{
    int nargs = node->nargs;
    
    gr_ResizeNodeArgs (node, nargs + 1);
    node->args[nargs] = exp;
}

/*
 * Build an arc from one node to another.
 *
 * Returns 0 if ok, else 1 if arc exists.
 */

int
gr_BuildAnArc (g, fromNode, toNode)
Graph g;
Node fromNode, toNode;
{
    rbNode rbFrom, rbTo;
    int foundFromNode, foundToNode;

    if (fromNode == toNode)
	return 1;

    rbTo = rb_Find (fromNode->children, (void *) &(toNode->nk), &foundToNode);
    rbFrom = rb_Find (toNode->parents, (void *) &(fromNode->nk),
		      &foundFromNode);

    if (foundToNode && foundFromNode) {
	msg_Format ("warning: Arc from %d to %d already defined\n",
		    fromNode->nk.id, toNode->nk.id);
	return 1;
    }

    if (foundToNode)
	msg_Format ("bogus: One side of arc from %d to %d already defined\n",
		    fromNode->nk.id, toNode->nk.id);
    else {
	rb_InsertBefore(rbTo, (void *) &(toNode->nk), toNode);
    }

    if (foundFromNode)
	msg_Format ("bogus: One side of arc from %d to %d already defined\n",
		    fromNode->nk.id, toNode->nk.id);
    else {
	rb_InsertBefore(rbFrom, (void *) &(fromNode->nk), fromNode);
    }

    if (toNode->nparents == 0) {
	rbNode r;
	r = rb_Find (g->heads, (void *) &(toNode->nk), &foundToNode);
	if (!foundToNode)
	    msg_Format ("bogus: %d should be in g->heads but is not\n",
			toNode->nk.id);
	else
	    rb_DeleteNode (r);
    }
    toNode->nparents++;
    fromNode->nchildren++;
    return 0;
}


/*
 * Build an arc from one node to another, given node id's
 *
 * Returns 0 if ok, else 1 if arc exists, 2 if node not found.
 */

int
gr_BuildArc (g, from, to)
Graph g;
int from;
int to;
{
    Node fromNode, toNode;

    if (!(fromNode = gr_GetNodeById (g, from))) {
	msg_Format("Node %d is not defined\n", from);
	return 2;
    }
    if (!(toNode = gr_GetNodeById (g, to))) {
	msg_Format("Node %d is not defined\n", to);
	return 2;
    }
    return gr_BuildAnArc (g, fromNode, toNode);
}

/*
 *  Remove an arc from one node to another.
 */

int
gr_RemAnArc (g, from_n, to_n)
Graph g;
Node from_n, to_n;
{
    rbNode tn;
    int fnd;

#if 1
    fprintf (stderr, "gr_RemAnArc (g, from %d, to %d)\n",
	     from_n->nk.id, to_n->nk.id);
#endif

    tn = rb_Find (from_n->children, (void *) &(to_n->nk), &fnd);
    if (fnd) {
	rb_DeleteNode (tn);
	from_n->nchildren--;
    }

    tn = rb_Find (to_n->parents, (void *) &(from_n->nk), &fnd);
    if (fnd) {
	rb_DeleteNode (tn);
	if (--to_n->nparents < 1) {
	    rb_Insert (g->heads, (void *) &(to_n->nk), (void *) to_n);
	}
    }
    return 0;
}

/*
 * Remove an arc from one node to another, given node id's
 *  XXX in the process of being thrown out
 */

int
gr_RemArc (g, from, to)
Graph g;
int from;
int to;
{
    Node from_n, to_n;

    if (!(from_n = gr_GetNodeById (g, from))) {
	msg_Format ("Node %d is not defined\n", from);
	return 2;
    }
    if (!(to_n = gr_GetNodeById (g, to))) {
	msg_Format ("Node %d is not defined\n", to);
	return 2;
    }
    return gr_RemAnArc (g, from_n, to_n);
}

int
gr_IsArc (from, to)
Node from;
Node to;
{
    rbTree tr = from->children;
    rbNode tn;

    for (tn = rb_First (tr); !rb_Done (tn, tr); tn = rb_Next (tn))
	if ((Node) rb_Value (tn) == to)
	    return 1;
    return 0;
}

int
gr_HiNodeId(g)
Graph g;
{
    rbTree tr = g->nlist;
    rbNode tn;
    int hi_id = 0;
    int i;

    for (tn = rb_First (tr); !rb_Done (tn, tr); tn = rb_Next(tn)) {
	i = ((Node) rb_Value (tn))->nk.id;
	if (i > hi_id)
	    hi_id = i;
    }
    return hi_id;
}

int
gr_GetNodeLoc (n, x, y)
Node n;
int *x;
int *y;
{
    if (n->xy.set) {
	*x = n->xy.x;
	*y = n->xy.y;
	return 0;
    }
    return -1;
}

void
gr_SetNodeLoc (n, x, y)
Node n;
int x, y;
{
    n->xy.set = 1;
    n->xy.x = x;
    n->xy.y = y;
}

void
gr_AddNodeParam (n, p)
Node n;
Param p;
{
    int strcmp ();

    rb_Insert (n->params, (void *) p->name, (void *) p);
}


int
gr_Empty (g)
Graph g;
{
    if (g == (Graph) NULL)
	return 1;
    return (rb_Empty (g->nlist));
}


void
gr_ResetGraph (g, state)
Graph g;
int state;
{
    rbNode tn;
    Node n;

    if (g == (Graph) NULL || g->nlist == (rbTree) NULL || rb_Empty (g->nlist))
	return;

    for (tn = rb_First (g->nlist); !rb_Done (tn, g->nlist);
	 tn = rb_Next (tn)) {
	n = (Node) rb_Value (tn);
	n->state = state;
	n->numRunning = 0;
	n->numIdle = 0;
    }
}

static void
gr_ResetNodeAndChildren (n1, n2, state)
Node n1, n2;
int state;
{
    rbNode tn;

    n1->state = state;
    n1->numRunning = 0;
    n1->numIdle = 0;

    if (!rb_Empty (n1->children)) {
	for (tn = rb_First (n1->children); !rb_Done (tn, n1->children);
	     tn = rb_Next (tn)) {
	    Node n = (Node) rb_Value (tn);

	    if (n != n2)
		gr_ResetNodeAndChildren (n, n2, state);
	}
    }
}

void
gr_ResetSubGraph (n, state)
Node n;
{
    if (n == (Node) NULL)
	return;

    switch (n->node_type) {
    case NODE_COND:
    case NODE_LOOP:
    case NODE_FANOUT:
    case NODE_PIPE:
	gr_ResetNodeAndChildren (n, n->pair, state);
	return;
    default:
	return;
    }
}

/*
 * What follows is left over from an earlier version of htool that
 * displayed expanded HeNCE graphs as they were being traced, until
 * it became apparent that this was too unwieldy to use.  The
 * current htool doesn't use this stuff.  Someday it may reappear.
 */

/*
 * gr_PatchNode()
 *
 * 'Replace' node n1 guts with n2, i.e. leave only arcs and node id
 * alone.
 */

int
gr_PatchNode (n1, n2)
Node n1, n2;
{
    if (n1->node_type != n2->node_type)
	return 1;

#define ANYSWAP(x,y,t) do{t temp;temp = x;x = y;y = temp;}while(0)
    ANYSWAP(n1->nparams, n2->nparams, int);
    ANYSWAP(n1->params, n2->params, rbTree);
    ANYSWAP(n1->sub_name, n2->sub_name, char*);
    ANYSWAP(n1->ret_param, n2->ret_param, Param);
    ANYSWAP(n1->arrays, n2->arrays, char);
    ANYSWAP(n1->nargs, n2->nargs, int);
    ANYSWAP(n1->args, n2->args, struct expression**);
#undef ANYSWAP
    return 0;
}

/*
 * gr_CopyGraph()
 *
 * Make an independant copy of a graph.  All nodes in src graph must
 * have only a single instance.
 */

Graph
gr_CopyGraph(g1)
Graph g1;
{
    Graph g2 = gr_New();
    rbTree ntree = g1->nlist;
    rbTree atree;
    rbNode tn, atn;
    Node n1, n2;

    for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next (tn)) {
	n1 = (Node) rb_Value (tn);
	n2 = gr_NewNode (n1->nk.id);
	n2->node_type = n1->node_type;
	n2->xy.x = n1->xy.x;
	n2->xy.y = n1->xy.y;
	n2->xy.set = n1->xy.set;
	if (n1->sub_name)
	    n2->sub_name = STRDUP (n1->sub_name);
	else
	    n2->sub_name = NULL;
	gr_AddNode (g2, n2);
	n1->mirror = n2;
    }
    for (tn = rb_First (ntree); !rb_Done (tn, ntree); tn = rb_Next (tn)) {
	n1 = (Node) rb_Value (tn);
	atree = n1->children;
	for (atn = rb_First (atree); ! rb_Done(atn, atree);
	     atn = rb_Next (atn)) {
	    n2 = (Node) rb_Value (atn);
	    gr_BuildAnArc (g2, n1->mirror, n2->mirror);
	}
    }
    return g2;
}

#ifdef TRACE_EXPANSION
/*
 * gr_CopyNode()
 *
 * Make a new instance of a node.
 */

Node
gr_CopyNode(n1)
Node n1;
{
    Node n2 = TALLOC (1, struct node);

    /*
     * XXX we are waiting on some sort of memory mgmt stuff here to be
     * able to reuse things like params, etc
     */

    n2->node_type = n1->node_type;
    n2->xy.x = n1->xy.x;
    n2->xy.y = n1->xy.y;
    n2->xy.set = n1->xy.set;
    n2->nk.id = n1->nk.id;
    n2->nk.inst = n1->instgen[0]++;
    n2->instgen = n1->instgen;
    n2->nparams = 0;
    n2->params = 0;
    n2->sub_name = 0;
    n2->ret_param = 0;
    n2->arrays = 0;
    n2->nargs = 0;
    n2->args = 0;
    n2->parents = rb_Create (compare_nodes);
    n2->nparents = 0;
    n2->children = rb_Create(compare_nodes);
    n2->nchildren = 0;
    n2->pair = 0;
    n2->scope = 0;
    n2->flags = 0;
    n2->pvminum = 0;
    n2->state = 0;
    n2->numRunning = 0;		/* XXX not sure about this */
    n2->numIdle = 0;
    
    n2->mirror = n1;
    
    n1->mirror = n2;

    return n2;
}

/*
 * copySGNodes()
 *
 * Recursively descends from a given node, creating a new instance
 * of each node encountered.  Stops descent if endnode is reached
 * or BEEN_HERE is set.
 *
 * If 'only' is true, only copies interior nodes (not virtual
 * pair around subgraph).
 */

static void
copySGNodes(g, n, endnode, only, xof, yof)
Graph g;
Node n, endnode;
int only;
int xof, yof;
{
    Node n2, n3;
    rbTree atree;
    rbNode atn;

/*
fprintf(stderr, "copySGNodes(%d, %d) @ %d, %d\n",
n->nk.id, endnode->nk.id, xof, yof);
*/
    n->mirror = 0;
    if (!only) {
	n2 = gr_CopyNode (n);
	n2->xy.x += xof;
	n2->xy.y += yof;
	gr_AddNode (g, n2);
    }
    n->flags |= BEEN_HERE;
    if (n == endnode)
	return;
    atree = n->children;
    for (atn = rb_First(atree); !rb_Done(atn, atree); atn = rb_Next(atn)) {
	n2 = (Node) rb_Value (atn);
	if (!(n2->flags & BEEN_HERE)) {
	    copySGNodes (g, n2, endnode, only, xof, yof);
	    if (only && n2 != endnode) {
		n3 = gr_CopyNode (n2);
		n3->xy.x += xof;
		n3->xy.y += yof;
		gr_AddNode (g, n3);
	    }
	}
    }
}

/*
 * resetSGNodes()
 *
 * Recursively descend from a given node, resetting BEEN_HERE
 * of each node encountered.  Stop descent if endnode reached or
 * BEEN_HERE is already reset.
 */

static void
resetSGNodes (n, endnode)
Node n, endnode;
{
    rbTree atree;
    rbNode atn;
    Node n2;

/*
printf("resetSGNodes(%d, %d)\n", n->nk.id, endnode->nk.id);
*/
    n->flags &= ~BEEN_HERE;
    if (n != endnode) {
	atree = n->children;
	for (atn = rb_First (atree); !rb_Done (atn, atree);
	     atn = rb_Next (atn)) {
	    n2 = (Node)rb_Value(atn);
	    if (n2->flags & BEEN_HERE)
		resetSGNodes (n2, endnode);
	}
    }
}

/*
 * connectSGNodes()
 *
 * Recursively descend from a given node, connecting up the mirror
 * copies of each node encountered.  Stop descent if BEEN_HERE is set.
 */

static void
connectSGNodes (g, n, endnode)
Graph g;
Node n, endnode;
{
    Node n2, n3;
    rbTree atree;
    rbNode atn;

#if 0
    printf ("connectSGNodes(%d-%d, %d-%d)\n", n->nk.id, n->nk.inst,
	    endnode->nk.id, endnode->nk.inst);
#endif

    if (n2 = n->mirror) {

	/* Connect n2's children */
	if (n != endnode) {
	    atree = n->children;
	    for (atn = rb_First(atree); atn != atree; atn = rb_Next(atn)) {
		n3 = (Node)rb_Value(atn);
		if (n3->mirror) {
		    gr_BuildAnArc(g, n2, n3->mirror);
#if 0
		      printf("CONNECTING %d-%d, %d-%d)\n", n2->nk.id,
			     n2->nk.inst, n3->mirror->nk.id,
			     n3->mirror->nk.inst);
#endif
		}
	    }
	}
	
	/* Set n2's pair */
	n2->pair = n->pair ? n->pair->mirror : 0;
    }
    
    n->flags |= BEEN_HERE;
    if (n == endnode)
	return;
    
    atree = n->children;
    for (atn = rb_First(atree); atn != atree; atn = rb_Next(atn)) {
	n2 = (Node)rb_Value(atn);
	if (!(n2->flags & BEEN_HERE))
	    connectSGNodes(g, n2, endnode);
    }
}

/*
 * gr_CopySubgraph()
 *
 * Make a new instance of a subgraph.  Must be handed a paired node.
 * Returns pointer to the new instance of the node originally passed.
 *
 * If 'sep' is true, makes a complete separate copy, else copies
 * interior nodes only and links them in.
 */

Node
gr_CopySubgraph (g, n, sep, xof, yof)
Graph g;
Node n;
int sep;
int xof, yof;
{
    Node other = n->pair;	/* the closing virnode */
#if 0
    Node n2;
#endif

    if (sep) {
	copySGNodes(g, n, other, 0, xof, yof);
	resetSGNodes(n, other);
	connectSGNodes(g, n, other);
	resetSGNodes(n, other);
	return n->mirror;
	
    }
    else {
#if 0
	rbTree atree;
	rbNode atn;
#endif
	copySGNodes(g, n, other, 1, xof, yof);
	resetSGNodes(n, other);
	connectSGNodes(g, n, other);
	resetSGNodes(n, other);

#if 0
/*
 * XXX this is broken --
 * can't traverse our child tree and also add stuff to it
 */
	atree = n->children;
	for (atn = rb_First(atree); !rb_Done (atn, atree); atn = rb_Next(atn))
	    if (n2 = ((Node)rb_Value(atn))->mirror) {
		gr_BuildAnArc(g, n, n2);
		printf ("CONN %d-%d %d-%d\n", n->nk.id, n->nk.inst,
			n2->nk.id, n2->nk.inst);
	    }
	atree = other->parents;
	for (atn = rb_First(atree); !rb_Done (atn, atree); atn = rb_Next(atn))
	    if (n2 = ((Node) rb_Value (atn))->mirror) {
		gr_BuildAnArc (g, n2, other);
		printf ("CONN %d-%d %d-%d\n", n->nk.id, n->nk.inst,
			other->nk.id, other->nk.inst);
	    }
#endif
	return 0;
    }
}

static void
removeSGNodes(g, n, endnode)
Graph g;
Node n, endnode;
{
    Node n2;
    rbTree atree;
    rbNode atn;

#if 0
    printf("removeSGNodes(%d-%d, %d-%d)\n",
	   n->nk.id, n->nk.inst, endnode->nk.id, endnode->nk.inst);
#endif

    atree = n->children;
    while (!rb_Done (atn = rb_First(atree), atree)) {
	n2 = (Node) rb_Value (atn);
	if (n2 == endnode)
	    gr_RemAnArc (g, n, n2);
	else {
	    removeSGNodes (g, n2, endnode);
	    gr_RemNode (g, n2);
	    gr_FreeNode (n2);
	}
    }
}

void
gr_DeleteSubgraph(g, n)
Graph g;
Node n;
{
    Node n2 = n->pair;

    removeSGNodes (g, n, n2);
    gr_NullifyNode (g, n);
    gr_NullifyNode (g, n2);
}

/*
 *	gr_NullifyNode()
 *
 *	Remove a node from the graph, but short ckt arcs going through
 *	it to the proper nodes.
 */

void
gr_NullifyNode (g, n)
Graph g;
Node n;
{
    rbTree ptree = n->parents;
    rbTree ctree = n->children;
    rbNode ptn, ctn;
    Node pn, cn;

#if 0
    puts("NullifyNode:");
#endif

    for (ptn = rb_First (ptree); !rb_Done (ptn, ptree); ptn = rb_Next (ptn)) {
	pn = (Node) rb_Value (ptn);
	for (ctn = rb_First (ctree); !rb_Done (ctn, ctree);
	     ctn = rb_Next (ctn)) {
	    cn = (Node) rb_Value (ctn);
	    gr_BuildAnArc (g, pn, cn);
#if 0
	    printf ("%d-%d --> %d-%d\n", pn->nk.id, pn->nk.inst,
		    cn->nk.id, cn->nk.inst);
#endif
	}
    }
    gr_RemNode (g, n);
    gr_FreeNode (n);
}

/*
 *	gr_NullifySubgraph()
 *
 *	Remove a subgraph from the graph, but short ckt arcs going through
 *	it to the proper nodes.
 */

void
gr_NullifySubgraph(g, n)
Graph g;
Node n;
{
    rbTree ptree = n->parents;
    rbTree ctree = n->pair->children;
    rbNode ptn, ctn;
    Node pn, cn;

#if 0
    puts("NullifySubgraph:");
#endif
    for (ptn = rb_First (ptree); !rb_Done (ptn, ptree); ptn = rb_Next(ptn)) {
	pn = (Node)rb_Value(ptn);
	for (ctn = rb_First (ctree); !rb_Done (ctn, ctree);
	     ctn = rb_Next(ctn)) {
	    cn = (Node) rb_Value (ctn);
	    gr_BuildAnArc (g, pn, cn);
#if 0
	    printf ("%d-%d --> %d-%d\n", pn->nk.id, pn->nk.inst,
		    cn->nk.id, cn->nk.inst);
#endif
	}
    }
    gr_DeleteSubgraph (g, n);
}
#endif /* TRACE_EXPANSION */
