/*
 * routines for dealing with cost matrices
 *
 * XXX eventually, when comparing host names, compare their IP addresses
 * rather than their names.  Or find some way to designate one name for
 * each host.
 * 
 * $Id: costmat.c,v 1.1 1994/02/17 20:19:29 moore Exp $
 *
 * $Log: costmat.c,v $
 * Revision 1.1  1994/02/17  20:19:29  moore
 * Initial revision
 *
 */

#include <stdio.h>
#include <ctype.h>
#include <netdb.h>
#include <sys/types.h>
#include <netinet/in.h>

#include <X11/Intrinsic.h>	/* needed by config.h (sigh) */

#include "config.h"
#include "costmat.h"
#include "global.h"
#include "graph.h"
#include "param.h"
#include "rb.h"
#include "tilde.h"

/*
 * Bondo
 */
#include <Malloc.h>
#include <String.h>

static int
hostcmp (h1, h2)
char *h1, *h2;
{
    return strcasecmp (h1, h2);
}

/*
 * cm_New ()
 *
 * Make a new cost matrix.  Add a single entry with null hostname
 * and function name.
 */

struct cost_matrix*
cm_New ()
{
    struct cost_matrix *cm;

    if (cm = TALLOC (1, struct cost_matrix)) {
	cm->nhost = 0;
	cm->nsub = 0;
	cm->hosts = (char **) NULL;
	cm->hflag = (int *) NULL;
	cm->subs = (char **) NULL;
	cm->sflag = (int *) NULL;
	cm->mat = (int *) NULL;
	cm->colors = (int *) NULL;
	cm->hostColors = (int *) NULL;
	cm->subColors = (int *) NULL;
	cm_AddEntry (cm, (char*) NULL, (char*) NULL, 0);
	cm->modified = 0;
    }
    return cm;
}

/*
 * cm_Free ()
 *
 * Free any storage associated with a cost matrix.
 */

void
cm_Free (cm)
struct cost_matrix *cm;
{
    int i;

    if (cm) {
	if (cm->hosts) {
	    for (i = cm->nhost; i-- > 0; )
		if (cm->hosts[i]) {
		    FREE (cm->hosts[i]);
		    cm->hosts[i] = (char *) NULL;
		}
	    FREE ((char *) cm->hosts);
	    cm->hosts = (char **) NULL;
	}
	if (cm->subs) {
	    for (i = cm->nsub; i-- > 0; )
		if (cm->subs[i]) {
		    FREE (cm->subs[i]);
		    cm->subs[i] = NULL;
		}
	    FREE ((char *) cm->subs);
	    cm->subs = (char **) NULL;
	}
	if (cm->sflag) {
	    FREE ((char *) cm->sflag);
	    cm->sflag = (int *) NULL;
	}
	if (cm->mat) {
	    FREE ((char *) cm->mat);
	    cm->mat = (int *) NULL;
	}
	if (cm->colors) {
	    FREE ((char *) cm->colors);
	    cm->colors = (int *) NULL;
	}
	if (cm->hostColors) {
	    FREE ((char *) cm->hostColors);
	    cm->hostColors = (int *) NULL;
	}
	if (cm->subColors) {
	    FREE ((char *) cm->subColors);
	    cm->subColors = (int *) NULL;
	}
	FREE ((char *) cm);
	cm = NULL;
    }
}

/*
 * cm_AddEntry ()
 *
 * Add (or change) the cost of a host/subroutine pair in a cost matrix.
 * Creates a new row or column if necessary.
 *
 * Returns 0 if ok
 *         1 if can't get memory.
 */

int
cm_AddEntry (cm, host, sub, val)
struct cost_matrix *cm;
char *host;
char *sub;
int val;
{
    int h, s;
    char **pp;
    int *ip;
    int *cp;
    int i, j, k;
    int n;

    cm->modified = 1;
    /*
     * special case: if the cost matrix is empty, i.e.,
     * has a single NULL host and a single NULL subr,
     * don't create a new entry
     */
    if (cm->nhost == 1 && cm->nsub == 1 &&
	cm->hosts[0] == NULL && cm->subs[0] == NULL &&
	(cm->mat == NULL || cm->mat[0] == 0)) {
	cm->hosts[0] = host ? STRDUP(host) : NULL;
	cm->subs[0] = sub ? STRDUP(sub) : NULL;
	if (cm->mat == NULL)
	    cm->mat = TALLOC (1, int);
	cm->mat[0] = val;
	if (cm->colors == NULL)
	    cm->colors = TALLOC (1, int);
	cm->colors[0] = 0;
	if (cm->hostColors == NULL)
	    cm->hostColors = TALLOC (1, int);
	cm->hostColors[0] = 0;
	if (cm->subColors == NULL)
	    cm->subColors = TALLOC (1, int);
	cm->subColors[0] = 0;
	return;
    }
    if (host) {
	for (h = cm->nhost; h-- > 0; )
	    if (cm->hosts[h] && !hostcmp (cm->hosts[h], host))
		break;
    }
    else {
	for (h = cm->nhost; h-- > 0; )
	    if (!cm->hosts[h])
		break;
    }
    if (sub) {
	for (s = cm->nsub; s-- > 0; )
	    if (cm->subs[s] && !strcmp (cm->subs[s], sub))
		break;
    }
    else {
	for (s = cm->nsub; s-- > 0; )
	    if (!cm->subs[s])
		break;
    }

    if (h < 0 || s < 0) {
	n = (k = ((s < 0) ? cm->nsub + 1 : cm->nsub))
	    * ((h < 0) ? cm->nhost + 1 : cm->nhost);
	if (!(ip = TALLOC (n, int)))
	    return 1;
	bzero (ip, n * sizeof (int));
	if ((cp = TALLOC (n, int)) == NULL)
	    return 1;
	bzero (cp, n * sizeof (int));
	for (i = cm->nsub; i-- > 0; )
	    for (j = cm->nhost; j-- > 0; ) {
		ip[i + j * k] = cm->mat[i + j * cm->nsub];
		cp[i + j * k] = cm->colors[i + j * cm->nsub];
	    }
	if (cm->mat)
	    FREE ((char *) cm->mat);
	cm->mat = ip;
	if (cm->colors)
	    FREE ((char *) cm->colors);
	cm->colors = cp;
    }
    if (h < 0) {
	h = cm->nhost++;
	if (!(pp = TALLOC (h + 1, char*))
	    || !(ip = TALLOC (h + 1, int)))
	    return 1;
	if ((cp = TALLOC (h + 1, int)) == NULL)
	    return 1;
	if (h) {
	    if (cm->hosts) {
		bcopy (cm->hosts, pp, h * sizeof (char*));
		FREE ((char *) cm->hosts);
	    }
	    if (cm->hflag) {
		bcopy (cm->hflag, ip, h * sizeof (int));
		FREE ((char *) cm->hflag);
	    }
	    if (cm->hostColors) {
		bcopy (cm->hostColors, cp, h * sizeof (int));
		FREE ((char *) cm->hostColors);
	    }
	}
	cm->hosts = pp;
	cm->hflag = ip;
	cm->hosts[h] = host ? STRDUP (host) : 0;
	cm->hflag[h] = SUBHOST_EN;
	cm->hostColors = cp;
	cm->hostColors[h] = 0;
    }
    if (s < 0) {
	s = cm->nsub++;
	if (!(pp = TALLOC (s + 1, char*))
	    || !(ip = TALLOC (s + 1, int)))
	    return 1;
	if ((cp = TALLOC (s + 1, int)) == NULL)
	    return 1;
	if (s) {
	    if (cm->subs) {
		bcopy (cm->subs, pp, s * sizeof(char*));
		FREE ((char *) cm->subs);
	    }
	    if (cm->sflag) {
		bcopy (cm->sflag, ip, s * sizeof(int));
		FREE ((char *) cm->sflag);
	    }
	    if (cm->subColors) {
		bcopy (cm->subColors, cp, s * sizeof (int));
		FREE ((char *) cm->subColors);
	    }
	}
	cm->subs = pp;
	cm->sflag = ip;
	cm->subs[s] = sub ? STRDUP (sub) : 0;
	cm->sflag[s] = 0;
	cm->subColors = cp;
	cm->subColors[s] = 0;
    }
    cm->mat[s + h * cm->nsub] = val;
    cm->colors[s + h * cm->nsub] = 0;
    return 0;
}

/*
 * set the color of a particular cell.  h and s are indices returned
 * by cm_SubrNum and cm_HostNum, respectively.
 *
 * return 1 iff the color is different than the old color and the
 * cell needs to be repainted, else return 0.
 */

int
cm_SetColor (cm, h, s, color)
struct cost_matrix *cm;
int h, s;
int color;
{
    int old_color;

    if (h >= 0 && h < cm->nhost && s >= 0 && s < cm->nsub) {
	old_color = cm->colors[h * cm->nsub + s];
	cm->colors[h * cm->nsub + s] = color;
    }
    else if (h >= 0 && h < cm->nhost) {
	old_color = cm->hostColors[h];
	cm->hostColors[h] = color;
    }
    else if (s >= 0 && s < cm->nsub) {
	old_color = cm->subColors[s];
	cm->subColors[s] = color;
    }
    return old_color != color ? 1 : 0;
}


void
cm_SetAllColors (cm, color)
struct cost_matrix *cm;
int color;
{
    int s, h;

    for (s = 0; s < cm->nsub; ++s)
	for (h = 0; h < cm->nhost; ++h)
	    cm_SetColor (cm, h, s, color);
}

/*
 * return the index of a particular subroutine in the cost matrix,
 * or -1 if not found.  Used by the stuff that updates the colors.
 */

int
cm_SubrNum (cm, sub)
struct cost_matrix *cm;
char *sub;
{
    int s;
    
    if (sub == NULL)
	return -1;
    for (s = 0; s < cm->nsub; ++s)
	if (cm->subs[s] && strcmp (cm->subs[s], sub) == 0)
	    return s;
    return -1;
}


/*
 * return the index of a particular host in the cost matrix,
 * or -1 if not found.  Used by the stuff that updates the colors.
 */

int
cm_HostNum (cm, host)
struct cost_matrix *cm;
char *host;
{
    int h;

    if (host == NULL)
	return -1;
    for (h = 0; h < cm->nhost; ++h)
	if (cm->hosts[h] && hostcmp (cm->hosts[h], host) == 0)
	    return h;
    return -1;
}

/*
 * cm_ReadFile ()
 *
 * Read a file into a new cost matrix and return it.
 *
 * Returns 0 if ok
 *         1 if can't read file
 *	   2 if errors in file
 *	   3 if can't get memory
 */

int
cm_ReadFile (fn, cmp)
char *fn;
struct cost_matrix **cmp;
{
    struct cost_matrix *cm;
    FILE *ff;
    char buf[1024];
    int ac;
    char *av[3];
    char *p;
    int err = 0;
    int line = 0;

    if (!(cm = cm_New ()))
	return 3;
    if (!(ff = tfopen (fn, "r")))
	return 1;

    while (fgets (buf, sizeof (buf), ff)) {
	char host[256];
	char sub[256];
	int cost;

	++line;
	for (p = buf; *p && isspace (*p); p++);
	if (!*p || *p == '#')	/* skip comments */
	    continue;
	ac = 3;
	if (sscanf (buf, "%s %s %d", host, sub, &cost) != 3) {
	    msg_Format ("Cost matrix \"%s\" (line %d): syntax error\n",
			fn, line);
	    err++;
	    continue;
	}
	cm_AddEntry (cm, host, sub, cost);
    }
    fclose (ff);
    cm->modified = 0;
    
    if (err) {
	cm_Free (cm);
	return 2;
    }

    if (cmp)
	*cmp = cm;
    else
	cm_Free (cm);
    return 0;
}

/*
 * write out a cost matrix.
 *
 * if write_always == 1, write all entries for which the cost is nonzero.
 * 			 (this is used to "save" cost matrices.)
 * if write_always == 0, only write those entries used to run the graph.
 *                       (this is used to write out temp cost matrices
 *                       for use by the HeNCE executioner.)
 */

static int
cm_really_write_file (fn, cmp, write_always)
char *fn;
struct cost_matrix *cmp;
int write_always;
{
    FILE *ff;
    int h, s;

    if (!cmp)
	return 2;
    if (!(ff = tfopen (fn, "w")))
	return 1;
    
    for (h = 0; h < cmp->nhost; h++) {
	/*
	 * don't write out entries if host is NULL
	 */
	if (cmp->hosts[h] == NULL || *cmp->hosts[h] == '\0')
	    continue;
	/*
	 * unless write_always is set, don't write out any entries
	 * for a host that isn't going to be used.
	 */
	if ((write_always == 0) && (cmp->hflag[h] & SUBHOST_HOST) == 0)
	    continue;
	for (s = 0; s < cmp->nsub; s++)
	    /* only write out the entry if cost > 0 */
	    if (cmp->subs[s] && *cmp->subs[s] &&
		cmp->mat[s + h * cmp->nsub] > 0)
		fprintf (ff, "%s %s %d\n", cmp->hosts[h], cmp->subs[s],
			 cmp->mat[s + h * cmp->nsub]);
    }
    (void)fclose (ff);
    return 0;
}

/*	cm_WriteFile ()
 *
 *	Write a cost matrix into a file.
 *
 *	Returns 0 if ok
 *		1 if can't write file.
 *		2 if null cost matrix.
 */

int
cm_WriteFile (fn, cmp)
char *fn;
struct cost_matrix *cmp;
{
    int foo;

    foo = cm_really_write_file (fn, cmp, 1);
    if (foo == 0)
	cmp->modified = 0;
    return foo;
}


/*
 * return true iff the cost matrix is empty, that is, there are
 * no nonzero entries defined for any (host,subr) pair
 */

int
cm_IsEmpty (cm)
struct cost_matrix *cm;
{
    int s, h;

    if (cm == NULL || cm->nhost == 0 || cm->nsub == 0 || cm->mat == NULL)
	return 1;
    for (s = 0; s < cm->nsub; ++s) {
	if (cm->subs[s] == NULL)
	    continue;
	for (h = 0; h < cm->nhost; ++h) {
	    if (cm->hosts[h] == NULL)
		continue;
	    if (cm->mat[s + h * cm->nsub] != 0)
		return 0;
	}
    }
    return 1;
}

void
cm_BuildDefaultCostMatrix (g, cm)
Graph g;
struct cost_matrix *cm;
{
    rbNode tn;
    char hostname[1024];

    gethostname (hostname, sizeof (hostname));
    for (tn = rb_First (g->nlist); !rb_Done(tn, g->nlist); tn = rb_Next (tn)) {
	Node n = (Node) rb_Value (tn);

	if (n == NULL || n->node_type != NODE_NORMAL || n->sub_name == NULL)
	    continue;
	cm_AddEntry (cm, hostname, n->sub_name, 1);
    }
}

/*
 * write out a "temp" cost matrix to be used when running the
 * HeNCE executioner.  This must be called immediately
 * after cm_AddHostsToPvm() so that the "host is needed"
 * flags will be set in the cost matrix.
 *
 * called by pvm_RunProgram ();
 *
 * return values are the same as for cm_WriteFile ();
 */

int
cm_WriteTempFile (fn, cmp)
char *fn;
struct cost_matrix *cmp;
{
    struct cost_matrix *tmp;
    Graph g = global.graph;
    int x;
    char hostname[1024];

    if (!cm_IsEmpty (cmp))
	return cm_really_write_file (fn, cmp, 0);

    /*
     * if there's no cost matrix, write out one with the current
     * host as the only hosts, all subroutines, and all costs
     * of 1.
     */

    if (g == NULL || g->nlist == NULL || rb_Empty (g->nlist)) {
	msg_Format ("graph is empty\n");
	return 1;
    }

    tmp = cm_New ();
    cm_BuildDefaultCostMatrix (g, tmp);
    msg_Format ("Warning:  no cost matrix defined, running all nodes on %s\n",
		tmp->hosts[0]);
    x = cm_really_write_file (fn, tmp, 1);
    cm_Free (tmp);
    return x;
}


/*
 * find all hosts necessary to run the program, and make sure they are
 * part of the virtual machine.
 *
 * NB: the pvm should already be running.
 *
 * make sure:
 * (a) every subr called by the graph is represented in the cost matrix
 * (b) every subr can be run on at least one host
 * (c) we only care about the hosts for which there is a nonzero entry
 *     in the cost matrix, for a subr we are running; but make sure
 *     all of these are configured in the pvm.
 *
 * return:
 * 0 if successful;
 * -1 if there is an error which would keep the program from running.
 *
 * TODO:
 * (a) mark errors for display on graph.
 * (b) mark "dead" hosts (those we can't add) on cost matrix
 * (c) mark unused hosts on cost matrix (in gray?)
 * (d) mark unused subrs on cost matrix (in gray?)
 */

int
cm_AddHostsToPvm (g, cm)
Graph g;
struct cost_matrix *cm;
{
    int s;			/* subr iteration variable */
    int h;			/* host iteration variable */
    int err = 0;		/* error count */
    rbNode tn;
    Node n;

    /*
     * sanity checks.
     */
    if (g == NULL || gr_Empty (g)) {
	msg_Format ("Graph is empty.\n");
	return -1;
    }
    if (cm_IsEmpty (cm)) {
#if 0
	msg_Format ("Cost matrix is empty.\n");
	return -1;
#else
	/*
	 * if there's no cost matrix at all, return 0;
	 * cm_WriteTempFile() will write out a default one.
	 */
	return 0;
#endif
    }

    /*
     * 1. clear all 'host needed' flags
     */
    for (h = 0; h < cm->nhost; ++h)
	cm->hflag[h] &= ~SUBHOST_HOST;

    /*
     * 2. for each "normal" node in the graph, look through the cost
     *    matrix for the subr called by that node.
     *    (if the function is not in the cost matrix; it's an error.)
     */

    for (tn = rb_First(g->nlist); !rb_Done (tn, g->nlist); tn = rb_Next (tn)) {
	int foundit = 0;

	n = (Node) rb_Value (tn);
	
	if (n == NULL || n->node_type != NODE_NORMAL)
	    continue;

	if (n->sub_name == NULL) {
	    msg_Format ("node %d: no function call specified.\n", n->nk.id);
	    /* XXX mark node as being in error */
	    ++err;
	    continue;
	}
	
	/*
	 * look for subr in cost matrix
	 */
	for (s = 0; s < cm->nsub; ++s) {

	    if (cm->subs[s] == NULL)
		continue;
	    if (strcmp (n->sub_name, cm->subs[s]) == 0) {
		/*
		 * found it; mark so we can look at it later.
		 */
		foundit = 1;
		cm->sflag[s] |= SUBHOST_SUB;
	    }
	}
	if (!foundit) {
	    msg_Format ("node %d: subr %s not found in cost matrix.\n",
			n->nk.id, n->sub_name);
	    /* XXX mark node as being in error */
	    ++err;
	}
    }
    if (err > 0)
	return -1;

    /*
     * 3. for each subr in use, make sure there is at least one host
     * with nonzero cost for that subr.  Mark all hosts that qualify.
     * If no hosts are configured, it's an error.
     */
    for (s = 0; s < cm->nsub; ++s) {
	int num_matching_hosts = 0;

	if ((cm->sflag[s] & SUBHOST_SUB) == 0)
	    continue;
	for (h = 0; h < cm->nhost; ++h) {
	    if (cm->hosts[h] == NULL || *cm->hosts[h] == '\0')
		continue;
	    if (cm->mat[s + h * cm->nsub] > 0)
		cm->hflag[h] |= SUBHOST_HOST;
	    num_matching_hosts++;
	}
	if (num_matching_hosts == 0) {
	    msg_Format ("No hosts are configured to run function %s.\n",
			cm->subs[s]);
	    if (cm_SetColor (cm, -1, s, COLOR_BROKEN))
		config_repaint_cells (-1, s);
	    ++err;
	}
    }
    if (err > 0)
	return -1;

    /*
     * 4. add each of the marked hosts to the pvm.  Fail if we can't
     * add all of them.
     */
    for (h = 0; h < cm->nhost; ++h) {
	if (cm->hflag[h] & SUBHOST_HOST) {
	    if (pvm_AddHost (cm->hosts[h]) != 0) {
		if (cm_SetColor (cm, h, -1, COLOR_BROKEN))
		    config_repaint_cells (h, -1);
		++err;
	    }
	}
    }

    return err ? -1 : 0;
}


/*
 * scan the graph and add any subr to the cost matrix that isn't already
 * there.
 *
 * this is called from the pvm panel pull-down Edit menu.
 */

int
cm_AddNamesFromGraph (g, cm)
Graph g;
struct cost_matrix *cm;
{
    rbNode tn;
    int modified = 0;

    if (g == NULL || g->nlist == NULL || rb_Empty (g->nlist))
	return;
    for (tn = rb_First (g->nlist); !rb_Done (tn, g->nlist);
	 tn = rb_Next (tn)) {
	Node n = (Node) rb_Value (tn);
	
	if (n->node_type == NODE_NORMAL && n->sub_name && *n->sub_name) {
	    int s;

	    for (s = cm->nsub; s-- > 0; )
		if (cm->subs[s] && strcmp (cm->subs[s], n->sub_name) == 0)
		    break;
	    if (s <= 0) {
		cm_AddEntry (cm, (char *) NULL, n->sub_name, 0);
		modified = 1;
	    }
	}
    }
    return modified;
}

/****************************************************************************
 *	       NOTHING BELOW THIS LINE IS USED ANYMORE                      *
 ****************************************************************************/
#if 0
/*	cm_CheckGraph ()
 *
 *	Check that every node in a graph has is mapped to at least one
 *	host (SUBHOST_EN is true and the cost is > 0).
 *	Mark all needed hosts with SUBHOST_HOST and needed subs with
 *	SUBHOST_SUB.
 *
 *	Returns 0 if ok, else 1;
 */

/* DEAD */
int
cm_CheckGraph (grf, cm)
Graph grf;
struct cost_matrix *cm;
{
    Node n;
    rbNode tn;
    int err = 0;
    int i, j, k;

    if (grf == NULL) 
	return 0;

    if (grf->heads == NULL || gr_Empty (grf))
	return 0;

    /* clear all 'needed' flags */

    for (i = cm->nsub; i-- > 0; )
	cm->sflag[i] &= ~SUBHOST_SUB;
    for (i = cm->nhost; i-- > 0; )
	cm->hflag[i] &= ~SUBHOST_HOST;
    
    /*	for each node, mark function used and also
	all enabled hosts with positive cost */

    for (tn = rb_First(grf->nlist); !rb_Done (tn, grf->nlist);
	 tn = rb_Next (tn)) {
	n = (Node) rb_Value (tn);
	
	if (n == NULL || n->node_type != NODE_NORMAL)
	    continue;

	for (i = cm->nsub; i-- > 0; )
	    if (cm->subs[i] != NULL && n->sub_name != NULL &&
		strcmp (n->sub_name, cm->subs[i]) == 0) {
		cm->sflag[i] |= SUBHOST_SUB;
		k = 0;
		for (j = cm->nhost; j-- > 0; )
		    if ((cm->hflag[j] & SUBHOST_EN)
			&& cm->hosts[j]
			&& cm->mat[i + j * cm->nsub] > 0) {
			cm->hflag[j] |= SUBHOST_HOST;
			k++;
		    }
		if (!k)
		    i = -1;
		break;
	    }
	if (i < 0) {
	    if (n->sub_name == NULL)
		msg_Format ("node %d: missing function name\n", n->nk.id);
	    else
		msg_Format ("node %d: no mapping for function %s\n",
			    n->nk.id, n->sub_name);
	    err++;
	}
    }
    return err ? 1 : 0;
}

/*
 * return 1 if hostname is mentioned in cost matrix, else 0
 * This is called by cm_BuildPvmHostFile().
 *
 * Also, set the INPVM bit as a side effect.  cm_BuildPvmHostFile()
 * will call cm_ClearPvmFlag() before starting, and will call
 * cm_CheckPvmFlag() after building the host file.  If there is
 * a host that is defined in the cost matrix, but not in the pvm
 * hosts file, cm_CheckPvmFlag() will complain about it.
 */

/* DEAD */
int
cm_HostIsPresent (cm, hostname)
struct cost_matrix *cm;
char *hostname;
{
    int h;
    for (h = cm->nhost; h-- > 0; ) {
	if ((cm->hflag[h] & SUBHOST_HOST) &&
	    hostcmp (cm->hosts[h], hostname) == 0) {
	    cm->hflag[h] |= SUBHOST_INPVM;
	    return 1;
	}
    }
    return 0;
}

/* DEAD */
void
cm_ClearPvmFlag (cm)
struct cost_matrix *cm;
{
    int h;
    for (h = cm->nhost; h-- > 0; )
	cm->hflag[h] &= ~SUBHOST_INPVM;
}

/* DEAD */
int
cm_CheckPvmFlag (cm)
struct cost_matrix *cm;
{
    int h;
    int error = 0;

    for (h = cm->nhost; h-- > 0; ) {
	if (cm->hosts[h] && (cm->hflag[h] & SUBHOST_INPVM) == 0) {
	    msg_Format ("\
%s appears in the cost matrix, but not in the pvm hosts file.\n",
			cm->hosts[h]);
	    error = -1;
	}
    }
    return error;
}

/* DEAD */
int
cm_SubIsPresent (cm, sub)
struct cost_matrix *cm;
char *sub;
{
    int s;
    for (s = cm->nsub; s-- > 0; ) {
	if ((cm->sflag[s] & SUBHOST_SUB) &&
	    strcmp (cm->subs[s], sub) == 0) {
	    return 1;
	}
    }
    return 0;
}


/*
 * make sure there is a "sub" entry in the cost matrix for every
 * function called by the HeNCE graph.  Don't call cm_AddEntry()
 * for any functions that area already there, since this has the
 * side effect of setting the modified bit.
 *
 * returns 1 if any functions were added, else 0.
 */

/* DEAD */
int
cm_AddNamesFromGraph (g, cm)
Graph g;
struct cost_matrix *cm;
{
    rbNode tn;
    int modified = 0;

    if (g->nlist == NULL || rb_Empty (g->nlist))
	return 0;
    for (tn = rb_First (g->nlist); !rb_Done (tn, g->nlist);
	 tn = rb_Next (tn)) {
	Node n = (Node) rb_Value (tn);
	
	if (n->node_type == NODE_NORMAL && n->sub_name && *n->sub_name) {
	    int s;

	    for (s = cm->nsub; s-- > 0; )
		if (cm->subs[s] && strcmp (cm->subs[s], n->sub_name) == 0)
		    break;
	    if (s <= 0) {
		cm_AddEntry (cm, (char *) NULL, n->sub_name, 0);
		modified = 1;
	    }
	}
    }
    return modified;
}

/*
 * Given a "master" pvm hosts file, emit a replica of that host file
 * containing only those hosts we need for this HeNCE program, as
 * determined by the cost matrix.
 *
 * if "in" is NULL, it means the pvm hosts file did not exist, so
 * we simply write out a host file containing everything in the
 * cost matrix.
 *
 * XXX old versions of pvm require that the local host name appear
 * first.  This tries to ensure this by always emitting the local
 * hostname (as determined by gethostbyname()) first, and refusing
 * to copy it if it appears later in the hosts file.  This doesn't
 * cause any problems with current pvm implementations, since all
 * of the per-host "options" in a pvm hosts file are only relevant
 * to remote hosts.  This might change in the future.  Also, the
 * check to see if a hostname in the pvm hosts file is "local" is
 * a simple strcmp () of the hostname from the master pvm file with
 * the one from gethostname().  It would be better done with a
 * case-insensitive string comparision, or perhaps by comparing IP
 * addresses.
 */

/* DEAD */
int
cm_BuildPvmHostFile (in, out, cm)
FILE *in, *out;
struct cost_matrix *cm;
{
    char line[1024];
    char hostname[1024];
    char myhostname[256];
    int h;

    gethostname (myhostname, sizeof myhostname);

    /*
     * reset all "host is in pvm" flags in the cost matrix, then
     * set that bit for the local host, since we always copy it.
     */

    cm_ClearPvmFlag (cm);
    (void) cm_HostIsPresent (cm, myhostname);

    /*
     * pass 1: read and parse the master pvm hosts file, setting
     * the "host is in pvm" flag for every host that appears and
     * is also in the cost matrix.
     */

    if (in) {
	while (fgets (line, sizeof (line), in)) {
	    int hostIsMe;
	    
	    if (*line == '\0' || *line == '#')
		continue;
	    sscanf (line, "%[^# \t\n] %*[^\n]", hostname);
	    hostIsMe = (hostcmp (hostname, myhostname) == 0);
	    
	    if (hostIsMe)
		continue;
	    
	    (void) cm_HostIsPresent (cm, hostname);
	}
	
	if (ferror (in) || ferror (out))
	    return -1;
    }


    /*
     * Now, prepare to write the temp pvm hosts file.  First the
     * local host name is written to the file, then the names of
     * any hosts that were in the cost matrix but were not in the
     * master pvm hosts file.
     */

    fprintf (out, "%s\n", myhostname);
    for (h = cm->nhost; h-- > 0; ) {
	if (cm->hosts[h] && (cm->hflag[h] & SUBHOST_INPVM) == 0) {
	    fprintf (out, "%s\n", cm->hosts[h]);
	}
    }
    
    /*
     * pass 2: copy the line from the pvm hosts file for any host
     * (other than the local host) that exists in the cost matrix.
     */
    
    if (in) {
	rewind (in);
	while (fgets (line, sizeof (line), in)) {
	    int hostIsMe;
	    
	    if (*line == '\0' || *line == '#')
		continue;
	    sscanf (line, "%[^# \t\n] %*[^\n]", hostname);
	    hostIsMe = (hostcmp (hostname, myhostname) == 0);
	    
	    if (hostIsMe)
		continue;
	    
	    if (strcmp (hostname, "*") == 0 ||
		cm_HostIsPresent (cm, hostname)) {
		fputs (line, out);
	    }
	}
	
	if (ferror (in) || ferror (out))
	    return -1;
    }
    return 0;
}

/* DEAD */
int
cm_WriteHosts (fp, cm)
FILE *fp;
struct cost_matrix *cm;
{
    int h;

    for (h = cm->nhost; h-- > 0; ) {
	if (cm->hosts[h] && ((cm->hflag[h] & SUBHOST_EN) != 0)) {
	    fprintf (fp, "%s\n", cm->hosts[h]);
	}
    }
    return 0;
}
#endif

#if 0
/*
 * struct used to keep track of network addresses for each host
 * these are stored in an rb-tree, indexed by ip address.
 *
 * XXX if a host has > 1 ip address, each gets a separate copy
 * of this structure...fix this.
 */

struct host_addr_entry {
    struct in_addr in_addr;
    char *cm_hostname;	/* hostname used in cost matrix */
    int flags;
    int cm_host_num;	/* index into cost matrix */
#define HOST_IS_ME   01
#define HOST_WRITTEN 02
};

/*
 * utility function to compare two IP addresses
 */

/* DEAD */
static int
compare_ip_addresses (k1, k2)
void *k1, *k2;
{
    return memcmp ((struct in_addr *) k1, (struct in_addr *) k2,
		   sizeof (struct in_addr));
}

/*
 * look up the ip addresses for a particular hostname and store them
 * in the tree.	 if (islocal == 1), we are looking up the result of
 * gethostname() so we will know our own address.  hostnum is the
 * index of the host in the current cost matrix -- or -1 if the
 * hostname didn't come from the cost matrix.
 */

/* DEAD */
static void
add_ip_addresses (t, hostname, hostnum, islocal)
rbTree t;
char *hostname;
int hostnum;
int islocal;
{
    struct hostent *hp;
    int i;
    char *STRDUP ();
    
    if ((hp = gethostbyname (hostname)) == NULL) {
	msg_Format ("can't find address for host %s -- ignored\n", hostname);
	return;
    }
    for (i = 0; hp->h_addr_list[i]; ++i) {
	struct host_addr_entry *ptr;
	rbNode tn;
	int fnd = 0;

	tn = rb_Find (t, (void *) hp->h_addr_list[i], &fnd);
	if (fnd) {
	    ptr = rb_Value (tn);
	    if (ptr->flags & HOST_IS_ME) {
		if (ptr->cm_hostname)
		    FREE (ptr->cm_hostname);
		ptr->cm_hostname = STRDUP (hostname);
		ptr->cm_host_num = hostnum;
		continue;
	    }
	    msg_Format ("warning: host %s (net address %s) appears %s\n",
			hostname, inet_ntoa (hp->h_addr_list[i]),
			"more than once in cost matrix");
	    continue;
	}
#if 0
	msg_Format ("adding host %s (%s)\n", hostname,
		    inet_ntoa (hp->h_addr_list[i]));
#endif
	ptr = TALLOC (1, struct host_addr_entry);
	memcpy (&(ptr->in_addr), hp->h_addr_list[i], sizeof (struct in_addr));
	ptr->flags = islocal ? HOST_IS_ME : 0;
	ptr->cm_hostname = STRDUP (hostname);
	ptr->cm_host_num = hostnum;
	rb_InsertBefore (tn, &(ptr->in_addr), ptr);
    }
}

/*
 * given a line from the user's pvm hosts file, write it out
 * (using the name given in the cost matrix rather than the one
 * in the host file) and set the HOST_WRITTEN bit.  Also mark
 * the host as being in the current pvm machine definition;
 * so that later on we can add any hosts that were in the
 * cost matrix but weren't in the pvm hosts file.
 *
 * return 1 iff the host is the local host.
 */

/* DEAD */
static int
write_host_if_needed (out, t, line, cm)
FILE *out;
rbTree t;
char *line;
struct cost_matrix *cm;
{
    char *p;
    struct hostent *hp;
    int i;
    char tmp;

    for (p = line; *p && *p != ' ' && *p != '\t' && *p != '\n'; ++p);
    tmp = *p;
    *p = '\0';
    if ((hp = gethostbyname (line)) == NULL) {
	msg_Format ("warning: can't find address for host %s\n",
		    line);
	return 0;
    }
    *p = tmp;
    for (i = 0; hp->h_addr_list[i]; ++i) {
	rbNode tn;
	int fnd;

	tn = rb_Find (t, (void *) hp->h_addr_list[i], &fnd);
	if (fnd) {
	    struct host_addr_entry *ptr;

	    ptr = rb_Value (tn);
	    if (ptr->flags & HOST_WRITTEN) {
		msg_Format ("warning: host %s appears twice in pvm host file\n",
			    line);
		return 0;
	    }
	    fprintf (out, "%s%s", ptr->cm_hostname, p);
	    if (ptr->cm_host_num >= 0)
		cm->hflag[ptr->cm_host_num] |= SUBHOST_INPVM;
	    ptr->flags |= HOST_WRITTEN;
	    return ptr->flags & HOST_IS_ME ? 1 : 0;
	}
    }
    return 0;
}

/*
 * return 1 iff the host named is the local host.
 */

/* DEAD */
static int
host_is_me (t, name)
rbTree t;
char *name;
{
    struct hostent *hp;
    int i;

    if ((hp = gethostbyname (name)) == NULL)
	return 0;
    for (i = 0; hp->h_addr_list[i]; ++i) {
	rbNode tn;
	int fnd;

	tn = rb_Find (t, (void *) hp->h_addr_list[i], &fnd);
	if (fnd) {
	    struct host_addr_entry *ptr;

	    ptr = rb_Value (tn);
	    if (ptr->flags & HOST_IS_ME)
		return 1;
	}
    }
    return 0;
}

/*
 * delete the tree and every object allocated by entries in the tree.
 */

/* DEAD */
static void
delete_tree (t)
rbTree t;
{
    rbNode tn, next_tn;
    struct host_addr_entry *ptr;

    for (tn = rb_First (t); !rb_Done (tn, t); tn = next_tn) {
	ptr = (struct host_addr_entry *) rb_Value (tn);
	FREE (ptr->cm_hostname);
	FREE ((void *) ptr);
	next_tn = rb_Next (tn);
	rb_DeleteNode (tn);
    }
    rb_DeleteTree (t);
}

/*
 * given the current cost matrix and optionally a "master" pvm hosts
 * file, emit a replica of that hosts file containing only those hosts
 * specified by the cost matrix (plus the local host).
 *
 * If "in" is NULL, it means the htool user did not specify a pvm
 * hosts file, so we just write out a host file containing everything
 * in the cost matrix.
 *
 * returns 0 on success, -1 on error.
 *
 */

/* DEAD */
int
cm_BuildPvmHostFile (in, out, cm)
FILE *in, *out;
struct cost_matrix *cm;
{
    int h, s;
    int found_local;
    rbTree hostAddrList;
    struct hostent *hp;
    int have_localhost;
    char myhostname[1024];
    char line[1024];

    /*
     * for each host in the cost matrix, reset the "host is needed"
     * and the "host is in pvm" bits...then set the "host is needed"
     * bit if there are any nonzero entries in the cost matrix.
     *
     * XXX maybe should check first to see if any of the named subs
     * are actually used in the graph?
     */
    for (h = 0; h < cm->nhost; ++h) {
	cm->hflag[h] &= ~(SUBHOST_HOST | SUBHOST_INPVM);
	if (cm->hosts[h] == NULL)
	    continue;
	for (s = 0; s < cm->nsub; ++s) {
	    if (cm->mat[h * cm->nsub + s] != 0) {
		cm->hflag[h] |= SUBHOST_HOST;
		break;
	    }
	}
    }
    /*
     * start building a tree to keep track of everybody's IP address
     * initialize by putting the local host in the tree.
     */
    hostAddrList = rb_Create (compare_ip_addresses);

    gethostname (myhostname, sizeof (myhostname));
    add_ip_addresses (hostAddrList, myhostname, -1, 1);

    /*
     * now add the addresses for each host needed to run the program
     */
    for (h = 0; h < cm->nhost; ++h) {
	if (cm->hflag[h] & SUBHOST_HOST)
	    add_ip_addresses (hostAddrList, cm->hosts[h], h, 0);
    }

    /*
     * scan each entry in the "master" pvm hosts file.	if we need
     * that host to run the program, copy its entry to the hosts file
     * we are going to feed to pvm.  Keep track of whether we've
     * actually seen the local host -- pvm requires that it be there.
     * If the local host is in the master hosts file we want to use
     * it (if only so the ep= option will have an effect), but if
     * the host doesn't appear there we have to copy out a dummy line
     * later.
     */
    have_localhost = 0;
    if (in) {
	while (fgets (line, sizeof (line), in) != NULL) {
	    if (*line == '#')
		continue;
	    else if (*line == '*') {
		fprintf (out, "%s", line);
		continue;
	    }
	    if (write_host_if_needed (out, hostAddrList, line, cm))
		have_localhost = 1;
	}
    }
    /*
     * for each host defined in the cost matrix,
     * if that host is not already in the pvm, write out a line containing
     * that host's name to the pvm host file.  Finally, if we haven't
     * output the name of the local host, do that last.
     */
    fprintf (out, "*\n");	/* disable any defaults */
    for (h = 0; h < cm->nhost; ++h) {
	if ((cm->hosts[h] == NULL) ||
	    (cm->hflag[h] & SUBHOST_HOST) == 0 ||
	    (cm->hflag[h] & SUBHOST_INPVM))
	    continue;
	if (have_localhost == 0 && host_is_me (hostAddrList, cm->hosts[h]))
	    have_localhost = 1;
	fprintf (out, "%s\n", cm->hosts[h]);
    }
    if (have_localhost == 0)
	fprintf (out, "%s\n", myhostname);

    /*
     * clean up
     */
    delete_tree (hostAddrList);
    return 0;
}
#endif
