/* This program studies the nonlinear gain of a 1-D FEL */

#include <stdio.h>
#include <math.h>
#define INITENG     11.0    /* gamma */
#define CAVILEN     500    /* cavity length in unit of radiation wavelength */
#define STEP        0.005   /* kL(beta-betap)STEP << 2pi; i.e. STEP << 1 */
#define TWOPI       6.28318530717959
#define KL          314159.3 /* TWOPI*CAVILEN */
#define DEBUG       0
#define NUMPART     100
#define PHI0TRAP    0.25    /* determines max E; 0.25 for theta = 1.303 */
#define MARGIN      0.2     /* region around the separatrix; < PHI0TRAP */
#define OFFSET      1.303   /* the detuning factor */
#define NUMPOINTS   100     /* number of points in each curve */
#define gam(beta)       (1.0/sqrt(1.0 - beta*beta))
#define sepx(b,bp)  (KL*((1.0-bp*b)/sqrt(1.0-b*b)-sqrt(1.0-bp*bp)))

double  beta0, betap, field, trap, zmax;  /* global */
double  engLost();

main(argc, argv)
int     argc;
char    *argv[];
{
    beta0 = sqrt(1.0 - 1.0/(INITENG*INITENG));
    betap = beta0*(1.0 - 2.0*OFFSET/KL);
    zmax = 1.0;
    if ((field = sepx(beta0,betap)/(1.0-cos(PHI0TRAP))) <= 0.0){        
        fprintf(stderr, "error in main(): E negative\n");
        exit(1);
    }/* traps any particle with phi0 = (PHI0TRAP, 2*pi-PHI0TRAP) */
    trap = 0.5*field*(1.0-cos(PHI0TRAP)); /* phi0trap = 0, i.e. no trapping */
    switch(*argv[1]){
        case 'd':   /* debug */
            field = 10.0*trap;
            engLost();
            break;
        case 'f':
            scanfield();
            break;
        case 't':
            field = 0.1*trap;
            scantheta();
            break;
        case 'l':
            scanlen();
            break;
        default:
            exit(1);
    }
    fprintf(stderr, "mission completed\n");
    fputc('\007', stderr);
}

scantheta()
{   /* has not been fully debugged---use at your own peril ! */
    register double     theta;
    double              range, step;

    range = 4.0;
    step = 2.0*range/(double)NUMPOINTS;
    for (theta = -range; theta < range; theta += step){
        betap = beta0*(1.0 - 2.0*theta/KL);
        printf("%g  %g\n", theta, engLost());
    }
}
scanfield()
{
    double      step, dgam, max, g0, ssgain, tmp;

    g0 = KL/(INITENG*INITENG*INITENG*beta0*beta0);
    tmp = 2.0*OFFSET;
    ssgain = g0*(betap/beta0)*(1.0-cos(tmp)-OFFSET*sin(tmp))/(tmp*tmp*tmp);
    /* tmp = g0/(8.0*OFFSET); */
    tmp = g0/8.0;
    max = 9.0/tmp;
    step = max/(double)NUMPOINTS;
    for (field = trap; field < max; field += step){
        dgam = engLost();
        printf("%0.4g  %0.4g  %0.4g\n", field*tmp, dgam/(ssgain*field*field), dgam/(INITENG-1.0));
    }
}
scanlen()
{
    double      step;

    step = 1.0/NUMPOINTS;
    printf("%d  %d\n", 0, 0);
    for (zmax = step; zmax < 1.0; zmax += step){
        printf("%g  %0.4g\n", zmax, engLost()/(INITENG-1.0));
    }
}

double  onePass(), phi0;

/* Computes the average energy given up by the electrons over a distance zmax */
double engLost()
{
    register double gamma;
    double          tmp, sep1, sep2, dphi, mar;
    double          avg();

    dphi = TWOPI/(double)NUMPART;
    if (field < trap)
        return (INITENG - avg(0.0, TWOPI, 5.0*dphi)/TWOPI);
    if (fabs(tmp = 1.0-sepx(beta0,betap)/field) > 1.0){
        fprintf(stderr, "error in engLost():  cosine = %0.4g\n", tmp);
        exit(1);
    }
    sep1 = acos(tmp);
    sep2 = TWOPI - sep1;
    /* if there is singularity near separatrix, more points are computed */     
    if (sing(sep1)){    /* singular at sep1 and sep2 */
        mar = 0.2*MARGIN;
        /* MARGIN < PHI0TRAP! otherwise the 4 regions may overlap */
        gamma = avg(sep1-MARGIN, sep1+MARGIN, 0.05*dphi);
        gamma += avg(sep1+MARGIN, sep2-mar, 0.1*dphi);
        gamma += avg(sep2-mar, sep2+mar, 0.003*dphi);
        gamma += avg(sep2+mar, TWOPI+sep1-MARGIN, 0.1*dphi);
    } else if (sing(sep2)){    /* singular at sep2 */
        gamma = avg(sep2-MARGIN, sep2+MARGIN, 0.1*dphi);
        gamma += avg(sep2+MARGIN, TWOPI+sep2-MARGIN, dphi);
    } else  /* smooth curve */
        gamma = avg(0.0, TWOPI, 5.0*dphi);
    return (INITENG - gamma/TWOPI);
}
/* detects singularity near separatrix */
sing(sep)
double  sep;
{
    double      beta, gamma, tmp, maxgap, bpsq, dphi;

    dphi = 0.03;
    phi0 = sep - dphi;
    beta = onePass();
    gamma = gam(beta);
    bpsq = betap*betap;
    tmp = 2.0*field/KL + sqrt(1.0 - bpsq);
    tmp = (tmp - betap*sqrt(tmp*tmp+bpsq-1.0))/(1.0-bpsq);
    if ((maxgap = 0.1*(INITENG - tmp)) < 0.0){
        fprintf(stderr, "error in sing(): beta_ > 1.0\n");
        exit(1);
    }   
    phi0 = sep + dphi;
    beta = onePass();
    return (fabs(gam(beta)-gamma) > maxgap);
}
/* computes the average energy loss */
double avg(phi1, phi2, dphi)
double  phi1, phi2, dphi;
{
    register short      ct;
    register double     beta, gamma;
    double              stop;

    ct = 0;
    gamma = 0.0;
    /* stop half-a-step earlier to prevent round-off error from
        introducing an extra point */
    stop = phi2 - 0.5*dphi;
    for (phi0 = phi1; phi0 < stop; phi0 += dphi){
        beta = onePass();
#if DEBUG
        printf("%g  %g\n", phi0, INITENG-gam(beta));
#else
        gamma += gam(beta);
#endif
        ct++;
    }
    return ((phi2-phi1)*gamma/(double)ct);
}               

/* returns the exit velocity--beta in our scaling--of an electron */
double onePass()
{
    register double     v, z, t;
    register double     dv1, dv2, dv3, dv4, dz1, dz2, dz3, dz4;
    double              a(), half;

    z = 0.0;
    t = 0.0;
    v = beta0;
    half = 0.5*STEP;
    while (z < zmax){
        dz1 = STEP*v;
        dv1 = STEP*a(t, v, z);
        dz2 = STEP*(v + 0.5*dv1);
        t += half;
        dv2 = STEP*a(t, v+0.5*dv1, z+0.5*dz1);
        dz3 = STEP*(v + 0.5*dv2);
        dv3 = STEP*a(t, v+0.5*dv2, z+0.5*dz2);
        dz4 = STEP*(v + dv3);
        t += half;
        dv4 = STEP*a(t, v+dv3, z+dz3);
        v += (dv1 + 2.0*(dv2 + dv3) + dv4)/6.0;
        z += (dz1 + 2.0*(dz2 + dz3) + dz4)/6.0;
    }
    return (v);
}

/* computes the acceleration */
double a(t, beta, z)
double  t, beta, z;
{
    register double   gamma;

    gamma = gam(beta);
    return (field*sin(KL*(z-betap*t)+phi0)/(gamma*gamma*gamma));
}
