/*
 * test-fftn.c: a small test of 2D real FFT
 *
 * Usage: test-fftn [xsize [ysize]]
 *
 * -----------------------------------------------------------------------
 * This is a simple program to demonstrate the usage of the Singleton
 * split-radix n-dimensional FFT routine. For N dimensions, the fft routine
 * must be called N times (with appropriate arguments each time) -- the
 * function fftn() does that.
 *
 * This routine demonstrates fwd and inverse transform of a small floating-
 * point array. You could use double-precision, and there are other options-
 * for instance using a combined real/imag data array instead of separate.
 * for a more extensive example, and more notes, see the test-all.c program.
 *
 *   - 7/31/95 jpb
 */

#include <stdio.h>
#include <stdlib.h>
#include "fftn.h"

#define PLIM	30	/* array size above which it's too large to print */

#undef REAL
#define REAL	float
#define fftn	fftnf

/* define some convenient macros for printing out values */

static REAL *Re_data = NULL, *Im_data = NULL;
#define Real_El(r,c)	Re_data [(r)*xsize + (c)]
#define Imag_El(r,c)	Im_data [(r)*xsize + (c)]

void
fill_array (int xsize, int ysize)
{
   int r, c, index;

   printf ("Example 2D fft: filling data array [%dx%d].\n", xsize, ysize);
   index = 0;
   for (r = 0; r < ysize; r++) {    /* fill array with a sequence of #s */
      for (c = 0; c < xsize; c++) {
	 Real_El (r,c) = index++;
	 Imag_El (r,c) = 0.0;
	 if (xsize < PLIM) printf ("%.1f ", Real_El (r,c));
      }
      if (xsize < PLIM) printf ("\n");
   }
}

void
print_array (const char *msg, int xsize, int ysize)
{
   int r, c;

   /* print out small arrays */
   if (xsize < PLIM) {
      if (msg && *msg) printf ("%s\n", msg);
      for (r = 0; r < ysize; r++) {
	 for (c = 0; c < xsize; c++)
	   printf ("%.1f,%.1f ", Real_El (r,c), Imag_El (r,c));
	 printf ("\n");
      }
   }
}

int
main (int argc, char **argv)
{
   int ret;
   int dims [2];		/* pass fft dimensions */
   int xsize = 4, ysize = 5;	/* matrix dimensions */

   if (argc > 1) xsize = ysize = atoi (argv[1]);    /* square */
   if (argc > 2) ysize = atoi (argv[2]);

   Re_data = calloc (xsize * ysize, sizeof(REAL));
   Im_data = calloc (xsize * ysize, sizeof(REAL));
   if (Re_data == NULL || Im_data == NULL) {
      fprintf (stderr, "Unable to allocate memory for data storage.\n");
      return 1;
   }

   fill_array (xsize, ysize);
   dims [0] = xsize;		/* x,y array dimensions to pass */
   dims [1] = ysize;

   /* scale one of these ways: */
#if 0
#define FORWARD_SCALE	0.0
#define INVERSE_SCALE	-1.0
#endif

#if 0
#define FORWARD_SCALE	-1.0
#define INVERSE_SCALE	0.0
#endif

#if 1
#define FORWARD_SCALE	-2.0
#define INVERSE_SCALE	-2.0
#endif

   /* 2D forward fft */
   printf ("Forward fft ..."); fflush (stdout);
   ret = fftn (2, dims, Re_data, Im_data, 1, FORWARD_SCALE);
   if (ret) return 1;

   printf ("done.\n");	/* 2D forward FFT is now completed */
   print_array ("Transformed data:", xsize, ysize);

   /* -------------now do the inverse----------------- */

   /* 2D inverse fft */
   printf ("Inverse fft ..."); fflush (stdout);
   ret = fftn (2, dims, Re_data, Im_data, -1, INVERSE_SCALE);
   if (ret) return 1;

   /* ---------------------  Array should be un-transformed now ---- */
   print_array ("Re-transformed data:", xsize, ysize);

   printf ("done.\nFreeing dynamically allocated memory\n");
   fft_free ();

   free (Im_data);
   free (Re_data);
   return 0;
}
/* ---------------------- end-of-file (c source) ---------------------- */
