      SUBROUTINE DAREX(NO, N, M, P, NPAR, DPARAM, A, LDA, B, LDB, C,
     1                 LDC, Q, LDQ, R, LDR, S, LDS, X, LDX, NOTE, 
     2                 STORE, WITHC, WITHG, WITHS, RWORK, IERR)
C     
C     PURPOSE   
C
C     To generate the benchmark examples for the numerical solution of 
C     the discrete-time algebraic Riccati equation (DARE)
C
C            T                T               T    -1  T       T
C     0  =  A X A  -  X  -  (A X B + S) (R + B X B)  (B X A + S )  +  Q 
C
C     as presented in [1]. Here, A,Q,X are real N-by-N matrices, B,S are
C     N-by-M, and R is M-by-M. The matrices Q and R are symmetric and Q
C     may be given in factored form 
C
C                   T                         
C     (I)    Q  =  C Q0 C .
C    
C     Here, C is P-by-N and Q0 is P-by-P. If R is nonsingular, the DARE
C     can be rewritten equivalently as
C
C                                -1
C     0  =  X  -  A X (I_n + G X)  A  -  Q 
C      
C     where I_n is the N-by-N identity matrix and 
C
C                   -1  T
C     (II)   G = B R   B .
C
C     ARGUMENT LIST 
C       ARGUMENTS IN
C 
C         NO - INTEGER.
C              The number of the benchmark example to generate according  
C              to [1]. 
C 
C          N - INTEGER.
C              This integer determines the actual state dimension, i.e., 
C              the order of the matrix A as follows:  
C              N is fixed for the examples of Sections 2 and 3 of [1],
C              i.e., currently Examples 1-14.
C              NOTE that N is overwritten for Examples 1-14 and for the
C              other example(s) if N is set by default.
C
C       M, P - INTEGER.
C              M is the number of columns in the matrix B and the order
C              of the matrix R (in control problems, the number of
C              inputs of the system).  
C              P is the number of rows in the matrix C from (I) (in
C              control problems, the number of outputs of the system).
C              Currently, M and P are fixed or determined by N for all
C              examples and thus are not referenced on input.  
C              NOTE that M and P are overwritten and  M .LE. N  and
C              P .LE. N  for all examples. 
C
C       NPAR - INTEGER.
C              Number of input parameters supplied by the user.
C              Examples  1-11 (Section 2 of [1]) have no parameters.
C              Examples 12-13 (Section 3 of [1]) each have one DOUBLE
C              PRECISION parameter which may be supplied in DPARAM(1).
C              Example 14 has 4 DOUBLE PRECISION parameters which may
C              be supplied in DPARAM(1) - DPARAM(4).
C              Example 15 has one INTEGER parameter which determines the
C              size of the problem. This parameter may be supplied in 
C              the input argument N. In addition, this example has one
C              DOUBLE PRECISION parameter which may be supplied in
C              DPARAM(1).   
C              If the input value of NPAR is less than the number of
C              parameters of the Example NO (according to [1]), the
C              missing parameters are set by default.
C
C     DPARAM - DOUBLE PRECISION array of DIMENSION at least ndp.
C              Double precision parameter vector where ndp is the
C              number of DOUBLE PRECISION parameters of Example NO
C              (according to [1]). For all examples, ndp <= 4. For 
C              explanation of the parameters see [1].
C              DPARAM(1) defines the parameters 'epsilon' for the
C              examples in Section 3 (NO = 12,13), the parameter 'tau'  
C              for NO = 14, and the parameter 'r' for NO = 15.
C              For Example 14, DPARAM(2) - DPARAM(4) define in
C              consecutive order 'D', 'K', and 'r'.
C              If NPAR is smaller than the number of used parameters in
C              Example NO (as described in [1]), default values are
C              used and returned in corresponding components of DPARAM.
C              NOTE that those entries of DPARAM are overwritten which
C              are used to generate the example but were not supplied by
C              the user.
C
C        LDA - INTEGER.
C              The leading dimension of array A as declared in the 
C              calling program.
C              LDA .GE. N  where N is the order of the matrix A, i.e.,
C              the output value of the integer N.  
C        
C        LDB - INTEGER.
C              The leading dimension of array B as declared in the 
C              calling program.
C              LDB .GE. N  (output value of N).
C
C        LDC - INTEGER.
C              The leading dimension of array C as declared in the 
C              calling program.
C              LDC .GE. P  where P is either defined by default or
C              depends upon N. (For all examples, P .LE. N, where N is
C              the output value of the argument N.)   
C
C        LDQ - INTEGER.
C              If full storage mode is used for Q, i.e., STORE = 'F' 
C              or 'f', then Q is stored like a 2-dimensional array
C              with leading dimension LDQ. If packed symmetric storage
C              mode is used, then LDQ is not referenced.
C              That is, if STORE = 'F' or STORE = 'f', then
C              LDQ .GE. N  if WITHC = .FALSE. 
C              LDQ .GE. P  if WITHC =  .TRUE. 
C 
C        LDR - INTEGER.
C              If full storage mode is used for the array R, i.e., 
C              STORE = 'F' or 'f', then R is stored like a 2-dimensional
C              array with leading dimension LDR. If packed symmetric
C              storage mode is used, then LDR is not referenced.
C              That is, if STORE = 'F' or STORE = 'f', then
C              LDR .GE. M  if WITHG = .FALSE.
C              LDR .GE. N  if WITHG =  .TRUE.  
C
C        LDS - INTEGER.
C              The leading dimension of array S as declared in the 
C              calling program.
C              LDS .GE. N  if S is to be returned (see MODE PARAMETER
C              WITHS). Otherwise, LDS is not referenced. 
C
C        LDX - INTEGER.
C              The leading dimension of array X as declared in the 
C              calling program.
C              LDX .GE. N  if an exact solution is available (Examples
C              1,3,5,12-15). Otherwise, X is not referenced.  
C
C       ARGUMENTS OUT
C
C          N - INTEGER.
C              The order of the matrix A. 
C
C          M - INTEGER.
C              The number of columns of matrix B and the order of the 
C              matrix R. 
C
C          P - INTEGER.
C              The number of rows of the matrix C from (I).
C 
C     DPARAM - DOUBLE PRECISION array of DIMENSION at least 7.
C              Double precision parameter vector. For explanation of the
C              parameters see [1].
C              DPARAM(1) defines the parameters 'epsilon' for the
C              examples in Section 3 (NO = 12,13), the parameter 'tau'  
C              if NO = 14, and the parameter 'r' if NO = 15.
C              For Example 14, DPARAM(2) - DPARAM(4) define in
C              consecutive order 'D', 'K', and 'r'.
C
C          A - DOUBLE PRECISION array of DIMENSION (LDA,N).
C              The leading N-by-N part of this array contains the
C              coefficient matrix A of the DARE.
C        
C          B - DOUBLE PRECISION array of DIMENSION (LDB,M).
C              If WITHG = .FALSE., then the leading N-by-M part of array 
C              B contains the coefficient matrix B of the DARE.  
C              Otherwise, B is used as workspace. 
C
C          C - DOUBLE PRECISION array of DIMENSION (LDC,N).
C              If WITHC = .TRUE., then the leading P-by-N part of array
C              C contains the matrix C of the factored form (I) of Q. 
C              Otherwise, C is used as workspace.  
C
C          Q - DOUBLE PRECISION array of DIMENSION at least qdim.
C              If STORE = 'F' or 'f',           then qdim = LDQ*nq.
C              If STORE = 'U', 'u', 'L' or 'l', then qdim = nq*(nq+1)/2.
C              If WITHC = .FALSE., then nq = N and the array Q
C              contains the coefficient matrix Q of the DARE.  
C              If WITHC = .TRUE., then nq = P and the array Q contains
C              the matrix Q0 from (I).  
C              The symmetric matrix contained in array Q is stored
C              according to MODE PARAMETER STORE. 
C
C          R - DOUBLE PRECISION array of DIMENSION at least rdim.
C              If STORE = 'F' or 'f'           then rdim = LDR*nr.
C              If STORE = 'U', 'u', 'L' or 'l' then rdim = nr*(nr+1)/2.
C              If WITHG = .FALSE., then nr = M and the array R
C              contains the coefficient matrix R of the DARE.  
C              If WITHG = .TRUE., then nr = N and the array R contains
C              the matrix G from (II).  
C              The symmetric matrix contained in array R is stored
C              according to MODE PARAMETER STORE. 
C
C          X - DOUBLE PRECISION array of DIMENSION (LDX,xdim).
C              If an exact solution is available (NO = 1,3,5,12-15),
C              then xdim = N and the leading N-by-N part of this array
C              contains the solution matrix X. Otherwise, X is not
C              referenced. 
C
C       NOTE - CHARACTER*70.
C              String containing short information about the chosen 
C              example.
C
C     WORK SPACE
C 
C      RWORK - DOUBLE PRECISION array of DIMENSION at least N*N.
C
C     MODE PARAMETERS
C
C      STORE - CHARACTER.
C              Specifies the storage mode for arrays Q and R.
C              STORE = 'F' or 'f': Full symmetric matrices are stored in
C                                  Q and R, i.e., the leading N-by-N
C                                  (M-by-M, P-by-P) parts of these
C                                  arrays each contain a symmetric
C                                  matrix. 
C              STORE = 'L' or 'l': Matrices contained in arrays Q and R
C                                  are stored in lower packed mode, that
C                                  is, the lower triangle of a k-by-k 
C                                  (k=N,M,P) symmetric matrix is stored
C                                  by columns, i.e., the matrix entry 
C                                  Q(i,j) is stored in the array entry
C                                  Q(i+(2*k-j)*(j-1)/2)  for  j <= i. 
C              STORE = 'U' or 'u': Matrices contained in arrays Q and R 
C                                  are stored in upper packed mode, that  
C                                  is, the upper triangle of a k-by-k
C                                  (k=N,M,P) symmetric matrix is stored
C                                  by columns, i.e., the matrix entry 
C                                  G(i,j) is stored in the array entry 
C                                  G(i+j*(j-1)/2)  for  i <= j.
C              Otherwise, DAREX returns with an error.
C
C      WITHC - LOGICAL.
C              Indicates whether the matrices C, Q0 as in (I) are to be
C              returned as follows. 
C              WITHC =  .TRUE., C is returned in array C and Q0 is
C                               returned in array Q.
C              WITHC = .FALSE., the coefficient matrix Q of the DARE is
C                               returned in array Q, whereas C and Q0
C                               are not returned. 
C
C      WITHG - LOGICAL.
C              Indicates whether the matrix G in (II) or the matrices B
C              and R are returned as follows. 
C              WITHG =  .TRUE., the matrix G from (II) is returned in
C                               array R, whereas the matrices B and R
C                               are not returned. 
C              WITHG = .FALSE., the coefficient matrices B and R of the
C                               DARE are returned in arrays B and R.
C
C      WITHS - LOGICAL.
C              Indicates whether the coefficient matrix S of the DARE
C              is returned as follows. 
C              WITHS =  .TRUE., the coefficient matrix S of the DARE is
C                               returned in array S.
C              WITHS = .FALSE., the coefficient matrix S of the DARE is 
C                               not returned.
C
C     ERROR INDICATOR
C     
C       IERR - INTEGER. 
C              Unless the routine detects an error (see next section),
C              IERR contains 0 on exit.
C
C     WARNINGS AND ERRORS DETECTED BY THE ROUTINE
C
C     IERR = 1 : (NO .LT. 1) or (NO .GT. NEX).  
C                (NEX = number of available examples.)
C     IERR = 2 : (N .LT. 1) or (N .GT. LDA) or (N .GT. LDB) or 
C                or (P .GT. LDC) or (WITHS and N .GT. LDS) or
C                (N .GT. LDX  and solution is available) or
C                ((STORE = 'F' or STORE = 'f') and
C                 ((WITHC .EQ. .FALSE. and N .GT. LDQ) or          
C                  (WITHC .EQ.  .TRUE. and P .GT. LDQ)) or          
C                 ((WITHG .EQ. .FALSE. and M .GT. LDR) or          
C                  (WITHG .EQ.  .TRUE. and N .GT. LDR))).          
C     IERR = 3 : MODE PARAMETER STORE had an illegal value on input.
C     IERR = 4 : Data file could not be opened or had wrong format.
C     IERR = 5 : Division by zero.  
C     IERR = 6 : G can not be computed as in (II) due to a singular R
C                matrix. This error can only occur if 
C                (WITHG .EQ. .TRUE.).
C
C     REFERENCES
C
C     [1] P. BENNER, A.J. LAUB and V. MEHRMANN
C         A Collection of Benchmark Examples for the Numerical Solution
C         of Algebraic Riccati Equations II: Discrete-Time Case.
C         Technical Report SPC 95_23, Fak. f. Mathematik, 
C         TU Chemnitz-Zwickau (Germany), December 1995.
C     [2] E. ANDERSON ET AL.
C         LAPACK Users' Guide, second edition.
C         SIAM, Philadelphia, PA (1994).
C
C     CONTRIBUTOR
C     
C     Peter Benner and Volker Mehrmann (TU Chemnitz-Zwickau)
C     Alan J. Laub (University of California, Santa Barbara) 
C
C     KEYWORDS
C    
C     discrete-time, algebraic Riccati equation
C
C     REVISIONS
C
C     1995, December 14,
C     1996, February 28.
C
C***********************************************************************    
C
C     .. Parameters ..
C     . # of examples available , # of examples with fixed size. .
      INTEGER          NEX, NFSEX
      PARAMETER        (NEX = 15, NFSEX = 14)
      DOUBLE PRECISION ZERO, ONE, TWO, THREE, FOUR, FIVE
      PARAMETER        (ZERO = .0D0, ONE = .1D1, TWO = .2D1, 
     1                  THREE = .3D1, FOUR = .4D1, FIVE = .5D1) 
C
C     .. Scalar Arguments ..
      INTEGER          NO, N, M, P, NPAR, LDA, LDB, LDC, LDQ, LDR, LDS,
     1                 LDX, IERR 
      LOGICAL          WITHC, WITHG, WITHS
      CHARACTER        STORE
C
C     .. Array Arguments ..
      DOUBLE PRECISION A(LDA,*), B(LDB,*), C(LDC,*), Q(*), R(*), 
     1                 S(LDS,*), X(LDX,*), DPARAM(*), RWORK(*)
      CHARACTER        NOTE*70
C      
C     .. Local Scalars ..
      INTEGER          I, ISYMM, INFO, IOS, J, MSYMM, NSYMM, PSYMM, 
     1                 QDIMM, RDIMM
      DOUBLE PRECISION ALPHA, BETA, TEMP
C
C     ..Local Arrays ..
      INTEGER          NDEF(NEX), MDEF(NFSEX), PDEF(NFSEX)
      CHARACTER        IDENT*4, DATAF*11
      CHARACTER*70     NOTES(NEX)
C      
C     .. External Functions ..
C     . BLAS, LAPACK .
      DOUBLE PRECISION DDOT, DLAMCH, DLAPY2
      EXTERNAL         DDOT, DLAMCH, DLAPY2
C     . LAPACK .
      LOGICAL          LSAME
      EXTERNAL         LSAME
C
C     .. External Subroutines ..
C     . BLAS .
      EXTERNAL         DCOPY, DGEMV, DSPMV, DSPR, DSYMM
C     . LAPACK .
      EXTERNAL         DLASET, DPPTRF, DPPTRI, DPTTRF, DPTTRS, DSYRK 
C     . DAREX .
      EXTERNAL         SP2SY, SY2SP
C     
C     .. Intrinsic Functions ..
      INTRINSIC        SQRT
C      
C     .. Data Statements ..
C     . default values for dimensions .
      DATA NDEF /2, 2, 2, 2, 2, 4, 4, 4, 5, 6, 9, 2, 3, 4, 100 /  
      DATA MDEF /1, 2, 1, 2, 1, 2, 2, 4, 2, 2, 3, 1, 3, 1 /
      DATA PDEF /2, 2, 1, 2, 2, 4, 4, 4, 5, 2, 2, 2, 3, 1 /
C     . comments on examples .
      DATA NOTES /
     1'Laub 1979, Ex. 2: uncontrollable-unobservable data', 
     2'Laub 1979, Ex. 3',
     3'Van Dooren 1981, Ex. II: singular R matrix',
     4'Ionescu/Weiss 1992: singular R matrix, nonzero S matrix',
     5'Jonckheere 1981: (A,B) controllable, no solution X <= 0',
     6'Ackerson/Fu 1970: satellite control problem',
     7'Litkouhi 1983: system with slow and fast modes',
     8'Lu/Lin 1993, Ex. 4.3',
     9'Gajic/Shen 1993, Section 2.7.4: chemical plant',
     A'Davison/Wang 1974: nonzero S matrix',
     B'Patnaik et al. 1980: tubular ammonia reactor',
     C'increasingly bad scaled system as eps -> oo',
     D'Petkov et al. 1989: increasingly bad scaling as eps -> oo',
     E'Pappas et al. 1980: process control of paper machine',
     F'Pappas et al. 1980, Ex. 3'/
C
C     .. Executable Statements ..
C
      IERR = 0
C
      IF ((NO .LT. 1) .OR. (NO .GT. NEX)) THEN
	IERR = 1
      ELSE IF (NO .GT. NFSEX) THEN
        IF (NPAR .LE. 0)  N = NDEF(NO)
      ELSE IF ((NO .GT. 0) .AND. (NO .LE. NFSEX)) THEN
        N = NDEF(NO)  
        M = MDEF(NO)
        P = PDEF(NO) 
      END IF
      IF (IERR .NE. 0)  GOTO 2001
      IF ((N .LT. 1) .OR. (N .GT. LDA) .OR. (N .GT. LDB) .OR. 
     1    (P .GT. LDC) .OR. (WITHS .AND. (N .GT. LDS)) .OR.
     2    (LSAME(STORE,'F') .AND. ((WITHC .AND. (P .GT. LDQ)) .OR. 
     3    ((.NOT. WITHC) .AND. (N .GT. LDQ)) .OR. 
     4    (WITHG .AND. (N .GT. LDR)) .OR. 
     5    ((.NOT. WITHG) .AND. (M .GT. LDR)))))  THEN 
        IERR = 2 
      ELSE IF (.NOT. (LSAME(STORE,'F') .OR. LSAME(STORE,'L') .OR. 
     1    LSAME(STORE,'U'))) THEN
        IERR = 3
      ELSE IF ((NO .EQ. 1) .OR. (NO .EQ. 3) .OR. (NO .EQ. 5) .OR. 
     1         ((NO .GE. 12) .AND. (NO .LE. 15))) THEN 
C    .. solution X available ..
        IF (N .GT. LDX) THEN
          IERR = 2  
        ELSE
          CALL DLASET('A', N, N, ZERO, ZERO, X, LDX)
        END IF
      END IF
      IF (IERR .NE. 0)  GOTO 2001
C      
      NOTE  = NOTES(NO)
      NSYMM = N*(N+1)/2
      MSYMM = M*(M+1)/2
      PSYMM = P*(P+1)/2
C     
      CALL DLASET('A', N, N, ZERO, ZERO, A, LDA)
      CALL DLASET('A', N, M, ZERO, ZERO, B, LDB)
      CALL DLASET('A', P, N, ZERO, ZERO, C, LDC)
      CALL DLASET('L', PSYMM, 1, ZERO, ZERO, Q, 1)
      CALL DLASET('L', MSYMM, 1, ZERO, ZERO, R, 1)
      IF (WITHS)  CALL DLASET('A', N, M, ZERO, ZERO, S, LDS)
C
      IF (NO .EQ. 1) THEN
        A(1,1) = FOUR
        A(2,1) = -.45D1
        A(1,2) = THREE
        A(2,2) = -.35D1
        CALL DLASET('A', N, M, -ONE, ONE, B, LDB)
        Q(1)  = .9D1
        Q(2)  = .6D1
        Q(3)  = FOUR
        IDENT = '0101'
        TEMP  = (ONE + SQRT(FIVE)) / TWO
        CALL DLASET('A', N, N, .6D1*TEMP, FOUR*TEMP, X, LDX)
        X(1,1) = .9D1*TEMP
C
      ELSE IF (NO .EQ. 2) THEN
        A(1,1) = .9512D0
        A(2,2) = .9048D0
        CALL DLASET('A', 1, M, .4877D1, .4877D1, B, LDB)
        B(2,1) = -.11895D1
        B(2,2) = .3569D1
        R(1)   = ONE / THREE
        R(3)   = THREE
        Q(1)   = .5D-2
        Q(3)   = .2D-1
        IDENT  = '0100' 
C       
      ELSE IF (NO .EQ. 3) THEN
        A(1,1) =  TWO
	A(2,1) =  ONE
        A(1,2) = -ONE
 	B(1,1) =  ONE
        Q(1)   = ONE
        C(1,2) = ONE
        R(1)   = ZERO
        IDENT  = '0000'
        CALL DLASET('A', N, N, ZERO, ONE, X, LDX)
C
      ELSE IF (NO .EQ. 4) THEN
        A(1,2) =  ONE
	A(2,2) = -ONE
	B(1,1) =  ONE
	B(2,1) =  TWO
	B(2,2) =  ONE
        R(1)   = .9D1
        R(2)   = THREE
        R(3)   = ONE
        CALL DLASET('A', PSYMM, 1, -FOUR, -FOUR, Q, PSYMM)
        Q(3)   = .7D1
        CALL DRSCL(MSYMM, .11D2, Q, 1)
        IF (WITHS) THEN
          S(1,1) =  THREE
          S(2,1) = -ONE
          S(1,2) =  ONE
          S(2,2) = .7D1
        END IF
        IDENT  = '0100'
C
      ELSE IF (NO .EQ. 5) THEN
        A(1,2) = ONE
	B(2,1) = ONE
        Q(1)   = ONE
        Q(2)   = TWO
        Q(3)   = FOUR
        IDENT  = '0101'
        X(1,1) = ONE
        X(2,1) = TWO
        X(1,2) = TWO
        X(2,2) = TWO + SQRT(FIVE)
C
      ELSE IF (((NO .GE. 6) .AND. (NO .LE. 9)) .OR. (NO. EQ. 11)) THEN
        IF (NO .LT. 10) THEN
          WRITE (DATAF(1:10), '(A,I1,A)') 'DAREX', NO, '.DAT'
          OPEN(1, IOSTAT = IOS, STATUS = 'OLD', FILE = DATAF(1:10))
        ELSE
          WRITE (DATAF(1:11), '(A,I2,A)') 'DAREX', NO, '.DAT'
          OPEN(1, IOSTAT = IOS, STATUS = 'OLD', FILE = DATAF(1:11))
        END IF
        IF (IOS .NE. 0) THEN
          IERR = 4
        ELSE
          DO 10  I = 1, N
            READ (1, FMT = *, IOSTAT = IOS) (A(I,J), J = 1, N)
            IF (IOS .NE. 0)  IERR = 4
10        CONTINUE
          DO 20  I = 1, N
            READ (1, FMT = *, IOSTAT = IOS) (B(I,J), J = 1, M)
            IF (IOS .NE. 0)  IERR = 4
20        CONTINUE          
          IF (NO .EQ. 6) THEN
            Q(1)  =  .187D1
            Q(4)  = -.244D0
            Q(5)  =  .744D0
            Q(6)  =  .205D0
            Q(8)  =  .589D0
            Q(10) =  .1048D1            
          ELSE IF (NO .EQ. 7) THEN
            Q(1)  = .1D-1
            Q(5)  = .1D-1
            Q(8)  = .1D-1
            Q(10) = .1D-1
          ELSE IF (NO .EQ. 8) THEN
            CALL DLASET('U', P, N, ONE, ONE, C, LDC)
            C(1,3) =  TWO
            C(1,4) =  FOUR
            C(2,4) =  TWO            
            Q(1)   =  TWO
            Q(2)   = -ONE
            Q(5)   =  TWO
            Q(6)   = -ONE
            Q(8)   =  TWO
          ELSE IF (NO .EQ. 11) THEN
            C(1,1) = ONE
            C(2,5) = ONE
            Q(1)   = .5D2
            Q(3)   = .5D2 
          END IF
        END IF
        CLOSE(1)
        IF ((NO .EQ. 6) .OR. (NO .EQ. 7)) THEN
          IDENT = '0101'
        ELSE IF ((NO .EQ. 8) .OR. (NO .EQ. 11)) THEN
          IDENT = '0001'
        ELSE IF (NO .EQ. 9) THEN
          IDENT = '0111'
        END IF        
C
      ELSE IF (NO. EQ. 10) THEN
        A(1,2) = ONE
        A(2,3) = ONE
        A(4,5) = ONE
        A(5,6) = ONE
        B(3,1) = ONE
        B(6,2) = ONE
        C(1,1) = ONE
        C(1,2) = ONE
        C(2,4) = ONE
        C(2,5) = -ONE
        R(1)   = THREE
        R(3)   = ONE
        IF (WITHS) THEN
          S(1,1) = ONE
          S(2,1) = ONE
          S(4,1) = ONE
          S(5,1) = -ONE
        END IF
        IDENT  = '0010'
C
      ELSE IF (NO .EQ. 12) THEN
        IF (NPAR .LT. 1)  DPARAM(1) = .1D7
        A(1,2) = DPARAM(1)
        B(2,1) = ONE
        IDENT  = '0111'
        X(1,1) = ONE
        X(2,2) = ONE + DPARAM(1)*DPARAM(1)
C
      ELSE IF (NO .EQ. 13) THEN
        IF (NPAR .LT. 1)  DPARAM(1) = .1D7
        A(2,2) = ONE    
        A(3,3) = THREE
        R(1)   = DPARAM(1)
        R(4)   = DPARAM(1)
        R(6)   = DPARAM(1)
C     .. set C = V ..
        TEMP   = TWO/THREE
        CALL DLASET('A', P, N, -TEMP, ONE - TEMP, C, LDC)
C     .. and compute A <- C' A C
        CALL DSYMM('L','L', N, N, ONE, C, LDC, A, LDA, ZERO, RWORK, N)    
        CALL DSYMM('R','L', N, N, ONE, C, LDC, RWORK, N, ZERO, A, LDA) 
        Q(1)   = DPARAM(1)
        Q(4)   = DPARAM(1)
        Q(6)   = DPARAM(1)
        IDENT  = '1000'
        TEMP   = DPARAM(1)**2
        X(1,1) = DPARAM(1)
        X(2,2) = DPARAM(1) * (ONE + SQRT(FIVE)) / TWO     
        X(3,3) = DPARAM(1) * (.9D1 + SQRT(.85D2)) / TWO
        CALL DSYMM('L','L', N, N, ONE, C, LDC, X, LDX, ZERO, RWORK, N)    
        CALL DSYMM('R','L', N, N, ONE, C, LDC, RWORK, N, ZERO, X, LDX)
C
      ELSE IF (NO .EQ. 14) THEN
        IF (NPAR .LT. 4)  DPARAM(4) = .25D0
        IF (NPAR .LT. 3)  DPARAM(3) = ONE
        IF (NPAR .LT. 2)  DPARAM(2) = ONE
        IF (NPAR .LT. 1)  DPARAM(1) = .1D9
        IF (DPARAM(1) .EQ. ZERO) THEN
          IERR = 5
        ELSE
          TEMP  = DPARAM(2) / DPARAM(1)
          BETA  = DPARAM(3) * TEMP
          ALPHA = ONE - TEMP
          A(1,1) = ALPHA
          CALL DLASET('A', N-1, N-1, ZERO, ONE, A(2,1), LDA) 
          B(1,1) = BETA
          C(1,4) = ONE
          R(1)  = DPARAM(4)
          IDENT = '0010'
          IF (BETA .EQ. ZERO) THEN
            IERR = 5
          ELSE
            CALL DLASET('A', N, N, ZERO, ONE, X, LDX)
            BETA   = BETA * BETA 
            TEMP   = DPARAM(4) * (ALPHA + ONE) * (ALPHA - ONE) + BETA
            X(1,1) = (TEMP + SQRT(TEMP*TEMP + FOUR*BETA*DPARAM(4)))
            X(1,1) = X(1,1) / TWO / BETA
          END IF
        END IF
C 
      ELSE IF (NO .EQ. 15) THEN
        IF (NPAR .LT. 2)  DPARAM(1) = ONE
        M = 1
        P = N        
        CALL DLASET('A', N-1, N-1, ZERO, ONE, A(1,2), LDA)
        B(N,1) = ONE
        R(1)   = DPARAM(1)
        IDENT  = '0110'
        DO 40  I = 1, N
          X(I,I) = DBLE(I)
40      CONTINUE
      END IF
C
      IF (IERR .NE. 0)  GOTO 2001
C     .. set up data in required format ..
C
      IF (WITHG) THEN
C     .. G is to be returned in product form ..
        RDIMM = N
        IF (IDENT(4:4) .EQ. '0') THEN
C       .. invert R using Cholesky factorization, ..
          CALL DPPTRF('L', M, R, INFO)
          IF (INFO .EQ. 0) THEN
            CALL DPPTRI('L', M, R, INFO)
            IF (IDENT(1:1) .EQ. '0') THEN
C           .. B is not identity matrix ..
              DO 100  I = 1, N
                CALL DSPMV('L', M, ONE, R, B(I,1), LDB, ZERO, 
     1                     RWORK((I-1)*N+1), 1)  
100           CONTINUE 
              CALL DGEMV('T', M, N, ONE, RWORK, N, B(1,1), LDB, ZERO,
     1                   R, 1) 
              ISYMM = N + 1
              DO 110  I = 2, N 
                CALL DGEMV('T', M, N, ONE, RWORK, N, B(I,1), LDB, ZERO,
     1                     B(1,1), LDB)
                CALL DCOPY(N - I + 1, B(1,I), LDB, R(ISYMM), 1)
                ISYMM = ISYMM + (N - I + 1)
110           CONTINUE 
            END IF
          ELSE
            IF (INFO .GT. 0)  IERR = 6
          END IF
        ELSE
C       .. R = identity ..
          IF (IDENT(1:1) .EQ. '0') THEN
C         .. B not identity matrix ..
            IF (M .EQ. 1) THEN
              CALL DLASET('L', NSYMM, 1, ZERO, ZERO, R, 1)
              CALL DSPR('L', N, ONE, B, 1, R)
            ELSE
              CALL DSYRK('L', 'N', N, M, ONE, B, LDB, ZERO, RWORK, N) 
              CALL SY2SP(N, RWORK, N, 'L', 'L', INFO)
              CALL DCOPY(NSYMM, RWORK, 1, R, 1)
            END IF
          ELSE
C         .. B = R = identity ..
            ISYMM = 1
            DO 120  I = N, 1, -1
              R(ISYMM) = ONE
              ISYMM = ISYMM + I
120         CONTINUE 
          END IF 
        END IF         
      ELSE
        RDIMM = M
        IF (IDENT(1:1) .EQ. '1') 
     1    CALL DLASET('A', N, M, ZERO, ONE, B, LDB) 
        IF (IDENT(4:4) .EQ. '1') THEN
          ISYMM = 1
          DO 130  I = M, 1, -1
            R(ISYMM) = ONE
            ISYMM = ISYMM + I
130       CONTINUE
        END IF 
      END IF
C 
      IF (.NOT. WITHC) THEN
C     .. Q is to be returned in product form ..
        QDIMM = N
        IF (IDENT(3:3) .EQ. '0') THEN
          IF (IDENT(2:2) .EQ. '0') THEN
C         .. C is not identity matrix ..
            DO 140  I = 1, N
              CALL DSPMV('L', P, ONE, Q, C(1,I), 1, ZERO, 
     1                   RWORK((I-1)*N+1), 1)  
140         CONTINUE 
C         .. use Q(1:N) as workspace and compute the first column of Q
C            at the end .. 
            ISYMM = N + 1
            DO 150  I = 2, N 
              CALL DGEMV('T', P, N, ONE, RWORK, N, C(1,I), 1, ZERO,
     1                   Q(1), 1)
              CALL DCOPY(N - I + 1, Q(I), 1, Q(ISYMM), 1)
              ISYMM = ISYMM + (N - I + 1)
150         CONTINUE 
            CALL DGEMV('T', P, N, ONE, RWORK, N, C(1,1), 1, ZERO, Q, 1) 
          END IF 
        ELSE
C       .. Q = identity ..
          IF (IDENT(2:2) .EQ. '0') THEN
C         .. C is not identity matrix ..
            IF (P .EQ. 1) THEN
              CALL DLASET('L', NSYMM, 1, ZERO, ZERO, Q, 1)
              CALL DSPR('L', N, ONE, C, LDC, Q)
            ELSE
              CALL DSYRK('L', 'T', N, P, ONE, C, LDC, ZERO, RWORK, N)
              CALL SY2SP(N, RWORK, N, 'L', 'L', INFO)
              CALL DCOPY(NSYMM, RWORK, 1, Q, 1)
            END IF
          ELSE
C         .. C = Q = identity ..
            ISYMM = 1
            DO 160  I = N, 1, -1
              Q(ISYMM) = ONE
              ISYMM    = ISYMM + I
160         CONTINUE 
          END IF 
        END IF
      ELSE
        QDIMM = P
        IF (IDENT(2:2) .EQ. '1')
     1    CALL DLASET('A', P, N, ZERO, ONE, C, LDC)
        IF (IDENT(3:3) .EQ. '1') THEN
          ISYMM = 1
          DO 170  I = P, 1, -1
            Q(ISYMM) = ONE
            ISYMM    = ISYMM + I
170       CONTINUE 
        END IF 
      END IF
C
C     .. unpack symmetric matrices if required ..
      IF (LSAME(STORE,'F')) THEN
        CALL SP2SY(RDIMM, R, LDR, 'L', INFO)
        CALL SP2SY(QDIMM, Q, LDQ, 'L', INFO)
      ELSE IF (LSAME(STORE,'U')) THEN
        ISYMM = RDIMM * (RDIMM + 1) / 2
        CALL DCOPY(ISYMM, R, 1, RWORK, 1)
        CALL SP2SY(RDIMM, RWORK, RDIMM, 'L', INFO)
        CALL SY2SP(RDIMM, RWORK, RDIMM, 'L', 'U', INFO)
        CALL DCOPY(ISYMM, RWORK, 1, R, 1)
        ISYMM = QDIMM * (QDIMM + 1) / 2
        CALL DCOPY(ISYMM, Q, 1, RWORK, 1)
        CALL SP2SY(QDIMM, RWORK, QDIMM, 'L', INFO)
        CALL SY2SP(QDIMM, RWORK, QDIMM, 'L', 'U', INFO)
        CALL DCOPY(ISYMM, RWORK, 1, Q, 1)
      END IF
C 
2001  CONTINUE
      RETURN
C *** Last Line of DAREX ***
      END
 

