/*********************************************************************
 *        LINEAR TREE for Supervised Learning                        *
 *        Versao 1.0 (10/12/1997)                                    *
 *        Developed by: Joao Gama                                    *
 *                LIACC - Uni.do Porto                               *
 *                jgama@ncc.up.pt                                    *
 *********************************************************************
**                                                                  **
** Copyright Joao Gama                                              **
** LIACC, University of Porto                                       **
**                                                                  **
** Please fell free to make use of this code for personal use.      **
** Express written consent from the author is required to use       **
** this program in a commercial setting.                            **
**                                                                  **
** Comments and suggestions are very welcome!                       **
** Please report any bug (I'm sorry about that) to the author       **
**                                                                  **
** for further information please refer to                          **
** http://www.ncc.up.pt/~jgama                                      **
**                                                                  **
 *-------------------------------------------------------------------*
 *  File: Ltree.c                                                    *
 *********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <sys/times.h>
#include <time.h>
#include "Ci_instances.h"
#include "BuildTree.h"
#include "tree.h"
#include "prune.h"
#include "classify.h"
#include "discrim.h"
#include "utils.h"

#define FALSE      0
#define TRUE       1
#define KH_DEFAULT 2.5
 
short  VERBOSITY   = FALSE;

short  WRITE       = FALSE;
short  GAIN_RATIO  = TRUE;
short  UNSEENS     = FALSE;
int    MINIMUM_EXAMPLES = 2;
int    MAX_DEPTH   = 5;
double MIN_SPLIT   = 0.98;
double CF          = 0.10;
double WEIGHT      = 5.0;
double KH          = KH_DEFAULT;

#define VERBOSE(v) if(VERBOSITY >= v)

int main(int argn, char *argl[])
{
  char opt, dummy[20], *FileName = NULL, *optarg = NULL;
  int test_erros = 0, **mconf;
  double  prune_errors;
  DomainInfo *domain;
  FILE *fdata, *ftest;
  CiDs *ds, *ds_test;
  Tree *tree;
  TreeInfo *tree_pruned, *tree_info = MakeTreeInfo();
  struct tms before, after;

  while ((opt = mygetopt(argn, argl, "f:v:m:c:w:k:d:l:s:Wgu", &optarg)) != EOF ) {
    switch (opt)      {
    case 'f':
      FileName = optarg;
      break;
    case 'g':
      GAIN_RATIO = FALSE;
      break;
    case 'u':
      UNSEENS = TRUE;
      break;
    case 'W':
      WRITE = TRUE;
      break;
    case 'm':
      MINIMUM_EXAMPLES = atoi(optarg);
      break;
    case 'd':
      MAX_DEPTH = atoi(optarg);
      break;
    case 'k':
      KH = atof(optarg);
      break;
    case 's':
      if (atof(optarg) > 0.0 && atof(optarg) <= 100.0)
	MIN_SPLIT = atof(optarg) / 100.0;
      break;
    case 'w':
      WEIGHT = atof(optarg);
      break;
    case 'c':
      if (atoi(optarg) >= 0 && atoi(optarg) <= 100) 
	CF = atoi(optarg) / 100.0;
      break;
    case 'v':
      VERBOSITY = atoi(optarg);
      break;
    case 'l':
      if(atoi(optarg) > 0) {
	sprintf(dummy,"1.0e-%d",atoi(optarg));
	svd_treshold(atof(dummy));
      }
      break;
    default:
      fprintf(stderr, "Invalid Option\n");
    }
  }

  if (!FileName)
    fprintf(stderr, "\nUse:%s -f dataset [-u][-v Verbose](Version 1.0)\n", argl[0]);
  else {
    times(&before);    
    if ((domain = _ReadDomain(FileName, ".domain")) != NULL) {
      --NrAttrs(domain);
      if ((fdata = fopen(new_strcat(FileName, ".data"), "r")) != NULL) {
        ds = ReadCiDataset(fdata, domain);
	VERBOSE(4)
	  printf("Reading %ld Examples from File %s.data\n", Ci_NrExs(ds), FileName);
        VERBOSE(5) ShowDomain(domain);
        VERBOSE(5) Show_CiInstances(ds, 1, Ci_NrExs(ds)); 
	
	tree = BuildTree(ds, 1, Ci_NrExs(ds), NrAttrs(domain), tree_info);
	times(&after);
	printf("Decision Tree:\n(Nodes: %d, Leaves: %d, Depth: %d, Errors: %.3f)\n",
	       NODES(tree_info),LEAVES(tree_info), DEPTH(tree_info), ERRORS(tree_info));
	Show_Tree(tree, domain, (void *) ShowTree);
	tree_pruned = Prune_Tree(domain, tree, PruneTree, &prune_errors);
	RemoveCoefficients(domain, tree, NrAttrs(domain));

	if (NODES(tree_pruned) != NODES(tree_info)) {
	  printf("Pruned Decision Tree:\n(Nodes: %d, Leaves: %d, Depth: %d, Errors: %.3f )\n",
		 NODES(tree_pruned),LEAVES(tree_pruned), DEPTH(tree_pruned), ERRORS(tree_pruned));
	  Show_Tree(tree, domain, (void *) ShowTree);
	}
	Show_Coeficients(domain, tree);
        printf("\nLinear Tree Learning Time: %6.2f (sec)\n", 
	       (after.tms_utime - before.tms_utime) / 100.0);
	if (WRITE)
	  Write_Tree(FileName, tree, WriteData, Ci_NrClasses(domain));
	
	if (UNSEENS) {
	  times(&before);
	  if ((ftest = fopen(new_strcat(FileName, ".test"), "r")) != NULL) {
	    ds_test = ReadCiDataset(ftest, domain);
	    VERBOSE(4)
	      printf("Reading %ld Examples from File %s.test\n",Ci_NrExs(ds_test), FileName);
	    VERBOSE(5) Show_CiInstances(ds_test, 1, Ci_NrExs(ds_test)); 
	    mconf = classify(ds_test, tree, NULL, &test_erros);
	    times(&after);
	    VERBOSE(2) {
	      printf("Confusion Matrix:\n");
	      ShowIMatrix(mconf, 1, Ci_NrClasses(domain), 1, Ci_NrClasses(domain)); 
	    }
	    printf("Error rate on test set %f (%ld / %ld) (Nodes: %d, Leaves: %d, Depth: %d )\n",
		   test_erros/(double)Ci_NrExs(ds_test), test_erros, Ci_NrExs(ds_test),
		   NODES(tree_pruned),LEAVES(tree_pruned), DEPTH(tree_pruned));
	    printf("\nLinear Tree Test Time: %6.2f (sec)\n", 
		   (after.tms_utime - before.tms_utime) / 100.0);
	  }
	  else
	    fprintf(stderr, "File %s.test Not Found!\n", FileName);
	}
      }
      else
	fprintf(stderr, "File %s.data Not Found!\n", FileName);
    }
    else
      fprintf(stderr, "File %s.domain Not Found!\n", FileName);
  }
  exit(0);
}











