// ==========================================================================
//
//      FFFFF  RRRR   EEEEE  EEEEE  FFFFF   OOOO   RRRR   M    M 
//      F      R   R  E      E      F      O    O  R   R  MM  MM 
//      FFF    RRRR   EEE    EEE    FFF    O    O  RRRR   M MM M 
//      F      R  R   E      E      F      O    O  R  R   M    M 
//      F      R   R  E      E      F      O    O  R   R  M    M 
//      F      R   R  EEEEE  EEEEE  F       OOOO   R   R  M    M 
//
//
//           SSSSS  PPPPP   L      III  N    N  EEEEE   SSSSS    
//          S       P    P  L       I   NN   N  E      S
//           SSSS   PPPP    L       I   N N  N  EEE     SSSS
//               S  P       L       I   N  N N  E           S
//               S  P       L       I   N   NN  E           S
//          SSSSS   P       LLLLL  III  N    N  EEEEE  SSSSS 
//
// ==========================================================================
//  
//   Project:      "Freeform splines software package" of 
//
//                     University of Karlsruhe (TH)
//                     Institute for Operating and Dialog Systems
//                     Research Group on Computer Aided Geometric Design
//                     Head of research group: Prof. Dr. H. Prautzsch
//                     WWW: http://i33www.ira.uka.de
//
//
//   Program name: outvec2bez 
//
//   Version:      1.1
//
//   Author:       Thomas Mueller, Stephan Bischoff
//  
//   Date:         July, 1997
//
//   Changes:      September, 1999
//                 - The program can now handle patches of bidegree (5,5)
//                   and (13,13)
//
//   Usage:        outvec2bez OUTVEC-FILE BEZ-FILE
//
//   Description:  Converts filling surface in OUTVEC-FILE (generated using 
//                 fill_hole) to Geomview .bez-file and writes result to 
//                 BEZ-FILE.
//                
//                 Compile the program with  
//                        compile outvec2bez  
//                 or
//                        g++ -o outvec2bez outvec2bez.c++
//                 
//                 For details refer to the documentation provided in 
//                        user_manual.ps.
//
//   Examples:     An example is described in the user manual.
//
//                 Or take a look at our research-page:
//                        http://i33www.ira.uka.de
//
//   Contact:      Please let use hear from you in case you have any 
//                 suggestions or problems with our software:
//                        prau@ira.uka.de
//                        umlauf@ira.uka.de
//
//


// --------------------------------------------------------------------------
//
//                  INCLUDES
//
// --------------------------------------------------------------------------

#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#include <stdio.h>
#include "class_vec3d.c++"
#include "class_matrix.c++"


// --------------------------------------------------------------------------
//
//                  GLOBAL VARIABLES
//
// --------------------------------------------------------------------------

int DUMP = 0; // 0: no messages
              // 1: put out messages

const int MAX_EDGENUMBER = 8; // n-sided holes can be filled by program 
                              // fill_hole if n<=MAX_EDGENUMBER.

const int MAX_NUMBER_OF_POINTS = 7*MAX_EDGENUMBER*196; // this maximum is reached,
                                                       // when method 3 for the C^2-case
                                                       // is used:
                                                       // 7 patches * (14 * 14) points * MAX_EDGENUMBER

// --------------------------------------------------------------------------
//
//                  CLASS DECLARATIONS
//
// --------------------------------------------------------------------------

// A simple scene class:
class scene_class
{
private:

  int number_of_read_points; // Number of points in OUTVEC-FILE.
  int order_of_patches; // Polynomial degree+1 of the patches:
                        // =  5 in C^1-case, methods 0,1,2
                        // =  6 in C^1-case, method 3
                        // =  7 in C^2-case, methods 0,1,2
                        // = 14 in C^2-case, method 3
  
  vec3d points[MAX_NUMBER_OF_POINTS]; // The points from OUTVEC-FILE.
  
  
  // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Private Member Function: 
  //

  // Transforms a spline patch to a bezier patch.
  vec3d *spline2bezier(vec3d *); 

public:

  // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Public Member Functions: 
  //

  // Reads OUTVEC-FILE from disk.
  void read_vecfile(char *);  

  // Writes BEZ-FILE to disk.
  void write_bezfile(char *);      
};


// --------------------------------------------------------------------------
//
//                  PRIVATE MEMBER FUNCTION DEFINITION
//
// --------------------------------------------------------------------------

// Transformation matrices L for function scene_class::spline2bezier()
// bezier-row/line = L * spline row/line

double matrix_L4x4[5][5] = {
  {0.5,  0.5,  0,   0,    0},  
  { 0,    1,   0,   0,    0},  
  { 0,    0,   1,   0,    0},  
  { 0,    0,   0,   1,    0},  
  { 0,    0,   0,  0.5,  0.5}  
};

double matrix_L5x5[6][6] = {
  {0.5,  0.5,  0,   0,    0,    0},  
  { 0,    1,   0,   0,    0,    0},  
  { 0,    0,   1,   0,    0,    0},  
  { 0,    0,   0,   1,    0,    0},
  { 0,    0,   0,   0,    1,    0},
  { 0,    0,   0,   0,   0.5,  0.5}  
};

double matrix_L6x6[7][7] = {
  {0.25,   0.5,   0.25,   0,    0,     0,     0},  
  {  0,    0.5,    0.5,   0,    0,     0,     0},  
  {  0,     0,      1,    0,    0,     0,     0},  
  {  0,     0,      0,    1,    0,     0,     0},  
  {  0,     0,      0,    0,    1,     0,     0},  
  {  0,     0,      0,    0,   0.5,   0.5,    0},  
  {  0,     0,      0,    0,  0.25,   0.5,  0.25}  
}; 

double matrix_L13x13[14][14] = {
  {0.25,   0.5,   0.25,   0,    0,    0,    0,    0,    0,    0,    0,    0,     0,     0},  
  {  0,    0.5,    0.5,   0,    0,    0,    0,    0,    0,    0,    0,    0,     0,     0},  
  {  0,     0,      1,    0,    0,    0,    0,    0,    0,    0,    0,    0,     0,     0},  
  {  0,     0,      0,    1,    0,    0,    0,    0,    0,    0,    0,    0,     0,     0},  
  {  0,     0,      0,    0,    1,    0,    0,    0,    0,    0,    0,    0,     0,     0},  
  {  0,     0,      0,    0,    0,    1,    0,    0,    0,    0,    0,    0,     0,     0},  
  {  0,     0,      0,    0,    0,    0,    1,    0,    0,    0,    0,    0,     0,     0},  
  {  0,     0,      0,    0,    0,    0,    0,    1,    0,    0,    0,    0,     0,     0},  
  {  0,     0,      0,    0,    0,    0,    0,    0,    1,    0,    0,    0,     0,     0},  
  {  0,     0,      0,    0,    0,    0,    0,    0,    0,    1,    0,    0,     0,     0},  
  {  0,     0,      0,    0,    0,    0,    0,    0,    0,    0,    1,    0,     0,     0},  
  {  0,     0,      0,    0,    0,    0,    0,    0,    0,    0,    0,    1,     0,     0},  
  {  0,     0,      0,    0,    0,    0,    0,    0,    0,    0,    0,   0.5,   0.5,    0},  
  {  0,     0,      0,    0,    0,    0,    0,    0,    0,    0,    0,  0.25,   0.5,  0.25}  
}; 

// Given:  Spline control points in spline_control_points of order
//         order_of_patches (= degree+1) over the knot vector
//         (-1 -1 0 0 0 1 1 1 2 2) (C^1-case, methods 0,1,2)
//         (-1 -1 0 0 0 0 1 1 1 1 2 2) (C^1-case, method 3)
//         (-1 -1 -1 0 0 0 0 1 1 1 1 2 2 2) (C^2-case, methods 0,1,2)
//         (-1 -1 -1 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 1 1 1 2 2 2) (C^2-case, method 3)
// Result: Bezier points of the described surface.
vec3d *scene_class::spline2bezier(vec3d *spline_control_points)
{
  int i, j, l;
  int degree = order_of_patches-1;
  // needed for transformation
  Matrix L, d(order_of_patches), b(order_of_patches);
  // result
  static vec3d *bezier_points = new vec3d[order_of_patches*order_of_patches];


   // determine transformation matrix L

   if (order_of_patches == 5){
      Matrix help(5,5);

      for (i=0; i<=4; i++)
         for (j=0; j<=4; j++)
            help(i,j) = matrix_L4x4[i][j];
      L = help;
   }

   if (order_of_patches == 6){
     Matrix help(6,6);
     
     for (i=0; i<=5; i++)
       for (j=0; j<=5; j++)
            help(i,j) = matrix_L5x5[i][j];
     L = help;
   }
   
   if (order_of_patches == 7){
      Matrix help(7,7);
      
      for (i=0; i<=6; i++)
	for (j=0; j<=6; j++)
	  help(i,j) = matrix_L6x6[i][j];
      L = help;
   }

   if (order_of_patches == 14){
      Matrix help(14,14);
      
      for (i=0; i<=13; i++)
	for (j=0; j<=13; j++)
	  help(i,j) = matrix_L13x13[i][j];
      L = help;
   }

   // transform in direction v
   for (l=0; l<=2; l++)
      for (i=0; i<=degree; i++){
         for (j=0; j<=degree; j++)
            d(j) = spline_control_points[i*order_of_patches+j][l];
         b = L*d;
         for (j=0; j<=degree; j++)
            bezier_points[i*order_of_patches+j][l] = b(j);
      }

   // transform in direction u
   for (l=0; l<=2; l++)
      for (j=0; j<=degree; j++){
         for (i=0; i<=degree; i++)
            d(i) = bezier_points[i*order_of_patches+j][l];
         b = L*d;
         for (i=0; i<=degree; i++)
            bezier_points[i*order_of_patches+j][l] = b(i);
      }

   return bezier_points;
}


// --------------------------------------------------------------------------
//
//                  PUBLIC MEMBER FUNCTION DEFINITIONS
//
// --------------------------------------------------------------------------

// Reads OUTVEC-FILE from disk.
void scene_class::read_vecfile(char *name_vec)
{
  char buffer[120];
  
  ifstream file(name_vec);
  if (!file){
    cout << "Error: Can't open OUTVEC-FILE " << name_vec << "." << endl;
    exit(1);
  }
  
  istrstream *read;
  
  number_of_read_points = 0;  // Counts read points.
  while (file.good()){
    file.getline(buffer, sizeof(buffer));
    
    read = new istrstream(buffer);
    *read >> points[number_of_read_points][0]
	  >> points[number_of_read_points][1]
	  >> points[number_of_read_points][2];
    
    number_of_read_points++;
    if (number_of_read_points > MAX_NUMBER_OF_POINTS){
      cout << "OUTVEC-FILE " << name_vec << " has too many points" << endl;
      exit(1);
    }
    delete read;
  }
}

// Writes BEZ-FILE to disk.
void scene_class::write_bezfile(char *name_bez)
{   
  vec3d *bezier_points, *spline_control_points;
  int counter_points;    // Counts read points,
  int i, j, l;
  
  order_of_patches = 0;
  if (number_of_read_points%25 == 0)        // C^1-case, methods 0,1,2
    order_of_patches = 5;
  if (number_of_read_points%36 == 0)        // C^1-case, method 3
    order_of_patches = 6;
  if (number_of_read_points%49 == 0)        // C^2-case, methods 0,1,2
    order_of_patches = 7;
  if ((number_of_read_points%(196 * 4) == 0) &&// C^2-case, method 3
      (number_of_read_points/(196 * 4) > 2))
    order_of_patches = 14;
  if (number_of_read_points == 1225) // C^1-case, method 1 or 2, 7-sided hole
    order_of_patches = 5;

  if (order_of_patches == 0)
    {
      cout << "Wrong number of points in OUTVEC-FILE (= "
	   << number_of_read_points << ")." << endl;
      exit(1);
    }

  if (DUMP >= 1)
    cout << "Number of control points = " << number_of_read_points << ".\n"
	 << "The corresponding surface is C^"
	 << (order_of_patches < 7 ? 1 : 2) << "-continuous." << endl;
  
  
  // Open file for write operation:
  ofstream file(name_bez, ios::out, 0644);
  if (!file){
    cout << "Error: Can't open bez-file " << name_bez << "." << endl;
    exit(1);
  }
  
  counter_points = 0;
  spline_control_points = new vec3d[order_of_patches*order_of_patches];
  
  // Write bezier patch:
  if (order_of_patches == 14) 
    file << "BEZDD3" << endl;      // hexadecimal notation for patches of bidegree > (6,6)
  else
    file << "BEZ" << order_of_patches-1 << order_of_patches-1 << 3 << endl;

  // Write file:
  while (counter_points < number_of_read_points){
    // Read next patch:
    for (i=0; i<=order_of_patches-1; i++)
      for (j=0; j<=order_of_patches-1; j++){
	for (l=0; l<=2; l++)
	  spline_control_points[i*order_of_patches+j][l] =
	    points[counter_points][l];
	
	counter_points++;
      }
    
    // Transform spline surface to bezier surface.
    bezier_points = spline2bezier(spline_control_points);
    
    for (i=0; i<=order_of_patches-1; i++)
      for (j=0; j<=order_of_patches-1; j++)
	file << bezier_points[i*order_of_patches+j][0] << " "
	     << bezier_points[i*order_of_patches+j][1] << " "
	     << bezier_points[i*order_of_patches+j][2] << endl;
  }
  
  delete[] spline_control_points;
}


// ==========================================================================
//                       M    M    AA    III  N    N
//                       MM  MM   A  A    I   NN   N
//                       M MM M  A    A   I   N N  N
//                       M    M  AAAAAA   I   N  N N
//                       M    M  A    A   I   N   NN
//                       M    M  A    A  III  N    N     
// ==========================================================================

// Converts file OUTVEC-FILE to Geomview bez-file BEZ-FILE.
int main(int argc, char **argv)
{

  if (argc != 3)
    {
      cout << "\nUsage:  " << argv[0] << " OUTVEC-FILE BEZ-FILE\n" << endl;
      cout << "Converts filling surface in OUTVEC-FILE (generated using fill_hole)\n" 
	   << "to Geomview .bez-file and writes result to BEZ-FILE.\n\n"
	   << "For details refer to the documentation in user_manual.ps.\n"
	   << endl;
      
    } 
  else 
    {    
      scene_class scene;
      
      // Read OUTVEC-FILE.
      scene.read_vecfile(argv[1]);
      // Write BEZ-FILE.
      scene.write_bezfile(argv[2]);
    }
  
  return 0;
}


