// ==========================================================================
//
//      FFFFF  RRRR   EEEEE  EEEEE  FFFFF   OOOO   RRRR   M    M 
//      F      R   R  E      E      F      O    O  R   R  MM  MM 
//      FFF    RRRR   EEE    EEE    FFF    O    O  RRRR   M MM M 
//      F      R  R   E      E      F      O    O  R  R   M    M 
//      F      R   R  E      E      F      O    O  R   R  M    M 
//      F      R   R  EEEEE  EEEEE  F       OOOO   R   R  M    M 
//
//
//           SSSSS  PPPPP   L      III  N    N  EEEEE   SSSSS    
//          S       P    P  L       I   NN   N  E      S
//           SSSS   PPPP    L       I   N N  N  EEE     SSSS
//               S  P       L       I   N  N N  E           S
//               S  P       L       I   N   NN  E           S
//          SSSSS   P       LLLLL  III  N    N  EEEEE  SSSSS 
//
// ==========================================================================
//  
//   Project:      "Freeform splines software package" of 
//
//                     University of Karlsruhe (TH)
//                     Institute for Operating and Dialog Systems
//                     Research Group on Computer Aided Geometric Design
//                     Head of research group: Prof. Dr. H. Prautzsch
//                     WWW: http://i33www.ira.uka.de
//
//
//   Program name: off2invec
//
//   Version:      1.1
//
//   Author:       Thomas Mueller
//  
//   Date:         July, 1997
//
//   Usage:        off2invec OFF-FILE INVEC-FILE
//
//   Description:  Converts the control net of the rim surface given in the 
//                 Geomview .off-file OFF-FILE to the format used by fill_hole
//                 and writes the result to INVEC-FILE.
//                
//                 Compile the program with  
//                        compile off2invec
//                 or
//                        g++ -o off2invec off2invec.c++
//                 
//                 For details refer to the documentation provided in 
//                        user_manual.ps.
//
//   Examples:     An example is described in the user manual.
//
//                 Or take a look at our research-page:
//                        http://i33www.ira.uka.de
//
//   Contact:      Please let use hear from you in case you have any 
//                 suggestions or problems with our software:
//                        prau@ira.uka.de
//                        umlauf@ira.uka.de
//
//


// --------------------------------------------------------------------------
//
//                  INCLUDES
//
// --------------------------------------------------------------------------

#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#include "class_vec3d.c++"


// --------------------------------------------------------------------------
//
//                  GLOBAL VARIABLES
//
// --------------------------------------------------------------------------

const int DUMP = 1; // 0 = put out only errormessages
                    // 1 = put out some messages



// --------------------------------------------------------------------------
//
//                  CLASS DECLARATIONS
//
// --------------------------------------------------------------------------

// A simple edge class:
class edges
{
public:
  int *e;
};

// A control net class:
class controlnetclass
{
private:
  int number_knots, // number_knots   = Number of knots in the control net
      number_faces; // number_faces   = Number of faces in the control net

  vec3d *knot; // knotindex in {0...number_knots-1}
               // knot[knotindex]    : Point of the control net with 
               //                      index knotindex
               // knot[knotindex][j] : j-th coordinate of this point.
  
  int *numberedges_face; // numberedges_face[faceindex] = Number of edges in 
                         //                               face faceindex

  edges *face; // faceindex in {0...number_faces-1}
               // face[faceindex].e = new int[number of edges of face 
               //                             faceindex];
               // face[faceindex].e[edgeindex_face] = knotindex : 
               //                                     index of knot in edge
               //                                     edgeindex_face of face
               //                                     faceindex
               // edgeindex_face in {0...numberedges_face[faceindex]-1}
 
  int *count_of_knot; // Counts number of occurance of the knots in the faces
                      // (only if order_of_continuity = 2)

  int order_of_continuity; // 1 = create a C^1-continuous surface  
                           //     (there are n-sided faces with n != 4)
                           // 2 = create a C^2-continuous surface  
                           //     (there are only 4-sided faces)
   
  int number_of_edges_of_hole; // Number of edges in the resulting 
                               // n-sided hole.
 
  int *order_of_knots_in_vector; // Order of the control points.


  // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Private Member Functions: 
  //

  // Calculates adjacent face to f along edge k1, k2.
  int adjacent_face(int f, int k1, int k2);

  // Calculates adjacent face f2 (with knots k3, k4) to f along edge k1, k2.
  int next_face(int f, int k1, int k2, int &f2, int &k3, int &k4);

  // Moves net defined by f and k.
  int move_controlnet(int f[], int k[]);

  // Spins net defined by f and k.
  void spin_controlnet(int f[], int k[]);
  
  // Determines the ordering of the control points in the INVEC-FILE.
  int determine_order_knots_in_vector(int hole);

public:

  // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Public Member Functions: 
  //

  // Reads OFF-FILE
  void read_offFile(char*);  

  // Writes INVEC-FILE
  void write_vecFile(char*); 
};



// --------------------------------------------------------------------------
//
//                  PRIVATE MEMBER FUNCTION DEFINITION
//
// --------------------------------------------------------------------------

// Result: In k1 and k2 adjacent face of face f.
//         The result is -1 if this face does not exist or f=-1.
//         (k1=k2 is allowed: return first face that has knot k1.)
int controlnetclass::adjacent_face(int f, int k1, int k2)
{
  if (f == -1) return -1;
  
  int i, j, k, 
      erg = -1;
  for (i=0; i<=number_faces-1 && erg == -1; i++){
    if (i == f)
      continue;
    for (j=0; j<=numberedges_face[i]-1; j++)
      if (face[i].e[j] == k1)
	for (k=0; k<=numberedges_face[i]-1; k++)
	  if (face[i].e[k] == k2)
	    erg = i;
  }
  
  return erg;
}

//  Given:  Face f and two of its neighbouring knots k1, k2.
//  Result: In k1 and k2 adjacent face f2 with other knots k3 and k4.
//          If f2 does not exist or f2 is not 4-sided or f=-1, then f2=-1 is
//          returned.
//          (Ordering of k1,k2 and k3,k4 is important in function call !)
//
//                k2         k4
//      -----------------------
//      |          |          |
//      |          |          |
//      |    f     |    f2    |
//      |          |          |
//      |          |          |
//      -----------------------
//                k1         k3
int controlnetclass::next_face(int f, int k1, int k2,
			       int &f2, int &k3, int &k4)
{
  int j;
  
  if (f == -1){
    f2 = -1;
    return f2;
  }
  f2 = adjacent_face(f, k1, k2);
  if (f2 == -1)
    return f2;
  if (numberedges_face[f2] != 4){
    f2 = -1;
    return f2;
  }
  
  for (j=0; j<=numberedges_face[f2]-1; j++)
    if (face[f2].e[j] == k2)
      break;
  
  k4 = face[f2].e[(j-1+numberedges_face[f2])%numberedges_face[f2]];
  k3 = face[f2].e[(j-2+numberedges_face[f2])%numberedges_face[f2]];
  if (k4 == k1){
    k4 = face[f2].e[(j+1)%numberedges_face[f2]];
    k3 = face[f2].e[(j+2)%numberedges_face[f2]];
  }
  
  return f2;
}

// Given:  Control net defined by f, k.
// Result: Moved net of same size, see figure.
//         The hatched region is used only in C^2-case.
//
//
//                      /|\                  
//                     / | \                
//                       |                   
//                       |                   
//                                         
//      k[13]       k[14]       k[15]       k[16]
//     -------------------------------------
//     | ///////// | ///////// | ///////// |
//     | ///////// | ///////// | ///////// |
//     | / f[7] // | / f[8] // | / f[9] // |
//     | ///////// | ///////// | ///////// |
//     |k[9]       |k[10]      |k[11]      |k[12]
//     -------------------------------------
//     |           |           | ///////// |
//     |           |           | ///////// |
//     |   f[4]    |   f[5]    | / f[6] // |
//     |           |           | ///////// |
//     |k[5]       |k[6]       |k[7]       |k[8]
//     -------------------------------------
//     |           |           | ///////// |
//     |           |           | ///////// |
//     |   f[1]    |   f[2]    | / f[3] // |
//     |           |           | ///////// |
//     |k[1]       |k[2]       |k[3]       |k[4]
//     -------------------------------------
//
int controlnetclass::move_controlnet(int f[], int k[])
{
  if (order_of_continuity == 1){
    f[1] = f[4]; f[2] = f[5];
    k[1] = k[5]; k[2] = k[6];  k[3] = k[7]; 
    k[5] = k[9]; k[6] = k[10]; k[7] = k[11];
    
    next_face(f[1], k[6], k[5], f[4], k[10], k[9] );
    next_face(f[2], k[7], k[6], f[5], k[11], k[10]);
    
    if (f[4] == -1 || f[5] == -1)
      return -1;
    
  } else {
    f[1] = f[4]; f[4] = f[7];
    f[2] = f[5]; f[5] = f[8];
    f[3] = f[6]; f[6] = f[9];
    
    k[1] = k[5];   k[2]  = k[6];   k[3]  = k[7];   k[4]  = k[8]; 
    k[5] = k[9];   k[6]  = k[10];  k[7]  = k[11];  k[8]  = k[12];
    k[9] = k[13];  k[10] = k[14];  k[11] = k[15];  k[12] = k[16];
    
    next_face(f[4], k[10], k[9],  f[7], k[14], k[13]);
    next_face(f[5], k[11], k[10], f[8], k[15], k[14]);
    next_face(f[6], k[12], k[11], f[9], k[16], k[15]);
    
    if (f[7] == -1 || f[8] == -1 || f[9] == -1)
      return -1;
  }
  
  return 0;
}

// Given:  Control net defined by f, k
// Result: Spinned net (90 degree to the left) of same size,
//         See figure above. 
//         The result is still the same net, only the ordering
//         of the faces and knots in memory is different.
void controlnetclass::spin_controlnet(int f[], int k[])
{ 
  int h1, h2, h3;
  
  if (order_of_continuity == 1){
    h1 = f[1];
    f[1] = f[2];  f[2] = f[5];  f[5] = f[4];  f[4] = h1;
    
    h1 = k[1]; h2 = k[2];
    k[1]  = k[3];   k[2]  = k[7];   k[3]  = k[11]; 
    k[7]  = k[10];  k[11] = k[9];   k[10] = k[5];
    k[9]  = h1;     k[5]  = h2;
    
  } else {
    h1 = f[1]; h2 = f[2];
    f[1] = f[3];  f[2] = f[6];  f[3] = f[9];
    f[6] = f[8];  f[9] = f[7];  f[8] = f[4];
    f[7] = h1;    f[4] = h2; 
    
    h1 = k[1]; h2 = k[2]; h3 = k[3];
    k[1]  = k[4];   k[2]  = k[8];   k[3]  = k[12];  k[4]  = k[16]; 
    k[8]  = k[15];  k[12] = k[14];  k[16] = k[13];  k[15] = k[9];
    k[14] = k[5];   k[13] = h1;     k[9]  = h2;     k[5]  = h3;
    
    h1 = k[10];
    k[10] = k[6];  k[6] = k[7];  k[7] = k[11];  k[11] = h1;
  }
}

// Determines the ordering of the control points in the INVEC-FILE
// Returns  0 if there was no error,
// Returns -1 if there was an error
//
// Illustration of the used variables:
// The hatched region is used only in C^2-case
// 
//                                         
//          k[13]       k[14]       k[15]       k[16]
//         -------------------------------------
//         | ///////// | ///////// | ///////// |
//         | ///////// | ///////// | ///////// |
//         | / f[7] // | / f[8] // | / f[9] // |
//         | ///////// | ///////// | ///////// |
//         |k[9]       |k[10]      |k[11]      |k[12]
//        -------------------------------------
//         |           |           | ///////// |
//         |           |           | ///////// |
//         |   f[4]    |   f[5]    | / f[6] // |
//         |           |           | ///////// |
//         |k[5]       |k[6]       |k[7]       |k[8]
//         -------------------------------------
//         |           |           | ///////// |
//         |           |           | ///////// |
//         |   f[1]    |   f[2]    | / f[3] // |
//         |           |           | ///////// |
//         |k[1]       |k[2]       |k[3]       |k[4]
//    ------------------------------------------
//     hole|           |           |           | 
//   f[0]  |           |           |           | 
//         |   h[1]    |   h[2]    |   h[3]    |
//         |           |           |           |
//         |l[1]       |l[2]       |l[3]       |l[4]
//         -------------------------------------
//  
int controlnetclass::determine_order_knots_in_vector(int hole)
{
  int i, j, f[13], k[21], h[4], l[5];
  
  if (order_of_continuity == 2){  // hole is around extraordinary vertex
    if (DUMP == 1)
      cout << "This surface has a"
	   << (number_of_edges_of_hole == 8 ? "n " : " ")
	   << number_of_edges_of_hole << "-sided hole." << endl;
    
    k[1] = hole;
    f[1] = -1;
    for (i=0; i<=number_faces-1 && f[1]<0; i++)
      for (j=0; j<=numberedges_face[i]-1 && f[1]<0; j++)
	if (face[i].e[j] == k[1]){
	  f[1] = i;
	  break;
	}
    
    k[2] = face[f[1]].e[ (j+1)%numberedges_face[f[1]] ];
    h[1] = adjacent_face(f[1], k[1], k[2]);
    next_face(h[1], k[2], k[1], f[1], k[6], k[5]);
    if (adjacent_face(f[1], k[1], k[5]) == -1)
      next_face(h[1], k[1], k[2], f[1], k[6], k[5]);
    if (adjacent_face(f[1], k[1], k[5]) == -1){
      return -1;
    }
    // now f[1], k[1], k[2], k[5] and k[6] are determined
    
  } else {  // hole is around n-sided face
    
    if (DUMP == 1)
      cout << "This surface has a "
	   << number_of_edges_of_hole << "-sided hole." << endl;
    
    f[0] = hole;
    l[1] = face[f[0]].e[0];
    k[1] = face[f[0]].e[1];
    
    next_face(f[0], l[1], k[1], h[1], l[2], k[2]);
    next_face(h[1], k[2], k[1], f[1], k[6], k[5]);
    // now f[1], k[1], k[2], k[5] and k[6]  are determined
  }
  
  // determine all other f[i] und k[i]
  next_face(f[1], k[2],  k[6],  f[2], k[3],  k[7] );
  next_face(f[1], k[6],  k[5],  f[4], k[10], k[9] );
  next_face(f[4], k[6],  k[10], f[5], k[7],  k[11]);
  if (order_of_continuity == 2){
    next_face(f[2], k[3],  k[7],  f[3], k[4],  k[8] );
    next_face(f[5], k[7],  k[11], f[6], k[8],  k[12]);
    next_face(f[4], k[10], k[9],  f[7], k[14], k[13]);
    next_face(f[7], k[10], k[14], f[8], k[11], k[15]);
    next_face(f[8], k[11], k[15], f[9], k[12], k[16]);
  }
  
  // OK?
  if (order_of_continuity == 1){
    if (f[1] == -1 || f[2] == -1 || f[4] == -1 || f[5] == -1)
      return -1;
  } else {  // order_of_continuity = 2
    for (i=1; i<=9; i++)
      if (f[i] == -1)
	return -1;
    if (count_of_knot[k[2]]  != 4 ||
	count_of_knot[k[3]]  != 4 || count_of_knot[k[5]]  != 4 ||
	count_of_knot[k[6]]  != 4 || count_of_knot[k[7]]  != 4 ||
	count_of_knot[k[9]]  != 4 || count_of_knot[k[10]] != 4 ||
	count_of_knot[k[11]] != 4)
      return -1;
  }
  
  
  order_of_knots_in_vector = new int[order_of_continuity == 1
				    ? 9*number_of_edges_of_hole
				    : 12*number_of_edges_of_hole+1];
  for (i=0; i<=number_of_edges_of_hole-1; i++){
    if (order_of_continuity == 1){
      order_of_knots_in_vector[i                               ] = k[1];
      
      order_of_knots_in_vector[number_of_edges_of_hole+3*i     ] = k[6];
      order_of_knots_in_vector[number_of_edges_of_hole+3*i+1   ] = k[5];
      order_of_knots_in_vector[number_of_edges_of_hole+
            (3*i-1 >= 0 ? 3*i-1 : 3*number_of_edges_of_hole-1) ] = k[2];
      
      order_of_knots_in_vector[4*number_of_edges_of_hole+5*i   ] = k[11];
      order_of_knots_in_vector[4*number_of_edges_of_hole+5*i+1 ] = k[10];
      order_of_knots_in_vector[4*number_of_edges_of_hole+5*i+2 ] = k[9];
      order_of_knots_in_vector[4*number_of_edges_of_hole+
	    (5*i-1 >= 0 ? 5*i-1 : 5*number_of_edges_of_hole-1) ] = k[7];
      order_of_knots_in_vector[4*number_of_edges_of_hole+
	    (5*i-2 >= 0 ? 5*i-2 : 5*number_of_edges_of_hole-2) ] = k[3];
      
      
    } else {  // order_of_continuity == 2
      order_of_knots_in_vector[0                                  ] = k[1];
      
      order_of_knots_in_vector[1 + 2*i                            ] = k[6];
      order_of_knots_in_vector[1 + 2*i + 1                        ] = k[5];
      
      order_of_knots_in_vector[2*number_of_edges_of_hole+1 + 4*i  ] = k[11];
      order_of_knots_in_vector[2*number_of_edges_of_hole+1 + 4*i+1] = k[10];
      order_of_knots_in_vector[2*number_of_edges_of_hole+1 + 4*i+2] = k[9];
      order_of_knots_in_vector[2*number_of_edges_of_hole+1 +
	    (4*i-1 >= 0 ? 4*i-1 : 4*number_of_edges_of_hole-1)    ] = k[7];
      
      order_of_knots_in_vector[6*number_of_edges_of_hole+1 + 6*i  ] = k[16];
      order_of_knots_in_vector[6*number_of_edges_of_hole+1 + 6*i+1] = k[15];
      order_of_knots_in_vector[6*number_of_edges_of_hole+1 + 6*i+2] = k[14];
      order_of_knots_in_vector[6*number_of_edges_of_hole+1 + 6*i+3] = k[13];
      
      order_of_knots_in_vector[6*number_of_edges_of_hole+1 +
	     (6*i-1 >= 0 ? 6*i-1 : 6*number_of_edges_of_hole-1)   ] = k[12];
      order_of_knots_in_vector[6*number_of_edges_of_hole+1 +
	     (6*i-2 >= 0 ? 6*i-2 : 6*number_of_edges_of_hole-2)   ] = k[8];
    }
    
    spin_controlnet(f, k);
    if (move_controlnet(f, k) == -1)
      return -1;
    if (move_controlnet(f, k) == -1)
      return -1;
    if (move_controlnet(f, k) == -1)
      return -1;
  }
  
  return 0;
}


// --------------------------------------------------------------------------
//
//                  PUBLIC MEMBER FUNCTION DEFINITION
//
// --------------------------------------------------------------------------

// Reads Geomview .off-file with filename Name_off from disk.
void controlnetclass::read_offFile(char* name_off)
{
  int i, j;
  char buffer[120];
  ifstream file(name_off);
  order_of_continuity = 2;
  
  if (!file){
    cout << "Error: Can't open off-file " << name_off << "." << endl;
    exit(1);
  }
  
  istrstream *read;
  
  if (file.good()){
    buffer[0] = '\0';
    file.getline(buffer, sizeof(buffer));
    while ((buffer[0] == '#' || buffer[1] == '#' || buffer[2] == '#' ||
	    buffer[3] == '#' || buffer[4] == '#' || buffer[5] == '#' ||
	    buffer[0] == '\0' ||
	    buffer[0] == 'O' || buffer[1] == 'O' || buffer[2] == 'O' ||
	    buffer[3] == 'O' || buffer[4] == 'O' ||
	    buffer[0] == 'o' || buffer[1] == 'o' || buffer[2] == 'o' ||
	    buffer[3] == 'o' || buffer[4] == 'o') && file.good())
      file.getline(buffer, sizeof(buffer)); // Overread #- and OFF-lines.
                                    // Do not put too many spaces before !!
    if (!file){
      cout << "Error: Can't read off-file " << name_off << "." << endl;
      exit(1);
    }
    
    read = new istrstream(buffer);
    
    *read >> number_knots >> number_faces >> i;  // i is not used.
    if (number_knots <= 0 || number_faces <= 0){
      cout << "Error in file " << name_off << " (number of edges or faces)."
	   << endl;
      exit(1);
    }
    
    delete read;
  }
  
  
  knot = new vec3d[number_knots];
  numberedges_face = new int[number_faces];
  face = new edges[number_faces];
  
  // Read the knots: 
  for (i=0; i<=number_knots-1; i++){
    buffer[0] = '\0';
    if (file.good())
      file.getline(buffer, sizeof(buffer));
    while ((buffer[0] == '#' || buffer[1] == '#' || buffer[2] == '#' ||
	    buffer[3] == '#' || buffer[4] == '#' || buffer[5] == '#' ||
	    buffer[0] == '\0') && file.good())
      file.getline(buffer, sizeof(buffer)); // Overread #- and empty-lines.
                                  // Do not put too many spaces before # !!
    if (!file.good()){
      cout << "Error in off-File " << name_off << " (number of knots)."
	   << endl;
      exit(1);
    }
    
    read = new istrstream(buffer);
    *read >> knot[i][0] >> knot[i][1] >> knot[i][2];     
    delete read;
  }
  
  // Read the faces:
  for (i=0; i<=number_faces-1; i++){
    buffer[0] = '\0';
    if (file.good())
      file.getline(buffer, sizeof(buffer));
    while ((buffer[0] == '#' || buffer[1] == '#' || buffer[2] == '#' ||
	    buffer[3] == '#' || buffer[4] == '#' || buffer[5] == '#' ||
	    buffer[0] == '\0') && file.good())
      file.getline(buffer, sizeof(buffer)); // Overread #- and empty-lines.
                                  // Do not put too many spaces before # !!
    if (!file.good()){
      cout << "Error in off-File " << name_off
	   << " (number of knots or faces is wrong)."
	   << endl;
      exit(1);
    }
    
    read = new istrstream(buffer);
    
    *read >> numberedges_face[i];
    if (numberedges_face[i] != 4)  // There are n-sided faces with n != 4.
      order_of_continuity = 1;
    
    // Read face:
    face[i].e = new int[numberedges_face[i]];     
    for (j=0; j<=numberedges_face[i]-1; j++)
      *read >> face[i].e[j];
    
    delete read;
  } 
  
  while (file.good()){
    file.getline(buffer, sizeof(buffer));
    int empty_line = 1;
    for (j=0; j<sizeof(buffer) && buffer[j] != '\0'; j++)
      if (buffer[j] >= '0' && buffer[j] <= '9'){
	empty_line = 0;
	break;
      }
    if (empty_line == 0){
      cout << "Error in off-File " << name_off
	   << " (number of knots or faces is wrong)."
	   << endl;
      exit(1);
    }
  }
  
  if (DUMP == 1)
    cout << "The net is treated as the control net of a C^"
	 << order_of_continuity << "-continuous surface." << endl;
}

// Writes result into file name_vec in INVEC-format.
void controlnetclass::write_vecFile(char *name_vec)
{
  int i, j, error;
  
  if (order_of_continuity == 1){  // C^1-case, look for n-sided faces.
    for (i=0; i<=number_faces-1; i++)
      if (numberedges_face[i] != 4){  // Around i there is a hole.
	number_of_edges_of_hole = numberedges_face[i];
	break;
      }
    
    if (determine_order_knots_in_vector(i) != 0){
      cout << "Control net has wrong topologie." << endl;
      exit(1);
    }
    
    
  } else { // C^2-case, look for knots who occur n times, n != 4.

    // Count how often the knots occur in the faces.
    count_of_knot = new int[number_knots];
    for (i=0; i<=number_knots-1; i++) // init
      count_of_knot[i] = 0;
    for (i=0; i<=number_faces-1; i++) // count
      for (j=0; j<=numberedges_face[i]-1; j++)
	count_of_knot[face[i].e[j]] += 1;
    
    for (i=0; i<=number_knots-1; i++)
      if (count_of_knot[i] == 3 || count_of_knot[i] > 4){// Around i is hole.
	number_of_edges_of_hole = count_of_knot[i];
	error = determine_order_knots_in_vector(i);
	if (error == 0)
	  break;
      }
    
    if (error != 0){
      cout << "Control net has wrong topology." << endl;
      exit(1);
    }
    
    delete count_of_knot;
  }
  

  // Write vector file:
  int size_of_invec = (order_of_continuity == 1
		       ? 9*number_of_edges_of_hole-1
		       : 12*number_of_edges_of_hole
                       );
  ofstream file(name_vec, ios::out, 0644);
  if (!file){
    cout << "Error: Can't open file " << name_vec << "." << endl;
    exit(1);
  }
  
  for (i=0; i<=size_of_invec; i++){
    file << knot[ order_of_knots_in_vector[i] ][0] << " "
	 << knot[ order_of_knots_in_vector[i] ][1] << " "
	 << knot[ order_of_knots_in_vector[i] ][2];
    if (i != size_of_invec)
      file << endl;
  }
  
  delete order_of_knots_in_vector;
}


// ==========================================================================
//                       M    M    AA    III  N    N
//                       MM  MM   A  A    I   NN   N
//                       M MM M  A    A   I   N N  N
//                       M    M  AAAAAA   I   N  N N
//                       M    M  A    A   I   N   NN
//                       M    M  A    A  III  N    N     
// ==========================================================================

int main(int argc, char **argv)
{

  if (argc != 3)
    {
      cout << "\nUsage:  " << argv[0] << " OFF-FILE INVEC-FILE\n" << endl;
      cout << "Converts the control net of the rim surface given in the\n"
	   << "Geomview .off-file OFF-FILE to the format used by fill_hole\n"
	   << "and writes the result to INVEC-FILE.\n\n"
	   << "For details refer to the documentation in user_manual.ps.\n" 
	   << endl;
      
    } 
  else 
    {
      controlnetclass net;
      
      // Read .off-file.
      net.read_offFile(argv[1]);
      // Write INVEC-FILE.
      net.write_vecFile(argv[2]);
    }
  
  return 0;
}


