// ==========================================================================
//
//      FFFFF  RRRR   EEEEE  EEEEE  FFFFF   OOOO   RRRR   M    M 
//      F      R   R  E      E      F      O    O  R   R  MM  MM 
//      FFF    RRRR   EEE    EEE    FFF    O    O  RRRR   M MM M 
//      F      R  R   E      E      F      O    O  R  R   M    M 
//      F      R   R  E      E      F      O    O  R   R  M    M 
//      F      R   R  EEEEE  EEEEE  F       OOOO   R   R  M    M 
//
//
//           SSSSS  PPPPP   L      III  N    N  EEEEE   SSSSS    
//          S       P    P  L       I   NN   N  E      S
//           SSSS   PPPP    L       I   N N  N  EEE     SSSS
//               S  P       L       I   N  N N  E           S
//               S  P       L       I   N   NN  E           S
//          SSSSS   P       LLLLL  III  N    N  EEEEE  SSSSS 
//
// ==========================================================================
//  
//   Project:      "Freeform splines software package" of 
//
//                     University of Karlsruhe (TH)
//                     Institute for Operating and Dialog Systems
//                     Research Group on Computer Aided Geometric Design
//                     Head of research group: Prof. Dr. H. Prautzsch
//                     WWW: http://i33www.ira.uka.de
//
//
//   Program name: fill_hole
//
//   Version:      1.1
//
//   Author:       Thomas Mueller, Stephan Bischoff
//  
//   Date:         July, 1997
//
//   Changes:      September, 1999
//                 - method 3 implemented
//                 - matrices are assembled on run time
//
//   Usage:        fill_hole INVEC-FILE OUTVEC-FILE METHOD
//
//   Description:  The Program reads a control net from the file INVEC-FILE, 
//                 fills the corresponding n-sided hole depending on the 
//                 chosen METHOD and writes the resulting surface patches 
//                 to file OUTVEC-FILE. 
//                 Attention: n has to be <= 8, other holes can not be filled!!
//
//
//                 The file INVEC-FILE has to be as follows:
//                 =========================================
//                 The i-th line contains the coordinates of the i-th 
//                 control point separated by 'spaces'. Coordinates of 
//                 consecutive control points are separated by 'line breaks':
//              
//                 For example:
//              
//                      0.707107 2 0
//                      0.707107 2.70711 0.707107
//                      0 2 0.707107
//                      1.70711 1 0
//                      ...
//              
//                 Empty lines are allowed.
//              
//                 The points have to be ordered according to the following 
//                 scheme:
//                      Count the point from inner to outer rings around the
//                      n-sided face (n!=4) resp. the extraordinary point.
//                      Number counterclockwise within the rings.
//                 Thus there are 9*n control points resp.12*n+1 control 
//                 points in the INVEC-FILE. The following figures 
//                 depict this ordering:
//
//              
//                                          /|4n+6                      
//                                         / |                           
//                                        /  |                           
//                                       /   |                            
//                                  4n+7/    |                            
//                                     /|    |                            
//                                    / |   /|4n+5                       
//                                   /  |  / |                         
//                                  /   | /  |                            
//                             4n+8/    |/   |                   
//                      4n+9|------| n+4|    |                          
//                          |      |   /|    |4n+4                  
//                          |      |  / |   /\                       
//                          |      | /  |  /  \                        
//                          |   n+5|/   | /    \                     
//                       n+6|------|    |/      \ 4n+3                    
//              .  .  .     |      | n+3|        \        
//                          |      |   / \      / \                 
//                          |      |  /   \    /   \                 
//                          |      | /     \  /     \              
//                          |      |/       \/       \4n+2     
//                          |------|     n+2/\       /\      
//                         /      2 \      /  \     /  \        
//                        /          \    /    \   /    \             
//                       /            \  /      \ /      \  
//                      /   n-sided   1\/     n+1/        \   --> C^1-surface 
//                      \     face     /\       / \       /4n+1   
//                       \            /  \     /   \     /                 
//                        \          /    \   /     \   /                
//                         \n-1    n/      \ /       \ /                 
//                          \------/    n+3n/         /4n+5n           
//                                  \      / \       /              
//                                   \    /   \     /                 
//                                    \  /     \   /                    
//                                     \/       \ /                 
//                                n+3n-1\        /4n+5n-1             
//                                       \      /            
//                                        \    /                    
//                             .           \  /                      
//                          .               \/                         
//                       .                    4n+5n-2          
//                                                                          
//                                                                           
//              
//              
//                 respectively
//              
//              
//                                         6n+11    6n+10    6n+9             
//                                          ---------------------------6n+8   
//                                         /        /        /        /       
//                                        /        /        /        /        
//                                       /        /        /        /         
//                         .            /2n+8    /2n+7    /2n+6    /          
//                           .         /--------/--------/--------/6n+7       
//                             .      /        /        /        /            
//                                   /        /        /        /             
//                                  /        /        /        /             
//                                 /5       /4       /2n+5    /               
//                      \         /--------/--------/--------/6n+6           
//                       \       /        /        /        /                
//                        \     /        /        /        /                 
//                         \   /        /        /        /                  
//                          \ /1       /3       /2n+4    /                    
//                           ---------------------------- 6n+5  --> C^2-surface
//                          / \         \        \       \                   
//                         /   \         \        \       \                  
//                        /     \         \        \       \                 
//                       /       \2n+1     \ 2      \2n+3   \                
//                      /         \---------\--------\-------\6n+4           
//                                 \         \        \       \              
//                                  \         \        \       \             
//                                   \         \        \       \            
//                               .    \2n+4n    2n+4n+1  \2n+2   \           
//                             .       \---------\--------\-------\6n+3      
//                           .          \         \        \       \         
//                                       \         \        \       \        
//                                        \         \        \       \       
//                                         \         \        \       \      
//                                          \--------------------------6n+2  
//                                        6n+6n-1    6n+6n   6n+6n+1          
//              
//              
//              
//              
//                 Each subnet of size 3x3 (C^1-case) resp. 4x4 (C^2-case) 
//                 points is interpreted as the control net of a biquadratic 
//                 resp. bicubic tensorproduct B-spline surface. The 
//                 resulting surface ring surrounds an n-sided hole. This 
//                 program fills this surface ring.
//              
//                 As result (OUTVEC-FILE), you get all control points of the 
//                 patches which fill the hole plus the control points of the 
//                 patches forming the surface ring. The patches described 
//                 by INVEC-FILE are degree elevated and, in case method 0 
//                 is used modified. In case method 1,2 or 3 is used the control
//                 points for the surface ring describe the same surface as 
//                 the INVEC-FILE.
//
//                 Each block of 25 (C^1-case, upper figure above) resp.
//                 49 points (C^2-case, lower figure above) describes a 
//                 biquartic resp. bisextic uniform tensorproduct B-spline 
//                 surface patch. Points 1 to 25 resp. 49 form patch one, 
//                 points 26 resp. 50 to 50 resp. 98 patch two etc. 
//              
//                 The parameter METHOD describes which method shall be 
//                 used to fill the n-sided hole. Possible values are 0, 1,
//                 2 or 3.
//                     0 : The surface ring surrounding the n-sided hole 
//                         is filled with n patches. The surface ring 
//                         described by INVEC-FILE is modified.
//                     1 : The surface ring surrounding the n-sided hole is 
//                         filled with 3n patches. The surface described by 
//                         INVEC-FILE is not modified. 
//                     2 : same as 1, but with different sizes of the 
//                         fill-patches
//                     3 : The surface ring surrounding the n-sided hole
//                         is filled with n patches. The surface ring
//                         described by INVEC-FILE is not modified.
//
//                 The following table shows the bidegree of the patches:    
//
//
//                                  |     method    |
//                                  +-------+-------+
//                                  | 0,1,2 |   3   |
//                 -----------+-----+-------+-------+
//                            | C^1 |  4,4  |  5,5  |
//                 continuity +-----+-------+-------+
//                            | C^2 |  6,6  | 13,13 |
//                 -----------+-----+-------+-------+
//
//                 For further details refer to the documentation provided in 
//                        user_manual.ps.
//
//                 Compile the program with  
//                        compile fill_hole
//                 or
//                        g++ -o fill_hole fill_hole.c+
//                 
//   Examples:     An example is described in the user manual.
//
//                 Or take a look at our research-page:
//                        http://i33www.ira.uka.de
//
//   Contact:      Please let use hear from you in case you have any 
//                 suggestions or problems with our software:
//                        prau@ira.uka.de
//                        umlauf@ira.uka.de
//
//


// --------------------------------------------------------------------------
//
//                  INCLUDES
//
// --------------------------------------------------------------------------

#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include <fstream.h>
#include <strstream.h>
#include <stdio.h>
#include "class_matrix.c++"


// --------------------------------------------------------------------------
//
//                  GLOBAL VARIABLES
//
// --------------------------------------------------------------------------

int DUMP = 1; // =0: Program puts out only error messages.
              // =1: Put out some other messages, too.
              // =2: Put out more messages.

const int MAX_EDGENUMBER = 8; // n-sided holes with n>MAX_EDGENUMBER 
                              // can't be filled.

const int MAX_NUMBER_OF_INVEC_POINTS = 12*MAX_EDGENUMBER+1; // Maximum number
                                                            // of points in INVEC-FILE. 


// --------------------------------------------------------------------------
//
//                  CLASS DECLARATIONS
//
// --------------------------------------------------------------------------

// A simple scene class:
class scene_class
{
private:

  int number_of_invec_points;  // Number of read points in INVEC-file.
  Matrix invec_points[3];      // for INVEC-FILE
  Matrix outvec_points[3];     // for OUTVEC-FILE

 
  // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Private Member Function: 
  //

  // Reads matrix A from disk, then: OUTVEC = A * INVEC (coordinate-wise).
  Matrix read_matrix(int, int, int);  

public:

  // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Public Member Functions: 
  //

  // Reads INVEC-FILE from disk.
  void read_invecfile(char *);       
  
  // Calculates fill patches.
  void calculate_fillpatches(int);   

  // Writes OUTEC-FILE to disk.
  void write_outvecfile(char *);   
};





// --------------------------------------------------------------------------
//
//                  PRIVATE MEMBER FUNCTION DEFINITION
//
// --------------------------------------------------------------------------

// Reads matrix A for multiplication with the 3 vectors in INVEC-FILE.
Matrix scene_class::read_matrix(int number_of_edges, 
				int order_of_continuity,
                                int method)
{
  int m,   // number of rows 
      n,   // number of columns 
      deg; // degree of the patches 

  if (method < 3)
    {
      m = (method == 0 ? 4 : 7)*number_of_edges*
	     (order_of_continuity == 1 ? 25 : 49);
      n = (order_of_continuity == 1
	      ? 9*number_of_edges : 12*number_of_edges+1);
      deg = 2 * order_of_continuity + 2;
    }

  if (method > 2)
    {
      m =     4  * number_of_edges * (order_of_continuity == 1 ? 36 : 196);
      n = (order_of_continuity == 1
	      ? 9*number_of_edges : 12*number_of_edges+1);
      deg = (order_of_continuity == 1 ? 5 : 13);
    }

  int pl = (deg+1) * (deg+1);   /* number of coefficients/patch */

  Matrix A(m,n);
  
  // Determine the name of the matrix.
  char name_matrix[40];
  sprintf(name_matrix, "matrices/c%imt%i_%i.mat",
	  order_of_continuity, method, number_of_edges);

  // Read matrix A from disk.

  ifstream file(name_matrix);
  
  if (!file){
    cout << "Can't open matrix " << name_matrix << "." << endl;
    exit(1);
  }

  Matrix s;
  cerr << "Assembling the \"filling\"-matrix...\n";



  /*************************************
   *                                   *
   *  Assembly of the filling matrix.  *
   *                                   *
   *************************************/

  /*
    In case of (order_of_continuity == 1), there are 3 rings of invec-points.
   */

  if (order_of_continuity == 1)
    {
      /* method 0,1,2,3 */
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+j*pl)*A.rows+jj+((i+j)%number_of_edges)] = s.el[ii*s.rows+jj];
	//	    A.putMatrix(j*pl, (i+j)%number_of_edges, s);
	}
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+j*pl)*A.rows+jj+(number_of_edges+((i+j)%number_of_edges)*3)] = 
		s.el[ii*s.rows+jj];
	// A.putMatrix(j*pl, number_of_edges+((i+j)%number_of_edges)*3, s);
      }
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+j*pl)*A.rows+jj+(4*number_of_edges+ ((i+j)%number_of_edges)*5)] = 
		s.el[ii*s.rows+jj];
	//A.putMatrix(j*pl, 4*number_of_edges+((i+j)%number_of_edges)*5, s);
      }
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+(number_of_edges+3*j)*pl)*A.rows+jj+((i+j)%number_of_edges)] = 
		s.el[ii*s.rows+jj];
	//	  A.putMatrix((number_of_edges+3*j)*pl, (i+j)%number_of_edges, s);
      }
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+(number_of_edges+3*j)*pl)*A.rows+jj+(number_of_edges+((i+j)%number_of_edges)*3)] = 
		s.el[ii*s.rows+jj];
	//A.putMatrix((number_of_edges+3*j)*pl, number_of_edges+((i+j)%number_of_edges)*3, s);
      }
      for (int i = 0; i < number_of_edges; i++) {
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+(number_of_edges+3*j)*pl)*A.rows+jj+(4*number_of_edges +((i+j)%number_of_edges)*5)] = 
		s.el[ii*s.rows+jj];
	// A.putMatrix((number_of_edges+3*j)*pl, 4*number_of_edges+((i+j)%number_of_edges)*5, s);
      }
      /* method 1 and 2 */
      if ((method == 1) || (method == 2))
	{
	  for (int i = 0; i < number_of_edges; i++) {
	    file >> s;
	    for (int j = 0; j < number_of_edges; j++)
	      for (int ii = 0; ii < s.lines; ii++)
		for (int jj = 0; jj < s.rows; jj++)
		  A.el[(ii+(4*number_of_edges+3*j)*pl)*A.rows+jj+((i+j)%number_of_edges)] = 
		    s.el[ii*s.rows+jj];
	    //A.putMatrix((4*number_of_edges+3*j)*pl, (i+j)%number_of_edges, s);
	  }
	  for (int i = 0; i < number_of_edges; i++) {
	    file >> s;
	    for (int j = 0; j < number_of_edges; j++)
	      for (int ii = 0; ii < s.lines; ii++)
		for (int jj = 0; jj < s.rows; jj++)
		  A.el[(ii+(4*number_of_edges+3*j)*pl)*A.rows+jj+(number_of_edges+((i+j)%number_of_edges)*3)] = 
		    s.el[ii*s.rows+jj];
	    //A.putMatrix((4*number_of_edges+3*j)*pl, number_of_edges+((i+j)%number_of_edges)*3, s);
	  }
	  for (int i = 0; i < number_of_edges; i++) {
	    file >> s;
	    for (int j = 0; j < number_of_edges; j++)
	      for (int ii = 0; ii < s.lines; ii++)
		for (int jj = 0; jj < s.rows; jj++)
		  A.el[(ii+(4*number_of_edges+3*j)*pl)*A.rows+jj+(4*number_of_edges+((i+j)%number_of_edges)*5)] = 
		    s.el[ii*s.rows+jj];
	    //A.putMatrix((4*number_of_edges+3*j)*pl,
	    //    4*number_of_edges+((i+j)%number_of_edges)*5, s);
	  }
	}
    }
  
  /*
    In case of (order_of_continuity == 2) there are 4 rings of invec-points.
    */
  
  if (order_of_continuity == 2)
    {
      /* method 0,1,2,3 */
      file >> s;
      for (int j = 0; j < number_of_edges; j++)
	for (int ii = 0; ii < s.lines; ii++)
	  for (int jj = 0; jj < s.rows; jj++)
	    A.el[(ii+j*pl)*A.rows+jj] = 
	      s.el[ii*s.rows+jj];
      //	A.putMatrix(j*pl ,0 , s);
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+j*pl)*A.rows+jj+1+((i+j)%number_of_edges)*2] = 
		s.el[ii*s.rows+jj];
	//	    A.putMatrix(j*pl, 1+((i+j)%number_of_edges)*2, s);
      }
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+j*pl)*A.rows+jj+1+2*number_of_edges+((i+j)%number_of_edges)*4] = 
		s.el[ii*s.rows+jj];
	//	    A.putMatrix(j*pl, 1+2*number_of_edges+((i+j)%number_of_edges)*4, s);
      }
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+j*pl)*A.rows+jj+1+6*number_of_edges+((i+j)%number_of_edges)*6] = 
		s.el[ii*s.rows+jj];
	//	    A.putMatrix(j*pl, 1+2*number_of_edges+4*number_of_edges+((i+j)%number_of_edges)*6, s);
      }
      file >> s;
      for (int j = 0; j < number_of_edges; j++)
	for (int ii = 0; ii < s.lines; ii++)
	  for (int jj = 0; jj < s.rows; jj++)
	    A.el[(ii+(number_of_edges+3*j)*pl)*A.rows+jj] = 
	      s.el[ii*s.rows+jj];
      //	A.putMatrix((number_of_edges+3*j)*pl ,0, s);
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+(number_of_edges+3*j)*pl)*A.rows+jj+1+((i+j)%number_of_edges)*2] = 
		s.el[ii*s.rows+jj];
	//	    A.putMatrix((number_of_edges+3*j)*pl, 1+((i+j)%number_of_edges)*2, s);
      }
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+(number_of_edges+3*j)*pl)*A.rows+jj+1+2*number_of_edges+((i+j)%number_of_edges)*4] = 
		s.el[ii*s.rows+jj];
	//	    A.putMatrix((number_of_edges+3*j)*pl, 1+2*number_of_edges+((i+j)%number_of_edges)*4, s);
      }
      for (int i = 0; i < number_of_edges; i++)	{
	file >> s;
	for (int j = 0; j < number_of_edges; j++)
	  for (int ii = 0; ii < s.lines; ii++)
	    for (int jj = 0; jj < s.rows; jj++)
	      A.el[(ii+(number_of_edges+3*j)*pl)*A.rows+jj+1+6*number_of_edges+((i+j)%number_of_edges)*6] = 
		s.el[ii*s.rows+jj];
	//	    A.putMatrix((number_of_edges+3*j)*pl,
	//	1+2*number_of_edges+4*number_of_edges+((i+j)%number_of_edges)*6, s);
      }
      /* method 1 and 2 */
      if ((method == 1) || (method == 2))
	{
	  file >> s;
	  for (int j = 0; j < number_of_edges; j++)
	    A.putMatrix((4*number_of_edges+3*j)*pl ,0, s);
	  for (int i = 0; i < number_of_edges; i++) {
	    file >> s;
	    for (int j = 0; j < number_of_edges; j++)
	      for (int ii = 0; ii < s.lines; ii++)
		for (int jj = 0; jj < s.rows; jj++)
		  A.el[(ii+(4*number_of_edges+ 3*j)*pl)*A.rows+jj+1+((i+j)%number_of_edges)*2] = 
		    s.el[ii*s.rows+jj];
	    //		A.putMatrix((4*number_of_edges+3*j)*pl, 1+((i+j)%number_of_edges)*2, s);
	  }
	  for (int i = 0; i < number_of_edges; i++) {
	    file >> s;
	    for (int j = 0; j < number_of_edges; j++)
	      for (int ii = 0; ii < s.lines; ii++)
		for (int jj = 0; jj < s.rows; jj++)
		  A.el[(ii+(4*number_of_edges+3*j)*pl)*A.rows+jj+1+2*number_of_edges+((i+j)%number_of_edges)*4] = 
		    s.el[ii*s.rows+jj];
	    //		A.putMatrix((4*number_of_edges+3*j)*pl,
	    //    1+2*number_of_edges+((i+j)%number_of_edges)*4, s);
	  }
	  for (int i = 0; i < number_of_edges; i++) {
	    file >> s;
	    for (int j = 0; j < number_of_edges; j++)
	      for (int ii = 0; ii < s.lines; ii++)
		for (int jj = 0; jj < s.rows; jj++)
		  A.el[(ii+(4*number_of_edges+3*j)*pl)*A.rows+jj+1+6*number_of_edges+((i+j)%number_of_edges)*6] = 
		    s.el[ii*s.rows+jj];
	    //		A.putMatrix((4*number_of_edges+3*j)*pl,
	    //    1+2*number_of_edges+4*number_of_edges+((i+j)%number_of_edges)*6, s);
	  }
	}
    }
  
  return A;
}


// --------------------------------------------------------------------------
//
//                  PUBLIC MEMBER FUNCTION DEFINITIONS
//
// --------------------------------------------------------------------------

// Read INVEC-file from disk.
void scene_class::read_invecfile(char *name_invec)
{
  int coordinate;
  
  Matrix help(MAX_NUMBER_OF_INVEC_POINTS);  // 0-vector of maximal size.
  for (coordinate=0; coordinate<=2; coordinate++)
    invec_points[coordinate] = help;       // Set vector size to maximum.
  
  char buffer[120];
  
  ifstream file(name_invec);
  if (!file){
    cout << "Error: Can't open INVEC-FILE " << name_invec << "." << endl;
    exit(1);
  }
  
  istrstream *read;
  
  number_of_invec_points = 0;
  while (file.good()){
    buffer[0] = '\0';
    file.getline(buffer, sizeof(buffer));
    if (buffer[0] == '\0')   // Overread empty lines.
      continue;
    
    read = new istrstream(buffer);
    *read >> invec_points[0](number_of_invec_points)
	  >> invec_points[1](number_of_invec_points)
	  >> invec_points[2](number_of_invec_points);
    
    number_of_invec_points++;
    if (number_of_invec_points > MAX_NUMBER_OF_INVEC_POINTS){
      cout << "The INVEC-FILE has too many points, possibly it\n"
	   << "describes an n-sided hole with n>" << MAX_EDGENUMBER
	   << "." << endl;
      exit(1);
    }
    
    delete read;
  }
  
  // Give the invec_points-vectors the right size.
  for (coordinate=0; coordinate<=2; coordinate++)
    invec_points[coordinate].cut_vector(number_of_invec_points);
}


// Calculates patch points (OUTVEC) from the given control points (INVEC).
// The 3 rows of the INVEC-FILE are interpretated as vectors.
// Each of these vectors is multiplicated with A  ->  row of OUTVEC-FILE.
// I.e. OUTVEC = A * INVEC  coordinate-wise.
void scene_class::calculate_fillpatches(int method)
{
  Matrix A;
  int order_of_continuity,  // 1 or 2 for C^1-case resp. C^2-case.
      number_of_edges;      // =n if there is an n-sided hole.
  int coordinate;
  
  if (number_of_invec_points%9 == 0){
    // The control net around an n-sided face has 9*n control points
    // --> C^1-case.
    order_of_continuity = 1;
    number_of_edges = number_of_invec_points/9;  // = n
    
  } else if ((number_of_invec_points-1)%12 == 0){
    // The control net around an extraordinary point of order n has
    // 12*n+1 control points
    // --> C^2-case.
    order_of_continuity = 2;
    number_of_edges = (number_of_invec_points-1)/12;  // = n
    
  } else {
    cout << "Number of control points in the INVEC-FILE is wrong: = "
	 << number_of_invec_points << "." << endl;
    cout << "The number should be 27, 45, 54, 63 or 72 (C^1-case) resp. "
	 << "\n37, 61, 73, 85 or 97 (C^2-case)." << endl;
    exit(1);
  }
  
  
  if (DUMP >= 1)
    cout << "Filling C^" << order_of_continuity
	 << "-continuous surface ring with a"
	 << (number_of_edges == 8 ? "n " : " ") << number_of_edges
	 << "-sided hole." << endl;
  

  // Read matrix for multiplication.
  A = read_matrix(number_of_edges, order_of_continuity, method);
  
  if (DUMP >= 2)
    cout << "Size of matrix: " << A.number_of_lines() << " x "
	 << A.number_of_rows() << endl;

  cerr << "Calculating the fill patches...\n";
  // Calculate the fill patches by matrix multiplication.
  for (coordinate=0; coordinate<=2; coordinate++)
    outvec_points[coordinate] = A * invec_points[coordinate];
}

// Writes result to file OUTVEC-FILE.
void scene_class::write_outvecfile(char *name_outvec)
{
  int k;  // Counter for the outvec-points.
  
  ofstream file(name_outvec, ios::out, 0644);
  if (!file){
    cout << "Error: Can't open file " << name_outvec << "." << endl;
    exit(1);
  }
  
  cerr << "Writing result to file...\n";
  for (k=0; k<=outvec_points[0].number_of_lines()-1; k++){
    file << outvec_points[0](k) << " "
	 << outvec_points[1](k) << " "
	 << outvec_points[2](k);
    if (k != outvec_points[0].number_of_lines()-1)
      file << endl;
  }
}


// ==========================================================================
//                       M    M    AA    III  N    N
//                       MM  MM   A  A    I   NN   N
//                       M MM M  A    A   I   N N  N
//                       M    M  AAAAAA   I   N  N N
//                       M    M  A    A   I   N   NN
//                       M    M  A    A  III  N    N     
// ==========================================================================

// Fills hole described by INVEC-FILE. Result: OUTVEC-FILE.
int main(int argc, char **argv)
{   
  if (argc != 4 || argv[3][0] < '0' || argv[3][0] > '4')
    {
      cout << "\nUsage:  " << argv[0] << " INVEC-FILE OUTVEC-FILE METHOD\n"
	   << endl;
      cout << "Reads the control net in file INVEC-FILE and fills the\n"
	   << "corresponding surface ring with method METHOD. Possible \n"
	   << "values for METHOD are 0, 1 or 2:\n"
	   << "\t0/3:\tfill with n patches, \n"
	   << "\t1/2:\tfill with 4n patches of same/different size.\n"
	   << "The result is written to file OUTVEC-FILE.\n\n"
	   << "For details refer to the documentation in user_manual.ps.\n"
	   << endl;
      
    } 
  else 
    {
      scene_class scene;
      
      // Read the file INVEC-FILE.
      scene.read_invecfile(argv[1]);
      // Determine surface ring and fill n-sided hole.
      scene.calculate_fillpatches(argv[3][0] - '0');
      // Write result to file OFF-FILE.
      scene.write_outvecfile(argv[2]);
    }
  
  return 0;
}











