// ==========================================================================
//
//      FFFFF  RRRR   EEEEE  EEEEE  FFFFF   OOOO   RRRR   M    M 
//      F      R   R  E      E      F      O    O  R   R  MM  MM 
//      FFF    RRRR   EEE    EEE    FFF    O    O  RRRR   M MM M 
//      F      R  R   E      E      F      O    O  R  R   M    M 
//      F      R   R  E      E      F      O    O  R   R  M    M 
//      F      R   R  EEEEE  EEEEE  F       OOOO   R   R  M    M 
//
//
//           SSSSS  PPPPP   L      III  N    N  EEEEE   SSSSS    
//          S       P    P  L       I   NN   N  E      S
//           SSSS   PPPP    L       I   N N  N  EEE     SSSS
//               S  P       L       I   N  N N  E           S
//               S  P       L       I   N   NN  E           S
//          SSSSS   P       LLLLL  III  N    N  EEEEE  SSSSS 
//
// ==========================================================================
//  
//   Project:      "Freeform splines software package" of 
//
//                     University of Karlsruhe (TH)
//                     Institute for Operating and Dialog Systems
//                     Research Group on Computer Aided Geometric Design
//                     Head of research group: Prof. Dr. H. Prautzsch
//                     WWW: http://i33www.ira.uka.de
//
//
//   Class name:   Martix
//
//   Parents:      --
//
//   Version:      1.1
//
//   Author:       Thomas Mueller, Stephan Bischoff
//  
//   Date:         July, 1997
//
//   Changes:      September 1999
//                 - Added the streaming operators "<<" and ">>". A matrix
//                   is stored in the following format:
//                     number_of_rows
//                     number_of columns
//                     (number_of_rows * number_of_columns) numbers
//                 - Added the method "putMatrix".
//
//   Description:  Implements a basic matrix class with matrix multiplication,
//                 assignment, indexing operations and a cutting function.
//                 (z,1) matrices are understood as vectors. 
//                
//
//   Contact:      Please let use hear from you in case you have any 
//                 suggestions or problems with our software:
//                        prau@ira.uka.de
//                        umlauf@ira.uka.de
//


// --------------------------------------------------------------------------
//
//                  CLASS DECLARATION
//
// --------------------------------------------------------------------------
class Matrix
{
public:

  int lines;   // number of lines
  int rows;    // number of rows
  double *el;  // elements


  // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Constructors:
  //
  
  // Construct (z,s) 0-matrix. A (z,1) matrix is understood as vector.
  Matrix(int z, int s = 1) : lines(z), rows(s)
  {  
    int i;
    
    el = new double[z*s];       
    for (i=0; i<=z*s-1; i++)   
      el[i] = 0;
  }

  // Construct (1,1) 0-matrix.
  Matrix() : lines(1), rows(1)
  {
    el = new double[1];       
    el[0] = 0;                
  }
  
  // Copy-Constructor.  
  Matrix(const Matrix &m) : lines(m.lines), rows(m.rows)
  {
    int i;
    
    el = new double[m.lines*m.rows];
    for (i=0; i<=m.lines*m.rows-1; i++)
      el[i] = m.el[i];
  }


  // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Destruktor:
  //
  ~Matrix()
  {
    delete el;
  }


  // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Operators:
  //

  // Matrix-Product-Operator: matrix * matrix.  
  friend Matrix operator*(const Matrix &, const Matrix &);  

  // Assignment-Operator: matrix = matrix. 
  Matrix & operator=(const Matrix &);                       

  // Indexing-Operator: (i,j)-th element. 
  double & operator()(int, int) const;

  // Indexing-Operator: i-th element of (z,1) matrix.
  double & operator()(int);                        

  // Streaming-Operators
  friend ostream& operator << (ostream& os, const Matrix& a);
  friend istream& operator >> (istream& is, Matrix& a);

  // ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Member Functions: 
  //

  // Extracts first k elements of (z,1) matrix.
  void cut_vector(int k);  
  
  // Returns number of lines.
  int number_of_lines() const {return lines;};
  
  // Returns number of rows.
  int number_of_rows() const {return rows;};  

  // Puts matrix on a the specified position
  Matrix putMatrix(int z, int s, const Matrix& a);
};



// --------------------------------------------------------------------------
//
//                  OPERATOR DEFINITIONS
//
// --------------------------------------------------------------------------

// Matrix-Product-Operator: matrix * matrix.  
Matrix operator*(const Matrix &m, const Matrix &n)
{   
  if (m.rows != n.lines){
    cout << "Matrices cannot be multiplied." << endl;
    exit(1);
  }
  
  int i, j, k;
  Matrix result(m.lines, n.rows);
  
  for (i=0; i<=result.lines-1; i++)
    for (j=0; j<=result.rows-1; j++){
      result.el[i*result.rows+j] = 0;
      for (k=0; k<=m.rows-1; k++)
	result.el[i*result.rows+j] += m.el[i*m.rows+k] * n.el[k*n.rows+j];
    }
  
  return result;
}

// Assignment-Operator: matrix = matrix. 
Matrix & Matrix::operator=(const Matrix &m)
{
  int i;
  
  if (lines != m.lines || rows != m.rows){
    lines = m.lines;
    rows = m.rows;
    delete el;
    el = new double[lines * rows];
  }
  
  for (i=0; i<=lines*rows-1; i++)
    el[i] = m.el[i];
  
  return *this;
}

// Indexing-Operator: (i,j)-th element. 
inline double & Matrix::operator()(int i, int j) const
{
  if (i<0 || i>lines-1 || j<0 || j>rows-1){
    cout << "Matrix operator (): wrong line or row." << endl;
    exit(1);
  }

  return el[i*rows + j];
} 

// Indexing-Operator: i-th element of (z,1) matrix.
inline double & Matrix::operator()(int i)
{  
  if (i<0 || i>lines-1 || rows != 1){
    cout << "Matrix operator (): not a vector or wrong linenumber." << endl;
    exit(1);
  }   
  
  return el[i];
}

// Streaming-Operators
ostream& operator << (ostream& os, const Matrix& a)
{
  double t;
  os << a.lines << "\n" << a.rows << "\n";
  for (int i = 0; i < a.lines; i++)
    {
      for (int j = 0; j < a.rows-1; j++)
	{
	  t = a(i,j);
	  os.form("%2.5f ", t);
	}
      os.form("%2.5f\n", a(i,a.rows-1));
    }
}

istream& operator >> (istream& is, Matrix& a)
{
  is >> a.lines;
  is >> a.rows;
  
  a.el = new double[a.lines*a.rows];

  for (int i = 0; i < a.lines; i++)
    for (int j = 0; j < a.rows; j++)
      is >> a(i,j);
}

// --------------------------------------------------------------------------
//
//                  MEMBER FUNCTION DEFINITION
//
// --------------------------------------------------------------------------

// Extracts first k elements of (z,1) matrix.
void Matrix::cut_vector(int k)
{
  if (k > lines){
    cout << "cut_vector: impossible to cut to bigger vector." << endl;
    exit(1);
  }
  
  lines = k;
}

// Puts matrix a on the position (z,s)
Matrix Matrix::putMatrix(int z, int s, const Matrix& a)
{
  for (int i = 0; i < a.number_of_lines(); i++)
    for (int j = 0; j < a.number_of_rows(); j++)
      (*this)(z + i, s + j) = a(i,j);
  return *this;
}






