/* -*- Mode: C; c-basic-offset:4 ; -*- */
/*  
 *  (C) 2003 by Argonne National Laboratory.
 *      See COPYRIGHT in top-level directory.
 */

#ifndef LWLOG_H_INCLUDED
#define LWLOG_H_INCLUDED

#include "lwlogconf.h"
#include "lwnames.h"

/* Set the number of states */
#ifndef MPIU_MAX_STATE_NAMES
#define MPIU_MAX_STATE_NAMES 1024
#endif

/* 
 * This implements a simple mechanism for logging states.  See lwlog.c for
 * details.  The structures are
 *     state records
 *     output buffer
 * If USE_LWLOG_NEST is defined, we also keep track of the nesting
 * level of the state at the moment that the state *finishes*.
 */
typedef struct {
    MPID_Time_t start, end;
    int         state;
#ifdef USE_LWLOG_NEST
    int         nest;
#endif    
} LWLog_state_t;

typedef struct {
    LWLog_state_t *state_array;   /* Array of states */
    int           cur;            /* Index of next free state */
    int           nrec;           /* Number of states in array */
    FILE          *fp;            /* Pointer to file for writing out data */
} LWLog_t;

extern LWLog_t *MPIU_LWlog;
extern int MPIU_LWlog_flag;
#ifdef USE_LWLOG_NEST
extern int MPIU_LWlog_nest;
#endif

extern void MPIU_LWLog_write( void );
extern void MPIU_LWLog_init( int );
extern void MPIU_LWLog_finalize( void );
extern void MPIU_LWLog_set_flag( int );

#define MPIDU_STATE_DECL(a) MPID_Time_t time_stamp_in##a

#ifdef USE_LWLOG_NEST
#define MPIU_LWLOG_NEST(cs) cs->nest = MPIU_LWlog_nest--;
#define MPIU_LWLOG_INCR_NEST MPIU_LWlog_nest++;
#else
#define MPIU_LWLOG_NEST(cs)
#define MPIU_LWLOG_INCR_NEST
#endif

#define MPIDU_FUNC_ENTER(a) {                                    \
    if (MPIU_LWlog_flag) {                                       \
        MPID_Wtime( &time_stamp_in##a );                         \
        MPIU_LWLOG_INCR_NEST                                     \
    }                                                            \
}

/* 
   To make this thread safe, use a fetch and increment on the 
   cur pointer and combine this with a test against nrec.  
 */
#define MPIDU_FUNC_EXIT(a) {                                     \
    if (MPIU_LWlog_flag) {                                       \
        register LWLog_state_t  *cs;                             \
	if (MPIU_LWlog->cur == MPIU_LWlog->nrec) {               \
	    MPIU_LWLog_write();                                  \
	}                                                        \
    cs = MPIU_LWlog->state_array + MPIU_LWlog->cur++;            \
    cs->start = time_stamp_in##a;                                \
    MPID_Wtime( &cs->end );                                      \
    MPIU_LWLOG_NEST(cs)                                          \
    cs->state = a;                                               \
    }                                                            \
}

#define MPIDU_PT2PT_FUNC_ENTER(a)     MPIDU_FUNC_ENTER(a)
#define MPIDU_PT2PT_FUNC_EXIT(a)      MPIDU_FUNC_EXIT(a)
#define MPIDU_COLL_FUNC_ENTER(a)      MPIDU_FUNC_ENTER(a)
#define MPIDU_COLL_FUNC_EXIT(a)       MPIDU_FUNC_EXIT(a)
#define MPIDU_RMA_FUNC_ENTER(a)       MPIDU_FUNC_ENTER(a)
#define MPIDU_RMA_FUNC_EXIT(a)        MPIDU_FUNC_EXIT(a)
#define MPIDU_INIT_FUNC_ENTER(a)     
#define MPIDU_INIT_FUNC_EXIT(a)      
#define MPIDU_FINALIZE_FUNC_ENTER(a) 
#define MPIDU_FINALIZE_FUNC_EXIT(a)  

#define MPIDU_PT2PT_FUNC_ENTER_FRONT(a)
#define MPIDU_PT2PT_FUNC_ENTER_BACK(a)
#define MPIDU_PT2PT_FUNC_EXIT_FRONT(a)
#define MPIDU_PT2PT_FUNC_EXIT_BACK(a)
#define MPIDU_PT2PT_FUNC_ENTER_BOTH(a)
#define MPIDU_PT2PT_FUNC_EXIT_BOTH(a)

#define MPIDU_INIT_STATE_DECL(a)
#define MPIDU_FINALIZE_STATE_DECL(a)

#endif
