/* -*- Mode: C; c-basic-offset:4 ; -*- */
/*  
 *  (C) 2003 by Argonne National Laboratory.
 *      See COPYRIGHT in top-level directory.
 */
#include "mpiimpl.h"
#include <stdio.h>
#include <stdlib.h>

#include "lwlog.h"

/* style: allow:fprintf:3 sig:0 */   /* For logging file output only */

/*
  Write out the contents of the log buffer to the open file attached to the
  lwlog buffer 
*/

/* 
 * offset is an offset applied to times on output to remove a constant
 * offset.
 */
static double offset = 0; 
/*
 * MPIU_States is an array that maps state numbers onto string names for the
 * states, usually the functions.
 */

char *(MPIU_States[MPIU_MAX_STATE_NAMES]);
#ifdef USE_LWLOG_NEST
int MPIU_LWlog_nest = 0;
#endif

void MPIU_LWLog_write( void )
{
    LWLog_state_t *cs;
    char          *name=0;
    int           i;

    if (!MPIU_LWlog || !MPIU_LWlog->fp) {
	/* this should not happen; this routine should never be called
	 if there is no MPIU_LWlog structure or no file */
	return;
    }
    cs = MPIU_LWlog->state_array;
    for (i=0; i<MPIU_LWlog->cur; i++) {
	double t1, t2;
	/* Convert time stamps to doubles and write out the 
	   state */
	MPID_Wtime_todouble( &cs->start, &t1 );
	MPID_Wtime_todouble( &cs->end, &t2 );
	t1 -= offset;
	t2 -= offset;
	
	/* Get the state name, use blank if none */
	if (cs->state < 0 || cs->state >MPIU_MAX_STATE_NAMES) 
	    name = " ";
	else
	    name = MPIU_States[cs->state];
	if (!name || !name[0]) { name = " "; }
	/* This version outputs the difference, just to make it easier to
	   interpret the results by looking at the text.  It
           also adds the name of the */
#ifdef USE_LWLOG_NEST
	fprintf( MPIU_LWlog->fp, "%d\t%d\t%20.12g\t%20.12g\t%20.12g\t%s\n", 
		 cs->state, cs->next, t1, t2, t2-t1, name );
#else
	fprintf( MPIU_LWlog->fp, "%d\t%20.12g\t%20.12g\t%20.12g\t%s\n", 
		 cs->state, t1, t2, t2-t1, name );
#endif
	cs++;
    }

    fflush( MPIU_LWlog->fp );
    /* Reset the buffer */
    MPIU_LWlog->cur = 0;
}

/* Global variables (and their storage) */
static LWLog_t LWLogHeader = {0,0,0,0};
LWLog_t *MPIU_LWlog = &LWLogHeader;
int      MPIU_LWlog_flag = 0;

#ifndef MAXPATHLEN 
#define MAXPATHLEN 1024
#endif
#ifndef MAXLWLOG
/* 1M records by default */
#define MAXLWLOG 1024*1024
#endif

void MPIU_LWLog_init( int rank )
{
    char logfile[MAXPATHLEN+1];

    MPIU_Snprintf( logfile, MAXPATHLEN, "log%d.lwlog", rank );
    MPIU_LWlog->fp = fopen( logfile, "w" );
    if (MPIU_LWlog->fp) {
	MPIU_LWlog->state_array = 
	    (LWLog_state_t *)MPIU_Malloc( MAXLWLOG * sizeof(LWLog_state_t) );
	MPIU_LWlog->nrec = MAXLWLOG;
	MPIU_LWlog_flag  = 1;
    }
    /* Initialize the state names */
    MPIU_Describe_states();

    /* Get an estimate of the timing overhead */
    {
	MPID_Time_t start, cur;
	int          i;
	double       diff;

	MPID_Wtime( &start );
	/* 1000 is pretty arbitrary but should be neither too short nor
	   grossly too long */
	for (i=0; i<1000; i++) {
	    MPID_Wtime( &cur );
	}
	/* Use the last value of cur */
	MPID_Wtime_diff( &start, &cur, &diff );
	
	/* Actual overhead is divided by i */
	diff = diff / (double)i ;

	fprintf( MPIU_LWlog->fp, "#Timer Overhead\t%20.12g\n", diff );
    }
}

void MPIU_LWLog_finalize( void )
{
    /* Flush any remaining data */
    MPIU_LWLog_write( );
    /* Close the file and clear the variables just in case */
    fclose( MPIU_LWlog->fp );
    MPIU_LWlog->fp  = 0;
    MPIU_LWlog_flag = 0;
}

void MPIU_LWLog_set_flag( int flag )
{
    MPIU_LWlog_flag = flag;
}

/* These routines are the ones that are called in initthread.c and finalize.c
   to provide the handling for any logging/timer routines */

/* 
 * Timer init is call at the end of initthread, so MPI is ready.  
 * This allows us to set an origin for the timestamps, which helps
 * make the time stamps easier to read (and maximizes the number of 
 * significant digits in fixed width output).
 */
int MPIU_Timer_init( int rank, int size )
{
    double t1;
    MPID_Time_t timestamp;

    /* Get the timer offset */
    MPID_Wtime( &timestamp );
    MPID_Wtime_todouble( &timestamp, &t1 );
    /* Grrr.  Initthread calls Timer_init before setting the "mpi 
       initialized flag. */  
/*     PMPI_Allreduce( &t1, &offset, 1, MPI_DOUBLE, MPI_MIN, MPI_COMM_WORLD ); */
    offset = t1;

    /* Initialize logging */
    MPIU_LWLog_init( rank );
    return 0;
}

int MPIU_Timer_finalize(void) {
    MPIU_LWLog_finalize();
    return 0;
}
