/* ***** BEGIN LICENSE BLOCK *****
   - Version: MPL 1.1/GPL 2.0/LGPL 2.1
   -
   - The contents of this file are subject to the Mozilla Public License Version
   - 1.1 (the "License"); you may not use this file except in compliance with
   - the License. You may obtain a copy of the License at
   - http://www.mozilla.org/MPL/
   -
   - Software distributed under the License is distributed on an "AS IS" basis,
   - WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
   - for the specific language governing rights and limitations under the
   - License.
   -
   - The Original Code is MozTweak.
   -
   - The Initial Developer of the Original Code is Marek Wawoczny.
   - Portions created by the Initial Developer are Copyright (C) 2002
   - the Initial Developer. All Rights Reserved.
   -
   - Contributor(s):
   - Marek Wawoczny (Marek.Wawoczny@GmbH.rubikon.pl)
   -
   - Alternatively, the contents of this file may be used under the terms of
   - either the GNU General Public License Version 2 or later (the "GPL"), or
   - the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
   - in which case the provisions of the GPL or the LGPL are applicable instead
   - of those above. If you wish to allow use of your version of this file only
   - under the terms of either the GPL or the LGPL, and not to allow others to
   - use your version of this file under the terms of the MPL, indicate your
   - decision by deleting the provisions above and replace them with the notice
   - and other provisions required by the LGPL or the GPL. If you do not delete
   - the provisions above, a recipient may use your version of this file under
   - the terms of any one of the MPL, the GPL or the LGPL.
   -
   - ***** END LICENSE BLOCK ***** */

function MozTweakRDF(aBlocking, aDataSourceFile) {
  this.nsIRDFService = Components.classes["@mozilla.org/rdf/rdf-service;1"].getService();
  this.nsIRDFService = this.nsIRDFService.QueryInterface(Components.interfaces.nsIRDFService);
  this.nsIRDFContainer = Components.classes["@mozilla.org/rdf/container;1"].getService();
  this.nsIRDFContainer = this.nsIRDFContainer.QueryInterface(Components.interfaces.nsIRDFContainer);
  this.nsIRDFContainerUtils = Components.classes["@mozilla.org/rdf/container-utils;1"].getService();
  this.nsIRDFContainerUtils = this.nsIRDFContainerUtils.QueryInterface(Components.interfaces.nsIRDFContainerUtils);

  this.mBlocking = aBlocking;
  this.mDataSource = null;
  this.mDataSourceLoaded = false;
  this.mDataSourceState = 0x0;
  this.mDataSourceFile = aDataSourceFile;
  this.mDataSourceURI = this.getDataSourceURI();
  this.mNamespace  = "http://MozillaPL.org/rdf#";
  this.mSinkObserverRegistered = false;

  this.loadDataSource();
}

MozTweakRDF.prototype = { 
  addDataSource: function(aDataElement) {
    if (this.mDataSourceLoaded) {
      if (aDataElement && aDataElement.database) {
        aDataElement.database.AddDataSource(this.mDataSource);
        aDataElement.builder.rebuild();
      }
    }
    else {
      setTimeout("gMozTweakSignaturesRDF.addDataSource('" + aDataElement + "')", 125);
    }
  },

  removeDataSource: function(aDataElement) {
    if (this.mDataSourceLoaded) {
      if (aDataElement) {
        aDataElement.database.RemoveDataSource(this.mDataSource);
        aDataElement.builder.rebuild();
      }
    }
    else {
      setTimeout("gMozTweakSignaturesRDF.removeDataSource('" + aDataElement + "')", 125);
    }
  },

  loadDataSource: function() {
    if (this.mDataSource != null && this.mDataSourceLoaded) {
      return;
    }

    if (this.mSinkObserverRegistered) {
      return;
    }

    if (this.mBlocking) {
      this.mDataSource = this.nsIRDFService.GetDataSourceBlocking(this.mDataSourceURI);
    }
    else {
      this.mDataSource = this.nsIRDFService.GetDataSource(this.mDataSourceURI);
    }

    var sinkObserver = {
      RDF: this,
      onBeginLoad: function(aSink) { this.RDF.mDataSourceState = (this.RDF.mDataSourceState | 1); },
      onInterrupt: function(aSink) { this.RDF.mDataSourceState = (this.RDF.mDataSourceState | 2); },
      onResume: function(aSink) { this.RDF.mDataSourceState = (this.RDF.mDataSourceState & ~2); },
      
      onError: function(aSink, aStatus, aErrorMsg) { 
        this.RDF.mDataSourceState = (this.RDF.mDataSourceState | 8);
        alert(aStatus + ": " + aErrorMsg);
      },

      onEndLoad: function(aSink) { 
        this.RDF.mDataSourceState = (this.RDF.mDataSourceState | 4);
        this.RDF.mDataSourceLoaded = (this.RDF.mDataSourceState == 5);
      }
    }

    if (this.mDataSource.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).loaded) {
      this.mDataSourceLoaded = (this.mDataSource != null);
    }
    else {
      this.mDataSourceLoaded = false;
      this.mDataSourceState = 0x0;
      this.mSinkObserverRegistered = true;
      this.mDataSource.QueryInterface(Components.interfaces.nsIRDFXMLSink).addXMLSinkObserver(sinkObserver);
    }
  },

  refreshDataSource: function() {
    this.mDataSource.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Refresh(false);
  },

  changeItemValue: function(aSource, aProperty, aNewValue) {
    var oldValue = this.mDataSource.GetTarget(this.nsIRDFService.GetResource(aSource), this.nsIRDFService.GetResource(this.mNamespace + aProperty), true);

    if (oldValue) {
      oldValue = oldValue.QueryInterface(Components.interfaces.nsIRDFLiteral);
    }

    if (aNewValue) {
      aNewValue = this.nsIRDFService.GetLiteral(aNewValue);
    }

    if (oldValue && !aNewValue) {
      this.mDataSource.Unassert(this.nsIRDFService.GetResource(aSource), this.nsIRDFService.GetResource(this.mNamespace + aProperty), oldValue);
    }
    else if (!oldValue && aNewValue) {
      this.mDataSource.Assert(this.nsIRDFService.GetResource(aSource), this.nsIRDFService.GetResource(this.mNamespace + aProperty), aNewValue, true);
    }
    else {
      this.mDataSource.Change(this.nsIRDFService.GetResource(aSource), this.nsIRDFService.GetResource(this.mNamespace + aProperty), oldValue, aNewValue);
    }
  },

  getItemValue: function(aSource, aProperty) {
    var value = this.mDataSource.GetTarget(this.nsIRDFService.GetResource(aSource), this.nsIRDFService.GetResource(this.mNamespace + aProperty), true);

    if (value) {
      value = value.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
      return value;
    }

    return null;
  },

  removeNode: function(aSource, aProperty) {
    var removeNode = this.nsIRDFService.GetResource(aProperty);
    var ArcLabelsOut = this.mDataSource.ArcLabelsOut(removeNode);

    while(ArcLabelsOut.hasMoreElements()) {
      var Arc = ArcLabelsOut.getNext();
      var Targets = this.mDataSource.GetTargets(removeNode, Arc, true);

      while (Targets.hasMoreElements()) {
        var Target = Targets.getNext();

        this.mDataSource.Unassert(removeNode, Arc, Target, true);
      }
    }

    this.nsIRDFContainer.Init(this.mDataSource, this.nsIRDFService.GetResource(aSource));
    this.nsIRDFContainer.RemoveElement(this.nsIRDFService.GetResource(aProperty), false);
  },

  removeContainer: function(aSource) {
    this.nsIRDFContainer.Init(this.mDataSource, this.nsIRDFService.GetResource(aSource));

    var Elements = this.nsIRDFContainer.GetElements();
    while(Elements.hasMoreElements()) {
      var Element = Elements.getNext();
      if(this.nsIRDFContainerUtils.IsSeq(this.mDataSource, Element)) {
        this.removeContainer(Element);
        this.nsIRDFContainer.Init(this.mDataSource, this.nsIRDFService.GetResource(aSource));
      }
      var ArcLabelsOut = this.mDataSource.ArcLabelsOut(Element);
      while(ArcLabelsOut.hasMoreElements()) {
        var Arc = ArcLabelsOut.getNext();
        var Targets = this.mDataSource.GetTargets(Element, Arc, true);
        while (Targets.hasMoreElements()) {
          var Target = Targets.getNext();

          this.mDataSource.Unassert(Element, Arc, Target, true);
        }
      }
      this.nsIRDFContainer.RemoveElement(Element, false);
    }

    this.nsIRDFContainer.RemoveElement(this.nsIRDFService.GetResource(aSource), false);
    
    // Begin Hack
    this.mDataSource.Unassert(this.nsIRDFService.GetResource(aSource), this.nsIRDFService.GetResource("http://www.w3.org/1999/02/22-rdf-syntax-ns#instanceOf"), this.nsIRDFService.GetResource("http://www.w3.org/1999/02/22-rdf-syntax-ns#Seq"));
    // End Hack  
  },

  addNode: function(aSource, aElementArray) {
    var newNode = this.nsIRDFService.GetResource("rdf:" + this.generateRandomSalt());

    for (var item in aElementArray) {
      this.mDataSource.Assert(newNode, this.nsIRDFService.GetResource(this.mNamespace + item), this.nsIRDFService.GetLiteral(aElementArray[item]), true);
    }

    this.nsIRDFContainer.Init(this.mDataSource, this.nsIRDFService.GetResource(aSource));
    this.nsIRDFContainer.AppendElement(newNode);

    return newNode.Value;
  },

  addContainer: function(aContainer) {
    this.nsIRDFContainerUtils.MakeSeq(this.mDataSource, this.nsIRDFService.GetResource(aContainer));
	},

  flushDataSource: function() {
    this.mDataSource.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Flush();
  },

  generateRandomSalt: function() {
    var charSet = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    var randomSalt = charSet.charAt(Math.floor(Math.random() * charSet.length));

    for (var i = 1; i < 12; ++i) {
      randomSalt = randomSalt + charSet.charAt(Math.floor(Math.random() * charSet.length));
    }
    
    return randomSalt;
  },

  getDataSourceURI: function() {
    var directoryServices = Components.classes["@mozilla.org/file/directory_service;1"].getService()

    var profileDirectory = directoryServices.QueryInterface(Components.interfaces.nsIProperties).get("ProfD", Components.interfaces.nsIFile);
    var profileDefaultsDirectory = directoryServices.QueryInterface(Components.interfaces.nsIProperties).get("profDef", Components.interfaces.nsIFile);
    
    var ioService = Components.classes["@mozilla.org/network/io-service;1"].getService(Components.interfaces.nsIIOService);
    
    var File = new Components.Constructor("@mozilla.org/file/local;1", "nsILocalFile", "initWithPath");

    try {
      var MozTweakFile = new File(profileDirectory.path);

      MozTweakFile.append(this.mDataSourceFile);

      if (MozTweakFile.exists()) {
        return ioService.newFileURI(MozTweakFile).spec;
      }
      else {
        var defaultMozTweakFile = new File(profileDefaultsDirectory.path);

        defaultMozTweakFile.append(this.mDataSourceFile);
        defaultMozTweakFile.copyTo(profileDirectory, null);
        return ioService.newFileURI(defaultMozTweakFile).spec;
      }
    }
    catch (err) {
      return null;
    }
  }
};

var gMozTweakSignaturesRDF = new MozTweakRDF(false, "moztweakSignatures.rdf");
var gMozTweakHeadersRDF = new MozTweakRDF(false, "moztweakHeaders.rdf");
