/* @(#)mp.c	16.1.1.1 (ES0-DMD) 06/19/01 15:30:10 */
/*===========================================================================
  Copyright (C) 1995 European Southern Observatory (ESO)
 
  This program is free software; you can redistribute it and/or 
  modify it under the terms of the GNU General Public License as 
  published by the Free Software Foundation; either version 2 of 
  the License, or (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public 
  License along with this program; if not, write to the Free 
  Software Foundation, Inc., 675 Massachusetss Ave, Cambridge, 
  MA 02139, USA.
 
  Corresponding concerning ESO-MIDAS should be addressed as follows:
	Internet e-mail: midas@eso.org
	Postal address: European Southern Observatory
			Data Management Division 
			Karl-Schwarzschild-Strasse 2
			D 85748 Garching bei Muenchen 
			GERMANY
===========================================================================*/

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.TYPE		Module
.IDENTIFICATION	mp.c
.AUTHOR    	Francois Ochsenbein [ESO-IPG]
.LANGUAGE  	C
.KEYWORDS	Main Parameters
.ENVIRONMENT 	
.VERSION 1.0 	06-Nov-1986: Creation
.VERSION 1.1 	05-Mar-1988: Adapted for MIDAS
.VERSION 1.2	13-Oct-1988: Cosmetic Modifications
.VERSION 1.3	22-Jun-1989: Added mp_set
.VERSION 1.4	23-Nov-1989: Filenames may start with / (Unix) 
.VERSION 1.5	08-Dec-1989: Added mp_gs   mp_gl  (Get Named Parameters)
.VERSION 1.6 	08-May-1990  Added mp_del (delete parms facility)
.VERSION 1.7 	08-Jun-1991: Be sure mp_gs does NOT modify argx...
.COMMENTS
	These routines allow the management of parameters to a main program.

\begin{TeX}
	Arguments to a main program start with a character having the
	following meaning:
\begin{itemize}
\item	{\tt <}\quad redirect input file
\item	{\tt >}\quad redirect output file
\item	{\tt \^{ }}\quad terminal name
\item	{\tt -}\quad option string 
\end{itemize}

An option string is made of an alphabetic (always in lower-case)
character, possibly followed by a number or a string, \eg
\begin{quote}
{\tt -t}	\\
{\tt -l80}
{\tt -fin}
\end{quote}

A named parameter is normally written \fbox{{\tt name=}{\em value}}.
It can be retrieved using mp\_gt.

A {\em special} argument starts with one of the characters
\fbox{{\tt -+`'\$\%\^{ }\&(),;:|\b }}
\end{TeX}

-----------------------*/

#define		DEBUG		0	/* For debugging purposes	*/

#define	PM_LEVEL	LEVEL_STR

#define PASCAL_DEF	0	/* Don't include pascalisation ... */
#include <stesodef.h>
#include <atype.h>
#include <tra.h>
#include <string.h>

static	int argc = 0;
static 	char **argv = (char **)0;
static 	char **envp = (char **)0;
static	int argx = 0;			/* Current argument index	*/
static  long	number = 0;

static char special[] = "-+`'$%^&(),;:|\\";
			/* Parameters starting whith those are special	*/

/*==========================================================================*/
int mp_save(np, p1, p2)    
/*+++
.PURPOSE Save the arguments of a main program.
.RETURNS Number of parameters
.REMARKS To be called at the beginning of a program
---*/
	int np;		/* IN: Argument count		*/
	char **p1;	/* IN: List of parameters		*/
	char **p2;	/* IN: List of environment parameters	*/
{
  ENTER("mp_save");
#if DEBUG
  TRACE_ED_I("Number of main program arguments=",np);
#endif

  argc = np;
  argv = p1;
  envp = p2;
  argx = 0;

  EXIT(argc) ;
}

/*==========================================================================*/
char *mp_get(n)
/*+++
.PURPOSE Get argument number n
.RETURNS Address of n.th argument, or NULL
.REMARKS 
---*/
	unsigned int	n;	/* IN: Number of wished argument	*/
{
	register char *p;
	
  ENTER("*mp_get");
#if DEBUG
  TRACE_ED_I("Main argument #",(int)n);
#endif

  if (n < argc)	 { p = *(argv+n); TRACE(p); }
   else		   p = NULL_PTR(char);

  EXITp(p);
}

/*==========================================================================*/
int mp_reset()
/*+++
.PURPOSE Reset the Argument List.
.RETURNS Number of Arguments
.REMARKS 
---*/
{
  ENTER("+mp_reset");

  argx = 0;

  EXIT(argc);
}

/*==========================================================================*/
char *mp_next()
/*+++
.PURPOSE Get next NORMAL parameter
.RETURNS Address of next NORMAL (starting with alphanumeric or slash) 
	parameter, or NULL
.REMARKS 
---*/
{
	char *mp_get(), *p;
	int  i;

  ENTER("*mp_next");

  i = argx;		/* The argument count will be reset if nothing found */
  while (++i < argc)   {
  	if_not (p = *(argv+i))	
  		continue;
  	if (special[strloc(special, *p)])
  		continue;
  	else	break;
  }

  argx = i;
  EXITp(mp_get(argx));
}

/*==========================================================================*/
char *mp_get0()
/*+++
.PURPOSE Get next parameter, whatever it is (option or normal)
.RETURNS Address of next NORMAL (starting with alphanumeric) parameter, or NULL
.REMARKS 
---*/
{
	char *mp_get();

  ENTER("*mp_get0");

  EXITp(mp_get(++argx));
}

/*==========================================================================*/
int mp_del()
/*+++
.PURPOSE Delete current parameter
.RETURNS Number of deleted parameter
.REMARKS 
---*/
{
  ENTER("+mp_del");
  
  if ((argx > 0) && (argx < argc))	/* ... Don't delete Program File    */
  	*(argv+argx) = NULL_PTR(char);

  EXIT(argx);
}

/*==========================================================================*/
char *mp_gs(name)
/*+++
.PURPOSE Retrieve a named option.
.RETURNS Pointer to found string / NULL if not found
.REMARKS Example:  parameter written as name="value", will retrieve value
---*/
	char	*name;	/* IN: Starting string to look for	*/
{
	register char **ps;
	register char *p;
	register int i, l;

  ENTER("*mp_gs");


	/* Method: scan parameters beginning with name, and compare */

  l = strlen(name);

  for (ps = argv, i = 1; i < argc; i++) {
  	if_not (p = *++ps)	continue;
	if (oscomp(p,name,l) == 0)
		break;
  }

  if ((i < argc) && (p))	/* The comparison was successfull. */
  	p += l;
  else	p = NULL_PTR(char);

  EXITp(p);
}

/*==========================================================================*/
char	*mp_flagged(ch)
/*+++
.PURPOSE Retrieve a flagged (starting with a specified char) parameter.
.RETURNS Address of retrieved parameter WITHOUT the flag, or NULL
.REMARKS 
---*/
	char	ch;	/* IN: Starting character	*/
{
	char	op[2];

  op[0] = ch; op[1] = '\0';
  
  return(mp_gs(op));
}

/*==========================================================================*/
long mp_gl(name)
/*+++
.PURPOSE Retrieve a named numeric option.
.RETURNS Value following name / 0 when not found.
.REMARKS Example:  parameter written as name="value", will retrieve value
---*/
	char	*name;	/* IN: Starting string to look for	*/
{
	register char *p;
	long	number;

  ENTER(".mp_gl");

  number = 0;
  if (p = mp_gs(name))
  	tr_al(p, strlen(p), &number);

  EXITl(number);
}

/*==========================================================================*/
int	mp_option(ch)
/*+++
.PURPOSE Retrieve an option in the parameter
.RETURNS 0 (non-existing option), or value following the option (1 is default)
.REMARKS Example is:
\begin{TeX}
$$\begin{tabular}{llr} \hline
Parameter  & Argument	& Result \\ 
\hline
{\tt	-x,-l-80} & {\tt 'l'} &	-80\\
{\tt	-x,-l-80} & {\tt 'x'} &	1\\
{\tt	-x,-l-80} & {\tt 'g'} &	0\\
\hline
\end{tabular}$$
\end{TeX}
---*/
	char	ch;	/* IN: Starting option character	*/
{
	register char *p;
	register int st;
	char 	optstr[3];
	long	number;

  ENTER("mp_option");

	/* Method: Use mp_gs, and then translate the number if any */

  optstr[0] = '-', optstr[1] = ch, optstr[2] = '\0';
  if (p = mp_gs(optstr))   {
  	st = tr_al(p, strlen(p), &number);
  	if (st == 0)	st = 1;		/* No number, default is 1	*/
  	else if (st < 0)st = 0;
  	else		st = number;
  }
  else	st = 0;				/* Option not present	*/

  EXIT(st);
}

/*==========================================================================*/
char *mp_poption(ch)
/*+++
.PURPOSE Retrieve an option string in the main parameters (option is e.g. -o)
.RETURNS Address of string following the option, or NULL
.REMARKS Example: the parameter -oXYZ will return XYZ.
---*/
	char	ch;	/* IN: Starting option character	*/
{
	char	op[3];

  op[0] = '-'; op[1] = ch; op[2] = '\0';
  
  return(mp_gs(op));
}

