/* @(#)scf.c	16.1.1.1 (ESO-DMD) 06/19/01 15:18:56 */
/*===========================================================================
  Copyright (C) 1995 European Southern Observatory (ESO)
 
  This program is free software; you can redistribute it and/or 
  modify it under the terms of the GNU General Public License as 
  published by the Free Software Foundation; either version 2 of 
  the License, or (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public 
  License along with this program; if not, write to the Free 
  Software Foundation, Inc., 675 Massachusetts Ave, Cambridge, 
  MA 02139, USA.
 
  Correspondence concerning ESO-MIDAS should be addressed as follows:
	Internet e-mail: midas@eso.org
	Postal address: European Southern Observatory
			Data Management Division 
			Karl-Schwarzschild-Strasse 2
			D 85748 Garching bei Muenchen 
			GERMANY
===========================================================================*/

/*+++++++++++++++++++++++ SC interface module SCF +++++++++++++++++++++++
.LANGUAGE C
.IDENTIFICATION Module SCF
.AUTHOR         K. Banse 		ESO - Garching
.KEYWORDS       standard interfaces, bulk data files
.ENVIRONMENT    VMS and UNIX
.COMMENTS 
holds SCFGET, SCFPUT, SCFDEL, SCFCRE, SCFRNM
.REMARKS
opening modes are: F_I_MODE, F_O_MODE, F_IO_MODE, F_X_MODE.                
data types are: D_R4_FORMAT, D_I2_FORMAT, D_I1_FORMAT, D_OLD_FORMAT.         
filetype no's are: F_IMA_TYPE, F_TBL_TYPE, F_FIT_TYPE.              
.VERSION  [1.20] 871109: creation
 
 010421		last modif

----------------------------------------------------------------------------*/

#include <fileexts.h>

#include <osyparms.h>




/*

*/

int SCFGET(imno,felem,size,actsize,bufadr)

/*++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE
read data from disk frame into memory 
.ALGORITHM
synchronous disk I/O
.RETURNS
return status ( 0 = o.k )
--------------------------------------------------------*/

int   imno		/* IN : file no. of data frame */;
int   felem	/* IN : 1st pixel to be accessed in data space */;
int   size    	/* IN : number of data values (pixels) to be read */;
int   *actsize 	/* OUT: actual no. of pixels read */;
char	*bufadr;	/* IN: address of data buffer  */

{
int  status;
register int  nr;

char   *pntrs[2];

struct FCT_STRUCT  *fctpntr;



 
/*  check 'imno' + 'felem' */
 
if ( (imno < 0) || (imno >= FCT.MAXENT) )
   {
   status = ERR_INPINV;
   goto end_of_it;
   }

fctpntr = FCT.ENTRIES + imno;

if ( (fctpntr->NAME[0] == ' ') || (felem <= 0) ) 
   {
   status = ERR_INPINV;
   goto end_of_it;
   }


if (fctpntr->FILTYP > 0)		/* FITS file */
   {
   char  *datpntr, *zpntr;
   int   mm, felbytes, nbytes;


   if (fctpntr->DATTYP != fctpntr->FORMAT)
      nbytes = get_byte(fctpntr->DATTYP);
   else
      nbytes = fctpntr->NOBYTE;

   felem --;
   felbytes = felem * nbytes;

   if (fctpntr->BDADDR[0] != (char *) 0)
      {
      status = ERR_NORMAL;
      datpntr = fctpntr->BDADDR[0] + felbytes;
      mm = fctpntr->KAUX[1];
      }
   else
      {
      status = SCFMAP(imno,F_I_MODE,1,-1,&mm,&datpntr);
      datpntr += felbytes;
      }

   *actsize = mm - felem;
   if (*actsize < 1) 
      status = ERR_INPINV;
   else
      {
      if (size < 1) size = fctpntr->SIZE;
      if (*actsize > size) *actsize = size;
      zpntr = bufadr;
      for (nr=0; nr<((*actsize)*nbytes); nr++) *zpntr++ = *datpntr++;
      }
   }

else
   {
   if (fctpntr->ACCESS == 'X')
      {
      felem --;
      *actsize = fctpntr->SIZE - felem;

      if ( (fctpntr->DATTYP != fctpntr->FORMAT) ||
           (fctpntr->BDADDR[0] == (char *) 0) ||
           (*actsize < 1) )
         status = ERR_INPINV;
      else
         {
         int   felbytes, nbytes;
         char  *datpntr, *zpntr;

         nbytes = fctpntr->NOBYTE;
         felbytes = felem * nbytes;
         datpntr = fctpntr->BDADDR[0] + felbytes;
         if (size < 1) size = fctpntr->SIZE;
         if (*actsize > size) *actsize = size;
         zpntr = bufadr;
         for (nr=0; nr<((*actsize)*nbytes); nr++) *zpntr++ = *datpntr++;
         status = ERR_NORMAL;
         }
      }
   else 
      {
      if (fctpntr->DATTYP != fctpntr->FORMAT)
         {				/* check if data type mismatch */
         pntrs[0] = bufadr;
         status = conv_dat(3,imno,felem,size,actsize,pntrs);
         }
      else
         status = rddisk(imno,felem,size,actsize,bufadr);
      }
   }


end_of_it:
if (status != ERR_NORMAL) MID_E2(10,imno,"...",status,1);
return (status);

}	

/*

*/

int SCFPUT(imno,felem,size,bufadr)

/*++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE
write data from memory into disk frame 
.ALGORITHM
synchronous disk I/O
.RETURNS
return status ( 0 = o.k )
--------------------------------------------------------*/

int   imno		/* IN : file no. of data frame */;
int   felem	/* IN : 1st pixel to be accessed in data space */;
int   size    	/* IN : no. of data values (pixels) to be written  */;
char	*bufadr;	/* IN: address of data buffer  */

{
int  status, mm;

char   *pntrs[2];

struct FCT_STRUCT  *fctpntr;


/*  check 'imno' + 'felem' */
 
if ( (imno < 0) || (imno >= FCT.MAXENT) )
   {
   status = ERR_INPINV;
   goto end_of_it;
   }

fctpntr = FCT.ENTRIES + imno;

if ((fctpntr->NAME[0] == ' ') || (felem <= 0) || (size <= 0))
   {
   status = ERR_INPINV;
   goto end_of_it;
   }
else if (fctpntr->PROT == 2) 
   {
   status = ERR_FILPRO;
   goto end_of_it;
   }

if (fctpntr->LINK[0] == 2) 
   fctpntr->ACCESS = 'O';		/* if son, we update later on ... */


if (fctpntr->FILTYP > 0)               /* FITS file */
   {
   char  *datpntr, *zpntr;
   int   mm, felbytes, nbytes;

   if (fctpntr->DATTYP != fctpntr->FORMAT)
      nbytes = get_byte(fctpntr->DATTYP);
   else
      nbytes = fctpntr->NOBYTE;

   felem --;
   felbytes = felem * nbytes;

   if (fctpntr->BDADDR[0] != (char *) 0)
      {						/* already mapped in */
      status = ERR_NORMAL;
      datpntr = fctpntr->BDADDR[0] + felbytes;
      mm = fctpntr->KAUX[1];
      fctpntr->ACCESS = 'O';			/* ensure update at SCFCLO */
      }
   else
      {
      status = SCFMAP(imno,F_O_MODE,1,-1,&mm,&datpntr);
      datpntr += felbytes;
      }

   if (felem+size > mm)
      status = ERR_INPINV;
   else
      {
      register int   nr;

      zpntr = bufadr;
      for (nr=0; nr<((size)*nbytes); nr++) *datpntr++ = *zpntr++;
      }
   }

else
   {
   if (fctpntr->DATTYP != fctpntr->FORMAT)
      {					/* check if data type mismatch */
      pntrs[0] = bufadr;
      status = conv_dat(2,imno,felem,size,&mm,pntrs);
      }

   else
      status = wrdisk(imno,felem,size,bufadr);
   }


end_of_it:
if (status != ERR_NORMAL) MID_E2(11,imno,"...",status,1);
return (status);

}	

/*

*/

int SCFDEL(name)

/*++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE
delete a frame
.ALGORITHM
use FSY modules to do the job
.RETURNS
return status ( 0 = o.k )
--------------------------------------------------*/

char   *name    /* IN : name of object */;

{
char   *mypntr, temp[200];

int   mm, namtyp, part_no;
int   status;

struct FCT_STRUCT  *fctpntr;

extern int  osfdelete();


/*  check file name  */
 
mypntr = &temp[8];		/* because of eventual error message ... */
CGN_CLEANF(name,0,mypntr,FCT_NAME_LEN,&namtyp,&part_no);


mm = MID_FINDFR(mypntr);		/*  look, if file still in FCT  */
if (mm >= 0) 
   {
   fctpntr = FCT.ENTRIES + mm;
   fctpntr->CATALOG[1] = 'N';		/* we do not add this file to a catal */
   status = SCFCLO(mm);
   }


/*  and delete it  */
 
status = osfdelete(mypntr);

if (status == 0)
   return (ERR_NORMAL);
else
   {
   strncpy(temp,"SCFDEL: ",8);
   MID_ERROR("FSY",temp,status,1);
   return (ERR_INPINV);
   }
}

/*

*/

int SCFCRE(name,dattype,iomode,filtype,size,imno)
/*++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE
create an object of given type
.ALGORITHM
set FCB_CATALOG, otherwise MID_INITFR does all the work...
.RETURNS
return status ( 0 = o.k )
--------------------------------------------------*/

char   *name       /* IN: name of object */;
int dattype;  /* IN: data types as defined in Module header  */
int iomode;   /* IN: opening mode, e.g. F_O_MODE, ...   */
int filtype;  /* IN: filetype no., e.g. F_TBL_TYPE, ... */
int size	   /* IN: size of object = no. of data values in file */;
int *imno	   /* OUT: file no.  */;

{
int   mm, namtyp, part_no;
int   entrx, status;

char   temp[200];

struct FCT_STRUCT  *fctpntr;


/* get cleand file name 'temp'  */
 
status = CGN_CLEANF(name,filtype,temp,FCT_NAME_LEN,&namtyp,&part_no);
if ((status != 0) || (part_no != 0))
   {
   status = ERR_FILNAM;
   strcpy(temp,name);
   goto end_of_it;
   }

if (iomode != F_H_MODE)	
   {
   if (size < 1) 		/* we need size > 0 !!  */
      {		
      status = ERR_OUTLIM;
      goto end_of_it;
      }
   }
else				/* unless we just want to */
   size = -1;			/* write the file header */


/*	
   if iomode = F_X_MODE,  just prepare entry in FCT
*/
	
if (iomode == F_X_MODE)
   {
   entrx = MID_FINDFR(temp);			/* look, if already in FCT  */
   if (entrx > -1) 				/* > -1 is the same as >= 0 */
      {
      status = ERR_INPINV;
      goto end_of_it;
      }


   /*  enter file also into FCT with size in bytes  */
 
   entrx = MID_CREFRE(temp,-1);
   if (entrx < 0)
      {
      status = ERR_FILNAM;
      goto end_of_it;
      }
  
   fctpntr = FCT.ENTRIES + entrx;
   fctpntr->CATALOG[1] = 'N';			/* no catalog...  */
   fctpntr->ACCESS = 'X';			/* set to scratch access */
   *imno = entrx;


   /* store required data type + default access mode  */

   fctpntr->FORMAT = dattype;
   fctpntr->NOBYTE = get_byte(fctpntr->FORMAT);
   fctpntr->SIZE = size;
   fctpntr->STBLOK = 1;			/* no FCB, nor descriptors... */

   status = ERR_NORMAL;
   }

else
   {			/*  create file on disk (store FCB + LDB's)  */
   infr_loop:
   status = MID_INITFR(temp,dattype,namtyp,size,imno);

   if (status == (-99))			   /* if an FCT entry already there, */
      {						/* get rid of it  */
      MID_ACCFRM(temp,0,imno,&mm);
      (void) SCFCLO(*imno);
      goto infr_loop;
      }
   
    if (status != ERR_NORMAL) 
      goto end_of_it;

   fctpntr = FCT.ENTRIES + (*imno);
   if (KEYALL.ORIGIN != (-1))
      {
      mm = filtype + 4 + OFF_CATINF;	/* point to relevant flag in CATALINF */
      if (KIWORDS[mm] == 1)
         fctpntr->CATALOG[1] = 'Y';	/* we want to catalog that one */
      }
   }


/*  that's it...  */
 
fctpntr->DATTYP = fctpntr->FORMAT;
return status;
 

end_of_it:
MID_E1(4,temp,status,1);
return status;
}

/*

*/

int SCFRNM(oldname,newname)
/*++++++++++++++++++++++++++++++++++++++++++++++++++
.PURPOSE
rename a disk file
.ALGORITHM
use CGN_CLEANF + osfrename
.RETURNS
return status ( 0 = o.k )
--------------------------------------------------------*/

char	*oldname;		/* IN: old name  */
char	*newname;		/* OUT: new name  */

{
char  temp[200];

int   namtyp, part_no;


CGN_CLEANF(oldname,0,temp,FCT_NAME_LEN,&namtyp,&part_no);

return (osfrename(temp,newname));
}	

