/**********************************************************************/
/*
/* $Date: 91/10/25 23:20:58 $
/* $Revision: 1.20.33.1 $
/*
/**********************************************************************/

/*****************************************************************************/
/***                                                                       ***/
/***             Copyright (c) 1988, Visual Edge Software Ltd.             ***/
/***                                                                       ***/
/***   ALL RIGHTS RESERVED.  This notice is  intended  as  a  precaution   ***/
/***   against  inadvertent publication, and shall not be deemed to con-   ***/
/***   stitute an acknowledgment that publication has  occurred  nor  to   ***/
/***   imply  any  waiver  of confidentiality.  The year included in the   ***/
/***   notice is the year of the creation of the work.                     ***/
/***                                                                       ***/
/*****************************************************************************/


#include <stdio.h>
#include <string.h>
#include <ctype.h>

#ifdef USE_STDLIB
#include <stdlib.h>
#endif /* USE_STDLIB */

#ifndef _IBMR2
#include <math.h>
#endif /* _IBMR2 */

#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#ifdef vms 
#include <decw$include/X.h>
#else
#include <X11/X.h>
#endif
#ifdef vms 
#include <decw$include/Xlib.h>
#else
#include <X11/Xlib.h>
#endif
#ifdef vms 
#include <decw$include/Intrinsic.h>
#else
#include <X11/Intrinsic.h>
#endif
#ifdef vms 
#include <decw$include/Xutil.h>
#else
#include <X11/Xutil.h>
#endif
#ifdef vms 
#include <decw$include/StringDefs.h>
#else
#include <X11/StringDefs.h>
#endif
#include <global.h>
#include <resload.h>
#include <uimx_cat.h>
#include <valloc.h>
#include <vtypes.h>
#ifndef	RUNTIME
#include <misc_ds.h>
#endif

/* X/OPEN message catalog macros. These will make the code more compact. */
#define CGETS(ms,ds_ms)                 UxCatGets(MC_SYSTEM_NAME,ms,ds_ms)
#define MCGETS(ms,ds_ms)                UxCatGets(MC_MISC,ms,ds_ms)


#ifdef OL_WIDGETS
#define APPDEF "/usr/X/lib/app-defaults"
#define DEFAULT_BPATH ". ~ /usr/X/include/X11/bitmaps/$UXAPP /usr/X/include/X11/bitmaps /usr/X/include/X11/pixmaps/$UXAPP /usr/X/include/X11/pixmaps"
#endif /* OL_WIDGETS */

#ifdef MOTIF_WIDGETS
#define APPDEF "/usr/lib/X11/app-defaults"
#define APPDEF2 "/usr/lib/X11/%L/app-defaults"
#define DEFAULT_BPATH ". ~ /usr/include/X11/bitmaps/$UXAPP /usr/include/X11/bitmaps"
#endif /* MOTIF_WIDGETS */

extern Display		*UxDisplay;
#define XTDATABASE	(UxDisplay->db)
pathlist		UxResourcePath= NULL, UxBitmapPath= NULL;

char		*UxUimxRFile= "app.rf",	 	 /* main resource file */
		*UxToolkitRFile= "defaults.rf",	 /* tookit defaults r file */
		*UxScreenRFile= "screen.rf",	 /* color/mono resource file */
		*UxSpecificRFile= "specific.rf"; /* size specific res. file */


#ifdef OL_WIDGETS 
/*******************************************************************/
/* announceResourceFiles determines whether or not resource files  */
/* are announced when read.                                        */
/*******************************************************************/

static Boolean announceResourceFiles = FALSE;
#endif /* OL_WIDGETS */

/*****************************************************************************
NAME:		void UxUpdateScreenSpecificResources()
		void UpdateResource(qualifier, rname)
		void UpdateFontResource(qualifier, rname, font1, font2)

		char	*qualifier	- "HighRes", "MediumRes", "LowRes"
					  "Color" or "Mono"
		char	*rname		- resource name
		char	*font1		- first default font name
		char	*font2		- second default font name

DESCRIPTION:	These functions together get from the resource database
		certain UIM/X screen specific resources for the current type
		of screen and then put them back in the database without the
		screen specification if they are not already defined in the
		database.  For example on a high res monitor
			Uimx*HighRes*FontList:
		gets written back to the resource database as
			Uimx*FontList:

		For fonts, if the requested ones aren't available,
		we attempt to substitute "variable" or "fixed".

LAST REV:	March 15, 1990		See bugfix 2111
		July 29, 1991		See fix2830
		September 10, 1991	See fix295
-----------------------------------------------------------------------------*/

void UxUpdateScreenSpecificResources()
{
	register char 	*type;
	extern	 void	UpdateResource();

#ifndef	RUNTIME
	register char	*res;
	register int	width, height, pixels;
	extern	 void	UpdateFontResource();

	res = UxGetAppResource("screenResolution");
	if (res == NULL) {
		width = DisplayWidth(UxDisplay, UxScreen);
		height = DisplayHeight(UxDisplay, UxScreen);
		pixels = width * height;
		if (pixels >= 1000 * 1000)
			res = "HighRes";
		else if (pixels >= 800 * 600)
			res = "MediumRes";
		else
			res = "LowRes";
	}

	UpdateFontResource(res, "SysFont1", "variable", "fixed");
	UpdateFontResource(res, "FontList", "variable", "fixed");
	UpdateFontResource(res, "XmText.fontList", "fixed", "variable");
	UpdateResource(res, "uimxScale");
	UpdateResource(res, "uimxXScale");
	UpdateResource(res, "uimxYScale");
#endif

	type = UxGetAppResource("screenType");
	if (type == NULL) {
		Visual *vis = DefaultVisual(UxDisplay, UxScreen);	

		if( vis->class == GrayScale || vis->class == StaticGray ) {
			type = "Mono";
		} else {
			type = "Color";
		}
	}

	UpdateResource(type, "foreground");
	UpdateResource(type, "background");
}

static void UpdateResource(qualifier, rname)
	char *qualifier, *rname;
{
	dstring		res;
	XrmValue	value, value2;
	char		*type, *type2;
	int		found;
	extern char	*UxProgramName, *UxProgramClassName;
	extern void	UxPutAnyResource();

	res = dcreate(qualifier);
	dappend(res, ".");
	dappend(res, rname);

	found = UxGetAnyResource(UxProgramName, dgetstr(res), &type, &value);
	if (found) {
		found = UxGetAnyResource(UxProgramName, rname, &type2, &value2);
		if (!found) {
			/*-------------------------------------------
	 		 * Install the qualified resource 
			 * under the unqualified name.
	 		 *-------------------------------------------*/

			UxPutAnyResource(UxProgramClassName, rname, type, &value);
		}
	}

	dfree(res);
}

#ifndef	RUNTIME
static void UpdateFontResource (qualifier, rname, font1, font2)
	char *qualifier, *rname, *font1, *font2;
{
    int		isGeneral;
    int		isSpecific = 0;
    char	*type;
    XrmValue	value, newvalue;
    extern char	*UxProgramName;

    /*-----------------------------------------------------
     * Get the resource value.
     *-----------------------------------------------------*/
    isGeneral = UxGetAnyResource (UxProgramName, rname, &type, &value);
    if (!isGeneral)
    {
	dstring	res;

	res = dcreate (qualifier);
	dappend (res, ".");
	dappend (res, rname);
	isSpecific = UxGetAnyResource(UxProgramName,dgetstr(res),&type,&value);
	dfree (res);
    }

    if (isGeneral || isSpecific) 
    {
    	int	isNewFont;
	/*-----------------------------------------------------
         * Validate the resource value.
	 *-----------------------------------------------------*/
	if (NO_ERROR == UxLoadOneFont ((char *)value.addr)) {
	    newvalue.addr = value.addr;
	    newvalue.size = strlen((char *)value.addr)+1;
	    isNewFont = 0;
	} else {
	    /*-----------------------------------------------------
	     * No luck.  Try the alternates....
	     *-----------------------------------------------------*/
	    UxStandardError (MCGETS (MS_MISC_FIND_FONT_CHANGE,
				    DS_MS_MISC_FIND_FONT_CHANGE), 
			     font1);

	    if (NO_ERROR == UxLoadOneFont (font1)) {
	        newvalue.addr = (caddr_t) font1;
	        newvalue.size = strlen(font1)+1;
	    } else {
	        UxStandardError (MCGETS (MS_MISC_FIND_FONT_CHANGE,
				        DS_MS_MISC_FIND_FONT_CHANGE), 
				 font2);
	        newvalue.addr = (caddr_t) font2;
	        newvalue.size = strlen(font2)+1;
	    }
	    isNewFont = 1;
	}

	/*-----------------------------------------------------
 	 * Install the validated font resource 
	 * under its unqualified name.
 	 *-----------------------------------------------------*/
	if (isSpecific || isNewFont) {

    	    extern char *UxProgramClassName;
    	    extern void	UxPutAnyResource();

	    UxPutAnyResource(UxProgramClassName, rname, type, &newvalue);
	}
    }
}
#endif


/*****************************************************************************
NAME:		void		UxInitializeResourcePath()

DESCRIPTION:	initialize the resource file load path

LAST REV:	March 13, 1991		See bugfix 2111.
		Modified to conform to X resource path.
-----------------------------------------------------------------------------*/

void UxInitializeResourcePath()
{
	static Boolean	initialized = FALSE;
	dstring		path, rdir, screen;
	char		res[32];
	char		*getenv();
	char		*appResDir;

	if( initialized == TRUE ) {
	   return;
	} else {
	   initialized = TRUE;
	}

	if(UxResourcePath != NULL)
		return;

	screen= dcreate(UxGetAppResource("screenType"));

	if(dlen(screen) == 0)
	{	
		Visual *vis = DefaultVisual(UxDisplay, UxScreen);	

		if( vis->class == GrayScale || vis->class == StaticGray ) {
			screen= dcreate("mono/");
		} else {
			screen= dcreate("color/");
		}
	} else {
		if (dnstr(screen)[0] == 'C')
			dnstr(screen)[0] = 'c';
		if (dnstr(screen)[0] == 'M')
			dnstr(screen)[0] = 'm';
		if( dnstr(screen)[dlen(screen)-1] != '/' )
			dappend(screen, "/");
	}

	sprintf(res, "%dx%d", DisplayWidth(UxDisplay, UxScreen),
		DisplayHeight(UxDisplay, UxScreen));

	path= dcreate(" ");

	/*
	 * The path is ordered such that if a file is present in more than
	 * one directory, the second copy of the file override the previously
	 * loaded copy.  The following order is used:
	 *
	 *	/usr/lib/X11/app-defaluts/
	 *	/usr/lib/X11/app-defaults/$UXAPP/...
	 *	/usr/lib/X11/LANG/app-defaluts/			(for MOTIF only)
	 *	/usr/lib/X11/LANG/app-defaults/$UXAPP/...	(for MOTIF only)
	 * if XAPPLRESDIR exists
	 *		XAPPLRESDIR/$UXAPP/...
	 * else
	 *		~/$UXAPP/...
	 * for compatibility with old versions of UIM/X add current directory
	 *	./$UXAPP/...
	 */

	/* /usr/lib/X11/app-defaults/ */
	rdir = dcreate(" ");
	dappend(rdir, APPDEF);
	dappend(rdir, "/");
	dconcat(path, rdir);

	/* /usr/lib/X11/app-defaults/<ClassName> */
	dappend(rdir, "$UXAPP/");
	dconcat(path, rdir);

	/* /usr/lib/X11/app-defaults/<ClassName>/color */
	dconcat(path, rdir);
	dconcat(path, screen);

	/* /usr/lib/X11/app-defaults/<ClassName>/color/1280x1024 */
	dconcat(path, rdir);
	dconcat(path, screen);
	dappend(path, res);

	dfree(rdir);

#ifdef MOTIF_WIDGETS
	/* /usr/lib/X11/%L/app-defaults/ */
	rdir = dcreate(" ");
	dappend(rdir, APPDEF2);
	dappend(rdir, "/");
	dconcat(path, rdir);

	/* /usr/lib/X11/%L/app-defaults/<ClassName> */
	dappend(rdir, "$UXAPP/");
	dconcat(path, rdir);

	/* /usr/lib/X11/%L/app-defaults/<ClassName>/color */
	dconcat(path, rdir);
	dconcat(path, screen);

	/* /usr/lib/X11/%L/app-defaults/<ClassName>/color/1280x1024 */
	dconcat(path, rdir);
	dconcat(path, screen);
	dappend(path, res);

	dfree(rdir);
#endif /* MOTIF_WIDGETS */	

	rdir = dcreate(" ");
	if ( (appResDir = getenv( "XAPPLRESDIR" )) != NULL ) {
		dappend(rdir, appResDir);
	} else {
		dappend(rdir, "~/");
	}

	dappend(rdir, "$UXAPP/");

	/* XAPPLRESDIR/<ClassName> or ~/<ClassName> */
	dconcat(path, rdir);

	/* XAPPLRESDIR/<ClassName>/color or ~/<ClassName>/color */
	dconcat(path, rdir);
	dconcat(path, screen);

	/* XAPPLRESDIR/<ClassName>/color/res or ~/<ClassName>/color/res */
	dconcat(path, rdir);
	dconcat(path, screen);
	dappend(path, res);

	dfree(rdir);

	/*
	 * Current directory for compatibility with old versions of UIM/X.
	 */ 
	rdir = dcreate(" ./$UXAPP/");

	/* ./<ClassName> */
	dconcat(path, rdir);

	/* ./<ClassName>/color */
	dconcat(path, rdir);
	dconcat(path, screen);

	/* ./<ClassName>/color/res */
	dconcat(path, rdir);
	dconcat(path, screen);
	dappend(path, res);

	UxResourcePath= UxInitPath(dnstr(path));

	dfree(rdir);
	dfree(screen);
	dfree(path);
}

/*****************************************************************************
NAME:		char		*UxExpandResourceFilename(fname)

INPUT:		char		*fname		- resource file name

RETURN:		expanded name in a static buffer

DESCRIPTION:	find the full path to the resource file using the current
		resource file path.

LAST REV:	Jan 3/1989
-----------------------------------------------------------------------------*/

char *UxExpandResourceFilename(fname)
	char		*fname;
{
	UxInitializeResourcePath();

	return(UxExpandFilename(UxResourcePath, fname));
}


/*****************************************************************************
NAME:		void		UxInitializeBitmapPath()

DESCRIPTION:	initialize the bitmap search path

LAST REV:	Feb 25/1989
-----------------------------------------------------------------------------*/

void UxInitializeBitmapPath()
{
	if(UxBitmapPath == NULL) {
#if     defined(MOTIF_WIDGETS) & ! defined(RUN_TIME)
		char *str, *fmt= "$%s/icons %s", *buf;
		str = CGETS(MS_SYS_U_TASKDIR,DS_MS_SYS_U_TASKDIR);
		buf = UxMalloc(strlen(fmt) + strlen(str) + strlen(DEFAULT_BPATH) + 1);
		sprintf(buf,fmt,str,DEFAULT_BPATH);
		UxBitmapPath= UxInitPath(buf);
		UxFree(buf);
#else 
		UxBitmapPath= UxInitPath(DEFAULT_BPATH);
#endif
	}
}

/*****************************************************************************
NAME:		char		*UxExpandBitmapFilename(fname)

INPUT:		char		*fname		- filename

RETURN:		full path to bitmap file

DESCRIPTION:	search for the bitmap file 'fname' in the dirs given in
		UxBitmapPath

LAST REV:	Feb 21/1989
-----------------------------------------------------------------------------*/

char *UxExpandBitmapFilename(fname)
	char		*fname;
{
	UxInitializeBitmapPath();

	return(UxExpandFilename(UxBitmapPath, fname));
}

/*****************************************************************************
NAME:		void		UxOverrideResources(fname)

INPUT:		char		*fname		- resource file name

DESCRIPTION:	Load the resource file 'fname' and don't override its values
		by loading .Xdefaults again.

LAST REV:	March 14, 1991		See bugfix 2111.
		Don't expand filename using UxResourcePath.  Either fname
		is an absolute path or a relative path to the current directory.
-----------------------------------------------------------------------------*/

void UxOverrideResources(fname)
	char		*fname;
{
	XrmDatabase	rdb = (XrmDatabase) 0;
	char		*name;
	pathlist	cur_dir;

	if(fname == NULL)
		return;

	cur_dir = UxInitPath(".");
	name = UxExpandFilename(cur_dir, fname);
	dfree(*cur_dir);

	if (name == NULL)
	{	
#ifdef OL_WIDGETS
		char	msg[282];

		sprintf(msg, "Resource file %.*s not found.\n", 
				sizeof(msg)-27,	fname);
		XtWarning(msg);
#endif /* OL_WIDGETS */
		return;
	}

#ifdef OL_WIDGETS
        if( announceResourceFiles == TRUE )
	   printf ("Reading resource file %s\n", name); 
#endif /* OL_WIDGETS */
	rdb = XrmGetFileDatabase(name);
	if (rdb) {
		XrmMergeDatabases(rdb, &XTDATABASE);
	}
}

/*****************************************************************************
NAME:		void		UxLoadResources(fname)

INPUT:		char		*fname		- name of resource file

DESCRIPTION:	Load 'fname' into the current toolkit resource database
		nondestructively.

LAST REV:	March 14, 1991		See bugfix 2111.
		Don't expand filename using UxResourcePath.  Either fname
		is an absolute path or a relative path to the current directory.
-----------------------------------------------------------------------------*/

void UxLoadResources(fname)
	char		*fname;
{
	XrmDatabase	rdb = (XrmDatabase)0;
	char		*name;
	pathlist	cur_dir;

	if(fname == NULL)
		return;

	cur_dir = UxInitPath(".");
	name = UxExpandFilename(cur_dir, fname);
	dfree(*cur_dir);

	if (name == NULL)
	{	
#ifdef OL_WIDGETS
		char	msg[282];

		sprintf(msg, "Resource file %.*s not found.\n", 
				sizeof(msg)-27,	fname);
		XtWarning(msg);
#endif /* OL_WIDGETS */
		return;
	}

#ifdef OL_WIDGETS
       	if( announceResourceFiles == TRUE )
	   printf ("Reading resource file %s\n", name); 
#endif /* OL_WIDGETS */

	rdb = XrmGetFileDatabase(name);
	if (rdb) {
		XrmMergeDatabases(XTDATABASE, &rdb);
		XTDATABASE = rdb;
	}
}

/*****************************************************************************
NAME:		void		UxLoadResourceTree()

DESCRIPTION:	Load all UIM/X resource files, that have not been loaded by
		the regular XtInitialize() function, into the current toolkit
		resource database nondestructively.
		This includes the files contained in the UIM/X resource tree:
			Uimx/app.rf
			     defaults.rf
			     color/screen.rf
				   specific.rf
				   1024x768/specific.rf
		plus the standard <ClassName> file contained in the users
		current directory.

LAST REV:	March 15, 1991		See bugfix 2111.
-----------------------------------------------------------------------------*/

void UxLoadResourceTree()
{
	XrmDatabase	new_db, system_db;
	void		LoadResourceFileFromTree();

	UxInitializeResourcePath();

	new_db = NULL;
	LoadResourceFileFromTree(UxUimxRFile, &new_db);
	LoadResourceFileFromTree(UxToolkitRFile, &new_db);
	LoadResourceFileFromTree(UxScreenRFile, &new_db);
	LoadResourceFileFromTree(UxSpecificRFile, &new_db);

	/*
	 * Override resources in new database with resources found
	 * in ./UXAPP.
	 */
	system_db =  XTDATABASE;	/* save system data base */
	XTDATABASE = new_db;
	UxOverrideResources("$UXAPP");

	/*
	 * Merge system database into new database because
	 * system information overrides application default
	 * information.
	 */
	XrmMergeDatabases(system_db, &XTDATABASE);
}


/*****************************************************************************
NAME:		static void	LoadResourceFileFromTree(fname, rdb)

INPUT:		char		*fname		- filename
		XrmDatabase	*rdb		- resource data base

DESCRIPTION:	Load the given file from the UIM/X resource tree (if found)
		into the given toolkit resource database destructively.

LAST REV:	March 15, 1991		See bugfix 2111.
-----------------------------------------------------------------------------*/

static	void LoadResourceFileFromTree(fname, rdb)
	char		*fname;
	XrmDatabase	*rdb;
{
	XrmDatabase	nrdb;
	char		*name;

	if(fname == NULL)
		return;

	name = UxExpandAllFilenames(UxResourcePath, fname);

	if (name == NULL)
	{	
		return;
	}

	do {
#ifdef OL_WIDGETS
        	if( announceResourceFiles == TRUE )
		   printf ("Reading resource file %s\n", name); 
#endif /* OL_WIDGETS */

		/*
		 * New database must overwrite previous one because files
		 * are ordered from least specific to most specific.
		 */
		if ((nrdb = XrmGetFileDatabase(name)) != NULL)
			XrmMergeDatabases(nrdb, rdb);

		name = UxExpandAllFilenames((pathlist)NULL, (char *)NULL);
	} while (name);
}


/*****************************************************************************
NAME:		char		*UxGetResource(prog, rname)

INPUT:		char		*prog		- program name
				*rname		- resource name

RETURN:		value of the requested resource in string form 
		when successful.
		NULL is returned if the resource is not found or
		if the type of the resource is not an XtRString.

DESCRIPTION:	given a program 'name' and a resource name, get the 
		associated resource from the resource database.

LAST REV:	Jan 10/91 - bug2006
-----------------------------------------------------------------------------*/

char *UxGetResource(prog, rname)
	char		*prog, *rname;
{
	dstring		ds;
	XrmString	type;
	XrmValue	rtrn;
	extern char	*UxProgramClassName;
	dstring		ds2;
				
	ds= dcreate(prog);
	dappend(ds, ".");
	dappend(ds, rname);

	ds2 = dcreate(UxProgramClassName);
	dappend(ds2, ".");
	dappend(ds2, rname);

	if( False == XrmGetResource( XTDATABASE, 
				     dnstr(ds), 
				     dnstr(ds2),
				     &type, 
				     &rtrn) ||
				     strcmp (type, XtRString) != 0 )
	{
	    rtrn.addr = (caddr_t) NULL;
	}

	dfree(ds);
	dfree(ds2);

	return(rtrn.addr);
}

/*****************************************************************************
NAME:		char		*UxGetAppResource(rname)
INPUT:		char		*rname		- resource name
RETURN:		resource value
DESCRIPTION:	get a resource for the program with name given by the
		environment variable UXAPP
LAST REV:	5 Sept 1990
		-- removed the ifdef's and both now use the MOTIF one;
-----------------------------------------------------------------------------*/

char *UxGetAppResource(rname)
	char		*rname;
{
	char		*rtrn;
	extern		char *UxProgramName;
	rtrn = UxGetResource(UxProgramName, rname);
	return(rtrn);
}

char *UxGetUimxResource(rname)
	char		*rname;
{
	return(UxGetAppResource(rname));
}

/*****************************************************************************
NAME:		char		*UxGetDefault(prog, rname, def)

INPUT:		char		*prog		- program name
				*rname		- resource name
				*def		- default value

RETURN:		resource value

DESCRIPTION:	return the resource 'rname'. If not found return 'def'

LAST REV:	Feb 25/1989
-----------------------------------------------------------------------------*/

char *UxGetDefault(prog, rname, def)
	char		*prog, *rname, *def;
{
	char		*rtrn;

	rtrn= UxGetResource(prog, rname);

	if(rtrn == NULL)
		return(def);

	return(rtrn);
}

/*****************************************************************************
NAME:		char		*UxGetUimxDefault(rname, def)

INPUT:		char		*rname		- resource name
				*def		- default value

RETURN:		resource value

DESCRIPTION:	return the Uimx resource 'rname'. If not found return 'def'

LAST REV:	Feb 25/1989
-----------------------------------------------------------------------------*/

char *UxGetAppDefault(rname, def)
	char		*rname, *def;
{
	char		*rtrn;

	rtrn= UxGetUimxResource(rname);

	if(rtrn == NULL)
		return(def);

	return(rtrn);
}

char *UxGetUimxDefault(rname, def)
	char		*rname, *def;
{
	return(UxGetAppDefault(rname, def));
}

static	int	scale_factors_uninitialized = 1;
static	float	uimxXScale = 1.0;
static	float	uimxYScale = 1.0;

static	int	InitScaleFactors()
{
	float	temp;

#ifdef OL_WIDGETS
	uimxXScale = atof(UxGetAppDefault("oleXScale", "1.0"));
	uimxYScale = atof(UxGetAppDefault("oleYScale", "1.0"));
#endif /* OL_WIDGETS */

#ifdef MOTIF_WIDGETS
	temp = atof(UxGetAppDefault("uimxScale", "1.0"));
	uimxXScale = temp * atof(UxGetAppDefault("uimxXScale", "1.0"));
	uimxYScale = temp * atof(UxGetAppDefault("uimxYScale", "1.0"));
#endif /* MOTIF_WIDGETS */
	scale_factors_uninitialized = 0;
}



int UxScaleUimxX(width)
	int		width;
{
	if (scale_factors_uninitialized)
		InitScaleFactors();
	return (width*uimxXScale + 0.5);
}

int UxScaleUimxY(height)
	int		height;
{
	if (scale_factors_uninitialized)
		InitScaleFactors();
	return (height*uimxYScale + 0.5);
}

#ifdef OL_WIDGETS
/*****************************************************************************
NAME:		void		UxAnnounceResourceFiles( verboseFlag )

INPUT:		Boolean		verboseFlag	- TRUE or FALSE

DESCRIPTION:	When verboseFlag is TRUE, the resource filenames
		will be announced when they are read at initialization
		time. If FALSE, they will not be announced. Note
		that the static variable announceResourceFiles defaults to
		FALSE.

		This function may be called during the startup procedure
		of ole and uimx.

LAST REV:	May 3/90
-----------------------------------------------------------------------------*/

void UxAnnounceResourceFiles( verboseFlag )
Boolean verboseFlag;
{
	if ( verboseFlag == TRUE || verboseFlag == FALSE )
		announceResourceFiles = verboseFlag;	
}
#endif /* OL_WIDGETS */

/*****************************************************************************
NAME:		int		*UxGetAnyResource(prog, rname, type, value)

INPUT:		char		*prog		- program name
				*rname		- resource name

OUTPUT:		char		**type		- type of resource value
		XrmValue	*value		- pointer to resource value

RETURN:		True if resource is found, False otherwise.

DESCRIPTION:	Given a program 'name' and a resource name, get the 
		associated resource from the resource database.
		Handles any type of resource.

LAST REV:	March 15, 1991		See bugfix 2111.
-----------------------------------------------------------------------------*/

int	UxGetAnyResource(prog, rname, type, value)
	char		*prog, *rname, **type;
	XrmValue	*value;
{
	dstring		ds, ds2;
	int		rc;
	extern char	*UxProgramClassName;
				
	ds= dcreate(prog);
	dappend(ds, ".");
	dappend(ds, rname);

	ds2 = dcreate(UxProgramClassName);
	dappend(ds2, ".");
	dappend(ds2, rname);

	rc = XrmGetResource(XTDATABASE, dnstr(ds), dnstr(ds2), type, value);

	dfree(ds);
	dfree(ds2);

	return(rc);
}

/*****************************************************************************
NAME:		void		UxPutAnyResource(prog, rname, type, value)

INPUT:		char		*prog		- program name or class name
				*rname		- resource name
				*type		- type of value
		XrmValue	*value		- value for resource

DESCRIPTION:	Given a program 'name' and a resource name, put the 
		associated resource into the resource database.
		Handles any type of resource.

LAST REV:	March 15, 1991		See bugfix 2111.
-----------------------------------------------------------------------------*/

void UxPutAnyResource(prog, rname, type, value)
	char		*prog, *rname, *type;
	XrmValue	*value;
{
	dstring		ds;
				
	ds = dcreate(prog);
	dappend(ds, "*");	/* Use '*' because rname may not be */
	dappend(ds, rname);	/* fully specified */

	XrmPutResource(&XTDATABASE, dnstr(ds), type, value);

	dfree(ds);
}
