
/*
 * This program is under the GNU GPL.
 * Use at your own risk.
 *
 * A simple water effect
 * written by Antonio Campos Hernndez (acampos@ceronet.com)
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <GL/glut.h>
/* Replace the following path with the place you store the 3Dfx demos of the Mesa distribution */
#include "image.c"

#ifdef XMESA
#include <GL/xmesa.h>
int fullscreen=1;
#endif

#define FRAME           50    /* Number of frames to show fps */
#define M_LENGTH        60    /* Number of QUAD per row or per column */

#define ALT             0.02  /* Altitude of elevated boxes */
#define INC_ANG_CIRCLE  0.06; /* Angular velocity for the animated focus */

#define INC_ANG_X       1     /* Acceleration for the rotation of the Grid */
#define INC_ANG_Y       1

GLdouble water[M_LENGTH][M_LENGTH][3]; /* Water has tree stages: current, velocity, after integration */
GLdouble ang_x=50,ang_y=0,inc_ang_x=0,inc_ang_y=0; /* variables related to the rotation of the grid */
int current=0; /* Current stage of water */
int px=0,py=0; /* Position of water distorsion focus */
GLdouble ang_circle=0; /* The angle of the animated focus */
IMAGE *texture;  /* guess yourself */

int help=1;     /*Show help or not */
int fog=1;      /*Show fog or not */
int anim_circle=0;  /* Animated focus or not */
int use_texture=1;  /* Use texture or not */

void printstring(void *font, char *string)
{
  int len,i;

  len=(int)strlen(string);
  for(i=0;i<len;i++)
    glutBitmapCharacter(font,string[i]);
}

void printhelp(void)
{
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
  glColor4f(0.0,0.0,0.0,0.5);
  glRecti(40,40,600,440);

  glColor3f(1.0,0.0,0.0);
  glRasterPos2i(300,420);
  printstring(GLUT_BITMAP_TIMES_ROMAN_24,"Help");

  glRasterPos2i(60,390);
  printstring(GLUT_BITMAP_TIMES_ROMAN_24,"h - Toggle Help");
  glRasterPos2i(60,360);
  printstring(GLUT_BITMAP_TIMES_ROMAN_24,"t - Toggle Textures");
  glRasterPos2i(60,330);
  printstring(GLUT_BITMAP_TIMES_ROMAN_24,"f - Toggle Fog");
  glRasterPos2i(60,300);
  printstring(GLUT_BITMAP_TIMES_ROMAN_24,"m - Toggle circular focus");
  glRasterPos2i(60,270);
  printstring(GLUT_BITMAP_TIMES_ROMAN_24,"i, k, j, l - Rotate Grid");
  glRasterPos2i(60,240);
  printstring(GLUT_BITMAP_TIMES_ROMAN_24,"Arrow Keys - Move Focus");
  glRasterPos2i(60,210);
  printstring(GLUT_BITMAP_TIMES_ROMAN_24,"a - Drop a random drop");
  glRasterPos2i(60,180);
  printstring(GLUT_BITMAP_TIMES_ROMAN_24,"Space - Toggle fullscreen mode");
  glRasterPos2i(60,120);
  printstring(GLUT_BITMAP_TIMES_ROMAN_24,"Esc - Leave. Oh! no :^(");
  glDisable(GL_BLEND);
}

double aleatorio(double rango) /* return a random number between 0 and rango */
{
  return rango*(rand()/(RAND_MAX+1.0));
}

/* elevate a box at [x,y,x+w,y+h] over the layer "layer" of the grid */
void draw_box(int x,int y,int w,int h,GLdouble height,int layer)
{
  int i,j;
  
  for (j=y; j<=y+h; j++)
    for (i=x; i<=x+w; i++)
      water[i%M_LENGTH][j%M_LENGTH][layer]+=height;
}

/* Make the water become less wild */
void calm_water(int layer)
{
int i,j;

  for (i=0; i<M_LENGTH; i++)
    for (j=0; j<M_LENGTH; j++)
      {
        water[i][j][layer]/=1.005;
      }
}

/* Make the integration step over the water */
void calculate_water(void)
{
int first=current,second=(current+1)%3,last=(current+2)%3;
int i,j;
  for (i=1; i<M_LENGTH-1; i++)
    for (j=1; j<M_LENGTH-1; j++)
      {
        water[i][j][last]=-water[i][j][first]+
        0.5*(water[i-1][j][second]+water[i+1][j][second]+
             water[i][j-1][second]+water[i][j+1][second]);
      }
}

void Idle( void )
{
   /* update animation vars */

   ang_x+=inc_ang_x;
   ang_y+=inc_ang_y;
   if (anim_circle)
   {
     draw_box(M_LENGTH/2+(int)(M_LENGTH/3*cos(ang_circle)),
              M_LENGTH/2+(int) (M_LENGTH/3*sin(ang_circle)),5,5,ALT,current);
     ang_circle+=INC_ANG_CIRCLE;
   }
   calculate_water();
   calm_water((current+1)%3);
   calm_water((current+2)%3);
   current=(current+1)%3;

   glutPostRedisplay();
}

float gettime(void)
{
  static clock_t told=0;
  clock_t tnew,ris;

  tnew=clock();

  ris=tnew-told;

  told=tnew;

  return(ris/(float)CLOCKS_PER_SEC);
}

void Draw( void )
{
int i,j;
static int count=0;
static char frbuf[80]="";
float fr;
 
   glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

   glShadeModel(GL_SMOOTH);
   if (fog)
     glEnable(GL_FOG);
   else
     glDisable(GL_FOG);

   if (use_texture)
     glEnable(GL_TEXTURE_2D);
   else
     glDisable(GL_TEXTURE_2D);

   glPushMatrix();
   /* draw stuff here */
   glRotated(ang_x,1,0,0); 
   glRotated(ang_y,0,1,0);
   glScaled(3,1,3);
   glTranslated(-.5,0,.5);
   for (j=1; j<M_LENGTH-1; j++)
   {
     glBegin(GL_TRIANGLE_STRIP);
     for (i=1; i<M_LENGTH-1; i++)
     {
       glColor3d(water[i][j][current]+1,water[i][j][current]+1,water[i][j][current]+1);
       glTexCoord2d((GLfloat) i/M_LENGTH+((water[i][j][current]-water[i+1][j][current])/(1.0/M_LENGTH))/35,
       (GLfloat) j/M_LENGTH+((water[i][j][current]-water[i][j+1][current])/(1.0/M_LENGTH))/35);
       glVertex3d(i*(1.0/M_LENGTH),water[i][j][current],-j*(1.0/M_LENGTH));

       glColor3d(water[i][j+1][current]+1,water[i][j+1][current]+1,water[i][j+1][current]+1);
       glTexCoord2d((GLfloat) i/M_LENGTH+((water[i][j+1][current]-water[i+1][j+1][current])/(1.0/M_LENGTH))/35,
       (GLfloat) (j+1)/M_LENGTH+((water[i][j+1][current]-water[i][j+2][current])/(1.0/M_LENGTH))/35);
       glVertex3d(i*(1.0/M_LENGTH),water[i][j+1][current],-(j+1)*(1.0/M_LENGTH));
     }
     glEnd();
   }

  glPopMatrix();

  if((count % FRAME)==0) {
    fr=gettime();
    sprintf(frbuf,"Frame rate: %.2f",FRAME/fr);
  }

  glDisable(GL_TEXTURE_2D);
  glDisable(GL_FOG);
  glShadeModel(GL_FLAT);

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(-0.5,639.5,-0.5,479.5,-1.0,1.0);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  glColor3f(1.0,0.0,0.0);
  glRasterPos2i(10,10);
  printstring(GLUT_BITMAP_HELVETICA_18,frbuf);
  glRasterPos2i(350,470);
  printstring(GLUT_BITMAP_HELVETICA_10,"Water Written by Antonio Campos (acampos@ceronet.com)");

  if(help)
    printhelp();

  glPopMatrix(); /* GL_MODELVIEW */
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  glutSwapBuffers();

  count++;
}


void Reshape( int width, int height )
{
   glViewport( 0, 0, width, height );
   glMatrixMode( GL_PROJECTION );
   glLoadIdentity();
   glFrustum( -.5, .5, -.5, .5, .5, 5.0 );
   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity();
   glTranslatef( 0, 0, -2 );
}


void Key( unsigned char key, int x, int y )
{
   switch (key) {
      case 27:
         exit(0);
         break;
      case 'a':
      case 'A':
         draw_box((int) aleatorio(M_LENGTH),(int) aleatorio(M_LENGTH),
         (int) aleatorio(10),(int) aleatorio(10),aleatorio(ALT*5),current);
         break;
#ifdef XMESA
      case ' ':
         fullscreen=!fullscreen;
         XMesaSetFXmode(fullscreen ? XMESA_FX_FULLSCREEN : XMESA_FX_WINDOW);
      break;
#endif
      case 'm':
      case 'M':
         anim_circle=!anim_circle;
         break;
      case 'h':
      case 'H':
         help=!help;
         break;
      case 'f':
      case 'F':
         fog=!fog;
         break;
      case 't':
      case 'T':
         use_texture=!use_texture;
         break;
      case 'i':
      case 'I':
          inc_ang_x+=INC_ANG_X;
          break;
      case 'k':
      case 'K':
          inc_ang_x-=INC_ANG_X;
          break;
      case 'j':
      case 'J':
          inc_ang_y-=INC_ANG_Y;
          break;
      case 'l':
      case 'L':
          inc_ang_y+=INC_ANG_Y;
          break;
   }
   glutPostRedisplay();
}


void SpecialKey( int key, int x, int y )
{
   switch (key) {
      case GLUT_KEY_DOWN:
         draw_box(px,py,5,5,ALT,(current+0)%3);
         py--;
         if (py<0) py=M_LENGTH-1;
         break;
      case GLUT_KEY_UP:
         draw_box(px,py,5,5,ALT,(current+0)%3);
         py++;
         py%=M_LENGTH;
         break;
      case GLUT_KEY_LEFT:
         draw_box(px,py,5,5,ALT,(current+0)%3);
         px--;
         if (px<0) px=M_LENGTH-1;
         px%=M_LENGTH;
         break;
      case GLUT_KEY_RIGHT:
         draw_box(px,py,5,5,ALT,(current+0)%3);
         px++;
         px%=M_LENGTH;
         break;
   }
   glutPostRedisplay();
}


void Init( void )
{
  GLfloat fog_color[]={.4,.3,.8,1};
  GLenum gluerr;

  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LESS);

  glPixelStorei(GL_UNPACK_ALIGNMENT,4);
  if (gluerr=gluBuild2DMipmaps(GL_TEXTURE_2D, 3, texture->sizeX, texture->sizeY, GL_RGB,
                               GL_UNSIGNED_BYTE, (GLvoid *)(texture->data)))
  {
      fprintf(stderr,"GLULib%s\n",gluErrorString(gluerr));
      exit(-1);
  }

  glFogi(GL_FOG_MODE,GL_LINEAR);
  glFogfv(GL_FOG_COLOR,fog_color);
  glClearColor(fog_color[0],fog_color[1],fog_color[2],fog_color[3]);
  glFogf(GL_FOG_START,1);
  glFogf(GL_FOG_END,2.7);

  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);          
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);  

  /*  glPolygonMode(GL_FRONT,GL_LINE);*/
}



int main( int argc, char *argv[] )
{
   fprintf(stderr,"Water By Antonio Campos (acampos@ceronet.com)\n");
   glutInitWindowSize( 640, 480 );

   glutInitDisplayMode( GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH );

   glutInit( &argc, argv );

   glutCreateWindow("A simple water effect...");

   
   if (argc!=2) 
   {
     fprintf(stderr,"Usage: %s [RGB image] \n",argv[0]);
     exit(0);
   }
   
   if (!(texture=ImageLoad(argv[1])))
   {
     fprintf(stderr,"Error loading texture: %s\n",argv[1]);
     exit(0);
   }
   
   Init();

   glutReshapeFunc( Reshape );
   glutKeyboardFunc( Key );
   glutSpecialFunc( SpecialKey );
   glutDisplayFunc( Draw );
   glutIdleFunc( Idle );

   glutMainLoop();
   return 0;
}
