/*
* Recreate packet stream recorded with rtpdump -w.
* Does not use RTP timestamps, but rather recreates arrival time
* sequence. Currently does not correct SR/RR absolute (NTP) timestamps,
* but should. Receiver reports are fairly meaningless.
*
* (c) 1994-1995 Henning Schulzrinne, GMD Fokus
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>    /* gettimeofday() */
#include <sys/socket.h>  /* struct sockaddr */
#include <netinet/in.h>
#include <arpa/inet.h>   /* inet_ntoa() */
#include <time.h>
#include <stdio.h>       /* stderr, printf() */
#include <string.h>
#include <stdlib.h>      /* perror() */
#include <unistd.h>      /* write() */
#include "notify.h"      /* notify_start(), ... */
#include "types.h"
#include "rtpdump.h"     /* RD_packet_t */
#include "multimer.h"    /* timer_set() */
#include "ansi.h"

static char rcsid[] = "$Id: rtpplay.c,v 1.2 1995/08/30 11:19:01 hgs Exp $";
static int verbose = 0;
static FILE *in = stdin;
static int sock[2];  /* output sockets */
RD_buffer_t buffer;

static void usage(char *argv0)
{
  fprintf(stderr, "Usage: %s destination/port[/ttl]\n", argv0);
  exit(1);
} /* usage */


static double tdbl(struct timeval *a)
{
  return a->tv_sec + a->tv_usec/1e6;
}


/*
* Transmit RTP/RTCP packet on output socket.
*/
static void play_transmit(void)
{
  if (write(sock[buffer.p.hdr.plen == 0], buffer.p.data, buffer.p.hdr.length) < 0) {
    perror("write");
  } 
} /* play_transmit */


/*
* Timer handler: read next record from file and insert into timer
* handler.
*/
static Notify_value play_handler(Notify_client client)
{
  static int first = -1;        /* offset of first packet */
  static struct timeval start;  /* generation time of first played back p. */
  struct timeval now;           /* current time */
  struct timeval next;          /* next packet generation time */

  gettimeofday(&now, 0);
  if (first < 0) {
    RD_read(in, &buffer);
    first = buffer.p.hdr.offset;
    start = now;
  }

  play_transmit();
  RD_read(in, &buffer);

  /* compute next playout time */
  buffer.p.hdr.offset -= first;
  next.tv_sec  = start.tv_sec  + buffer.p.hdr.offset/1000;
  next.tv_usec = start.tv_usec + (buffer.p.hdr.offset%1000) * 1000;
  if (next.tv_usec > 1000000) {
    next.tv_usec -= 1000000;
    next.tv_sec  += 1;
  }

  if (verbose > 0) {
    printf("%1.3f %s(%d;%d) t=%ld\n",
      tdbl(&now), buffer.p.hdr.plen ? "RTP" : "RTCP ",
      buffer.p.hdr.length, buffer.p.hdr.plen, buffer.p.hdr.offset);
  }

  timer_set(&next, play_handler, 0, 0);
  return NOTIFY_DONE;
} /* play_handler */


int main(int argc, char *argv[])
{
  char ttl = 16;
  static struct sockaddr_in sin;
  int i;
  int c;
  char *host;
  extern char *optarg;
  extern int optind;
  extern int hpt(char *h, struct sockaddr *sa, unsigned char *ttl);

  /* parse command line arguments */
  while ((c = getopt(argc, argv, "v")) != EOF) {
    switch(c) {
    case 'v':
      verbose = 1;
      break;
    case '?':
    case 'h':
      usage(argv[0]);
      break;
    }
  }

  if (optind < argc) {
    if (hpt(host, (struct sockaddr *)&sin, &ttl) < 0) {
      usage(argv[0]);
    }
  }

  /* read header of input file */
  if (RD_header(stdin, &sin, 0) < 0) {
    fprintf(stderr, "Invalid header\n");
    exit(1);
  }

  /* create/connect sockets */
  for (i = 0; i < 2; i++) {
    sock[i] = socket(PF_INET, SOCK_DGRAM, 0);
    if (sock[i] < 0) {
      perror("socket");
      exit(1);
    }
    sin.sin_port = htons(ntohs(sin.sin_port) + i);

    if (connect(sock[i], (struct sockaddr *)&sin, sizeof(sin)) < 0) {
      perror("connect");
      exit(1);
    }

    if (IN_CLASSD(sin.sin_addr.s_addr) && 
        (setsockopt(sock[i], IPPROTO_IP, IP_MULTICAST_TTL, &ttl,
                   sizeof(ttl)) < 0)) {
      perror("IP_MULTICAST_TTL");
      exit(1);
    }
  }

  play_handler(0);
  notify_start();
  return 0;
} /* main */
