/** This class defines the canvas area that represents the
  * range of pan and tilt motion available.  The vertical axis
  * corresponds to tilt and the horizontal axis to pan.  A user
  * can click anywhere on the canvas to request that the camera
  * move to that pan/tilt location.
  */

package CamClnt;

import java.awt.*;
import java.awt.event.*;

public class panTiltArea extends Canvas implements MouseListener {

  // define the class variables
    private boolean drawing = false;
    private int      px, py, pw, ph;
    private CamComm  camcomm;
    private double   maxpanspeed = 0.95;
    private double   maxtiltspeed = 0.95;
    private double   panmin = -100.0;
    private double   panmax = 100.0;
    private double   tiltmin = -25.0;
    private double   tiltmax = 25.0;
    private int      rectx = 2, recty = 2, rectwid = 200, recthght = 100;

  // define the class constructor and start listening for mouse clicks
    panTiltArea ( CamComm comm ) {
        super ();
	addMouseListener( this );
	camcomm = comm;
    }

  /** overriding getMinimumSize assures that this component will be big enough to
      be useable.
      */
  public Dimension getMinimumSize() {
      return new Dimension( 204, 105 );
  }
  /** overriding getPreferredSize assures that this component will be big enough to
      be useable when pack encounters this component.
      */
  public Dimension getPreferredSize() {
      return new Dimension( 204, 105 );
  }

  /** method called when the pan/tilt area needs to be redrawn
    */
    public void paint (Graphics g) {
      //g.drawString ("pan/tilt", 20, 115);
	g.setColor( Color.white );
	g.fill3DRect( rectx, recty, rectwid, recthght, true );
	g.setColor( Color.black );
	g.drawString ("<-- pan -->", 70, 13 );
	g.drawString ("^", 5, 18 );
	g.drawString ("|", 8, 23 );
	g.drawString ("t", 6, 38 );
	g.drawString ("i", 7, 50 );
	g.drawString ("l", 7, 63 );
	g.drawString ("t", 6, 74 );
	g.drawString ("|", 8, 88 );
	g.drawString ("v", 6, 93 );
	if( drawing == true ) {
	  g.drawRect(px, py, pw, ph );
	}
    }
  /** method called every time the mouse is pressed in the pan/tilt area
    */
    public void mousePressed (MouseEvent e) {
      drawing = false;
    }
  /** method called every time the mouse is released in the pan/tilt area
    */
    public void mouseReleased (MouseEvent e) {
        String temp;
        if( e.isMetaDown())
	  {
	  // Send a relative rather than an absolute command because
	  // the right mouse button was pressed.
	  double curtilt = -1.0 * (double)(e.getY() - py) * 2.0 * tiltmax / (double)recthght;
	  double curpan = (double)(e.getX() - px) * 2.0 * panmax / (double)rectwid;
	  temp = "tilt R " + curtilt + " " + maxtiltspeed +
	                "@pan R " + curpan + " " + maxpanspeed;
 	  System.out.println("Relative move of tilt = " + curtilt + " pan = " + curpan + " requested");
	  }
	else
	  {
	    // send an absolute command for the camera to move to the requested location
	  double curtilt = -1.0 * (double)(e.getY() - recty - (recthght / 2.0)) * 2.0 * tiltmax / (double)recthght;
	  double curpan  = (double)(e.getX() - rectx - (rectwid / 2.0 )) * 2.0 * panmax / (double)rectwid;
	  temp = "tilt A " + curtilt + " " + maxtiltspeed +
	                "@pan A " + curpan + " " + maxpanspeed;
	  }
	// send the command to move the camera
	camcomm.writeCamString( temp );

	// set the variables that define where to draw the little square showing
	// where the camera is at the moment.
	px = e.getX();
	py = e.getY();
	pw = 2;
	ph = 2;
	drawing = true;
	// redraw the area so that the rectangle will show up
        repaint ();
    }

  // the next three methods are only defined because the interface
  // definition requires them.  They do not in fact do anything when they 
  // are called.
  /** place holder method.  Not used */
    public  void mouseEntered (MouseEvent e) {
      //System.out.println( "Mouse entered" );
    }
  /** place holder method.  Not used */
    public  void mouseExited (MouseEvent e) {
      //System.out.println( "Mouse exited" );
      //drawing = false;
    }
  /** place holder method.  Not used */
    public  void mouseClicked (MouseEvent e) {
      //System.out.println( "Mouse clicked" );
    }

    // update the position of the camera shown in the pan/tilt area
    public void updatePosition( double pan, double tilt ) {
	
	py =  recty + (recthght / 2) - (int)(tilt * (double)recthght/(2.0 * tiltmax));
	px = rectx + (rectwid / 2 ) + (int)(pan * (double)rectwid/(2.0 * panmax));
	drawing = true;
	repaint();
    }

}


