// This file contains the definition of the camera control interface
// itself.  It is the top object of the heirarchy of the interface.  The
// main() procedure is in this object.

package CamClnt;

import java.awt.*;
import java.io.*;
import java.awt.event.*;
import java.lang.*;
import java.applet.*;
import java.util.*;
import java.net.*;

/**  This class defines the user interface for the client of the remote control 
  *  camera system.  It is the top-level class of the interface.
  */

public class camInterface extends Frame implements ActionListener
{
  // THE FOLLOWING VARIABLES PROVIDE DEFAULT VALUES FOR SOME THINGS
  // THAT YOU MIGHT WANT TO CUSTOMIZE TO YOUR SITE
  String curaddr = null;                       // default address for devserv

  // Number of cameras to assume if no devserv answers initial query and no 	
  // argument is supplied when the interface is invoked.  
  private static final int defaultnumcams = 3; 
  private static final boolean HAVEJMF = false;  // compile in the JMF players

  // compile in Akenti Authorization
  private static final boolean USE_AKENTI = false; 
  // compile in use of CIF for communication
  private static final boolean USE_CIF = false; 
  // END SITE CUSTOMIZATION SECTION

  // variables defined for the class
  camSelect viewCam, pnpCam, moveCam; // camera selectors
  mycamPanel camPanel;         // layout panel that holds the camera selectors
  String devservmultaddr = "224.35.36.37"; // default multicast address
  int  numvideoswitches = 1; // The number of video switchers (default=1)
  CamComm comm; // The communication channels for talking to the devserv
  DevservStatus multisend; // multicast channel for sending requests
  DevservStatus statcomm; // multicast channel for listening to devserv status

  // interface field that allows users to type in a new devserv address
  TextField addressField; 
  zoomSlider zoomAbs; // The absolute zoom slider
  pictureArea pict; // The interface area that contains the video image
  int  numcams; // The number of cameras in the system 
  String      cname = null;     // cname of the video stream we are using
  private static final boolean DEBUG = false; // to get debugging printouts
  PopupMenu videoRTP;
  PopupMenu audioRTP;
  MyPopup   devservMenu;
  panTiltArea  pantilt;

  /** Constructor for the camInterface object.  It sets the size of the
    * interface according to the number of cameras in the system.
    */
    camInterface( int value, String dsaddr ) {
        super();
	numcams = value;
	curaddr = dsaddr;
	if( curaddr == null )
	    curaddr = "localhost";

	/* Open the communication channels to the device server.  The UDP 
           connection will be used to send commands and the multicast connection
 	   will be used to receive status information and query responses from 
           the device server.
 	*/

	try {
	  comm = new CamComm(this, curaddr, 5555, "UDP", USE_AKENTI, USE_CIF);
	  statcomm = new DevservStatus(this, devservmultaddr, 5556, "UUM");
	  multisend = new DevservStatus(this, devservmultaddr, 5558, "UUM");
	  // Query all devservs to discover who's out there.
	  multisend.writeDevString("devserv description");
	  int x = statcomm.readDevString(500, false);
	  while (x>0)
	      x = statcomm.readDevString(500, false);
	}
	catch(IOException e) { System.err.println(e); }

	// create the area that contains the video image
	pict =  new pictureArea(comm);
	// create the pan/tilt absolute selection box
	pantilt = new panTiltArea (comm);
	// create the zoom control buttons and slider
	Button zoomIn = new Button("Zoom In");
	Button zoomOut = new Button( "Zoom Out" );
	zoomAbs = new zoomSlider(comm);

	// create the field used to specify the device server address and 
        // initialize it to the default address
	Panel devservAddr = new Panel();
	devservAddr.setLayout(new BorderLayout());
	devservAddr.add("West", new Label("Server Address"));
	
	addressField = new TextField(27);
	addressField.setText(curaddr);
	addressField.setBackground(Color.white);
	addressField.setBounds( 228, 168, 156, 28 );
	addressField.setFont(new Font("Dialog", Font.BOLD, 12));
	devservAddr.add("East", addressField);

	// Get the hashtable that contains DevservInfo objects.
	// The DevservInfo objects contain the number of cameras, etc.
	// for configuring the GUI to fit the devserv host's description.  
	Hashtable devservHash = statcomm.getDevservHash();
	Button servSelect = new Button("Select");
	devservMenu = new MyPopup(devservHash);
	servSelect.add(devservMenu);
	devservAddr.add("Center", servSelect);

	// define the action that occurs when the server address selection 
        // button is pressed
	servSelect.setActionCommand("servaddr");
	servSelect.addActionListener(this);
	    
	// define the layout type for the interface and start adding things to 
        // the interface
	setLayout(new BorderLayout(5, 10));
        add("East", pict);
 
	/* Create the camera selection menus.  These allow the user to choose 
	   which camera they are viewing, moving, and which camera is in the 
           picture-in-picture.  If the user specified a server on the command
	   line, get the DevservInfo object for that host and use its number of
	   cameras.  If the current devserv is "localhost" or we don't have
	   information for the specified host, use the default number of 
	   cameras. 
        */  
	if (!curaddr.equals("localhost"))
	{
	   DevservInfo devInfo = statcomm.getDevservInfo(curaddr);
	   if (devInfo != null)
	   {
	   	numcams = devInfo.getNumberDevices("camera", 3);		
		numvideoswitches = devInfo.getNumberDevices("videoswitch",3);
	   }
	}
	viewCam = new camSelect(numcams, "view", comm);
	pnpCam = new camSelect(numcams, "pnp", comm);
	moveCam = new camSelect(numcams, "move", comm);
	
	// add the camera selection menus to the layout
	camPanel = new mycamPanel();
	add("North", camPanel);

	// add the zoom slider and buttons and the device server address field
	Panel e = new Panel();
	e.setLayout(new BorderLayout());
	e.add("East", zoomIn);
	e.add("West", zoomOut);
	e.add("Center", zoomAbs);

	// if we have the JMF installed add buttons for launching the video 
        // and audio
	if( HAVEJMF ) {
	    // Create the video and audio display buttons
	    Button videoDisp = new Button( "Display Video" );
	    //videoRTP = new RTPStreamMenu( null, 0, 0, "video", comm );
	    videoRTP = new RTPStreamMenu("224.13.16.13",22334,1,"video", comm );
	    videoDisp.add( videoRTP );
	    Button audioDisp = new Button( "Display Audio" );
	    audioRTP = new RTPStreamMenu( null, 0, 0, "audio", comm );
	    audioDisp.add( audioRTP );

	    // define the action that occurs when the video display button is 
            // pressed
	    videoDisp.setActionCommand( "video" );
	    videoDisp.addActionListener( this );
	    
	    // define the action that occurs when the audio display button is 
            // pressed
	    audioDisp.setActionCommand( "audio" );
	    audioDisp.addActionListener( this );

	    Panel rtp = new Panel();
	    rtp.setLayout( new BorderLayout() );
	    rtp.add( "West", videoDisp );
	    rtp.add( "East", audioDisp );
	    rtp.add( "South", devservAddr );
	    e.add( "South", rtp );
	}
	else {
	    e.add( "South", devservAddr );
	}
	// define the action that occurs when the zoom buttons are pressed
	zoomIn.setActionCommand( "in" );
	zoomOut.setActionCommand( "out" );
	zoomIn.addActionListener( this );
	zoomOut.addActionListener( this );
	add("South", e );

	// color the interface something other than grey
	setBackground( Color.cyan );
	setForeground( Color.blue );

	// add a title bar menu to the interface
	CamMenu menu = new CamMenu( );
	setMenuBar( menu );
	enableEvents( AWTEvent.WINDOW_EVENT_MASK );

	// add the pan/tilt absolute selection box to the layout
	Panel ptPanel = new Panel();
	ptPanel.setLayout( new BorderLayout() );
	ptPanel.add( "Center", pantilt );
	add("West", ptPanel );

	// Let Java go through and figure out sizing and layout stuff
	validate();
	pack();

        // Create a thread to listen for devserv replies.
	Thread dssthread = new Thread((Runnable)statcomm);
	dssthread.start();
 
	// make the interface visible
	show();

	// code for closing the window
	this.addWindowListener( new WindowAdapter() {
	    public void windowClosing( WindowEvent e ) {
		String reason = "Shutdown of CamClnt";
		//camImage.closeRTP( reason );
	    }
	});
    }


  /** This is the method called when the camera control interface is invoked. 
    * It expects a single argument defining the number of cameras in the system.
    */
    public static void main (String [] args) 
    {
	Integer temp;
	String tempdsaddr;
	if( args.length > 0 ) {
	    tempdsaddr = args[0];
	}
	else
	    tempdsaddr = null;
	if( args.length > 1 ) {
	    temp = new Integer(args[0]);
	    if (DEBUG)
		System.out.println("Num cams is " + temp);
	}
	else
	    temp = new Integer(defaultnumcams);
        camInterface f = new camInterface(temp.intValue(), tempdsaddr);
	f.setLocation(20, 40);
    }

  /** called when the camera interface needs to be redrawn.
    */
    public void paint (Graphics g) {
        g.translate (getInsets().left, getInsets().top);
    }

  /** returns the current camera selected for moving.
    * All move comands will be for the currently selected camera for moving.
    * this method returns the number of that camera.
    */
    public int getCurrentCamera() {
        return moveCam.getCurrentCamera();
    }

  /** returns the current camera selected to be in the picture-in-picture.
    * The current camera selected to be in the smaller picture in
    * picture-in-picture is returned by this method.
    */

    public int getPnpCamera() {
        return pnpCam.getCurrentCamera();
    }

  /** internal method called when the zoom buttons are pressed on the interface,
    */
    public void actionPerformed( ActionEvent e ) {
        if (DEBUG)
	  System.out.println( "the zoom command " + e.getActionCommand());
	if( e.getActionCommand().equalsIgnoreCase( "in" ))
	  {
	    // send a request to zoom in by 1
	    comm.writeCamString( "zoom R 1.0" );
	    zoomAbs.modifyZoom( 1 );
	  }
	else if( e.getActionCommand().equalsIgnoreCase( "out" ))
	  {
	    // send a request to zoom out by 1
	    comm.writeCamString( "zoom R -1.0" );
	    zoomAbs.modifyZoom( -1 );
	  }
	else if( HAVEJMF && e.getActionCommand().equalsIgnoreCase( "video" ))
	  {
	    // it is a request to launch video
	    System.out.println( "video button pressed");
	    videoRTP.show( (Component)e.getSource(), 0 , 0  );
	  }
	else if (HAVEJMF && e.getActionCommand().equalsIgnoreCase( "audio" ) )
	 {
	     // it is a request to launch audio
	     if (DEBUG)
	        System.out.println("audio button pressed");
	     audioRTP.show( (Component)e.getSource(), 0 , 0  );
	 }
	else if (e.getActionCommand().equalsIgnoreCase("servaddr")) 
        {
	     if (DEBUG)	
	        System.out.println("Address selection button pressed");
	     devservMenu.show((Component)e.getSource(), 0, 0);
	}
    }

    // look for a window close event and close things cleanly
    protected void processWindowEvent(WindowEvent e) 
    {
	if (e.getID() == WindowEvent.WINDOW_CLOSING) 
	{
	    super.processWindowEvent(e);
	    System.exit(0);
	}
	else {
	    super.processWindowEvent(e);
	}
    }

    public void reconfigure( DevservInfo serv ) {
	if (serv == null) {
	  if (DEBUG)
	       System.out.println("Null server passed to reconfigure???");
	  return;
	}
	String servName = serv.getServerName();
	if (servName == null) {
	    if (DEBUG)
		System.out.println("Null server name in reconfigure??? ");
	    return;
	}

	if (DEBUG)
	    System.out.println( "servername = " + servName + "addrfield = " 
		+ addressField.getText());
	devservMenu.addParticipant(servName);
	if ((addressField.getText()).equals(servName)) {
	    camPanel.cpReconfigure(serv.getNumberDevices("camera",3), 
		serv.getNumberDevices("videoswitch", 3));
	    if (HAVEJMF) {
		// launch the appropriate RTPPlayer

		((RTPStreamMenu)videoRTP).setMediaInfo(serv.getMediaAddr(), serv.getMediaPort(), serv.getMediaTTL());
		((RTPStreamMenu)videoRTP).reLaunchRTP( serv.getMediaHost());
	    }
	}
    }

    public void updateStatus( DevservInfo serv ) {
	try{
	    InetAddress temp1 = InetAddress.getByName( addressField.getText());
	    InetAddress temp2 = InetAddress.getByName( serv.getServerName() );
	    if( (temp1.getHostAddress()).equals( temp2.getHostAddress())) {
		DevservInfo.DeviceInfo device = serv.getDevice("camera", getCurrentCamera());
		DevservInfo.DegreeOfFreedom pandof = device.getDOF( "pan" );
		DevservInfo.DegreeOfFreedom tiltdof = device.getDOF( "tilt" );
		DevservInfo.DegreeOfFreedom zoomdof = device.getDOF( "zoom" );
		pantilt.updatePosition( pandof.getCurrentPosn(), tiltdof.getCurrentPosn());
	    }
	}
	catch (UnknownHostException e ){
	    System.out.println( "Unknown host exception caught");
	}
    }

    protected class mycamPanel extends Panel {

	Label viewlbl, movelbl, pnplbl;
        GridLayout gc;
        int origvgap, compressedvgap = -10;

	mycamPanel() {
	
	    viewlbl = new Label( "Camera to View" );
	    movelbl = new Label( "Camera to Move" );
	    pnplbl = new Label( "Camera inside Pnp" );
	    // add the camera selection menus to the layout
	    gc = new GridLayout( 1, 2);
	    origvgap = gc.getVgap();
	    if( numcams > 1 || numvideoswitches > 0 )
		gc.setVgap( compressedvgap );
	    setLayout( gc );
	    if( numcams > 1 ) {
		add( movelbl );
		add( moveCam );
	    }
	    if( numvideoswitches > 0 ) {
		gc.setRows( 3 );
		add( viewlbl );
		add( viewCam );
		add( pnplbl );
		add( pnpCam );
	    }
	    //camPanel.validate();
	}

	protected void cpReconfigure(int newnumcams, int newnumvideoswitches) {

	    if (numcams != newnumcams) {
		if (DEBUG)
		   System.out.println("Reconfigure camera number from " 
			+ numcams + " to " + newnumcams);
		viewCam.changeNumCams(newnumcams);
		moveCam.changeNumCams(newnumcams);
		pnpCam.changeNumCams( newnumcams );
		if( numcams > 1 && newnumcams <= 1 ) {
		    remove( movelbl );
		    remove( moveCam );
		}
		else if( newnumcams > 1 && numcams <= 1 ){
		    add( movelbl );
		    add( moveCam );
		}
	    }
	    if (numvideoswitches != newnumvideoswitches) {
	        if (DEBUG)
		    System.out.println("Reconfiguring no. video switches from " 
			+ numvideoswitches + " to " + newnumvideoswitches );
		if( newnumvideoswitches < 1 && numvideoswitches > 0 ){
		    remove( viewlbl );
		    remove( viewCam );
		    remove( pnplbl );
		    remove( pnpCam );
		    gc.setRows( 1 );
		}
		else if( numvideoswitches < 1 && newnumvideoswitches > 0 ) {
		    add( viewlbl );
		    add( viewCam );
		    add( pnplbl );
		    add( pnpCam );
		    gc.setRows( 3 );
		}
	    }
	    if( newnumcams > 1 || newnumvideoswitches > 0 )
		gc.setVgap( compressedvgap );
	    else
		gc.setVgap( origvgap );
	    numcams = newnumcams;
	    numvideoswitches = newnumvideoswitches;
	    pack();
	}

    }
    protected class MyPopup extends PopupMenu 
	implements ActionListener {

	MyPopup(Hashtable ht) {
	    devservHash = ht;
	    if (devservHash != null)
	    {
		Enumeration e = devservHash.keys(); 
		while (e.hasMoreElements())
		    addParticipant((String)e.nextElement());
	    }
	}

	public void actionPerformed( ActionEvent e ) {
	    if (DEBUG)
	      System.out.println("User selected server " + e.getActionCommand());
	    choiceAction( e.getActionCommand() );
	}

	protected void choiceAction( String name ) {
	    // the user has selected a cname whose stream they wanted played
	    // so we need to start the player
	    if (DEBUG)
		System.out.println("Chose to switch to server " + name);
	    addressField.setText( name );
	    DevservInfo tempserv = statcomm.getDevservInfo( name );
	    reconfigure( tempserv );
	    updateStatus( tempserv );
	}

	public void rePopulate( Hashtable table ) {
	    removeAll();
	    //System.out.println("Need to add all known devserv hosts here");
	}
	
        public void addParticipant(String name) {
	    for( int i = 0; i < getItemCount(); i++ ) {
		if( name.equals((getItem(i)).getLabel())){
		    return;
		}
	    }
	    MenuItem temp;
	    add(temp = new MenuItem(name));
	    temp.addActionListener(this);
	    if (DEBUG)
		System.out.println("addParticipant: Added to the server list " 
			+ name);
	}

	private Hashtable devservHash;
    }
}
