/**
 * \file QAbout.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Kalle Dalheimer
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "QAbout.h"
#include "QAboutDialog.h"
#include "Qt2BC.h"
#include "qt_helpers.h"

#include "controllers/ButtonController.h"
#include "controllers/ControlAboutlyx.h"

#include "support/lstrings.h"

#include <sstream>

#include <qlabel.h>
#include <qpushbutton.h>
#include <qtextcodec.h>
#include <qtextview.h>

using lyx::support::prefixIs;

using std::getline;

using std::istringstream;
using std::ostringstream;
using std::string;

namespace lyx {
namespace frontend {

typedef QController<ControlAboutlyx, QView<QAboutDialog> > base_class;

QAbout::QAbout(Dialog & parent)
	: base_class(parent, _("About LyX"))
{
}


void QAbout::build_dialog()
{
	dialog_.reset(new QAboutDialog);
	connect(dialog_.get()->closePB, SIGNAL(clicked()),
		this, SLOT(slotClose()));

	dialog_->copyright->setText(toqstr(controller().getCopyright()));
	dialog_->copyright->append("\n");
	dialog_->copyright->append(toqstr(controller().getLicense()));
	dialog_->copyright->append("\n");
	dialog_->copyright->append(toqstr(controller().getDisclaimer()));

	dialog_->versionLA->setText(toqstr(controller().getVersion()));

	// The code below should depend on a autoconf test. (Lgb)
#if 1
	// There are a lot of buggy stringstream implementations..., but the
	// code below will work on all of them (I hope). The drawback with
	// this solutions os the extra copying. (Lgb)

	ostringstream in;
	controller().getCredits(in);

	istringstream ss(in.str());

	string s;
	ostringstream out;

	while (getline(ss, s)) {
		if (prefixIs(s, "@b"))
			out << "<b>" << s.substr(2) << "</b>";
		else if (prefixIs(s, "@i"))
			out << "<i>" << s.substr(2) << "</i>";
		else
			out << s;
		out << "<br>";
	}
#else
	// Good stringstream implementations can handle this. It avoids
	// some copying, and should thus be faster and use less memory. (Lgb)
	// I'll make this the default for a short while to see if anyone
	// see the error...
	stringstream in;
	controller().getCredits(in);
	in.seekg(0);
	string s;
	ostringstream out;

	while (getline(in, s)) {
		if (prefixIs(s, "@b"))
			out << "<b>" << s.substr(2) << "</b>";
		else if (prefixIs(s, "@i"))
			out << "<i>" << s.substr(2) << "</i>";
		else
			out << s;
		out << "<br>";
	}
#endif

	// Try and grab the latin1 codec
	QTextCodec * const codec =
		QTextCodec::codecForName("ISO8859-1");
	if (!codec)
		lyxerr << "Unable to find ISO8859-1 codec" << std::endl;

	QString const qtext = codec ?
		codec->toUnicode(out.str().c_str()) :
		toqstr(out.str());
	dialog_->creditsTV->setText(qtext);

	// try to resize to a good size
	dialog_->copyright->hide();
	dialog_->setMinimumSize(dialog_->copyright->sizeHint());
	dialog_->copyright->show();
	dialog_->setMinimumSize(dialog_->sizeHint());

	// Manage the cancel/close button
	bcview().setCancel(dialog_->closePB);
	bc().refresh();
}

} // namespace frontend
} // namespace lyx
