// -*- C++ -*-
/*
 *  Purpose:     A general purpose resizable array.
 *  Author:      Alejandro Aguilar Sierra <asierra@servidor.unam.mx>
 *  Created:     January 1996
 *
 *  Dependencies: None (almost)
 *
 *  Copyright: 1996, Alejandro Aguilar Sierra
 *                 1997  The LyX Team!
 *
 *   You are free to use and modify this code under the terms of
 *   the GNU General Public Licence version 2 or later.
 */

#ifndef MATH_DATA_H
#define MATH_DATA_H

#include <iosfwd>
#include <vector>

#include "math_atom.h"

class MathMacro;
class LaTeXFeatures;
class ReplaceData;


#ifdef __GNUG__
#pragma interface
#endif


/** \class MathArray
    \brief Low level container for math insets

    \author Alejandro Aguilar Sierra
    \author Andr Pnitz
    \author Lars Gullik Bjnnes
    \version February 2001
  */

class MathArray  {
public:
	///
	typedef std::vector<MathAtom>        buffer_type;
	///
	typedef buffer_type::const_iterator  const_iterator;
	///
	typedef buffer_type::iterator        iterator;
	///
	typedef buffer_type::size_type       size_type;
	///
	typedef buffer_type::difference_type difference_type;
	///
	typedef buffer_type::size_type       idx_type;
	///
	typedef buffer_type::size_type       pos_type;

public:
	///
	MathArray();
	///
	MathArray(MathArray const &, size_type from, size_type to);
	///
	MathArray(iterator from, iterator to);

	///
	size_type size() const;
	///
	bool empty() const;
	///
	void clear();
	///
	void swap(MathArray &);

	///
	void insert(size_type pos, MathAtom const &);
	///
	void insert(size_type pos, MathArray const &);

	///
	void erase(iterator pos1, iterator pos2);
	///
	void erase(iterator pos);
	///
	void erase(size_type pos1, size_type pos2);
	///
	void erase(size_type pos);
	///
	void erase();

	///
	void push_back(MathAtom const &);
	///
	void push_back(MathArray const &);
	///
	void pop_back();
	///
	MathAtom & back();

	///
	MathAtom & front();
	///
	MathAtom const & front() const;

	///
	void dump() const;
	///
	void dump2() const;
	///
	void substitute(MathMacro const &);
	/// looks for exact match
	bool match(MathArray const &) const;
	/// looks for inclusion match starting at pos
	bool matchpart(MathArray const &, pos_type pos) const;
	/// looks for containment
	bool contains(MathArray const &) const;
	///
	void replace(ReplaceData &);

	///
	MathAtom & at(size_type pos);
	///
	MathAtom const & at(size_type pos) const;
	///
	void validate(LaTeXFeatures &) const;
	///
	const_iterator begin() const;
	///
	const_iterator end() const;
	///
	iterator begin();
	///
	iterator end();

private:
	/// Buffer
	buffer_type bf_;
};

///
std::ostream & operator<<(std::ostream & os, MathArray const & ar);


#endif
